------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2014, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Conversions;  use Ada.Characters;
with Ada.Exceptions;              use Ada.Exceptions;

with Asis.Declarations;           use Asis.Declarations;
with Asis.Definitions;            use Asis.Definitions;
with Asis.Expressions;            use Asis.Expressions;
with Asis.Elements;               use Asis.Elements;
with Asis.Ids;                    use Asis.Ids;
with Asis.Iterator;               use Asis.Iterator;

with Asis.Extensions.Strings;     use Asis.Extensions.Strings;

with Ada2Java.Kernel;             use Ada2Java.Kernel;
with Ada2Java.Config;             use Ada2Java.Config;
with Ada2Java.Bound_Units;        use Ada2Java.Bound_Units;

with Ada.Strings;                 use Ada.Strings;

with Ada.Strings.Unbounded;       use Ada.Strings.Unbounded;

with Asis.Extensions;             use Asis.Extensions;
with Asis.Data_Decomposition;     use Asis.Data_Decomposition;
with Asis.Set_Get;                use  Asis.Set_Get;

package body Ada2Java.Simplifications is

   -- Concrete types --

   type Concrete_Type_View is new Simple_Type_View with null record;

   type Concrete_Type_View_Access is access all Concrete_Type_View'Class;

   function Copy
     (This : access Concrete_Type_View)
      return access Concrete_Type_View;

   type Concrete_Type_Wrapper_View is
     new Simple_Type_Wrapper_View with null record;

   type Concrete_Type_Wrapper_View_Access is
     access all Concrete_Type_Wrapper_View'Class;

   function Copy
     (This : access Concrete_Type_Wrapper_View)
      return access Concrete_Type_Wrapper_View;

   type Concrete_Object_View is new Simple_Object_View with null record;

   type Concrete_Object_View_Access is access all Concrete_Object_View'Class;

   function Copy
     (This : access Concrete_Object_View)
      return access Concrete_Object_View;

   type Concrete_Profile_Data_View is new Simple_Profile_Data_View
   with null record;

   type Concrete_Profile_Data_View_Access is access all
     Concrete_Profile_Data_View'Class;

   function Copy
     (This : access Concrete_Profile_Data_View)
      return access Concrete_Profile_Data_View;

   type Concrete_Parameter_View is new Parameter_Simple_View
   with null record;

   type Concrete_Parameter_View_Access is access all
     Concrete_Parameter_View'Class;

   function Copy
     (This : access Concrete_Parameter_View)
      return access Concrete_Parameter_View;

   type Concrete_Subprogram_View is new Simple_Subprogram_View
   with null record;

   type Concrete_Subprogram_View_Access is access all
     Concrete_Subprogram_View'Class;

   function Copy
     (This : access Concrete_Subprogram_View)
      return access Concrete_Subprogram_View;

   type Concrete_Exception_View is new Simple_Exception_View
   with null record;

   type Concrete_Exception_View_Access is access all
     Concrete_Exception_View'Class;

   function Copy
     (This : access Concrete_Exception_View)
      return access Concrete_Exception_View;

   function Simplify_Subprogram
     (Handle     : not null access Kernel.Kernel_Record;
      Subprogram : Asis.Element) return Simple_Subprogram_View_Access;
   --  Process the subprogram given in parameter and return a new instance of
   --  it. We don't check in the database if the object is already present.
   --  This is equivalent to calling Create_Simple_Type and Handle_Definition.

   procedure Create_Simple_Type
     (Handle          : access Kernel.Kernel_Record;
      Element         : Asis.Element;
      Type_Definition : out Asis.Element;
      Result          : out Simple_Type_Reference;
      Old_Type        : out Boolean);
   --  Create a simple type out of the asis element given in parameter, and
   --  return the corresponding type definition to be used in
   --  Handle_Definition. If Old_Type is true, then the type was already
   --  present in the database and Handle_Definition does not need to be
   --  called.

   procedure Simplify_Type_Expression
     (Element : in out Asis.Definition; Result : in out Simple_Type_Reference);
   --  Simplify the type expression, removing prefix & suffixes if any. Result
   --  may be updated for example if the type was a 'Class expression.

   procedure Handle_Definition
     (Element : Asis.Definition;
      Handle  : not null access Kernel.Kernel_Record;
      Success : out Boolean;
      Result  : in out Simple_Type_Reference);
   --  Handles the type definition, and complete the type reference
   --  accordingly.

   procedure Get_Type_Declaration
     (Element     : Asis.Element;
      Declaration : out Asis.Element;
      Result      : in out Simple_Type_Reference);
   --  Return the type declaration for the element given in parameter.

   function Is_Generic_Actual (Elem : Asis.Element) return Boolean;
   --  Return true if the element is the actual of a formal generic parameter.

   function Escape_Java_Identifier
     (Identifier : Wide_String) return Wide_String;
   --  Return the Java identifier, adding a _ in front of it if it matches a
   --  reserved word

   ------------------------
   -- Get_Simple_Element --
   ------------------------

   function Get_Simple_Element
     (Handle  : not null access Kernel.Kernel_Record;
      Element : Asis.Element) return Simple_Element_View_Access
   is
      Base_Package : constant Package_Handle := Get_Or_Create_Package
        (Handle, Element);
      Object_Index : constant Simple_Element_Index :=
        To_Simple_Element_Index (Element);
      Result       : Simple_Element_View_Access;

      function Is_Constant_Element (Element : Asis.Element) return Boolean;

      function Create_Simple_Object
        (Actual : Asis.Element; Element : Asis.Element)
         return Simple_Element_View_Access;

      function Create_Simple_Exception return Simple_Element_View_Access;

      -------------------------
      -- Is_Constant_Element --
      -------------------------

      function Is_Constant_Element (Element : Asis.Element) return Boolean is

         function Unroll_Renames (Element : Asis.Element) return Asis.Element;

         function Unroll_Renames (Element : Asis.Element) return Asis.Element
         is
            Renamed : Asis.Element;
         begin

            if Declaration_Kind (Element) not in A_Renaming_Declaration then
               return Element;
            end if;

            Renamed := Renamed_Entity (Element);

            if Expression_Kind (Renamed) = A_Selected_Component then
               Renamed := Selector (Renamed);
            end if;

            if Expression_Kind (Renamed) = An_Enumeration_Literal then
               return Renamed;
            end if;

            if Expression_Kind (Renamed) /= An_Explicit_Dereference then
               Renamed := Corresponding_Name_Declaration (Renamed);
            end if;

            return Unroll_Renames (Renamed);
         end Unroll_Renames;

         --  Special case for a renamed declaration
         --  Ensure constant attribute of the renamed entity is preserved
         --  if set
         Maybe_Renamed : constant Asis.Element := Unroll_Renames (Element);

      begin
         return (Declaration_Kind (Maybe_Renamed) = A_Constant_Declaration
                 or else Declaration_Kind (Maybe_Renamed)
                 = A_Deferred_Constant_Declaration
                 or else Expression_Kind (Maybe_Renamed)
                 = An_Enumeration_Literal);
      end Is_Constant_Element;

      --------------------------
      -- Create_Simple_Object --
      --------------------------

      function Create_Simple_Object
        (Actual : Asis.Element; Element : Asis.Element)
         return Simple_Element_View_Access
      is
         Type_Of : constant Simple_Type_Reference :=
           Simplify_Type_Of (Handle, Element);
         Result : Simple_Object_View_Access;
      begin
         if Type_Of /= Null_Type_Reference
           and then not Is_Generic_Actual (Element)
         then
            Result := new Concrete_Object_View;
            Initialize_Configurable_Properties (Handle, Result);

            Result.Index        := To_Element_Index (Actual);
            Result.Base_Package := Base_Package;
            Result.Type_Of      := Type_Of;
            Result.Name         := To_Dynamic_Expression
              (Get_First_Name (Actual));

            Result.Java_Name    := To_Dynamic_Expression
              (Escape_Java_Identifier (To_Wide_String (Result.Name)));

            Result.Is_Constant  := Is_Constant_Element (Element);
            Result.Is_Aliased   := Trait_Kind (Element) = An_Aliased_Trait;
            Result.Location     := Get_Source_Location_Of_Name (Element);

            if Type_Of.Ref.Kind in Access_Issue_Kinds
              and then Trait_Kind (Element) /= An_Aliased_Trait
              and then not Ada2Java.Allow_Unaliased_Access
            then
               Trace_With_Location
                 ("""" & Get_First_Name (Actual)
                  & """ should be aliased, see --no-unaliased-access",
                  Errors_And_Warnings);

               Result.Can_Be_Bound := No;
            elsif Type_Of = Null_Type_Reference
              or else Type_Of.Ref.Can_Be_Bound = No
            then
               Trace_With_Location
                 ("can't bind """ & Get_First_Name (Actual)
                  & """ because """
                  & To_Wide_String (Type_Of.Ref.Full_Ada_Name)
                  & """ can't be bound", Errors_And_Warnings);

               Result.Can_Be_Bound := No;
            end if;

            Add_Element
              (Get_Or_Create_Bound_Unit
                 (Handle, Simple_Element_View_Access (Result)),
               Simple_Element_View_Access (Result));

            return Simple_Element_View_Access (Result);
         else
            return null;
         end if;
      end Create_Simple_Object;

      function Create_Simple_Exception return Simple_Element_View_Access is
         Result : Simple_Exception_View_Access;
      begin
         Result := new Concrete_Exception_View;
         Initialize_Configurable_Properties (Handle, Result);
         Result.Name := To_Dynamic_Expression (Get_First_Name (Element));
         Result.Index := To_Element_Index (Element);
         Result.Base_Package := Base_Package;
         Result.Location := Get_Source_Location_Of_Name (Element);

         Add_Element
           (Get_Or_Create_Bound_Unit
              (Handle, Simple_Element_View_Access (Result)),
            Simple_Element_View_Access (Result));

         return Simple_Element_View_Access (Result);
      end Create_Simple_Exception;

   begin
      if Get_Simple_Elements_DB (Handle).all.Contains (Object_Index) then
         return Get_Simple_Elements_DB (Handle).all.Element
           (Object_Index);
      end if;

      case Declaration_Kind (Element) is
         when An_Ordinary_Type_Declaration | A_Subtype_Declaration  =>

            Result := Simple_Element_View_Access
              (Simplify_Type_Of (Handle, Element).Ref);

         when A_Private_Type_Declaration | A_Private_Extension_Declaration =>

            Result := Simple_Element_View_Access
                 (Simplify_Type_Of (Handle, Element).Ref);

         when A_Procedure_Declaration
            | A_Function_Declaration
            | An_Expression_Function_Declaration
            | A_Function_Renaming_Declaration
            | A_Procedure_Renaming_Declaration
            | A_Null_Procedure_Declaration =>

            Result := Simple_Element_View_Access
              (Simplify_Subprogram (Handle, Element));

            if Declaration_Origin (Element)
              /= An_Implicit_Inherited_Declaration
            then
               Add_Element
                 (Get_Or_Create_Bound_Unit (Handle, Result), Result);
            end if;

         when An_Object_Declaration | An_Object_Renaming_Declaration =>

            Result := Create_Simple_Object (Element, Element);

         when An_Incomplete_Type_Declaration
            | A_Tagged_Incomplete_Type_Declaration =>

            return null;

         when An_Exception_Declaration =>

            return Create_Simple_Exception;

         when others =>

            if Element_Kind (Element) = A_Defining_Name then
               Result := Create_Simple_Object
                 (Element, Enclosing_Element (Element));
            else
               raise Not_Supported with
                 "binding of "
                   & Declaration_Kind (Element)'Img & " not supported";
            end if;

      end case;

      if Result = null then
         return null;
      end if;

      if not
        Get_Simple_Elements_DB (Handle).all.Contains (Object_Index)
      then
         --  If the element has not been included in the db during the analysis
         --  process, then add it.

         if Result.Location = null then
            Result.Location := Get_Source_Location_Of_Name (Element);
         end if;

         Get_Simple_Elements_DB (Handle).all.Insert (Object_Index, Result);
      end if;

      return Result;

   exception
      when Silent_Not_Supported =>
         return null;

      when E : Not_Supported =>
         if Exception_Message (E) = "" then
            Trace_With_Location
              ("binding not supported", Errors_And_Warnings);
         else
            Trace_With_Location
              (Conversions.To_Wide_String (Exception_Message (E)),
               Errors_And_Warnings);
         end if;

         return null;
   end Get_Simple_Element;

   ----------
   -- Copy --
   ----------

   --  ??? These subprograms should perform the copy of the relevant
   --  information. They should be completed.

   function Copy
     (This : access Concrete_Type_View) return access Concrete_Type_View
   is
      Result : constant Concrete_Type_View_Access :=
        new Concrete_Type_View'(This.all);
   begin
      Result.Exported_Name := Deep_Copy (Result.Exported_Name);
      Result.Full_Ada_Name := Deep_Copy (Result.Full_Ada_Name);
      Result.Full_Java_Name := Deep_Copy (Result.Full_Java_Name);

      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Type_Wrapper_View)
      return access Concrete_Type_Wrapper_View
   is
      Result : constant Concrete_Type_Wrapper_View_Access :=
        new Concrete_Type_Wrapper_View'(This.all);
   begin
      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Object_View) return access Concrete_Object_View
   is
      Result : constant Concrete_Object_View_Access :=
        new Concrete_Object_View'(This.all);
   begin
      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Profile_Data_View)
      return access Concrete_Profile_Data_View
   is
      Result : constant Concrete_Profile_Data_View_Access :=
        new Concrete_Profile_Data_View'(This.all);
   begin
      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Parameter_View)
      return access Concrete_Parameter_View
   is
      Result : constant Concrete_Parameter_View_Access :=
        new Concrete_Parameter_View'(This.all);
   begin
      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Exception_View)
      return access Concrete_Exception_View
   is
      Result : constant Concrete_Exception_View_Access :=
        new Concrete_Exception_View'(This.all);
   begin
      return Result;
   end Copy;

   function Copy
     (This : access Concrete_Subprogram_View)
      return access Concrete_Subprogram_View
   is
      Copy_Obj : constant Concrete_Subprogram_View_Access :=
        new Concrete_Subprogram_View'(This.all);
   begin
      Copy_Obj.Parameters := new Parameter_Array'(Copy_Obj.Parameters.all);

      if Copy_Obj.Parameters /= null then
         for J in Copy_Obj.Parameters'Range loop
            Copy_Obj.Parameters (J) := Parameter_Simple_View_Access
              (Copy (Copy_Obj.Parameters (J)));
            Copy_Obj.Parameters (J).Enclosing_Sb :=
              Simple_Subprogram_View_Access (Copy_Obj);
         end loop;
      end if;

      if Copy_Obj.Returned_Type /= null then
         Copy_Obj.Returned_Type := Simple_Profile_Data_View_Access
           (Copy (Copy_Obj.Returned_Type));
         Copy_Obj.Returned_Type.Enclosing_Sb :=
           Simple_Subprogram_View_Access (Copy_Obj);
      end if;

      return Copy_Obj;
   end Copy;

   ------------
   -- Create --
   ------------

   procedure Initialize_Configurable_Properties
     (Handle : not null access Kernel.Kernel_Record;
      Object : access Simple_Element_View'Class) is
   begin
      if Ada2Java.Default_Lock = null then
         Object.Lock_Var :=
           Get_Library_Binding_Unit (Handle).Java_File.Full_Class_Name
           & ".lock";
      else
         Object.Lock_Var := To_Dynamic_Expression (Ada2Java.Default_Lock.all);
      end if;
   end Initialize_Configurable_Properties;

   procedure Copy_Configurable_Properties
     (Enclosing : not null access Simple_Element_View'Class;
      Object    : not null access Simple_Element_View'Class);

   procedure Copy_Configurable_Properties
     (Enclosing : not null access Simple_Element_View'Class;
      Object    : not null access Simple_Element_View'Class)
   is
   begin
      Object.Base_Package := Enclosing.Base_Package;
      Object.Lock_Var := Enclosing.Lock_Var;
   end Copy_Configurable_Properties;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Type_View_Access
   is
      Ret : constant Simple_Type_View_Access := new Concrete_Type_View;
   begin
      if Enclosing /= null then
         Copy_Configurable_Properties (Enclosing, Ret);
      end if;

      return Ret;
   end Create;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Type_Wrapper_Access
   is
      Ret : constant Simple_Type_Wrapper_Access :=
        new Concrete_Type_Wrapper_View;
   begin
      if Enclosing /= null then
         Copy_Configurable_Properties (Enclosing, Ret);
      end if;

      return Ret;
   end Create;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Object_View_Access
   is
      Ret : constant Simple_Object_View_Access := new Concrete_Object_View;
   begin
      if Enclosing /= null then
         Copy_Configurable_Properties (Enclosing, Ret);
      end if;

      return Ret;
   end Create;

   function Create
     (Enclosing : not null access Simple_Subprogram_View'Class)
      return Simple_Profile_Data_View_Access
   is
      Ret : constant Simple_Profile_Data_View_Access :=
        new Concrete_Profile_Data_View;
   begin
      Copy_Configurable_Properties (Enclosing, Ret);
      Ret.Enclosing_Sb := Simple_Subprogram_View_Access (Enclosing);

      return Ret;
   end Create;

   function Create
     (Enclosing : not null access Simple_Subprogram_View'Class)
      return Parameter_Simple_View_Access
   is
      Ret : constant Parameter_Simple_View_Access :=
        new Concrete_Parameter_View;
   begin
      Copy_Configurable_Properties (Enclosing, Ret);
      Ret.Enclosing_Sb := Simple_Subprogram_View_Access (Enclosing);

      return Ret;
   end Create;

   function Create
     (Enclosing : not null access Simple_Element_View'Class)
      return Simple_Subprogram_View_Access
   is
      Ret : constant Simple_Subprogram_View_Access :=
        new Concrete_Subprogram_View;
   begin
      Copy_Configurable_Properties (Enclosing, Ret);

      return Ret;
   end Create;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Exception_View_Access
   is
      Ret : constant Simple_Exception_View_Access :=
        new Concrete_Exception_View;
   begin
      if Enclosing /= null then
         Copy_Configurable_Properties (Enclosing, Ret);
      end if;

      return Ret;
   end Create;

   -------------------------
   -- Get_Source_Location --
   -------------------------

   function Get_Source_Location
     (Element : Asis.Element) return Source_Location
   is
   begin
      return new Wide_String'
        (Conversions.To_Wide_String (Build_GNAT_Location (Element)));
   end Get_Source_Location;

   ---------------------------------
   -- Get_Source_Location_Of_Name --
   ---------------------------------

   function Get_Source_Location_Of_Name
     (Element : Asis.Element) return Source_Location is
   begin
      if Element_Kind (Element) = A_Defining_Name then
         return Get_Source_Location (Element);
      elsif Element_Kind (Element) /= A_Declaration then
         return Get_Source_Location (Element);
      else
         return Get_Source_Location (Names (Element)(1));
      end if;
   end Get_Source_Location_Of_Name;

   ----------
   -- Free --
   ----------

   procedure Free (This : in out Simple_Type_View_Access) is
      pragma Unreferenced (This);
   begin
      null;
   end Free;

   ----------
   -- Dump --
   ----------

   function Dump (This : Simple_Type_View) return String is
      Value : Unbounded_String;
   begin
      Append (Value, "{");

      if This.Index.Location = null then
         Append (Value, "?");
      else
         Append (Value, This.Index.Location.all);
         Append (Value, ": " & This.Kind'Img);

         if This.Target_Type /= Null_Type_Reference then
            Append (Value, " -> [" & Dump (This.Target_Type.Ref.all) & "]");
         end if;
      end if;

      return To_String (Value);
   end Dump;

   -------------------
   -- Get_Java_Name --
   -------------------

   function Get_Java_Name
     (Element : Simple_Object_View'Class) return Dynamic_Expression is
   begin
      if Element.Java_Name = Empty_Dynamic_Expression then
         return Element.Name;
      else
         return Element.Java_Name;
      end if;
   end Get_Java_Name;

   -------------------
   -- Get_Glue_Name --
   -------------------

   function Get_Glue_Name
     (Element : Simple_Object_View'Class) return Dynamic_Expression
   is
   begin
      if Element.Glue_Name /= Empty_Dynamic_Expression then
         return Element.Glue_Name;
      else
         return Element.Name;
      end if;
   end Get_Glue_Name;

   ----------------------------
   -- Check_Type_Consistency --
   ----------------------------

   function Is_Type_Consistent
     (Ref : Simple_Type_Reference) return Boolean is
   begin
      if Ref /= Null_Type_Reference and then Ref.Ref.Can_Be_Bound /= No then
         if (Ref.Ref.Kind = Access_Kind
             or else Ref.Ref.Kind = Array_Kind)
           and then Ref.Ref.Target_Type.Ref = null
         then
            return False;
         end if;
      end if;

      return True;
   end Is_Type_Consistent;

   ------------------------------
   -- Simplify_Type_Expression --
   ------------------------------

   procedure Simplify_Type_Expression
     (Element : in out Asis.Definition; Result : in out Simple_Type_Reference)
   is
   begin
      loop
         if Expression_Kind (Element) = A_Selected_Component then
            Element := Selector (Element);
         elsif Expression_Kind (Element) = An_Attribute_Reference then
            --  We are in a case e.g. Cl'Class
            Result.Is_Class_Wide := True;
            Element := Prefix (Element);
         else
            exit;
         end if;
      end loop;
   end Simplify_Type_Expression;

   --------------------------
   -- Get_Type_Declaration --
   --------------------------

   procedure Get_Type_Declaration
     (Element     : Asis.Element;
      Declaration : out Asis.Element;
      Result      : in out Simple_Type_Reference)
   is
   begin
      if Element_Kind (Element) = A_Declaration then
         Declaration := Element;
      elsif Element_Kind (Element) = A_Definition then
         case Definition_Kind (Element) is
            when A_Subtype_Indication
               | A_Discrete_Subtype_Definition
               | A_Discrete_Range =>

               if Definition_Kind (Element) = A_Discrete_Subtype_Definition
                 and then Discrete_Range_Kind (Element)
                 /= A_Discrete_Subtype_Indication
               then
                  return;
               end if;

               Declaration :=
                 Asis.Definitions.Subtype_Mark (Element);

               Simplify_Type_Expression (Declaration, Result);

               Declaration := Enclosing_Element
                 (Corresponding_Name_Definition (Declaration));

               if Declaration_Kind (Declaration)
                 = An_Incomplete_Type_Declaration
                 or else Declaration_Kind (Declaration)
                 = A_Tagged_Incomplete_Type_Declaration
               then
                  Declaration :=
                    Corresponding_Type_Declaration (Declaration);
               end if;

            when others =>
               null;
         end case;
      elsif Element_Kind (Element) = An_Expression then
         Declaration := Element;

         Simplify_Type_Expression (Declaration, Result);

         Declaration := Enclosing_Element
           (Corresponding_Name_Definition (Declaration));
      end if;
   exception
      when Silent_Not_Supported =>
         raise;

      when E : others =>
         raise Ada2Java_Error with Exception_Information (E)
           & " raised for element " & Dump_Element (Element);
   end Get_Type_Declaration;

   -----------------------
   -- Handle_Definition --
   -----------------------

   procedure Handle_Definition
     (Element       : Asis.Definition;
      Handle        : not null access Kernel.Kernel_Record;
      Success       : out Boolean;
      Result        : in out Simple_Type_Reference)
   is
      Type_Definition  : Asis.Element := Element;

      procedure Handle_Enum_Values (Element : Asis.Definition);

      procedure Handle_Indexes (Element : Asis.Definition);
      --  Build a list of array indexes or type discriminants depending of
      --  the kind of type analysed.

      procedure Handle_Record_Discriminants (Element : Asis.Definition);
      --  Build a list of discriminants in the case we are handling a
      --  discriminated record.

      procedure Handle_Fields (Element : Asis.Definition);
      --  ???

      procedure Handle_Controlled_Primitives (Type_Def : Asis.Element);
      --  ???

      procedure Handle_Access_To_Subprogram;
      --  ???

      procedure Check_Access_Support;
      --  Given that the value of result is an access type, checks if this
      --  kind of access is supported, and put False in Success if not.

      ------------------------
      -- Handle_Enum_Values --
      ------------------------

      procedure Handle_Enum_Values (Element : Asis.Definition) is
         Values : constant Declaration_List :=
           Enumeration_Literal_Declarations
             (Element);
      begin
         Result.Ref.Enum_Values := new Values_Array (1 .. Values'Length);

         for J in Values'Range loop
            Result.Ref.Enum_Values (J) := To_Dynamic_Expression
              (Get_First_Name (Values (J)));
         end loop;
      end Handle_Enum_Values;

      --------------------
      -- Handle_Indexes --
      --------------------

      procedure Handle_Indexes (Element : Asis.Definition) is

         function Get_Indexes
           (Element : Asis.Element) return Element_List;
         --  Return the list of either an array indexes, or a record
         --  discriminants.

         procedure Set_Index
           (Index_Array : in out Simple_Object_View_Array;
            Index       : Integer;
            Element     : Asis.Element;
            Identifier  : Asis.Element);
         --  Initializes the index at the position given in parameter, given
         --  a type view and an identifier.

         -----------------
         -- Get_Indexes --
         -----------------

         function Get_Indexes
           (Element : Asis.Element) return Element_List
         is
            Nothing : Element_List (1 .. 0);
         begin
            if Type_Kind (Element) = An_Unconstrained_Array_Definition then
               return Index_Subtype_Definitions (Element);
            elsif Type_Kind (Element) = A_Constrained_Array_Definition then
               return Discrete_Subtype_Definitions (Element);
            elsif not
              Is_Nil (Discriminant_Part (Enclosing_Element (Element)))
            then
               if Definition_Kind
                 (Discriminant_Part (Enclosing_Element (Element)))
                 = An_Unknown_Discriminant_Part
               then
                  Result.Ref.Allow_Java_Creation := False;
                  Result.Ref.Allow_Java_Child_Types := False;

                  Trace_With_Location
                    ("""" & To_Wide_String
                       (Result.Ref.Exported_Name)
                     & """ can't be derived in Java",
                     Errors_And_Warnings);
                  Trace_With_Location
                    ("(derived Java types cannot specify unknown"
                     & " discriminants)",
                     Errors_And_Warnings);

                  return Nothing;
               else
                  return Discriminants
                    (Discriminant_Part (Enclosing_Element (Element)));
               end if;
            else
               return Nothing;
            end if;
         end Get_Indexes;

         ---------------
         -- Set_Index --
         ---------------

         procedure Set_Index
           (Index_Array : in out Simple_Object_View_Array;
            Index       : Integer;
            Element     : Asis.Element;
            Identifier  : Asis.Element) is
         begin
            Index_Array (Index) := Create (Result.Ref);
            Index_Array (Index).Base_Package :=
              Result.Ref.Base_Package;
            Index_Array (Index).Type_Of := Simplify_Type_Of
              (Handle, Element);
            Index_Array (Index).Name :=
              To_Dynamic_Expression (Get_First_Name (Identifier));
            Index_Array (Index).Java_Name := To_Dynamic_Expression
              (Escape_Java_Identifier
                 (To_Wide_String (Index_Array (Index).Name)));
            Index_Array (Index).Index := To_Element_Index (Identifier);

            if Element_Kind (Element) = A_Definition
              and then Definition_Kind (Element)
              = A_Discrete_Subtype_Definition
            then
               Index_Array (Index).Is_Constant := True;
            else
               Index_Array (Index).Is_Constant := False;
            end if;

            if Index_Array (Index).Type_Of.Ref.Can_Be_Bound = No then
               Index_Array (Index).Can_Be_Bound := No;
               Success := False;
            end if;
         end Set_Index;

         Indexes : constant Asis.Expression_List := Get_Indexes (Element);
         Index_Number : Integer := 0;
      begin
         for J in Indexes'Range loop
            if Element_Kind (Indexes (J)) = A_Declaration then
               Index_Number := Index_Number + Names (Indexes (J))'Length;
            else
               Index_Number := Index_Number + 1;
            end if;
         end loop;

         Result.Ref.Indexes :=
           new Simple_Object_View_Array (1 .. Index_Number);

         Index_Number := 0;

         for J in Indexes'Range loop

            if Element_Kind (Indexes (J)) = A_Declaration then
               declare
                  Index_Names : constant Defining_Name_List :=
                    Names (Indexes (J));
               begin
                  for K in Index_Names'Range loop
                     Index_Number := Index_Number + 1;
                     Set_Index
                       (Result.Ref.Indexes.all,
                        Index_Number,
                        Indexes (J),
                        Index_Names (K));
                  end loop;
               end;
            else
               Index_Number := Index_Number + 1;
               Set_Index
                 (Result.Ref.Indexes.all,
                  Index_Number,
                  Indexes (J),
                  Indexes (J));
            end if;
         end loop;

         --  If we are on a tagged derivation, and the parent has discriminants
         --  which are not given values by the derivation, then these
         --  discriminants are automatically reported to the child.

         if Result.Ref.Kind = Tagged_Record_Kind
           and then Result.Ref.Target_Type /= Null_Type_Reference
           and then (Result.Ref.Target_Type.Ref.Indexes = null
                     or else Result.Ref.Target_Type.Ref.Indexes'Length = 0)
           and then Result.Ref.Target_Type.Ref.Indexes /= null
           and then Result.Ref.Target_Type.Ref.Indexes'Length > 0
         then
            declare
               Parent : Asis.Element;
            begin
               if Definition_Kind (Type_Definition)
                 = A_Private_Extension_Definition
               then
                  Parent := Ancestor_Subtype_Indication (Type_Definition);
               else
                  Parent := Parent_Subtype_Indication (Type_Definition);
               end if;

               if Is_Nil (Subtype_Constraint (Parent)) then
                  Result.Ref.Indexes :=
                    Result.Ref.Target_Type.Ref.Indexes;

                  for J in Result.Ref.Indexes'Range loop
                     Result.Ref.Indexes (J) :=
                       Simple_Object_View_Access
                         (Copy (Result.Ref.Indexes (J)));
                  end loop;
               end if;
            end;
         end if;
      end Handle_Indexes;

      ---------------------------------
      -- Handle_Record_Discriminants --
      ---------------------------------

      procedure Handle_Record_Discriminants (Element : Asis.Definition) is
         Disc_Part : constant Asis.Element :=
           Discriminant_Part (Enclosing_Element (Element));
         Disc_Count : Positive := 1;
      begin
         if Is_Nil (Disc_Part) then
            return;
         end if;

         declare
            function Get_Discriminant_Name_Count
              (Specs : Asis.Discriminant_Specification_List) return Positive;
            --  Returns the number of dicriminant names for a given
            --  discriminant list. This is needed since the discriminant list
            --  has one entry per discriminant types, ie. cases like this:
            --    type R (A, B : T1; C : T2) is null record;
            --  results in:
            --    Specs'Length = 2
            --    Get_Discriminant_Name_Count = 3

            ---------------------------------
            -- Get_Discriminant_Name_Count --
            ---------------------------------

            function Get_Discriminant_Name_Count
              (Specs : Asis.Discriminant_Specification_List) return Positive
            is
               Count : Natural := 0;
            begin
               for J in Specs'Range loop
                  Count := Count + Names (Specs (J))'Length;
               end loop;

               pragma Assert (Count /= 0);
               --  We know that we have at least one discriminant. The Natural
               --  type is only needed to initialize the counter to 0.

               return Count;
            end Get_Discriminant_Name_Count;

            Specs : constant Asis.Discriminant_Specification_List :=
              Discriminants (Disc_Part);
         begin
            Result.Ref.Discriminants :=
              new Simple_Object_View_Array
                (1 .. Get_Discriminant_Name_Count (Specs));
            --  Array of different discriminant names.

            for J in Specs'Range loop
               declare
                  Discriminant_Type : constant Simple_Type_Reference :=
                    Simplify_Type_Of (Handle, Specs (J));
                  Discriminant_Names : constant Defining_Name_List :=
                    Names (Specs (J));
               begin
                  for K in Discriminant_Names'Range loop
                     declare
                        Obj : Simple_Object_View_Access renames
                          Result.Ref.Discriminants (Disc_Count);
                     begin
                        Obj := Create (Discriminant_Type.Ref);
                        Obj.Base_Package := Result.Ref.Base_Package;
                        Obj.Type_Of := Discriminant_Type;
                        Obj.Name := To_Dynamic_Expression
                          (Remove_Blanks (Defining_Name_Image
                           (Discriminant_Names (K))));
                        Obj.Java_Name := To_Dynamic_Expression
                          (Escape_Java_Identifier
                             (To_Wide_String (Obj.Name)));
                        Obj.Is_Constant := True;
                        Obj.Location :=
                          Get_Source_Location_Of_Name (Discriminant_Names (K));
                        Disc_Count := Disc_Count + 1;
                     end;
                  end loop;
               end;
            end loop;
         end;
      end Handle_Record_Discriminants;

      -------------------
      -- Handle_Fields --
      -------------------

      procedure Handle_Fields (Element : Asis.Definition) is
         Components_Def : constant Asis.Element :=
           Asis.Definitions.Record_Definition (Element);

         Not_Null_Components : Integer := 0;
         Component_Index : Integer := 1;
      begin
         Handle_Indexes (Element);

         if Is_Nil (Components_Def) or else
           Definition_Kind (Components_Def) = A_Null_Record_Definition
         then
            Result.Ref.Components := new Simple_Object_View_Array
              (1 .. 0);

            return;
         end if;

         declare
            Components : constant Asis.Record_Component_List :=
              Asis.Definitions.Record_Components (Components_Def);
            Components_Bound : Asis.Record_Component_List (Components'Range);
            Components_Bound_Index : Integer := Components'First;
            Loc : Location_Handle;
            Simple_Type : Simple_Type_Reference;
         begin
            for J in Components'Range loop
               Loc := Push_Location (Get_Source_Location (Components (J)));

               if Definition_Kind (Components (J)) = A_Null_Component then
                  null;
               elsif Definition_Kind (Components (J)) = A_Variant_Part then
                  Trace_With_Location
                    ("cannot bind variant parts", Errors_And_Warnings);
               else
                  begin
                     Simple_Type := Simplify_Type_Of (Handle, Components (J));

                     if Simple_Type = Null_Type_Reference
                       or else Simple_Type.Ref.Can_Be_Bound = No
                     then
                        Trace_With_Location
                          ("cannot bind type of field """
                           & Get_First_Name (Components (J))
                           & """", Errors_And_Warnings);
                     elsif Simple_Type.Ref.Kind in Access_Issue_Kinds
                       and then Trait_Kind
                         (Object_Declaration_View (Components (J)))
                       /= An_Aliased_Trait
                       and then not Ada2Java.Allow_Unaliased_Access
                     then
                        Trace_With_Location
                          (""""
                           & Get_First_Name (Components (J))
                           & """ should be aliased, "
                           & "see --no-unaliased-access",
                           Errors_And_Warnings);
                     else
                        Not_Null_Components := Not_Null_Components +
                          Names (Components (J))'Length;
                        Components_Bound (Components_Bound_Index) :=
                          Components (J);
                        Components_Bound_Index := Components_Bound_Index + 1;
                     end if;
                  exception
                     when Silent_Not_Supported =>
                        --  If the field is not supported, just continue to
                        --  analyze the next one.
                        null;

                     when E : Not_Supported =>
                        --  If the field is not supported, just continue to
                        --  analyze the next one.

                        Trace_With_Location
                          (Conversions.To_Wide_String
                             (Exception_Message (E)),
                           Errors_And_Warnings);
                  end;
               end if;

               Pop_Location (Loc);
            end loop;

            Result.Ref.Components := new Simple_Object_View_Array
              (1 ..  Not_Null_Components);

            for J in Components_Bound'First .. Components_Bound_Index - 1 loop
               Loc := Push_Location
                 (Get_Source_Location (Components_Bound (J)));

               Simple_Type := Simplify_Type_Of (Handle, Components_Bound (J));

               declare
                  Cpt_Names : constant Name_List :=
                    Names (Components_Bound (J));
               begin
                  for K in Cpt_Names'Range loop
                     Result.Ref.Components (Component_Index) :=
                       Create (Simple_Type.Ref);
                     Result.Ref.Components (Component_Index).
                       Base_Package := Result.Ref.Base_Package;
                     Result.Ref.Components (Component_Index).Type_Of :=
                       Simple_Type;
                     Result.Ref.Components (Component_Index).Name :=
                       To_Dynamic_Expression
                         (Remove_Blanks (Defining_Name_Image
                          (Cpt_Names (K))));
                     Result.Ref.Components (Component_Index).Java_Name :=
                       To_Dynamic_Expression
                         (Escape_Java_Identifier
                              (To_Wide_String
                                   (Result.Ref.Components
                                        (Component_Index).Name)));
                     Result.Ref.Components
                       (Component_Index).Is_Constant := False;
                     Result.Ref.Components (Component_Index).Index :=
                       To_Element_Index (Components_Bound (J));

                     Result.Ref.Components (Component_Index).Is_Aliased
                       := Trait_Kind
                         (Object_Declaration_View (Components_Bound (J)))
                       = An_Aliased_Trait;
                     Result.Ref.Components (Component_Index).Location :=
                       Get_Source_Location_Of_Name (Cpt_Names (K));
                     Component_Index := Component_Index + 1;
                  end loop;
               end;

               Pop_Location (Loc);
            end loop;
         end;
      exception
         when Silent_Not_Supported =>
            raise;

         when E : others =>
            raise Ada2Java_Error
              with "unexpected_Exception : " & Exception_Information (E);
      end Handle_Fields;

      ----------------------------------
      -- Handle_Controlled_Primitives --
      ----------------------------------

      procedure Handle_Controlled_Primitives (Type_Def : Asis.Element) is

         Type_Decl     : constant Asis.Element := Enclosing_Element (Type_Def);

         Primitives_Number : Integer := 0;
         Primitives_Buffer : Simple_Subprogram_View_Array (1 .. 1024);

         Declaration_Context : Asis.Element;

         procedure Pre_Procedure
           (Element :        Asis.Element;
            Control : in out Asis.Traverse_Control;
            State   : in out Integer);

         procedure Post_Procedure
           (Element :        Asis.Element;
            Control : in out Asis.Traverse_Control;
            State   : in out Integer);

         procedure Traverse_Node is new Asis.Iterator.Traverse_Element
           (Integer, Pre_Procedure, Post_Procedure);

         procedure Handle_Primitive
           (Element : Asis.Element; Implicit_Inherited : Boolean);

         -------------------
         -- Pre_Procedure --
         -------------------

         procedure Pre_Procedure
           (Element :        Asis.Element;
            Control : in out Asis.Traverse_Control;
            State   : in out Integer)
         is
            pragma Unreferenced (State);
         begin
            if Is_Private (Element) then
               Control := Abandon_Children;
               return;
            end if;

            if Declaration_Kind (Element) = A_Package_Declaration then
               Control := Continue;
            else
               Control := Abandon_Children;
            end if;

            if Declaration_Kind (Element) in
              A_Procedure_Declaration .. A_Function_Declaration
              or else Declaration_Kind (Element) =
                An_Expression_Function_Declaration
              or else Declaration_Kind (Element) = A_Null_Procedure_Declaration
            then
               if Is_Identical (Primitive_Owner (Element), Type_Def) then
                  Handle_Primitive (Element, False);
               end if;
            end if;
         end Pre_Procedure;

         --------------------
         -- Post_Procedure --
         --------------------

         procedure Post_Procedure
           (Element :        Asis.Element;
            Control : in out Asis.Traverse_Control;
            State   : in out Integer) is
            pragma Unreferenced (Element, Control, State);
         begin
            null;
         exception
            when others =>
               null;
         end Post_Procedure;

         ----------------------
         -- Handle_Primitive --
         ----------------------

         procedure Handle_Primitive
           (Element            : Asis.Element;
            Implicit_Inherited : Boolean)
         is
            Subprogram   : Simple_Subprogram_View_Access;
            Loc          : Location_Handle;
         begin
            Loc := Push_Location (Get_Source_Location (Element));

            Subprogram := Simple_Subprogram_View_Access
              (Get_Simple_Element (Handle, Element));

            if Subprogram /= null then
               Subprogram.Is_Final := Subprogram.Parameters = null
                 or else Subprogram.Parameters'Length = 0
                 or else Subprogram.Parameters (1).Type_Of.Is_Class_Wide;
            end if;

            if Subprogram = null or else Subprogram.Can_Be_Bound = No then
               Trace_With_Location
                 ("""" & To_Wide_String
                    (Result.Ref.Exported_Name)
                  & """ can't be derived in Java",
                  Errors_And_Warnings);
               Trace_With_Location
                 ("(cannot bind primitive """
                  & To_Wide_String (Subprogram.Name) & """)",
                  Errors_And_Warnings);

               Result.Ref.Allow_Java_Child_Types := False;
            elsif Subprogram.Parameters = null
              or else Subprogram.Parameters'Length = 0
              or else not Subprogram.Parameters (1).Is_Controlling
            then
               Trace_With_Location
                 ("""" & To_Wide_String
                    (Result.Ref.Exported_Name)
                  & """ can't be derived in Java",
                  Errors_And_Warnings);
               Trace_With_Location
                 ("(first parameter of primitive """
                  & To_Wide_String (Subprogram.Name)
                  & """ must be controlling)",
                  Errors_And_Warnings);

               Result.Ref.Allow_Java_Child_Types := False;
            else
               Primitives_Number := Primitives_Number + 1;

               Primitives_Buffer (Primitives_Number) := Subprogram;

               if Implicit_Inherited then
                  declare
                     Overidden_Sb : constant Simple_Subprogram_View_Access :=
                       Simple_Subprogram_View_Access
                         (Get_Simple_Element
                            (Handle, Corresponding_Declaration (Element)));
                  begin
                     --  Copy the parent properties changed by annotation to
                     --  the child.

                     --  ??? We should have a way to do that in a safe way, to
                     --  avoid forgetting one of these when adding new
                     --  modifiable attributes.

                     Subprogram.Name := Overidden_Sb.Name;

                     for J in Subprogram.Parameters'Range loop
                        declare
                           Parameter : constant Parameter_Simple_View_Access
                             := Overidden_Sb.Parameters (J);
                        begin
                           Subprogram.Parameters (J).Assume_Stored :=
                             Parameter.Assume_Stored;

                           if Parameter.Type_Of.Ref.Is_Anonymous then
                              Subprogram.Parameters (J).Type_Of.Ref :=
                                Parameter.Type_Of.Ref;
                           end if;
                        end;
                     end loop;

                     Subprogram.Overridden_Sb := Overidden_Sb;

                     Overidden_Sb.Overriding_Sbs.Append
                       (Simple_Element_View_Access (Subprogram));
                  end;
               end if;
            end if;

            Pop_Location (Loc);

         exception
            when Silent_Not_Supported | Not_Supported =>
               --  If there's at least one primitive that can't be bound, the
               --  creation of the java child will not be possible.

               Trace_With_Location
                 ("""" & To_Wide_String
                    (Result.Ref.Exported_Name)
                  & """ can't be derived in Java",
                  Errors_And_Warnings);

               Trace_With_Location
                 ("(cannot bind primitive """
                  & To_Wide_String (Subprogram.Name) & """)",
                  Errors_And_Warnings,
                  Get_Source_Location (Element));

               Result.Ref.Allow_Java_Child_Types := False;

               Pop_Location (Loc);
         end Handle_Primitive;

         State   : Integer := 0;
         Control : Asis.Traverse_Control := Asis.Continue;

      begin
         Declaration_Context := Enclosing_Element (Type_Decl);

         if Type_Kind (Type_Def) = A_Derived_Record_Extension_Definition
           or else Type_Kind (Type_Def) = A_Derived_Type_Definition
           or else Definition_Kind (Type_Def) = A_Private_Extension_Definition
         then
            declare
               Inherited : constant Declaration_List :=
                 Implicit_Inherited_Subprograms (Type_Def);
               Corresponding_Decl : Asis.Element;

            begin
               for J in Inherited'Range loop
                  Corresponding_Decl :=
                    Asis.Declarations.Corresponding_Declaration
                      (Inherited (J));

                  if not Asis.Extensions.Is_Private (Corresponding_Decl) then
                     Handle_Primitive (Inherited (J), True);
                  end if;
               end loop;
            end;
         end if;

         Traverse_Node (Declaration_Context, Control, State);

         Result.Ref.Primitives :=
           new Simple_Subprogram_View_Array'
             (Primitives_Buffer (1 .. Primitives_Number));
      end Handle_Controlled_Primitives;

      ---------------------------------
      -- Handle_Access_To_Subprogram --
      ---------------------------------

      procedure Handle_Access_To_Subprogram is
      begin
         Result.Ref.Kind := Access_Kind;
         Result.Ref.Target_Type.Ref := new Concrete_Type_View;
         Initialize_Configurable_Properties
           (Handle, Result.Ref.Target_Type.Ref);
         Result.Ref.Target_Type.Ref.Kind := Subprogram_Kind;

         if Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Procedure
           or else Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Function
         then
            Result.Ref.Target_Type.Ref.Target_Subprogram :=
              Simplify_Subprogram
                (Handle, Type_Definition);
         else
            Result.Ref.Target_Type.Ref.Target_Subprogram :=
              Simplify_Subprogram
                (Handle, Enclosing_Element (Type_Definition));
         end if;

         if Access_Type_Kind (Type_Definition)
           = An_Access_To_Function
           or else Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Function
         then
            Result.Ref.Target_Type.Ref.Target_Type :=
              Simplify_Type_Of
                (Handle,
                 Access_To_Function_Result_Profile
                   (Type_Definition));
         end if;

         for J in Result.Ref.Target_Type.Ref.Target_Subprogram
           .Parameters'Range
         loop
            Result.Ref.Target_Type.Ref.Target_Subprogram
              .Parameters (J).Is_Primitive := False;
         end loop;

         if Result.Ref.Target_Type.Ref.Target_Subprogram.Can_Be_Bound = No then
            Result.Ref.Can_Be_Bound := No;
         end if;
      end Handle_Access_To_Subprogram;

      --------------------------
      -- Check_Access_Support --
      --------------------------

      procedure Check_Access_Support is
      begin
         if not Success then
            --  If we're already in a case that is not successful, don't do
            --  anything

            return;
         end if;

         if Result.Ref.Target_Type.Ref.Can_Be_Bound = No then
            Trace_With_Location
              ("can't bind target access type", Errors_And_Warnings);

            Success := False;

            return;
         end if;

         case Result.Ref.Target_Type.Ref.Kind is
            when Array_Kind =>
               if Result.Ref.Target_Type
                 .Is_Constrained_St_From_Unconstrained
               then
                  Trace_With_Location
                    ("can't bind access to constrained arrays",
                     Errors_And_Warnings);

                  Success := False;
               end if;

            when Generic_Float_Kind
               | Generic_Integer_Kind
               | Standard_Boolean_Kind
               | Standard_Character_Kind
               | Enumeration_Kind =>

               Trace_With_Location
                 ("access to scalar not supported", Errors_And_Warnings);

               Success := False;

            when others =>
               null;

         end case;
      end Check_Access_Support;

      Loc : Location_Handle;
   begin
      Loc := Push_Location (Result.Ref.Location);
      Success := True;

      if not Is_Nil (Type_Definition) then
         if Type_Model_Kind (Type_Definition) = A_Simple_Static_Model then
            Result.Ref.Size := Size (Type_Definition);
         else
            Result.Ref.Size := 0;
         end if;

         case Definition_Kind (Type_Definition) is
            when An_Access_Definition =>
               declare
                  Decl        : Asis.Element;
                  Element_Def : Asis.Element;
                  Old_Type    : Boolean;
               begin
                  if Access_Definition_Kind (Type_Definition)
                    = An_Anonymous_Access_To_Variable
                    or else Access_Definition_Kind (Type_Definition)
                    = An_Anonymous_Access_To_Constant
                  then
                     --  In this case, we're in an anonymous access to data

                     Get_Type_Declaration
                       (Anonymous_Access_To_Object_Subtype_Mark
                          (Type_Definition), Decl, Result);

                     Result.Ref.Kind := Access_Kind;

                     Simplify_Type_Expression (Type_Definition, Result);

                     Create_Simple_Type
                       (Handle          => Handle,
                        Element         => Decl,
                        Type_Definition => Element_Def,
                        Result          => Result.Ref.Target_Type,
                        Old_Type        => Old_Type);

                     if Result.Is_Class_Wide then
                        --  If we determined that the anonymous access is a
                        --  class wide access, then mark the target reference
                        --  accordingly and set the correct access name
                        Result.Ref.Target_Type.Is_Class_Wide := True;

                        if Result.Ref.Is_Constant then
                           Result.Initial_Subtype_Name := Result.Ref.
                             Target_Type.Ref.Named_Constant_Class_Access;
                        else
                           Result.Initial_Subtype_Name :=
                             Result.Ref.Target_Type.Ref.Named_Class_Access;
                        end if;
                     elsif Result.Ref.Is_Constant then
                        Result.Initial_Subtype_Name :=
                          Result.Ref.Target_Type.Ref.Named_Constant_Access;
                     else
                        Result.Initial_Subtype_Name :=
                          Result.Ref.Target_Type.Ref.Named_Access;
                     end if;

                     if Result.Ref.Target_Type /= Null_Type_Reference
                       and then not Old_Type
                     then
                        Handle_Definition
                          (Element_Def,
                           Handle, Success, Result.Ref.Target_Type);
                     end if;

                     Result.Ref.Is_Anonymous := True;

                     Check_Access_Support;

                  elsif Access_Definition_Kind (Type_Definition)
                    = An_Anonymous_Access_To_Function
                    or else Access_Definition_Kind (Type_Definition)
                    = An_Anonymous_Access_To_Procedure
                  then
                     --  In this case, we're in an anonymous access to
                     --  subprogram.

                     Handle_Access_To_Subprogram;
                  end if;
               end;

            when A_Type_Definition =>
               case Type_Kind (Type_Definition) is
                  when An_Enumeration_Type_Definition =>
                     declare
                        Type_Declaration : constant Asis.Element :=
                          Enclosing_Element (Type_Definition);
                     begin
                        if Is_Standard_Boolean (Type_Declaration) then
                           Result.Ref.Kind := Standard_Boolean_Kind;
                        elsif Is_Standard_Character (Type_Declaration) then
                           Result.Ref.Kind := Standard_Character_Kind;
                        else
                           Result.Ref.Kind := Enumeration_Kind;
                           Handle_Enum_Values (Type_Definition);
                        end if;
                     end;

                  when A_Signed_Integer_Type_Definition
                     | A_Modular_Type_Definition =>

                     Result.Ref.Kind := Generic_Integer_Kind;

                  when A_Floating_Point_Definition
                     | An_Ordinary_Fixed_Point_Definition
                     | A_Decimal_Fixed_Point_Definition =>

                     Result.Ref.Kind := Generic_Float_Kind;

                  when A_Record_Type_Definition =>

                     Result.Ref.Kind := Record_Kind;
                     Handle_Record_Discriminants (Type_Definition);
                     Handle_Fields (Type_Definition);

                  when An_Interface_Type_Definition =>
                     Result.Ref.Kind := Tagged_Record_Kind;
                     Handle_Record_Discriminants (Type_Definition);
                     Handle_Controlled_Primitives (Type_Definition);

                  when A_Root_Type_Definition
                     | A_Tagged_Record_Type_Definition =>

                     Result.Ref.Kind := Tagged_Record_Kind;
                     Handle_Record_Discriminants (Type_Definition);
                     Handle_Fields (Type_Definition);
                     Handle_Controlled_Primitives (Type_Definition);

                  when A_Derived_Type_Definition =>

                     declare
                        Tmp  : Simple_Type_Reference;
                     begin
                        Tmp := Simplify_Type_Of
                          (Handle,
                           Corresponding_Parent_Subtype
                             (Type_Definition));

                        Result.Ref.Kind := Tmp.Ref.Kind;
                        Result.Ref.Target_Type := Tmp.Ref.Target_Type;
                        Result.Ref.Target_Subprogram :=
                          Tmp.Ref.Target_Subprogram;
                        Result.Ref.Enum_Values := Tmp.Ref.Enum_Values;

                        if Tmp.Ref.Kind = Array_Kind then
                           Result.Ref.Indexes :=
                             new Simple_Object_View_Array'
                               (Tmp.Ref.Indexes.all);

                           for J in Result.Ref.Indexes'Range loop
                              Result.Ref.Indexes (J) :=
                                Simple_Object_View_Access
                                  (Copy (Result.Ref.Indexes (J)));
                           end loop;

                           if Is_Nil
                             (Subtype_Constraint
                                (Parent_Subtype_Indication (Element)))
                           then
                              --  If there is no index specification, then
                              --  we implicitely use the parent ones.
                              null;

                           else
                              --  Otherwise, we're with a constrained array

                              for J in Result.Ref.Indexes'Range loop
                                 Result.Ref.Indexes (J).Is_Constant := True;
                              end loop;
                           end if;
                        else
                           Handle_Indexes (Type_Definition);
                        end if;
                     end;

                  when A_Derived_Record_Extension_Definition =>

                     Result.Ref.Kind := Tagged_Record_Kind;

                     declare
                        Decl : Asis.Element;
                     begin
                        Get_Type_Declaration
                          (Corresponding_Parent_Subtype
                             (Type_Definition), Decl, Result);

                        Result.Ref.Target_Type := Simplify_Type_Of
                          (Handle, Decl);
                     end;

                     Handle_Record_Discriminants (Type_Definition);
                     Handle_Fields (Type_Definition);
                     Handle_Controlled_Primitives (Type_Definition);

                     if Result.Ref.Target_Type.Ref.Is_Limited then
                        Result.Ref.Is_Limited := True;
                     end if;

                  when An_Unconstrained_Array_Definition
                     | A_Constrained_Array_Definition =>

                     Result.Ref.Kind := Array_Kind;

                     Simplify_Type_Of
                       (Handle,
                        Component_Definition_View
                          (Asis.Definitions.Array_Component_Definition
                             (Type_Definition)),
                        Result.Ref.Target_Type'Access);

                     if Result.Ref.Target_Type.Ref.Kind in Access_Issue_Kinds
                       and then Trait_Kind
                         (Asis.Definitions.Array_Component_Definition
                              (Type_Definition))
                       /= An_Aliased_Trait
                       and then not Ada2Java.Allow_Unaliased_Access
                     then
                        Trace_With_Location
                          ("components should be aliased, "
                           & "see --no-unaliased-access",
                           Errors_And_Warnings);

                        Success := False;
                     else
                        Handle_Indexes (Type_Definition);

                        for J in Result.Ref.Indexes'Range loop
                           --  We need to carry 'Base types on array indexes
                           --  instead of actual type, in order to handle
                           --  empty arrays.

                           Result.Ref.Indexes (J).Type_Of.Ref :=
                             Simple_Type_View_Access
                               (Copy (Result.Ref.Indexes (J).Type_Of.Ref));

                           if To_Wide_String
                             (Result.Ref.Indexes (J).Type_Of.Ref.Full_Ada_Name)
                             /= ""
                           then
                              Append
                                (Result.Ref.Indexes
                                   (J).Type_Of.Ref.Full_Ada_Name,
                                 "'Base");
                           end if;
                        end loop;
                     end if;

                  when An_Access_Type_Definition =>

                     Result.Ref.Kind := Access_Kind;

                     case Access_Type_Kind (Type_Definition) is
                        when A_Pool_Specific_Access_To_Variable =>
                           Trace_With_Location
                             ("pool specific pointers are not bound to java, "
                              & "add all to definition", Errors_And_Warnings);

                           Success := False;

                        when An_Access_To_Variable
                           | An_Access_To_Constant =>

                           Result.Ref.Is_Constant :=
                             Access_Type_Kind (Type_Definition)
                             = An_Access_To_Constant;

                           declare
                              Element_Def : Asis.Element;
                              Old_Type    : Boolean;
                           begin
                              Create_Simple_Type
                                (Handle          => Handle,
                                 Element         =>
                                   Asis.Definitions.Access_To_Object_Definition
                                     (Type_Definition),
                                 Type_Definition => Element_Def,
                                 Result          => Result.Ref.Target_Type,
                                 Old_Type        => Old_Type);

                              if Result.Ref.Target_Type /= Null_Type_Reference
                                and then not Old_Type
                              then
                                 Handle_Definition
                                   (Element_Def,
                                    Handle, Success, Result.Ref.Target_Type);
                              end if;

                              Check_Access_Support;
                           end;

                        when An_Access_To_Procedure
                           .. An_Access_To_Protected_Function =>

                           Handle_Access_To_Subprogram;

                        when others =>
                           null;
                     end case;

                  when others =>

                     raise Not_Supported
                       with Type_Kinds'Image (Type_Kind (Type_Definition))
                       & " is not supported.";

               end case;

            when A_Tagged_Private_Type_Definition
               | A_Private_Extension_Definition =>

               Result.Ref.Kind := Tagged_Record_Kind;
               Handle_Indexes (Type_Definition);
               Handle_Controlled_Primitives (Type_Definition);

            when A_Private_Type_Definition =>
               Result.Ref.Kind := Private_Kind;
               Handle_Indexes (Type_Definition);

            when A_Discrete_Subtype_Definition =>

               --  In this case, everything has already been properly done.
               null;

            when others =>

               raise Not_Supported
                 with Definition_Kinds'Image
                   (Definition_Kind (Type_Definition))
                 & " is not supported.";

         end case;

         Result.Ref.Is_Limited := Result.Ref.Is_Limited
           or else Trait_Kind (Element) = A_Limited_Private_Trait
           or else Trait_Kind (Element) = A_Limited_Trait
           or else Trait_Kind (Element) = An_Abstract_Limited_Private_Trait
           or else Trait_Kind (Element) = An_Abstract_Limited_Trait
           or else Interface_Kind (Element) = A_Limited_Interface;

         Result.Ref.Is_Abstract := Result.Ref.Is_Abstract
           or else Trait_Kind (Type_Definition) = An_Abstract_Trait
           or else Trait_Kind (Type_Definition) = An_Abstract_Private_Trait
           or else Trait_Kind (Type_Definition) = An_Abstract_Limited_Trait
           or else Trait_Kind (Type_Definition) =
              An_Abstract_Limited_Private_Trait
           or else Type_Kind (Type_Definition) = An_Interface_Type_Definition;
      end if;

      if Success then
         if Result.Ref.Kind = Tagged_Record_Kind then
            if Result.Ref.Is_Limited then
               Result.Ref.Allow_Java_Child_Types := False;
               Trace_With_Location
                 ("""" & To_Wide_String
                    (Result.Ref.Exported_Name)
                  & """ can't be derived in Java",
                  Errors_And_Warnings);
               Trace_With_Location
                 (Text  => "(limited types cannot be derived in Java)",
                  Level => Errors_And_Warnings);
            end if;

            if Result.Ref.Is_Abstract
              and then not Result.Ref.Allow_Java_Child_Types
            then
               --  If the object is abstract, we create dummy children with
               --  implementation for Java derivation. But if we can't even
               --  have java derivation, then there's no way any object of this
               --  kind will be initialized from Java, so deactivate java
               --  creation

               Result.Ref.Allow_Java_Creation := False;
            end if;
         end if;

         if Result.Ref.Allow_Java_Child_Types
           and then Result.Ref.Discriminants /= null
           and then Result.Ref.Primitives /= null
         then
            for J in Result.Ref.Primitives'Range loop
               if Result.Ref.Primitives (J).Returned_Type /= null
                 and then Result.Ref.Primitives (J).Returned_Type.
                 Is_Controlling
               then
                  Result.Ref.Allow_Java_Child_Types := False;
                  Trace_With_Location
                    ("""" & To_Wide_String
                       (Result.Ref.Exported_Name)
                     & """ can't be derived in Java",
                     Errors_And_Warnings);

                  Trace_With_Location
                    ("(discriminants tagged types having controlling "
                     & "results types in primitives cannot be derived "
                     & "in Java)",
                     Errors_And_Warnings);

                  exit;
               end if;
            end loop;
         end if;

         if To_Wide_String (Result.Ref.Exported_Name) /= "" then
            Add_Element
              (Get_Or_Create_Bound_Unit
                 (Handle, Simple_Element_View_Access (Result.Ref)),
               Simple_Element_View_Access (Result.Ref));
         end if;
      else
         Result.Ref.Can_Be_Bound := No;
      end if;

      Pop_Location (Loc);
   end Handle_Definition;

   ----------------------
   -- Simplify_Type_Of --
   ----------------------

   function Simplify_Type_Of
     (Handle  : access Kernel.Kernel_Record;
      Element : Asis.Element) return Simple_Type_Reference
   is
      Result  : aliased Simple_Type_Reference := Null_Type_Reference;
   begin
      Simplify_Type_Of (Handle, Element, Result'Access);

      return Result;
   end Simplify_Type_Of;

   ----------------------
   -- Simplify_Type_Of --
   ----------------------

   procedure Simplify_Type_Of
     (Handle  : access Kernel.Kernel_Record;
      Element : Asis.Element;
      Ref     : access Simple_Type_Reference)
   is
      Success     : Boolean := True;
      Element_Def : Asis.Element;
      Old_Type    : Boolean;
   begin
      Create_Simple_Type (Handle, Element, Element_Def, Ref.all, Old_Type);

      if Ref.all /= Null_Type_Reference then
         if not Old_Type then
            Handle_Definition (Element_Def, Handle, Success, Ref.all);
         end if;
      end if;
   end Simplify_Type_Of;

   ------------------------
   -- Create_Simple_Type --
   ------------------------

   procedure Create_Simple_Type
     (Handle          : access Kernel.Kernel_Record;
      Element         : Asis.Element;
      Type_Definition : out Asis.Element;
      Result          : out Simple_Type_Reference;
      Old_Type        : out Boolean)
   is
      Insertion_Done  : Boolean := False;
      Index_Inserted  : Simple_Element_Index;

      procedure Create_Simplified_Type
        (Element   : Asis.Element;
         Exists    : out Boolean;
         Anonymous : Boolean := False);
      --  Initializes the result given the element in parameter.

      procedure Skip_Subtypes (Element : in out Asis.Element);
      --  ???

      ----------------------------
      -- Create_Simplified_Type --
      ----------------------------

      procedure Create_Simplified_Type
        (Element   : Asis.Element;
         Exists    : out Boolean;
         Anonymous : Boolean := False)
      is
         Index        : Element_Index;
         Simple_Index : Simple_Element_Index;
      begin
         Exists := False;

         if Insertion_Done then
            return;
         end if;

         Index := To_Element_Index (Element);
         Simple_Index := To_Simple_Element_Index (Element);

         if Simple_Elements_DB_Pckg.Contains
           (Get_Simple_Elements_DB (Handle).all,
            Simple_Index)
         then
            Result.Ref := Simple_Type_View_Access
              (Simple_Elements_DB_Pckg.Element
                 (Get_Simple_Elements_DB (Handle).all,
                  Simple_Index));

            Exists := True;
            return;
         end if;

         Result.Ref.Base_Package := Get_Or_Create_Package (Handle, Element);

         if not Anonymous then
            Result.Ref.Exported_Name := To_Dynamic_Expression
              (Get_First_Name (Element));
            Result.Ref.Full_Ada_Name := To_Dynamic_Expression
              (Get_Full_Ada_Name (Handle, Element));

            if Is_Standard_String (Element) then
               Result.Ref.Exported_Name := To_Dynamic_Expression ("AdaString");
            end if;

         else
            Result.Ref.Exported_Name := To_Dynamic_Expression
              (Get_First_Name (Enclosing_Element (Element)))
              & "_Anonymous_Type";
            Result.Ref.Is_Anonymous := True;
         end if;

         Result.Ref.Full_Java_Name := To_Dynamic_Expression ("");

         if Get_Java_Base_Package
           (Get_Configuration (Result.Ref.Base_Package)) /= ""
         then
            Result.Ref.Full_Java_Name :=
              Get_Java_Base_Package
                (Get_Configuration (Result.Ref.Base_Package))
              & To_Dynamic_Expression (".");
         end if;

         Result.Ref.Full_Java_Name := Result.Ref.Full_Java_Name
           & Get_Bound_Package_Name (Result.Ref.Base_Package)
           & "." & Result.Ref.Exported_Name;

         Result.Ref.Index := Index;

         Result.Ref.Location := Get_Source_Location_Of_Name (Element);

         Simple_Elements_DB_Pckg.Insert
           (Container => Get_Simple_Elements_DB (Handle).all,
            Key       => Simple_Index,
            New_Item  => Simple_Element_View_Access (Result.Ref));
         Insertion_Done := True;
         Index_Inserted := Simple_Index;
      end Create_Simplified_Type;

      -------------------
      -- Skip_Subtypes --
      -------------------

      procedure Skip_Subtypes (Element : in out Asis.Element) is
         Actual_Is_Constrained : Boolean := False;
         Root_Is_Constrained : Boolean := False;

         Def : Asis.Element;
      begin
         loop
            case Declaration_Kind (Element) is
               when An_Ordinary_Type_Declaration
                  | A_Task_Type_Declaration
                  | A_Protected_Type_Declaration
                  | A_Private_Type_Declaration
                  | A_Private_Extension_Declaration
                  | A_Subtype_Declaration
                  | A_Formal_Type_Declaration =>

                  Def := Type_Declaration_View (Element);

               when others =>
                  Def := Nil_Element;
            end case;

            if not Is_Nil (Def) then
               if Definition_Kind (Def) = A_Type_Definition then
                  if Type_Kind (Def) = An_Unconstrained_Array_Definition then
                     Root_Is_Constrained := False;
                  else
                     Root_Is_Constrained := True;
                  end if;
               elsif Definition_Kind (Def) = A_Subtype_Indication then
                  if not
                    Is_Nil (Asis.Definitions.Subtype_Constraint (Def))
                  then
                     Actual_Is_Constrained := True;
                  end if;
               end if;
            end if;

            exit when Declaration_Kind (Element) /= A_Subtype_Declaration;

            if Declaration_Kind (Element) = A_Subtype_Declaration then
               Get_Type_Declaration
                 (Corresponding_First_Subtype (Element),
                  Element,
                  Result);
            end if;
         end loop;

         Result.Is_Constrained_St_From_Unconstrained :=
           not Root_Is_Constrained and then Actual_Is_Constrained;
      end Skip_Subtypes;

      Analyzed_Type : Asis.Element;

   begin
      Old_Type := False;
      Type_Definition := Nil_Element;

      if not (Declaration_Kind (Element) in A_Type_Declaration
              or else Declaration_Kind (Element) = A_Subtype_Declaration
              or else Declaration_Kind (Element) in An_Object_Declaration
              or else Declaration_Kind (Element)
                = An_Object_Renaming_Declaration
              or else Declaration_Kind (Element) in A_Number_Declaration
              or else Declaration_Kind (Element) = A_Parameter_Specification
              or else Declaration_Kind (Element) = A_Function_Declaration
              or else Declaration_Kind (Element) =
                An_Expression_Function_Declaration
              or else Declaration_Kind (Element) = A_Component_Declaration
              or else Declaration_Kind (Element) = A_Discriminant_Specification
              or else Definition_Kind (Element) = An_Access_Definition
              or else Definition_Kind (Element) = A_Subtype_Indication
              or else Definition_Kind (Element) = A_Type_Definition
              or else Definition_Kind (Element) = A_Discrete_Subtype_Definition
              or else Element_Kind (Element) = An_Expression)
      then
         Result := Null_Type_Reference;
         return;
      end if;

      Result := (new Concrete_Type_View, others => <>);
      Initialize_Configurable_Properties (Handle, Result.Ref);

      --  Initializes Analyzed_Type and Type_Definition

      if Declaration_Kind (Element) in A_Type_Declaration
        or else Declaration_Kind (Element) = A_Subtype_Declaration
      then
         Analyzed_Type := Element;

      elsif Declaration_Kind (Element) = An_Integer_Number_Declaration then
         Result.Ref.Kind := Generic_Integer_Kind;
         return;

      elsif Declaration_Kind (Element) = A_Real_Number_Declaration then
         Result.Ref.Kind := Generic_Float_Kind;
         return;

      elsif Element_Kind (Element) = A_Definition then
         Type_Definition := Element;

         if Definition_Kind (Element) = A_Discrete_Subtype_Definition
           and then Discrete_Range_Kind (Element)
           = A_Discrete_Simple_Expression_Range
         then
            Type_Definition := Corresponding_Expression_Type
                                                       (Lower_Bound (Element));
            if Is_Root_Num_Type (Type_Definition) then
               Type_Definition := Corresponding_Expression_Type
                                                       (Upper_Bound (Element));
            end if;
            if Is_Root_Num_Type (Type_Definition) then
               Type_Definition := Nil_Element;
               Result.Ref.Kind := Generic_Integer_Kind;
               return;
            end if;
         end if;

      elsif Element_Kind (Element) = An_Expression then
         Get_Type_Declaration (Element, Analyzed_Type, Result);
      else
         if Declaration_Kind (Element) = A_Function_Declaration
           or else Declaration_Kind (Element) =
             An_Expression_Function_Declaration
         then
            Type_Definition := Result_Profile (Element);
         else
            Type_Definition := Object_Declaration_View (Element);
         end if;

         if Definition_Kind (Type_Definition) = A_Component_Definition then
            Type_Definition :=
              Component_Definition_View (Type_Definition);
         end if;
      end if;

      if Definition_Kind (Type_Definition) = An_Access_Definition then
         if Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Variable
           or else Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Constant
         then
            --  In this case, we're on an anonymous access type on data -
            --  we will only analyze its definition and won't create any
            --  type.

            return;
         end if;
      end if;

      --  At this point, I have either a declaration of a definition...

      if Is_Nil (Analyzed_Type) and then not Is_Nil (Type_Definition) then
         Get_Type_Declaration (Type_Definition, Analyzed_Type, Result);
      end if;

      if not Is_Nil (Analyzed_Type) then
         if Declaration_Kind (Analyzed_Type) = A_Subtype_Declaration then
            declare
               Gen_Element : Asis.Element;
            begin
               if Names (Analyzed_Type)'Length >= 1 then
                  Gen_Element := Corresponding_Generic_Element
                    (Names (Analyzed_Type)(1));

                  if not Is_Nil (Gen_Element)
                    and then Declaration_Kind
                      (Enclosing_Element (Gen_Element))
                    = A_Formal_Type_Declaration
                  then
                     --  In this case, the name is a name of a formal generic,
                     --  we need to work on the actual instead

                     if Is_Class_Wide (Analyzed_Type) then
                        Result.Is_Class_Wide := True;
                     end if;

                     Analyzed_Type :=
                       Corresponding_First_Subtype (Analyzed_Type);
                  end if;
               end if;
            end;
         end if;

         Result.Initial_Subtype_Name := To_Dynamic_Expression
           (Get_Full_Ada_Name (Handle, Analyzed_Type));

         if Result.Is_Class_Wide then
            Append (Result.Initial_Subtype_Name, "'Class");
         end if;
      end if;

      Skip_Subtypes (Analyzed_Type);

      if Is_Nil (Analyzed_Type) then
         --  At this stage, we are either on a non-supported type or on an
         --  anonymous type - analyze the definition and create a dummy type
         --  if we are on an anonymous type that can be simplified & bound.

         if Is_Nil (Type_Definition) then
            raise Not_Supported with "";
         elsif Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Function
           or else Access_Definition_Kind (Type_Definition)
           = An_Anonymous_Access_To_Procedure
         then
            Create_Simplified_Type (Type_Definition, Old_Type, True);
         else
            raise Not_Supported with "";
         end if;
      elsif Declaration_Kind (Analyzed_Type)
        = A_Private_Extension_Declaration
      then
         Result.Ref.Kind := Tagged_Record_Kind;
         Create_Simplified_Type (Analyzed_Type, Old_Type);

         if Old_Type then
            return;
         end if;

         declare
            Decl : Asis.Element;
         begin
            Get_Type_Declaration
              (Ancestor_Subtype_Indication
                 (Type_Declaration_View (Analyzed_Type)),
               Decl,
               Result);

            Result.Ref.Target_Type := Simplify_Type_Of (Handle, Decl);
         end;

         if Result.Ref.Target_Type.Ref.Is_Limited then
            Result.Ref.Is_Limited := True;
            Result.Ref.Allow_Java_Child_Types := False;

            Trace_With_Location
              ("""" & To_Wide_String
                 (Result.Ref.Exported_Name)
               & """ can't be derived in Java",
               Errors_And_Warnings);
            Trace_With_Location
              (Text  => "(limited types cannot be derived in Java)",
               Level => Errors_And_Warnings);
         end if;

         Type_Definition := Type_Declaration_View (Analyzed_Type);
      else
         Create_Simplified_Type (Analyzed_Type, Old_Type);

         Type_Definition := Type_Declaration_View (Analyzed_Type);
      end if;

      if Is_Nil (Type_Definition) then
         Result := Null_Type_Reference;
      end if;
   exception
      when others =>
         if Insertion_Done then
            Simple_Elements_DB_Pckg.Delete
              (Container => Get_Simple_Elements_DB (Handle).all,
               Key       => Index_Inserted);
         end if;

         raise;
   end Create_Simple_Type;

   ------------------------------
   -- To_Parameter_Simple_View --
   ------------------------------

   function To_Parameter_Simple_View
     (Handle       : access Kernel.Kernel_Record;
      Enclosing_Sb : Simple_Subprogram_View_Access;
      Element      : Asis.Element;
      Name         : Asis.Element)
      return Parameter_Simple_View_Access
   is
      Simple_View : Parameter_Simple_View_Access;
      Type_Of     : Simple_Type_View_Access;
      Actual_Type : Simple_Type_View_Access;

   begin
      case Declaration_Kind (Element) is
         when A_Parameter_Specification =>
            Simple_View := Create (Enclosing_Sb);
            Simple_View.Name :=
              To_Dynamic_Expression (Get_First_Name (Name));
            Simple_View.Java_Name := To_Dynamic_Expression
              (Escape_Java_Identifier (To_Wide_String (Simple_View.Name)));

            Simple_View.Type_Of :=
              Simplify_Type_Of (Handle, Element);

            case Mode_Kind (Element) is
               when An_Out_Mode =>
                  Simple_View.Mode := Out_Mode;

               when An_In_Out_Mode =>
                  Simple_View.Mode := In_Out_Mode;

               when others =>
                  Simple_View.Mode := In_Mode;
            end case;

         when others =>
            return null;
      end case;

      --  On certain occasions, due to the fact that we use full names instead
      --  of partial names for types, we may be in the situation where the
      --  parameter name hides a package name. E.g.:
      --
      --  package P is
      --     type T is null record;
      --     procedure Proc (P : T);
      --  end P;
      --
      --  generates the following glue:
      --
      --  procedure Proc (P : P.T);
      --
      --  In order to havoid these (non-compilable) situation, we detect the
      --  problem here and add a discriminative suffix to the parameter name.
      --
      --  We're doing similar analysis for the package name, e.g. if the
      --  binding root is "J", the following will generate an error:
      --
      --  procedure Proc (J : Some_Type);

      declare
         Param_Name : constant Wide_String := To_Wide_String
           (Simple_View.Name);
         Type_Name  : constant Wide_String := To_Wide_String
           ((if Simple_View.Type_Of.Ref.Is_Anonymous then
                Simple_View.Type_Of.Ref.Target_Type.Ref.Full_Ada_Name
             else
            Simple_View.Type_Of.Ref.Full_Ada_Name));

         function Param_Confuses_With_Hiding_Name
           (Hiding_Name : Wide_String) return Boolean;
         --  ???

         -------------------------------------
         -- Param_Confuses_With_Hiding_Name --
         -------------------------------------

         function Param_Confuses_With_Hiding_Name
           (Hiding_Name : Wide_String) return Boolean is
         begin
            return Hiding_Name = Param_Name
              or else
                (Hiding_Name'Length > Param_Name'Length
                 and then Hiding_Name
                   (Hiding_Name'First
                    .. Hiding_Name'First + Param_Name'Length - 1)
                 = Param_Name
                 and then Hiding_Name
                   (Hiding_Name'First + Param_Name'Length) = '.');
         end Param_Confuses_With_Hiding_Name;

      begin
         Simple_View.Glue_Name := Simple_View.Name;

         if Param_Confuses_With_Hiding_Name (Type_Name)
           or else Param_Confuses_With_Hiding_Name
             (Ada2Java.Bound_Package_Root.all)
         then
            Simple_View.Glue_Name :=
              Simple_View.Glue_Name & "_" & Get_Unique_Id;
         end if;
      end;

      --  Checks if the type can be bound

      if Simple_View.Type_Of.Ref.Can_Be_Bound = No then
         Simple_View.Can_Be_Bound := No;

         return Simple_View;
      end if;

      --  Handles generation of the wrappers if needed

      Type_Of := Simple_View.Type_Of.Ref;

      if (Type_Of.Kind = Access_Kind
          or else
            (Type_Of.Kind = Enumeration_Kind
             and then Ada2Java.Use_Java_1_5_Enums))
        and then
          ((Simple_View.Mode = Out_Mode
            or else Simple_View.Mode = In_Out_Mode))
      then
         if Type_Of.Kind = Access_Kind then
            Actual_Type := Type_Of.Target_Type.Ref;
         else
            Actual_Type := Type_Of;
         end if;

         if Actual_Type.Wrapper = null then
            Actual_Type.Wrapper := Create (Type_Of);
            Actual_Type.Wrapper.Index := Type_Of.Index;
            Actual_Type.Wrapper.Index.Location :=
              new String'("ref:" & Type_Of.Index.Location.all);
            Actual_Type.Wrapper.Wrapped_Type := Actual_Type;

            Add_Element
              (Get_Or_Create_Bound_Unit
                 (Handle, Simple_Element_View_Access (Type_Of)),
               Simple_Element_View_Access (Actual_Type.Wrapper));
         end if;
      end if;

      --  Check if the element has a null exclusion

      if Trait_Kind (Element) = A_Null_Exclusion_Trait then
         Simple_View.Type_Of.Is_Not_Null := True;
      end if;

      return Simple_View;
   end To_Parameter_Simple_View;

   -------------------------
   -- Simplify_Subprogram --
   -------------------------

   function Simplify_Subprogram
     (Handle     : not null access Kernel.Kernel_Record;
      Subprogram : Asis.Element) return Simple_Subprogram_View_Access
   is
      Simple_View  : Simple_Subprogram_View_Access;

      Total_Parameters : Integer := 0;

      Current_Simple_Param : Integer := 0;

      function Get_Parameters return Parameter_Specification_List;
      --  ???

      function Get_Returned_Type return Simple_Profile_Data_View_Access;
      --  ???

      procedure Handle_Subprogram_Data
        (Data : Simple_Profile_Data_View_Access);
      --  ???

      --------------------
      -- Get_Parameters --
      --------------------

      function Get_Parameters return Parameter_Specification_List is
      begin
         if Declaration_Kind (Subprogram) = A_Function_Declaration
           or else Declaration_Kind (Subprogram)
             = An_Expression_Function_Declaration
           or else Declaration_Kind (Subprogram) = A_Procedure_Declaration
           or else Declaration_Kind (Subprogram)
             = A_Procedure_Renaming_Declaration
           or else Declaration_Kind (Subprogram)
             = A_Function_Renaming_Declaration
           or else Declaration_Kind (Subprogram)
             = A_Null_Procedure_Declaration
           or else Declaration_Kind (Subprogram)
             = An_Expression_Function_Declaration
         then
            return Parameter_Profile (Subprogram);
         elsif Access_Definition_Kind (Subprogram)
           = An_Anonymous_Access_To_Function
           or else Access_Definition_Kind (Subprogram)
           = An_Anonymous_Access_To_Procedure
         then
            return Access_To_Subprogram_Parameter_Profile (Subprogram);
         else
            return Access_To_Subprogram_Parameter_Profile
              (Type_Declaration_View (Subprogram));
         end if;
      end Get_Parameters;

      -----------------------
      -- Get_Returned_Type --
      -----------------------

      function Get_Returned_Type return Simple_Profile_Data_View_Access is
         Result : Simple_Profile_Data_View_Access;
      begin
         if Declaration_Kind (Subprogram) = A_Function_Declaration
           or else Declaration_Kind (Subprogram)
           = An_Expression_Function_Declaration
           or else Declaration_Kind (Subprogram)
           = A_Function_Renaming_Declaration
         then
            Result := Create (Simple_View);

            Result.Type_Of :=
              Simplify_Type_Of (Handle, Result_Profile (Subprogram));
         elsif Declaration_Kind (Subprogram) in A_Type_Declaration
           and then Access_Type_Kind (Type_Declaration_View (Subprogram))
           = An_Access_To_Function
         then
            Result := Create (Simple_View);

            Result.Type_Of := Simplify_Type_Of
              (Handle,
               Access_To_Function_Result_Profile
                 (Type_Declaration_View (Subprogram)));
         elsif Access_Definition_Kind (Subprogram)
           = An_Anonymous_Access_To_Function
         then
            Result := Create (Simple_View);

            Result.Type_Of := Simplify_Type_Of
              (Handle,
               Access_To_Function_Result_Profile (Subprogram));
         else
            return null;
         end if;

         Result.Enclosing_Sb := Simple_View;

         return Result;
      end Get_Returned_Type;

      ----------------------------
      -- Handle_Subprogram_Data --
      ----------------------------

      procedure Handle_Subprogram_Data
        (Data : Simple_Profile_Data_View_Access)
      is
         Primitive_View : Simple_Type_View_Access;
      begin
         Primitive_View := Data.Type_Of.Ref;

         if Primitive_View.Kind = Access_Kind then
            Primitive_View := Primitive_View.Target_Type.Ref;
         end if;

         if (Primitive_View.Kind = Record_Kind
             or else Primitive_View.Kind = Tagged_Record_Kind
             or else Primitive_View.Kind = Private_Kind)
           and then
             (Data.all not in Parameter_Simple_View
              or else Parameter_Simple_View (Data.all).Mode = In_Mode
              or else Data.Type_Of.Ref.Kind /= Access_Kind)
         then
            Data.Is_Primitive :=
                Primitive_View.Base_Package =
                  Simple_View.Base_Package;
         else
            Data.Is_Primitive := False;
         end if;

         if Simple_View.Is_Dispatching then
            declare
               Primitive_Owner_Index : Element_Index;
               Param_Type : Simple_Type_View_Access;
            begin
               if not Is_Nil (Primitive_Owner (Subprogram)) then
                  --  In this case, the subprogram is explicitely
                  --  declared, use the Primitive_Owner

                  Primitive_Owner_Index  :=
                    To_Element_Index
                      (Enclosing_Element
                           (Primitive_Owner (Subprogram)));
               else
                  --  In this case, the subprogram is implicitely
                  --  declared, use Declarations.Corresponding_Type

                  Primitive_Owner_Index :=
                    To_Element_Index
                      (Enclosing_Element
                         (Corresponding_Type (Subprogram)));
               end if;

               if Data.Type_Of.Ref.Kind = Access_Kind
                 and then Data.Type_Of.Ref.Exported_Name
                   = Empty_Dynamic_Expression
               then
                  Param_Type := Data.Type_Of.Ref.Target_Type.Ref;
               else
                  Param_Type := Data.Type_Of.Ref;
               end if;

               Data.Is_Controlling :=
                 Param_Type /= null and then
               Param_Type.Index = Primitive_Owner_Index;
            end;
         end if;
      end Handle_Subprogram_Data;

   begin
      if Is_Generic_Actual (Subprogram) then
         --  Simply ignore subprogram from actual generic parameters.

         raise Silent_Not_Supported;
      end if;

      Simple_View := new Concrete_Subprogram_View;
      Simple_View.Location := Get_Source_Location_Of_Name (Subprogram);

      Initialize_Configurable_Properties (Handle, Simple_View);

      Simple_View.Is_Dispatching :=
        Is_Dispatching_Operation (Subprogram);
      Simple_View.Call_Convention := Java_To_Ada;

      if Access_Definition_Kind (Subprogram)
        = An_Anonymous_Access_To_Procedure
        or else Access_Definition_Kind (Subprogram)
        = An_Anonymous_Access_To_Function
      then
         Simple_View.Original_Name := To_Dynamic_Expression (Get_Unique_Id);
      else
         Simple_View.Original_Name :=
           To_Dynamic_Expression (Get_First_Name (Subprogram));
      end if;

      if Definition_Kind (Subprogram) = An_Access_Definition
        and then Access_Definition_Kind (Subprogram)
        = An_Anonymous_Access_To_Procedure
      then
         Simple_View.Name := To_Dynamic_Expression ("");
      else
         declare
            Name : constant Wide_String := Get_First_Name (Subprogram);
         begin
            if Name (Name'First) = '"' then
               --  We're on an operator - need to use a different name for
               --  the binding to Java

               if Name = """=""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_EQUAL");
               elsif Name = """>""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_GT");
               elsif Name = """<""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_LT");
               elsif Name = """>=""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_GE");
               elsif Name = """<=""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_LE");
               elsif Name = """or""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_OR");
               elsif Name = """and""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_AND");
               elsif Name = """xor""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_XOR");
               elsif Name = """+""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_PLUS");
               elsif Name = """-""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_MINUS");
               elsif Name = """/""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_DIV");
               elsif Name = """*""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_MUL");
               elsif Name = """**""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_EXP");
               elsif Name = """&""" then
                  Simple_View.Name := To_Dynamic_Expression ("OP_CONCAT");
               end if;
            else
               Simple_View.Name := To_Dynamic_Expression
                 (Name);
            end if;
         end;
      end if;

      Simple_View.Is_Abstract := Trait_Kind (Subprogram) = An_Abstract_Trait;
      Simple_View.Base_Package := Get_Or_Create_Package (Handle, Subprogram);
      Simple_View.Index := To_Element_Index (Subprogram);

      declare
         Parameters  : constant Parameter_Specification_List := Get_Parameters;
      begin
         for J in Parameters'Range loop
            Total_Parameters :=
              Total_Parameters + Names (Parameters (J))'Length;
         end loop;

         Simple_View.Parameters := new Parameter_Array (1 .. Total_Parameters);

         Current_Simple_Param := 1;

         for J in Parameters'Range loop
            declare
               Param_Names : constant Defining_Name_List :=
                 Names (Parameters (J));
               Loc : Location_Handle;
            begin
               for K in Param_Names'Range loop
                  Loc := Push_Location (Get_Source_Location (Parameters (J)));
                  Simple_View.Parameters (Current_Simple_Param) :=
                    To_Parameter_Simple_View
                      (Handle, Simple_View, Parameters (J), Param_Names (K));

                  if Simple_View.Parameters
                    (Current_Simple_Param).Type_Of.Ref.Can_Be_Bound = No
                  then
                     Trace_With_Location
                       ("parameter type can't be bound", Errors_And_Warnings);

                     Simple_View.Can_Be_Bound := No;

                     return Simple_View;
                  end if;

                  if (Declaration_Kind (Subprogram) = A_Function_Declaration
                    or else Declaration_Kind (Subprogram) =
                      An_Expression_Function_Declaration)
                    and then Simple_View.Parameters
                      (Current_Simple_Param).Mode = In_Out_Mode
                  then
                     Trace_With_Location
                       ("in out parameters can't be bound",
                        Errors_And_Warnings);

                     Simple_View.Can_Be_Bound := No;

                     return Simple_View;
                  end if;

                  Handle_Subprogram_Data
                    (Simple_Profile_Data_View_Access
                       (Simple_View.Parameters (Current_Simple_Param)));

                  Current_Simple_Param := Current_Simple_Param + 1;
                  Pop_Location (Loc);
               end loop;
            end;
         end loop;

         Simple_View.Returned_Type := Get_Returned_Type;

         if Simple_View.Returned_Type /= null then
            if Simple_View.Returned_Type.Type_Of.Ref.Can_Be_Bound = No then
               Trace_With_Location
                 ("return type can't be bound", Errors_And_Warnings);

               Simple_View.Can_Be_Bound := No;

               return Simple_View;
            end if;

            Handle_Subprogram_Data (Simple_View.Returned_Type);
         end if;
      end;

      if Simple_View.Is_Abstract then
         --  If this is an abstract subprogram, then check that there's at
         --  least one controlling parameter. Otherwise, abstract means remove
         --  and there's nothing to bind.

         declare
            Controlling_Found : Boolean := False;
         begin
            for J in Simple_View.Parameters'Range loop
               if Simple_View.Parameters (J).Is_Controlling then
                  Controlling_Found := True;

                  exit;
               end if;
            end loop;

            if Simple_View.Returned_Type /= null
              and then Simple_View.Returned_Type.Is_Controlling
            then
               Controlling_Found := True;
            end if;

            if not Controlling_Found then
               --  An abstract subprogram with no controlling part cannot be
               --  bound.

               Simple_View.Can_Be_Bound := No;

               return Simple_View;
            end if;
         end;
      end if;

      if Declaration_Kind (Subprogram) = A_Function_Renaming_Declaration
        or else Declaration_Kind (Subprogram)
        = A_Procedure_Renaming_Declaration
      then
         declare
            Renamed : Asis.Element;
         begin
            Renamed := Renamed_Entity (Subprogram);

            if Expression_Kind (Renamed) = A_Selected_Component then
               Renamed := Selector (Renamed);
            end if;

            Renamed := Corresponding_Name_Declaration (Renamed);

            if Element_Kind (Renamed) /= Not_An_Element then
               --  If we could identify an explicit node for the renaming

               Simple_View.Renaming :=
                 Get_Simple_Element (Handle, Renamed);
            end if;
         end;
      end if;

      --  Computes subprogram attachement.

      if Simple_View.Can_Be_Bound /= No
        and then Simple_View.Parameters /= null
        and then Simple_View.Parameters'Length >= 1
        and then
          (Declaration_Kind (Subprogram) = A_Procedure_Declaration
           or else Declaration_Kind (Subprogram) = A_Function_Declaration
           or else Declaration_Kind (Subprogram) =
             An_Expression_Function_Declaration
           or else Declaration_Kind (Subprogram) =
             A_Null_Procedure_Declaration)
      then
         if Simple_View.Is_Dispatching then
            if Simple_View.Parameters (1).Is_Controlling then
               Simple_View.Parameters (1).Attached :=
                 Ada2Java.Default_Controlling_Attachement
                 or else Ada2Java.Default_Ada_2005_Attachement;
            else
               Simple_View.Parameters (1).Attached := False;
            end if;
         else
            declare
               Analyzed_Type : Simple_Type_Reference :=
                 Simple_View.Parameters (1).Type_Of;
               Is_Access : Boolean := False;
            begin
               if Analyzed_Type.Ref.Kind = Access_Kind then
                  Analyzed_Type := Analyzed_Type.Ref.Target_Type;
                  Is_Access := True;
               end if;

               if Analyzed_Type.Ref.Base_Package = Simple_View.Base_Package
                 and then
                   (not Is_Access
                    or else Simple_View.Parameters (1).Mode = In_Mode)
               then
                  --  We consider attachement only if the analyzed type is
                  --  on the same package as the subprogram. In addition, we
                  --  do not consider out & in out access types.

                  case Analyzed_Type.Ref.Kind is
                     when Private_Kind | Record_Kind | Tagged_Record_Kind =>

                        Simple_View.Parameters (1).Attached :=
                          (Is_Access
                           and then Ada2Java.Default_Access_Attachement)
                          or else
                            (not Is_Access
                             and then Ada2Java.Default_Attachement);

                        if not Simple_View.Parameters (1).Attached
                          and then Ada2Java.Default_Ada_2005_Attachement
                          and then Analyzed_Type.Is_Class_Wide
                          and then
                            (not Is_Access
                             or else Simple_View.Parameters (1).
                               Type_Of.Ref.Is_Anonymous)
                        then
                           --  We are in the Ada 2005 attachement case.

                           Simple_View.Parameters (1).Attached := True;
                        end if;

                     when others =>
                        Simple_View.Parameters (1).Attached := False;

                  end case;
               else
                  Simple_View.Parameters (1).Attached := False;
               end if;
            end;
         end if;
      end if;

      return Simple_View;
   end Simplify_Subprogram;

   -----------------
   -- Is_Attached --
   -----------------

   function Is_Attached (This : Simple_Subprogram_View) return Boolean is
   begin
      return This.Parameters /= null
        and then This.Parameters'Length >= 1
        and then This.Parameters (1).Attached;
   end Is_Attached;

   -----------------------
   -- Get_Attached_Type --
   -----------------------

   function Get_Attached_Type
     (This : Simple_Subprogram_View) return Simple_Type_View_Access
   is
   begin
      if This.Is_Attached then
         if This.Parameters (1).Type_Of.Ref.Kind = Access_Kind then
            return This.Parameters (1).Type_Of.Ref.Target_Type.Ref;
         else
            return This.Parameters (1).Type_Of.Ref;
         end if;
      elsif This.Force_Attach /= null then
         return This.Force_Attach;
      else
         return null;
      end if;
   end Get_Attached_Type;

   ---------------------------
   -- Get_Controlling_Param --
   ---------------------------

   function Get_Controlling_Param
     (This : Simple_Subprogram_View) return Parameter_Simple_View_Access is
   begin
      if This.Parameters /= null
        and then This.Parameters'Length >= 1
        and then This.Parameters (1).Is_Controlling
      then
         return This.Parameters (1);
      else
         return null;
      end if;
   end Get_Controlling_Param;

   -------------------
   -- Get_Java_Name --
   -------------------

   function Get_Java_Name
     (This : Simple_Subprogram_View) return Dynamic_Expression is
   begin
      if This.Java_Name = Empty_Dynamic_Expression then
         return This.Name;
      else
         return This.Java_Name;
      end if;
   end Get_Java_Name;

   -------------------------
   -- Is_Standard_Package --
   -------------------------

   function Is_Standard_Package (E : Asis.Element) return Boolean is
   begin
      return Get_First_Name (E) = "Standard"
        and then Element_Kind
          (Enclosing_Element (E)) = Not_An_Element;
   end Is_Standard_Package;

   -------------------------
   -- Is_Standard_Boolean --
   -------------------------

   function Is_Standard_Boolean (E : Asis.Element) return Boolean is
   begin
      return Get_First_Name (E) = "Boolean"
        and then Is_Standard_Package (Enclosing_Element (E));
   end Is_Standard_Boolean;

   ---------------------------
   -- Is_Standard_Character --
   ---------------------------

   function Is_Standard_Character (E : Asis.Element) return Boolean is
   begin
      return Get_First_Name (E) = "Character"
        and then Is_Standard_Package (Enclosing_Element (E));
   end Is_Standard_Character;

   ------------------------
   -- Is_Standard_String --
   ------------------------

   function Is_Standard_String (E : Asis.Element) return Boolean is
   begin
      return Get_First_Name (E) = "String"
        and then Is_Standard_Package (Enclosing_Element (E));
   end Is_Standard_String;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Asis.Element) return Boolean is
      Left_Id    : constant Asis.Ids.Id := Create_Id (Left);
      Right_Id   : constant Asis.Ids.Id := Create_Id (Right);
   begin
      return Left_Id < Right_Id;
   end "<";

   ----------------------
   -- Create_Access_To --
   ----------------------

   function Create_Access_To
     (Base_Type   : Simple_Type_Reference;
      Is_Constant : Boolean)
      return Simple_Type_Reference
   is
      Access_Type : Simple_Type_Reference;
   begin
      Access_Type.Ref := Create (Base_Type.Ref);
      Access_Type.Ref.Is_Constant := Is_Constant;
      Access_Type.Ref.Target_Type := Base_Type;
      Access_Type.Ref.Kind := Access_Kind;
      Access_Type.Ref.Index := Base_Type.Ref.Index;
      Access_Type.Ref.Index.Location := new String'
        ("acc:" & Access_Type.Ref.Index.Location.all);

      if Base_Type.Is_Class_Wide then
         if Is_Constant then
            Access_Type.Initial_Subtype_Name :=
              Base_Type.Ref.Named_Constant_Class_Access;
         else
            Access_Type.Initial_Subtype_Name :=
              Base_Type.Ref.Named_Class_Access;
         end if;
         Access_Type.Is_Class_Wide := True;
         --  ??? We shouldn't need to specify class wideness on the access
         --  type reference but only on the target type...
      elsif Is_Constant then
         Access_Type.Initial_Subtype_Name :=
           Base_Type.Ref.Named_Constant_Access;
         Access_Type.Is_Class_Wide := False;
      else
         Access_Type.Initial_Subtype_Name := Base_Type.Ref.Named_Access;
         Access_Type.Is_Class_Wide := False;
      end if;

      return Access_Type;
   end Create_Access_To;

   -----------------------------
   -- To_Simple_Element_Index --
   -----------------------------

   function To_Simple_Element_Index
     (Element : Asis.Element) return Simple_Element_Index
   is
   begin
      if Get_First_Name (Element) = "Exception_Occurrence"
        and then Get_First_Name (Enclosing_Element (Element))
        = "Ada.Exceptions"
      then
         return Exception_Occurence_Index;
      elsif Is_Standard_String (Element) then
         return String_Index;
      end if;

      return
        (Elem_Index         => To_Element_Index (Element),
         Instantiation_Loc  =>
           new String'(Build_GNAT_Location (Enclosing_Element (Element))));
   end To_Simple_Element_Index;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Simple_Element_Index) return Boolean is
   begin
      return Left.Elem_Index < Right.Elem_Index
        or else
          (Left.Elem_Index = Right.Elem_Index
           and then Left.Instantiation_Loc.all < Right.Instantiation_Loc.all);
   end "<";

   ----------------------------
   -- Is_Unconstrained_Array --
   ----------------------------

   function Is_Unconstrained_Array
     (Type_Of : Simple_Type_View_Access) return Boolean is
   begin
      if Type_Of.Kind = Array_Kind then
         for J in Type_Of.Indexes'Range loop
            if not Type_Of.Indexes (J).Is_Constant then
               return True;
            end if;
         end loop;
      end if;

      return False;
   end Is_Unconstrained_Array;

   -----------------------
   -- Is_Generic_Actual --
   -----------------------

   function Is_Generic_Actual (Elem : Asis.Element) return Boolean is
   begin
      if Element_Kind (Elem) = A_Declaration then
         declare
            Gen_Element : Asis.Element;
         begin
            if Names (Elem)'Length >= 1 then
               Gen_Element := Corresponding_Generic_Element
                 (Names (Elem)(1));

               if not Is_Nil (Gen_Element)
                 and then Declaration_Kind
                   (Enclosing_Element (Gen_Element))
               in A_Formal_Declaration
               then
                  return True;
               end if;
            end if;
         end;
      end if;

      return False;
   end Is_Generic_Actual;

   ---------------------------
   -- Escape_Java_Identifer --
   ---------------------------

   function Escape_Java_Identifier
     (Identifier : Wide_String) return Wide_String is
   begin
      if Identifier = "assert"
        or else Identifier = "boolean"
        or else Identifier = "break"
        or else Identifier = "byte"
        or else Identifier = "catch"
        or else Identifier = "char"
        or else Identifier = "class"
        or else Identifier = "const"
        or else Identifier = "continue"
        or else Identifier = "default"
        or else Identifier = "double"
        or else Identifier = "enum"
        or else Identifier = "extends"
        or else Identifier = "final"
        or else Identifier = "finally"
        or else Identifier = "float"
        or else Identifier = "implements"
        or else Identifier = "import"
        or else Identifier = "instanceof"
        or else Identifier = "int"
        or else Identifier = "long"
        or else Identifier = "native"
        or else Identifier = "public"
        or else Identifier = "short"
        or else Identifier = "static"
        or else Identifier = "strictfp"
        or else Identifier = "super"
        or else Identifier = "switch"
        or else Identifier = "this"
        or else Identifier = "throw"
        or else Identifier = "throws"
        or else Identifier = "transient"
        or else Identifier = "try"
        or else Identifier = "void"
        or else Identifier = "volatile"
      then
         return "_" & Identifier;
      else
         return Identifier;
      end if;
   end Escape_Java_Identifier;

end Ada2Java.Simplifications;
