------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                        Copyright (C) 2014, AdaCore                       --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters.Conversions;
with Ada.Strings.Wide_Unbounded; use Ada.Strings.Wide_Unbounded;

with GNAT.Directory_Operations;  use GNAT.Directory_Operations;
with GNAT.OS_Lib;                use GNAT.OS_Lib;

with Ada2Java;         use Ada2Java;
with Ada2Java.Kernel;  use Ada2Java.Kernel;

with Gnatvsn; use Gnatvsn;

with ASIS_UL.Common;             use ASIS_UL.Common;
with ASIS_UL.Compiler_Options;
with ASIS_UL.Environment;        use ASIS_UL.Environment;
with ASIS_UL.Options;            use ASIS_UL.Options;
with ASIS_UL.Output;             use ASIS_UL.Output;

package body Ada2Java.Projects is

   ----------------------
   -- Print_Tool_Usage --
   ----------------------

   overriding procedure Print_Tool_Usage
     (My_Project : Ada2Java_Project_Type)
   is
      pragma Unreferenced (My_Project);
   begin
      pragma Style_Checks (Off);

      Info ("");
      Info ("Usage: ada2java {<switch> | <input-file>} <input-file> " &
            "{<switch> | <input-file>}");
      Info ("");
      Info ("  Each <input-file> must be the name of a source");
      Info ("  file for an Ada package spec (including the extension).");
      Info ("");

      Info ("general options:");
      Info ("  --version   Display version and exit");
      Info ("  --help      Display usage and exit");
      Info ("  -debug-glue");
      Info ("     add debug comments");
      Info ("");
      Info ("  -verbose=(silent|errors_only|errors_and_warnings|" &
            "limited_verbosity|full_verbosity|debug)");
      Info ("     output verbosity level");
      Info ("  -b name");
      Info ("     Use this name as the base of generated java packages");
      Info ("  -c dir");
      Info ("     Generate the java interface in that directory");
      Info ("  -L name");
      Info ("     Set the native library name, generate build project & " &
            "automatic load");
      Info ("  -M name");
      Info ("     Set the main unit name, generate build project for an & " &
            "Ada main");
      Info ("  -o dir");
      Info ("     Generate the ada glue code in that directory");
      Info ("  -P prj");
      Info ("     Use this input project");

      Info ("");
      Info ("generation configuration options:");
      Info ("  --[no-]assume-escaped");
      Info ("     Set default assumption for access storing");
      Info ("  --bound-package-root=<package root name>");
      Info ("     Set the name of the root glue Ada packages (default is "
            & "JNI_Binding).");
      Info ("  --bound-package-suffix=<suffix name>");
      Info ("     Set the suffix of the glue Ada packages (default is _JNI).");
      Info ("  --[no-]attach-(Parameter (Parser => Parser)|access|controlling|ada2005)");
      Info ("     Set attachement policy");
      Info ("  --[no-]java-enum");
      Info ("     Enable or disable use of Java enumerations");
      Info ("  --link-mode=(export|register_natives)");
      Info ("     Set discovery method of native functions by Java");
      Info ("  --library-kind=(dynamic|encapsulated)");
      Info ("     Set library generation method: normal or standalone (used by"
            & " -L)");
      Info ("  --main-class=<java class>");
      Info ("     If the -M switch is used, changes the main java class to" &
            " use");
      Info ("  --no-locking[-finalize] / "
            & "--locking[-finalize]-(check|protect)");
      Info ("     Set default locking for subprograms");
      Info ("  --lock=<lock full name>");
      Info ("     Changes the default lock to the full qualified name.");
      Info ("  --[no-]unaliased-access");
      Info ("     Set access allowance on unaliased data");

      Info ("");

      pragma Style_Checks (On);
   end Print_Tool_Usage;

   --------------------
   -- Scan_Arguments --
   --------------------

   overriding procedure Scan_Arguments
     (My_Project  : in out Ada2Java_Project_Type;
      First_Pass  :        Boolean    := False;
      Parser      :        Opt_Parser := Command_Line_Parser;
      In_Switches :        Boolean    := False)
   is
      In_Project_File : constant Boolean := Parser /= Command_Line_Parser;
   begin

      loop
         case GNAT.Command_Line.Getopt
           --  !!! No real project support so far!
           --  -P Parameter (Parser => Parser) is used for some old-fasion
           --  usage of a project file
           ("v verbose= h d dd q P: b: c: o: L: M:"
            & " -assume-escaped"
            & " -bound-package-root="
            & " -bound-package-suffix="
            & " -link-mode="
            & " -library-kind="
            & " -no-assume-escaped"
            & " -no-locking"
            & " -locking-check"
            & " -locking-protect"
            & " -no-locking-finalize"
            & " -locking-finalize-check"
            & " -locking-finalize-protect"
            & " -lock="
            & " -attach-parameter"
            & " -attach-access"
            & " -attach-controlling"
            & " -attach-ada2005"
            & " -no-attach-parameter"
            & " -no-attach-access"
            & " -no-attach-controlling"
            & " -no-attach-ada2005"
            & " -java-enum"
            & " -no-java-enum"
            & " -unaliased-access"
            & " -no-unaliased-access"
            & " -main-class="
            & " -trace-checks"
            & " -trace-null-conversions"
            & " -debug-glue"
            & " -version"
            & " -help",
            Parser => Parser)
         is

            when ASCII.NUL =>
               exit when not
                 More_Arguments
                   (Store_Arguments => In_Project_File or else First_Pass,
                    In_Switches     => In_Switches);

            when 'h' =>
               if not First_Pass then
                  My_Project.Print_Tool_Usage;
               end if;
            when 'v' =>
               if not First_Pass then
                  if Full_Switch (Parser => Parser) = "v" then
                     Print_Tool_Version (2004);
                  else
                     Verbose_Mode := True;

                     if Parameter (Parser => Parser) = "" then
                        null;
                     elsif Parameter (Parser => Parser) = "silent" then
                        Current_Trace_Level := Silent;
                     elsif Parameter (Parser => Parser) = "errors_only" then
                        Current_Trace_Level := Errors_Only;
                     elsif Parameter (Parser => Parser) =
                       "errors_and_warnings"
                     then
                        Current_Trace_Level :=  Errors_And_Warnings;
                     elsif Parameter (Parser => Parser) =
                       "limited_verbosity"
                     then
                        Current_Trace_Level := Limited_Verbosity;
                     elsif Parameter (Parser => Parser) = "full_verbosity" then
                        Current_Trace_Level := Full_Verbosity;
                     elsif Parameter (Parser => Parser) = "debug" then
                        Current_Trace_Level := Ada2Java.Debug;
                     end if;
                  end if;
               end if;

            when 'q' =>
               if not First_Pass then
                  Quiet_Mode := True;
               end if;

            when 'd' =>
               if not First_Pass then
                  if Full_Switch (Parser => Parser) = "d" then
                     Debug_Mode := True;
                  elsif Full_Switch (Parser => Parser) = "dd" then
                     Progress_Indicator_Mode := True;
                  end if;
               end if;

            when 'P' =>
               if not First_Pass then
                  Project_File := new String'(Parameter (Parser => Parser));
                  --  We do not store the '-P' option here, because we need the
                  --  project file name with full path, we will do this in
                  --  Check_Parameters later on/
               end if;

            when 'b' =>
               if not First_Pass then
                  Default_Base_Package := To_Unbounded_Wide_String
                    (To_Wide_String (Parameter (Parser => Parser)));
               end if;

            when 'c' =>
               if not First_Pass then
                  if Is_Absolute_Path (Parameter (Parser => Parser)) then
                     Default_Class_Path := To_Unbounded_Wide_String
                       (To_Wide_String (Parameter (Parser => Parser)));
                  else
                     Default_Class_Path := To_Unbounded_Wide_String
                       (To_Wide_String
                          (Get_Current_Dir & Directory_Separator &
                             Parameter (Parser => Parser)));
                  end if;
               end if;

            when 'o' =>
               if not First_Pass then
                  if Is_Absolute_Path (Parameter (Parser => Parser)) then
                     Default_Ada_Base_Folder := To_Unbounded_Wide_String
                       (To_Wide_String (Parameter (Parser => Parser)));
                  else
                     Default_Ada_Base_Folder := To_Unbounded_Wide_String
                       (To_Wide_String
                          (Get_Current_Dir & Directory_Separator &
                             Parameter (Parser => Parser)));
                  end if;
               end if;

            when 'L' =>
               if not First_Pass then
                  Ada2Java.Native_Library_Name :=
                    new String'(Parameter (Parser => Parser));
               end if;

            when 'M' =>
               if not First_Pass then
                  Ada2Java.Main_Unit_Name :=
                    new String'(Parameter (Parser => Parser));
                  Ada2Java.Link_Method := Register_Natives;
               end if;

            when '-' =>
               if Full_Switch (Parser => Parser) = "-help" then
                  if In_Project_File then
                     Error ("project file should not contain '--help' option");
                     raise Parameter_Error;
                  end if;

                  Print_Usage := True;
                  return;
               elsif Full_Switch (Parser => Parser) = "-version" then
                  if In_Project_File then
                     Error
                       ("project file should not contain '--version' option");
                     raise Parameter_Error;
                  end if;

                  Print_Version := True;
                  return;
               end if;

               if not First_Pass then

                  if Full_Switch (Parser => Parser) = "-assume-escaped" then
                     Ada2Java.Default_Assumed_Stored := True;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-assume-escaped"
                  then
                     Ada2Java.Default_Assumed_Stored := False;
                  elsif Full_Switch (Parser => Parser) = "-link-mode" then
                     Ada2Java.Link_Method :=
                       Link_Method_Type'Value (Parameter (Parser => Parser));
                  elsif Full_Switch (Parser => Parser) = "-library-kind" then
                     Ada2Java.Library_Kind :=
                       Library_Kind_Type'Value (Parameter (Parser => Parser));
                  elsif Full_Switch (Parser => Parser) = "-no-locking" then
                     Ada2Java.Default_Locking_State := Disable;
                  elsif Full_Switch (Parser => Parser) = "-locking-check" then
                     Ada2Java.Default_Locking_State := Check;
                  elsif Full_Switch (Parser => Parser) =
                          "-locking-protect"
                  then
                     Ada2Java.Default_Locking_State := Protect;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-locking-finalize"
                  then
                     Ada2Java.Default_Finalize_Locking_State := Disable;
                  elsif Parameter (Parser => Parser) =
                          "-locking-finalize-check"
                  then
                     Ada2Java.Default_Finalize_Locking_State := Check;
                  elsif Full_Switch (Parser => Parser) =
                          "-locking-finalize-protect"
                  then
                     Ada2Java.Default_Finalize_Locking_State := Protect;
                  elsif Full_Switch (Parser => Parser) = "-lock" then
                     Ada2Java.Default_Lock := new Wide_String'
                       (Ada.Characters.Conversions.To_Wide_String
                         (Parameter (Parser => Parser)));
                  elsif Full_Switch (Parser => Parser) =
                          "-attach-parameter"
                  then
                     Ada2Java.Default_Attachement := True;
                     Ada2Java.Default_Access_Attachement := True;
                     Ada2Java.Default_Controlling_Attachement := True;
                     Ada2Java.Default_Ada_2005_Attachement := True;
                  elsif Full_Switch (Parser => Parser) = "-attach-access" then
                     Ada2Java.Default_Access_Attachement := True;
                  elsif Full_Switch (Parser => Parser) =
                          "-attach-controlling"
                  then
                     Ada2Java.Default_Controlling_Attachement := True;
                  elsif Full_Switch (Parser => Parser) = "-attach-ada2005" then
                     Ada2Java.Default_Ada_2005_Attachement := True;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-attach-parameter"
                  then
                     Ada2Java.Default_Attachement := False;
                     Ada2Java.Default_Access_Attachement := False;
                     Ada2Java.Default_Controlling_Attachement := False;
                     Ada2Java.Default_Ada_2005_Attachement := False;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-attach-access"
                  then
                     Ada2Java.Default_Access_Attachement := False;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-attach-controlling"
                  then
                     Ada2Java.Default_Controlling_Attachement := False;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-attach-ada2005"
                  then
                     Ada2Java.Default_Ada_2005_Attachement := False;
                  elsif Full_Switch (Parser => Parser) = "-java-enum" then
                     Ada2Java.Use_Java_1_5_Enums := True;
                  elsif Full_Switch (Parser => Parser) = "-no-java-enum" then
                     Ada2Java.Use_Java_1_5_Enums := False;
                  elsif Full_Switch (Parser => Parser) =
                          "-unaliased-access"
                  then
                     Ada2Java.Allow_Unaliased_Access := True;
                  elsif Full_Switch (Parser => Parser) =
                          "-no-unaliased-access"
                  then
                     Ada2Java.Allow_Unaliased_Access := False;
                  elsif Full_Switch (Parser => Parser) = "-main-class" then
                     Ada2Java.Java_Main_Class :=
                       new String'(Parameter (Parser => Parser));
                  elsif Full_Switch (Parser => Parser) =
                          "-bound-package-root"
                  then
                     Ada2Java.Bound_Package_Root :=
                       new Wide_String'(To_Wide_String
                             (Parameter (Parser => Parser)));
                  elsif Full_Switch (Parser => Parser) =
                          "-bound-package-suffix"
                  then
                     Ada2Java.Bound_Package_Suffix :=
                       new Wide_String'(To_Wide_String
                                          (Parameter (Parser => Parser)));
                  elsif Full_Switch (Parser => Parser) = "-trace-checks" then
                     Ada2Java.Trace_Checks := True;
                  elsif Full_Switch (Parser => Parser) =
                           "-trace-null-conversions"
                  then
                     Ada2Java.Trace_Null_Conversions := True;
                  elsif Full_Switch (Parser => Parser) = "-debug-glue" then
                     Ada2Java.Debug_Glue := True;
                  end if;

               end if;

            when others =>
               raise Parameter_Error;
         end case;
      end loop;

      Trace ("Scan parameters", Ada2Java.Debug);

      if not First_Pass then
         ASIS_UL.Compiler_Options.Process_cargs_Section (Parser);
      end if;

   exception
      when GNAT.Command_Line.Invalid_Switch =>
         Error ("invalid switch : " & Full_Switch (Parser => Parser));
         raise Parameter_Error;

      when GNAT.Command_Line.Invalid_Parameter =>
         Error ("Parameter missed for : -" & Full_Switch (Parser => Parser));
         raise Parameter_Error;

   end Scan_Arguments;

end Ada2Java.Projects;
