------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2014, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Exceptions; use Ada.Exceptions;

with Ada2Java.Utils; use Ada2Java.Utils;

with Ada2Java.Bound_Elements.Callbacks;
use Ada2Java.Bound_Elements.Callbacks;

with Ada2Java.Bound_Elements.Types;
use Ada2Java.Bound_Elements.Types;

with Ada2Java.Bound_Elements.References;
use Ada2Java.Bound_Elements.References;

with Ada.Characters.Conversions; use Ada.Characters;

package body Ada2Java.Bound_Elements.Subprograms.Data is

   Default_J_Object    : constant Dynamic_Expression :=
     JNI_Pckg & ".J_Null_Object";
   Default_J_Int_Array : constant Dynamic_Expression :=
     JNI_Pckg & ".J_Null_Int_J_Array";
   Default_J_Int       : constant Dynamic_Expression :=
     To_Dynamic_Expression ("0");
   Default_J_Float     : constant Dynamic_Expression :=
     To_Dynamic_Expression ("0.0");
   Default_J_Boolean   : constant Dynamic_Expression :=
     JNI_Pckg & ".J_Boolean (False)";
   Default_J_Char      : constant Dynamic_Expression := JNI_Pckg
     & ".J_Char (Ada.Characters.Conversions.To_Wide_Character (ASCII.NUL))";

   Exp_This : constant Dynamic_Expression :=
     To_Dynamic_Expression ("this");

   LongRef_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("com.adacore.ajis.LongRef");
   IntegerRef_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("com.adacore.ajis.IntegerRef");
   DoubleRef_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("com.adacore.ajis.DoubleRef");
   CharacterRef_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("com.adacore.ajis.CharacterRef");
   BooleanRef_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("com.adacore.ajis.BooleanRef");

   Long_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("long");
   Int_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("int");
   Double_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("double");
   Char_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("char");
   Boolean_Exp : constant Dynamic_Expression :=
     To_Dynamic_Expression ("boolean");

   Simple_Null_Access : constant Dynamic_Expression := To_Dynamic_Expression
     ("com.adacore.ajis.internal.ada.AdaAccess.simpleNullAccess ()");

   J_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("J");
   I_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("I");
   D_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("D");
   C_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("C");
   Z_Exp : constant Dynamic_Expression := To_Dynamic_Expression ("Z");

   procedure Generate_Ada_Array_Access
     (Handle           : not null access Kernel.Kernel_Record;
      Expression       : Wrapping_Expression;
      Type_View        : Simple_Type_View_Access;
      Access_Type_Name : Dynamic_Expression;
      Is_Constant      : Boolean);
   --  Extracts a pointer to array out of a J_Int_J_Array variable representing
   --  a wrapper to an access.

   function Get_Annot_Name (View : Simple_Type_Reference) return Wide_String;
   --  Return the name of the type used in the various functions, typically
   --  Integer or Long depending on the type size.

   function Get_JNI_Name
     (View : Simple_Type_Reference) return Dynamic_Expression;
   --  Return the JNI name of the type, typically J_Int or J_Long, depending
   --  on the type size.

   function Get_Null_Access_Reference
     (Handle : not null access Kernel.Kernel_Record;
      D      : Simple_Type_View_Access) return Dynamic_Expression;
   --  Return the reference to the java null access variable depending of the
   --  size of the pointer to the type (simple or fat pointer).

   procedure Insert_Trace
     (D     : Java_Data'Class;
      Label : Wide_String;
      To    : in out Dynamic_Expression);
   --  Inserts a trace in the code with the label given in parameter.
   --  Will display:
   --  <LABEL> ON "<subprogram name>"  ("<param name>"|RESULT)

   -------------------------------
   -- Generate_Ada_Array_Access --
   -------------------------------

   procedure Generate_Ada_Array_Access
     (Handle           : not null access Kernel.Kernel_Record;
      Expression       : Wrapping_Expression;
      Type_View        : Simple_Type_View_Access;
      Access_Type_Name : Dynamic_Expression;
      Is_Constant      : Boolean)
   is
      Conv : constant Type_Record :=
        Get_Or_Create_Conversion_Type (Handle, Type_View);

      Subtype_Id : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Access_Id : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Conv_Id     : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Result_Id : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Addr_Id   : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
   begin
      Append
        (Expression.Bloc.Declarations,
         New_Line & Addr_Id & " : constant System.Address := "
         &  Conv.Conversion_Package_Name.all
         & ".Get_Address (" & Expression.Expression & ");"
         & New_Line
         & "subtype " & Subtype_Id & " is "
         & Type_View.Full_Ada_Name
         & " (");

      for J in Type_View.Indexes'Range loop
         declare
            Bound_Number : constant Wide_String :=
              Ada.Characters.Conversions.To_Wide_String
                (Integer'Image (J));
            Trimed_Bound_Number : constant Wide_String :=
              Bound_Number
                (Bound_Number'First + 1 ..  Bound_Number'Last);
            Bound_Name   : constant Wide_String :=
              "Bound_" & Trimed_Bound_Number;
         begin
            if J > Type_View.Indexes'First then
               Append
                 (Expression.Bloc.Declarations,
                  ", ");
            end if;

            Append
              (Expression.Bloc.Declarations,
               Expression.Expression
               & "." & Bound_Name & "_Low .. "
               & Expression.Expression & "." & Bound_Name & "_High");
         end;
      end loop;

      Append
        (Expression.Bloc.Declarations,
         ");"
         & New_Line
         & "type " & Access_Id & " is access all " & Subtype_Id & ";"
         & New_Line
         & "package " & Conv_Id
         & " is new System.Address_To_Access_Conversions"
         & " (" & Subtype_Id & ");"
         & New_Line
         & Result_Id & " : " & Access_Type_Name & ";");

      Append
        (Expression.Bloc.Statements_Before,
         New_Line & "if " & AJIS_Pckg & ".""="" ("
         & Expression.Expression
         & ".Kind, " & AJIS_Pckg & ".Static) then"
         & New_Line (1)
         & Result_Id & " := "
         & Conv_Id & ".To_Pointer (" & Addr_Id & ").all'Unrestricted_Access;"
         & New_Line (-1) & "elsif " & AJIS_Pckg & ".""="" ("
         & Expression.Expression
         & ".Kind, " & AJIS_Pckg & ".General_Access) then"
         & New_Line (1) & Result_Id & " := " & Access_Type_Name & " ("
         & Expression.Expression & ".General_Pointer);"
         & New_Line (-1) & "else");

      if not Is_Constant then
         Append
           (Expression.Bloc.Statements_Before,
            New_Line (1)
            & "raise Standard.Constraint_Error with "
            & """Object stored is not access to constant."";");
      else
         Append
           (Expression.Bloc.Statements_Before,
            New_Line (1) & Result_Id & " := " & Access_Type_Name & " ("
            & Expression.Expression & ".Constant_Pointer);");
      end if;

      Append
        (Expression.Bloc.Statements_Before,
         New_Line (-1) & "end if;");

      Expression.Expression := Result_Id;

      Nest_Ada_Statements (Expression.Bloc);
   end Generate_Ada_Array_Access;

   --------------------
   -- Get_Annot_Name --
   --------------------

   function Get_Annot_Name
     (View : Simple_Type_Reference) return Wide_String is
   begin
      if View.Ref.Kind = Enumeration_Kind
        and then Ada2Java.Use_Java_1_5_Enums
      then
         return "Enum";
      elsif View.Ref.Size > 32 then
         return "Long";
      else
         return "Integer";
      end if;
   end Get_Annot_Name;

   function Get_JNI_Name
     (View : Simple_Type_Reference) return Dynamic_Expression is
   begin
      if View.Ref.Size > 32 then
         return JNI_Pckg & ".J_Long";
      else
         return JNI_Pckg & ".J_Int";
      end if;
   end Get_JNI_Name;

   -------------------------------
   -- Get_Null_Access_Reference --
   -------------------------------

   function Get_Null_Access_Reference
     (Handle : not null access Kernel.Kernel_Record;
      D      : Simple_Type_View_Access) return Dynamic_Expression
   is
      Real_Type : Simple_Type_View_Access;
   begin
      if D.Kind = Access_Kind then
         Real_Type := D.Target_Type.Ref;
      else
         Real_Type := D;
      end if;

      if Is_Unconstrained_Array (Real_Type) then
         declare
            Decl_Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
              (Handle, Simple_Element_View_Access (D.Target_Type.Ref));
         begin
            return Decl_Unit.Java_File.Full_Class_Name & ".nullArrayAccess ()";
         end;
      end if;

      return Simple_Null_Access;
   end Get_Null_Access_Reference;

   ------------------
   -- Insert_Trace --
   ------------------

   procedure Insert_Trace
     (D     : Java_Data'Class;
      Label : Wide_String;
      To    : in out Dynamic_Expression)
   is
   begin
      Append (To, New_Line &  "System.out.println (""" & Label & " ");

      if D.Decl_View.all in Simple_Profile_Data_View'Class then
         Append
           (To,
            "ON \"""
            & Simple_Profile_Data_View (D.Decl_View.all)
            .Enclosing_Sb.Name
            & "\"" (");
      else
         Append (To, "(");
      end if;

      if D.Decl_View.all in Parameter_Simple_View'Class then
         Append
           (To,
            "\""" & Parameter_Simple_View (D.Decl_View.all).Name
            & "\""");
      else
         Append (To, "RESULT");
      end if;

      Append (To, ")"");");
   end Insert_Trace;

   --------------------------
   -- Should_Be_Class_Wide --
   --------------------------

   function Should_Be_Class_Wide
     (D : Ada_Data'Class) return Boolean is
   begin
      if D.Simple_Type_View.Is_Class_Wide then
         --  Class wide types are bound into class wide types

         return True;
      elsif not
        ((D.Simple_Type_View.Ref.Kind = Access_Kind
          and then D.Simple_Type_View.Ref.Target_Type.Ref.Kind
          = Tagged_Record_Kind)
         or else (D.Simple_Type_View.Ref.Kind = Tagged_Record_Kind))
      then
         --  Non tagged types can't be bound into class wide types

         return False;
      elsif D.Decl_View.all not in Simple_Profile_Data_View'Class then
         --  Tagged non data types are bound into class wide types - in order
         --  to be able to store any kind of object

         return True;
      elsif Simple_Profile_Data_View
        (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
      then
         --  On real primitives, we are doing overridings at the Ada level -
         --  so we can't have any change, so types keep their signature.

         return False;
      elsif not Simple_Profile_Data_View (D.Decl_View.all).Is_Controlling then
         --  If this is not a controlling parameter, then we don't want to
         --  change anything either - we keep the constraints

         return False;
      else
         return True;
      end if;
   end Should_Be_Class_Wide;

   --------------------------
   -- Generate_Store_Check --
   --------------------------

   procedure Generate_Access_Checks
     (D              : in out Java_Data'Class;
      Expression     : Wrapping_Expression;
      Loc            : Check_Loc)
   is
      View : Simple_Profile_Data_View_Access;
      Indirect_Ref : Dynamic_Expression;

      Tmp_Name : constant Wide_String := Get_Unique_Id;

      Code_Node : Dynamic_Expression;
      Mode : constant Parameter_Mode_Type := D.Get_Mode;
   begin
      if D.Decl_View.all not in Simple_Profile_Data_View'Class
        or else Mode = Value_Native
      then
         return;
      end if;

      if Loc = Before then
         Code_Node := Expression.Bloc.Statements_Before;
      else
         Code_Node := Expression.Bloc.Statements_After;
      end if;

      View := Simple_Profile_Data_View_Access (D.Decl_View);

      case Mode is
         when Escapable_Reference =>
            if D.Decl_View.all in Parameter_Simple_View'Class
              and then Parameter_Simple_View (View.all).Mode /= In_Mode
            then
               Indirect_Ref := To_Dynamic_Expression (".getValue ()");
            else
               Indirect_Ref := To_Dynamic_Expression ("");
            end if;

            Append
              (Code_Node,
               New_Line & D.To_Type_Name & " " & Tmp_Name & " = "
               & Expression.Expression & ";");

            if Trace_Checks then
               Insert_Trace
                 (D,
                  "ESCAPABLE CHECK",
                  Expression.Bloc.Statements_Before);
            end if;

            Append
              (Code_Node,
               New_Line & "if (" & Tmp_Name
               & Indirect_Ref & " != null) {"
               & New_Line (1)
               & "com.adacore.ajis.internal.ada.Utils.checkEscapable ("
               & Tmp_Name & Indirect_Ref & ", ");

            if D.Decl_View.all in Parameter_Simple_View'Class then
               Append
                 (Code_Node,
                  """" & Parameter_Simple_View (View.all).Name & """");
            else
               Append
                 (Code_Node,
                  "null");
            end if;

            Append
              (Code_Node,
               ");" & New_Line (-1) & "}");

            Expression.Expression := To_Dynamic_Expression (Tmp_Name);
            Nest_Java_Statements (Expression.Bloc);

         when Value_Reference =>
            if View.all not in Parameter_Simple_View'Class
              or else not Parameter_Simple_View (View.all).Attached
            then
               Append
                 (Code_Node,
                  New_Line & D.To_Type_Name & " " & Tmp_Name & " = "
                  & Expression.Expression & ";");

               if Trace_Checks then
                  Insert_Trace
                    (D,
                     "NOT NULL CHECK",
                     Expression.Bloc.Statements_Before);
               end if;

               Append
                 (Code_Node,
                  New_Line & "if (" & Tmp_Name & " == null) {"
                  & New_Line (1)
                  & "throw new "
                  & "com.adacore.ajis.NativeException (""null not allowed");

               if D.Decl_View.all in Parameter_Simple_View'Class then
                  Append
                    (Code_Node,
                     " for " & Parameter_Simple_View (View.all).Name);
               elsif D.Decl_View.all in Simple_Profile_Data_View'Class then
                  Append
                    (Code_Node,
                     " for returned value");
               end if;

               Append
                 (Code_Node,
                  """);" & New_Line (-1) & "}");

               Expression.Expression := To_Dynamic_Expression (Tmp_Name);
               Nest_Java_Statements (Expression.Bloc);
            end if;

         when Value_Native | Static_Reference | Non_Escapable_Reference =>
            null;

      end case;
   end Generate_Access_Checks;

   --  ??? We should use more of the constants above, in order to minimize
   --  memory allocations.

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties)
   is
      pragma Unreferenced (D, Expression, Prop);
   begin
      null;
   end Create_Conversion_To_JNI;

   --------------
   -- Get_Mode --
   --------------

   function Get_Mode (D : Java_Data) return Parameter_Mode_Type is
   begin
      if D.Simple_Type_View.Ref.Kind in Unknown_Kind .. Enumeration_Kind then
         return Value_Native;
      end if;

      if D.Simple_Type_View.Ref.Kind = Access_Kind then
         if D.Simple_Type_View.Is_From_Unrestricted_Access then
            return Static_Reference;
         elsif D.Decl_View.all in Simple_Profile_Data_View'Class
           and then Simple_Profile_Data_View (D.Decl_View.all).Assume_Stored
         then
            return Escapable_Reference;
         else
            return Non_Escapable_Reference;
         end if;
      else
         return Value_Reference;
      end if;

   end Get_Mode;

   -------------------------
   -- Adjust_File_For_Use --
   -------------------------

   procedure Adjust_Unit_For_Use
     (D : in out Data_Record; Unit : Bound_Unit)
   is
      pragma Unreferenced (D, Unit);
   begin
      null;
   end Adjust_Unit_For_Use;

   -------------------
   -- Get_Type_View --
   -------------------

   function Get_Type_View
     (D : Data_Record) return Simple_Type_Reference is
   begin
      return D.Simple_Type_View;
   end Get_Type_View;

   -----------------------
   -- Create_Parameters --
   -----------------------

   function Create_Parameters
     (D    : Java_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access)
      return Dynamic_Expression
   is
      Complete_String : Dynamic_Expression := New_Dynamic_Expression;
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Attached
      then
         return To_Dynamic_Expression ("");
      else
         if Prof.Parameters_Written >= 2
           or else (Prof.Parameters_Written >= 1 and then not View.Is_Attached)
         then
            Append (Complete_String, ", ");
         end if;

         Append
           (Complete_String,
            To_Type_Name (Java_Data'Class (D)) & " " &
            To_Wide_String (D.Decl_View.Get_Java_Name));

         return Complete_String;
      end if;
   end Create_Parameters;

   ---------------------------
   -- Create_JNI_Parameters --
   ---------------------------

   function Create_JNI_Parameters
     (D    : Java_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression
   is
      pragma Unreferenced (View);

      Complete_String : Dynamic_Expression := New_Dynamic_Expression;
   begin
      if Prof.Parameters_Written >= 1 then
         Append (Complete_String, ", ");
      end if;

      Append
        (Complete_String,
         To_JNI_Type_Name (Java_Data'Class (D)) & " "
         & D.Decl_View.Get_Java_Name);

      return Complete_String;
   end Create_JNI_Parameters;

   -------------------------------
   -- To_Ada_Internal_Type_Name --
   -------------------------------

   function To_Ada_Internal_Type_Name
     (D : Ada_Data) return Dynamic_Expression is
   begin
      return To_Type_Name (Ada_Data'Class (D));
   end To_Ada_Internal_Type_Name;

   ----------------------------
   -- To_Qualified_Type_Name --
   ----------------------------

   function To_Qualified_Type_Name
     (D : Data_Record) return Dynamic_Expression is
   begin
      return To_Type_Name (Data_Record'Class (D)) & "'";
   end To_Qualified_Type_Name;

   -----------------------
   -- Create_Parameters --
   -----------------------

   function Create_Parameters
     (D    : Ada_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression
   is
      Complete_String : Dynamic_Expression := New_Dynamic_Expression;
   begin
      if Prof.Parameters_Written >= 1 or else View.High_Sb_With_Env then
         Append (Complete_String, "; ");
      else
         Append (Complete_String, " (");
      end if;

      Append
        (Complete_String,
         D.Decl_View.Get_Glue_Name & " : ");

      case Parameter_Simple_View (D.Decl_View.all).Mode is
         when In_Mode =>
            null;
         when Out_Mode =>
            Append (Complete_String, "out ");
         when In_Out_Mode =>
            Append (Complete_String, "in out ");
      end case;

      Append (Complete_String, To_Type_Name (Ada_Data'Class (D)));

      return Complete_String;
   end Create_Parameters;

   ---------------------------
   -- Create_JNI_Parameters --
   ---------------------------

   function Create_JNI_Parameters
     (D    : Ada_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression
   is
      pragma Unreferenced (Prof, View);

      Complete_String : Dynamic_Expression := New_Dynamic_Expression;
   begin
      Append
        (Complete_String,
         "; " & To_Wide_String (D.Decl_View.Get_Glue_Name)
         & " : " & To_JNI_Type_Name (Ada_Data'Class (D)));

      return Complete_String;
   end Create_JNI_Parameters;

   ------------------------
   -- Create_Bound_Data --
   ------------------------

   function Create_Bound_Data
     (View   : Simple_Object_View_Access;
      Unit   : Bound_Unit;
      Kernel : access Kernel_Record;
      Call_Convention : Call_Conventions) return Bound_Data
   is
      Returned_Data : Bound_Data;
      Simple_View   : constant Simple_Type_View_Access := View.Type_Of.Ref;
   begin
      if Simple_View.Kind = Access_Kind
        and then Simple_View.Target_Type /= Null_Type_Reference
        and then Simple_View.Target_Type.Ref.Kind = Access_Kind
      then
         Trace_With_Location
           ("cannot bind access to access type", Errors_And_Warnings);

         raise Silent_Not_Supported;
      end if;

      case Simple_View.Kind is
         when Enumeration_Kind =>

            if Ada2Java.Use_Java_1_5_Enums then
               Returned_Data.Java_Part := new Java_Enum_Data;
            else
               Returned_Data.Java_Part := new Java_Primitive_Data;
            end if;

            Returned_Data.Ada_Part := new Ada_Scalar_Data;

         when Generic_Integer_Kind
            | Generic_Float_Kind
            | Standard_Boolean_Kind
            | Standard_Character_Kind =>

            Returned_Data.Ada_Part := new Ada_Scalar_Data;
            Returned_Data.Java_Part := new Java_Primitive_Data;

         when Private_Kind | Record_Kind | Tagged_Record_Kind =>

            Returned_Data.Ada_Part := new Ada_Object_Data;
            Returned_Data.Java_Part := new Java_Reference_Data;

         when Access_Kind =>

            case Simple_View.Target_Type.Ref.Kind is
               when Subprogram_Kind =>
                  Returned_Data.Java_Part := new Java_Callback_Data;
                  Returned_Data.Ada_Part := new Ada_Callback_Data;

               when Generic_Float_Kind
                  | Generic_Integer_Kind
                  | Standard_Boolean_Kind
                  | Standard_Character_Kind
                  | Enumeration_Kind =>

                  Returned_Data.Ada_Part := new Ada_Pointer_To_Scalar_Data;
                  Returned_Data.Java_Part := new Java_Reference_Data;

               when others =>
                  Returned_Data.Ada_Part := new Ada_Pointer_To_Object_Data;
                  Returned_Data.Java_Part := new Java_Reference_Data;
            end case;

         when Array_Kind =>

            Returned_Data.Java_Part := new Java_Reference_Data;
            Returned_Data.Ada_Part := new Ada_Object_Data;

         when JNI_Kind =>

            Returned_Data.Java_Part := new Java_JNI_Data;
            Returned_Data.Ada_Part := new Ada_JNI_Data;

         when others =>

            raise Not_Supported with
               Simple_View.Kind'Img & " not supported.";
      end case;

      Returned_Data.Ada_Part.Simple_Type_View := View.Type_Of;
      Returned_Data.Java_Part.Simple_Type_View := View.Type_Of;

      Returned_Data.Ada_Part.Decl_View := View;
      Returned_Data.Java_Part.Decl_View := View;

      Returned_Data.Ada_Part.Kernel := Kernel;
      Returned_Data.Java_Part.Kernel := Kernel;

      Returned_Data.Ada_Part.Unit := Unit;
      Returned_Data.Java_Part.Unit := Unit;

      Returned_Data.Ada_Part.Call_Convention := Call_Convention;
      Returned_Data.Java_Part.Call_Convention := Call_Convention;

      return Returned_Data;

   exception
      when Silent_Not_Supported | Not_Supported =>
         raise;

      when E : others =>
         raise Ada2Java_Error with Exception_Information (E);
   end Create_Bound_Data;

   ------------------
   -- Has_Out_Mode --
   ------------------

   function Has_Out_Mode (D : Data_Record'Class) return Boolean is
   begin
      return D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode
        /= In_Mode;
   end Has_Out_Mode;

   -------------------------
   -- Java_Primitive_Data --
   -------------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Java_Primitive_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Integer_Kind | Enumeration_Kind =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return LongRef_Exp;
               else
                  return IntegerRef_Exp;
               end if;
            when Generic_Float_Kind =>
               return DoubleRef_Exp;
            when Standard_Character_Kind =>
               return CharacterRef_Exp;
            when Standard_Boolean_Kind =>
               return BooleanRef_Exp;
            when others =>
               null;
         end case;
      else
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Integer_Kind | Enumeration_Kind =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return Long_Exp;
               else
                  return Int_Exp;
               end if;
            when Generic_Float_Kind =>
               return Double_Exp;
            when Standard_Character_Kind =>
               return Char_Exp;
            when Standard_Boolean_Kind =>
               return Boolean_Exp;
            when others =>
               null;
         end case;
      end if;

      return To_Dynamic_Expression ("");
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Java_Primitive_Data) return Dynamic_Expression
   is
   begin
      return To_Type_Name (D);
   end To_JNI_Type_Name;

   --  Create_Converstion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Java_Primitive_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_From_JNI;

   --  Get_Mangled_Type  --

   function Get_JNI_Type (D : Java_Primitive_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Integer_Kind | Enumeration_Kind =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return To_Dynamic_Expression ("Lcom/adacore/ajis/LongRef;");
               else
                  return To_Dynamic_Expression
                    ("Lcom/adacore/ajis/IntegerRef;");
               end if;
            when Generic_Float_Kind =>
               return To_Dynamic_Expression ("Lcom/adacore/ajis/DoubleRef;");
            when Standard_Character_Kind =>
               return To_Dynamic_Expression
                 ("Lcom/adacore/ajis/CharacterRef;");
            when Standard_Boolean_Kind =>
               return To_Dynamic_Expression ("Lcom/adacore/ajis/BooleanRef;");
            when others =>
               null;
         end case;
      else
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Integer_Kind | Enumeration_Kind =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return J_Exp;
               else
                  return I_Exp;
               end if;
            when Generic_Float_Kind =>
               return D_Exp;
            when Standard_Character_Kind =>
               return C_Exp;
            when Standard_Boolean_Kind =>
               return Z_Exp;
            when others =>
               null;
         end case;
      end if;

      return To_Dynamic_Expression ("");
   end Get_JNI_Type;

   --  Adjust_File_For_Use  --

   procedure Adjust_Unit_For_Use
     (D : in out Java_Primitive_Data; Unit : Bound_Unit) is
   begin
      null;
   end Adjust_Unit_For_Use;

   -------------------------
   -- Java_Reference_Data --
   -------------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Java_Reference_Data) return Dynamic_Expression is
   begin
      if D.Simple_Type_View.Pass_By_Address then
         return To_Dynamic_Expression ("int []");
      elsif D.Simple_Type_View.Ref.Kind /= Access_Kind then
         return D.Simple_Type_View.Ref.Full_Java_Name;
      else
         if D.Decl_View.all in Parameter_Simple_View'Class
           and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
         then
            return D.Simple_Type_View.Ref.Target_Type.Ref.Full_Java_Name
              & ".Ref";
         else
            return D.Simple_Type_View.Ref.Target_Type.Ref.Full_Java_Name;
         end if;
      end if;
   end To_Type_Name;

   -- Adjust_File_For_Use --

   procedure Adjust_Unit_For_Use
     (D : in out Java_Reference_Data; Unit : Bound_Unit)
   is
      pragma Unreferenced (Unit);
   begin
      if D.Simple_Type_View.Ref.Kind /= Access_Kind then
         Add_Bound_Element
           (Kernel  => D.Kernel,
            Element => Simple_Element_View_Access (D.Simple_Type_View.Ref));
      else
         Add_Bound_Element
           (Kernel  => D.Kernel,
            Element => Simple_Element_View_Access
              (D.Simple_Type_View.Ref.Target_Type.Ref));
      end if;
   end Adjust_Unit_For_Use;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Java_Reference_Data) return Dynamic_Expression
   is
      Target_Type : Simple_Type_View_Access;
   begin
      if D.Simple_Type_View.Ref.Kind = Access_Kind then
         Target_Type := D.Simple_Type_View.Ref.Target_Type.Ref;
      else
         Target_Type := D.Simple_Type_View.Ref;
      end if;

      if D.Simple_Type_View.Pass_By_Address then
         return To_Dynamic_Expression ("int []");
      elsif D.Simple_Type_View.Ref.Kind = Access_Kind
        and then D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         return D.Simple_Type_View.Ref.Target_Type.Ref.Full_Java_Name
           & ".Ref";
      elsif Target_Type.Kind = Tagged_Record_Kind then
         return Target_Type.Full_Java_Name;
      else
         return To_Dynamic_Expression ("int []");
      end if;
   end To_JNI_Type_Name;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Reference_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties)
   is
      pragma Unreferenced (Prop);
      Target_Type : Simple_Type_View_Access;
   begin
      if D.Simple_Type_View.Ref.Kind = Access_Kind then
         Target_Type := D.Simple_Type_View.Ref.Target_Type.Ref;
      else
         Target_Type := D.Simple_Type_View.Ref;
      end if;

      if D.Simple_Type_View.Ref.Kind = Access_Kind
        and then D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         null;
      else
         if Target_Type.Kind /= Tagged_Record_Kind then
            declare
               Addr_Id : constant Wide_String := Get_Unique_Id;
               Obj_Id : constant Wide_String := Get_Unique_Id;
            begin
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & "int [] "
                  & Addr_Id & ";");
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & D.To_Type_Name
                  & " " & Obj_Id & " = " & Expression.Expression & ";");

               if D.Get_Mode /= Value_Reference then
                  if Ada2Java.Trace_Null_Conversions then
                     Insert_Trace
                       (D,
                        "NULL CONVERSION",
                        Expression.Bloc.Statements_Before);
                  end if;

                  Append
                    (Expression.Bloc.Statements_Before,
                     New_Line
                     & "if (" & Obj_Id & " == null) {"
                     & New_Line (1) & Addr_Id
                     & " = "
                     & Get_Null_Access_Reference
                       (D.Kernel, D.Simple_Type_View.Ref) & ";"
                     & New_Line (-1) & "} else {"
                     & New_Line (1) & Addr_Id
                     & " = ((com.adacore.ajis.internal.ada.AdaProxy) "
                     & Obj_Id & ").getAccess ();"
                     & New_Line (-1) & "}");

               else
                  Append
                    (Expression.Bloc.Statements_Before,
                     New_Line & Addr_Id
                     & " = ((com.adacore.ajis.internal.ada.AdaProxy) "
                     & Obj_Id & ").getAccess ();");
               end if;

               Expression.Expression := To_Dynamic_Expression (Addr_Id);
               Nest_Java_Statements (Expression.Bloc);
            end;
         end if;
      end if;

      if not D.Simple_Type_View.Pass_By_Address
        and then Target_Type.Kind /= Tagged_Record_Kind
        and then D.Simple_Type_View.Ref.Kind = Access_Kind
        and then D.Has_Out_Mode
      then
         declare
            Tmp : Dynamic_Expression := To_Dynamic_Expression (Get_Unique_Id);
         begin
            Append
              (Expression.Bloc.Declarations,
               New_Line & D.To_Type_Name & " "
               & Tmp & " = " & Expression.Expression & ";");
            Expression.Expression := Tmp;

            if D.Decl_View.all in Parameter_Simple_View'Class
              and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
            then
               Tmp := Tmp & ".getValue ()";
            end if;

            if D.Get_Mode = Static_Reference then
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Tmp
                  & ".myAllocator = "
                  & "com.adacore.ajis.IProxy.Allocator.STATIC;");
            else
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & "if (" & Tmp & " != null) {"
                  & New_Line (1) & Tmp
                  & ".myAllocator = "
                  & "com.adacore.ajis.IProxy.Allocator.DYNAMIC;"
                  & New_Line (-1) & "}");
            end if;
         end;
      end if;
   end Create_Conversion_To_JNI;

   --  Create_Converstion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Java_Reference_Data;
      Expression : Wrapping_Expression)
   is
      Input : constant Dynamic_Expression := Expression.Expression;
      Target_Type : Simple_Type_View_Access;
   begin
      if D.Simple_Type_View.Ref.Kind = Access_Kind then
         Target_Type := D.Simple_Type_View.Ref.Target_Type.Ref;
      else
         Target_Type := D.Simple_Type_View.Ref;
      end if;

      if D.Simple_Type_View.Ref.Kind = Access_Kind
        and then D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         null;
      elsif Target_Type.Kind = Tagged_Record_Kind
        or else D.Simple_Type_View.Pass_By_Address
      then
         --  We don't have any more transformation to do - just do the
         --  conversion to ensure that we've got the right type.

         Expression.Expression :=
           "((" & D.To_Type_Name & ") " & Input & ")";
      else
         declare
            Addr_Id : constant Wide_String := Get_Unique_Id;
            Obj_Id : constant Wide_String := Get_Unique_Id;
         begin
            Append
              (Expression.Bloc.Declarations,
               New_Line & "int [] "
               & Addr_Id & " = " & Expression.Expression & ";");
            Append
              (Expression.Bloc.Declarations,
               New_Line & D.To_Type_Name
               & " " & Obj_Id & ";");

            if D.Get_Mode /= Value_Reference then
               if Ada2Java.Trace_Null_Conversions then
                  Insert_Trace
                    (D,
                     "NULL CONVERSION",
                     Expression.Bloc.Statements_Before);
               end if;

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line
                  & "if (com.adacore.ajis.internal.ada.AdaAccess.isNull ("
                  & Addr_Id & ")) {"
                  & New_Line (1) & Obj_Id & " = null;"
                  & New_Line (-1) & "} else {"
                  & New_Line (1) & Obj_Id & " = new " & D.To_Type_Name & " ("
                  & "new com.adacore.ajis.internal.ada.AdaAccess ("
                  & Addr_Id & "));");
            else
               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line & Obj_Id & " = new " & D.To_Type_Name & " ("
                  & "new com.adacore.ajis.internal.ada.AdaAccess ("
                  & Addr_Id & "));");
            end if;

            if D.Simple_Type_View.Ref.Kind /= Access_Kind
              and then D.Decl_View.all not in Parameter_Simple_View'Class
            then
               --  In this case, we made a copy of the object, so we need to
               --  set the proxy so that it contains an address

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line & Obj_Id
                  & ".myOwner = com.adacore.ajis.IProxy.Owner.PROXY;");
            end if;

            if D.Get_Mode /= Value_Reference then
               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line (-1) & "}");
            end if;

            Expression.Expression := To_Dynamic_Expression (Obj_Id);

            Nest_Java_Statements (Expression.Bloc);
         end;
      end if;

      if not D.Simple_Type_View.Pass_By_Address
        and then Target_Type.Kind /= Tagged_Record_Kind
      then
         declare
            Tmp : Dynamic_Expression := To_Dynamic_Expression (Get_Unique_Id);
         begin
            Append
              (Expression.Bloc.Declarations,
               New_Line & D.To_Type_Name & " "
               & Tmp & " = " & Expression.Expression & ";");
            Expression.Expression := Tmp;

            if D.Decl_View.all in Parameter_Simple_View'Class
              and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
              and then D.Simple_Type_View.Ref.Kind = Access_Kind
            then
               Tmp := Tmp & ".getValue ()";
            end if;

            if D.Get_Mode = Static_Reference
              or else
                (D.Simple_Type_View.Ref.Kind /= Access_Kind
                 and then D.Decl_View.all in Parameter_Simple_View'Class)
            then
               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line & Tmp
                  & ".myAllocator = "
                  & "com.adacore.ajis.IProxy.Allocator.STATIC;");
            else
               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line & "if (" & Tmp & " != null) {"
                  & New_Line (1) & Tmp
                  & ".myAllocator = "
                  & "com.adacore.ajis.IProxy.Allocator.DYNAMIC;"
                  & New_Line (-1) & "}");
            end if;

            Nest_Java_Statements (Expression.Bloc);
         end;
      end if;
   end Create_Conversion_From_JNI;

   --  Get_JNI_Type  --

   function Get_JNI_Type (D : Java_Reference_Data) return Dynamic_Expression is
      Target_Type : Simple_Type_View_Access;
   begin
      if D.Simple_Type_View.Ref.Kind = Access_Kind then
         Target_Type := D.Simple_Type_View.Ref.Target_Type.Ref;
      else
         Target_Type := D.Simple_Type_View.Ref;
      end if;

      if D.Decl_View.Type_Of.Pass_By_Address then
         return To_Dynamic_Expression ("[I");
      elsif D.Simple_Type_View.Ref.Kind = Access_Kind
        and then D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         return To_Dynamic_Expression
           ("L" & Replace_Dots_By_Slashes
              (To_Wide_String
                 (D.Simple_Type_View.Ref.Target_Type.Ref.Full_Java_Name)
               & "$Ref")
            & ";");
      elsif Target_Type.Kind = Tagged_Record_Kind then
         return To_Dynamic_Expression
           ("L" & Replace_Dots_By_Slashes
              (To_Wide_String (Target_Type.Full_Java_Name))
            & ";");
      else
         return To_Dynamic_Expression ("[I");
      end if;
   end Get_JNI_Type;

   ------------------------
   -- Java_Callback_Data --
   ------------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Java_Callback_Data) return Dynamic_Expression is
   begin
      Add_Dependency_On_Type
        (D.Kernel,
         D.Unit.Root_Unit.Ada_Spec_File,
         D.Simple_Type_View.Ref);

      return D.Simple_Type_View.Ref.Full_Java_Name;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Java_Callback_Data) return Dynamic_Expression
   is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class and then
        Parameter_Simple_View_Access (D.Decl_View).Is_Controlling
      then
         return To_Dynamic_Expression ("int []");
      else
         return To_Type_Name (D);
      end if;
   end To_JNI_Type_Name;

   --  Adjust_File_For_Use  --

   procedure Adjust_Unit_For_Use
     (D : in out Java_Callback_Data; Unit : Bound_Unit) is
   begin
      Add_Import_Element
        (Unit.Java_File,
         To_Wide_String (D.Simple_Type_View.Ref.Full_Java_Name));
   end Adjust_Unit_For_Use;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Callback_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties) is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class and then
        Parameter_Simple_View_Access (D.Decl_View).Is_Controlling
      then
         if Prop.Java_Call_Parameters_Written = 0 then
            Expression.Expression := Exp_This;
         end if;

         Expression.Expression := Expression.Expression & ".adaAccess";
      end if;
   end Create_Conversion_To_JNI;

   --  Create_Converstion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Java_Callback_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_From_JNI;

   --  Get_Mangled_Type  --

   function Get_JNI_Type (D : Java_Callback_Data) return Dynamic_Expression is
      Decl_Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (D.Kernel, Simple_Element_View_Access (D.Simple_Type_View.Ref));
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class and then
        Parameter_Simple_View_Access (D.Decl_View).Is_Controlling
      then
         return To_Dynamic_Expression ("[I");
      else
         return To_Dynamic_Expression
           ("L" &  Replace_Dots_By_Slashes
              (To_Wide_String (Decl_Unit.Java_File.Full_Class_Name)) & ";");
      end if;
   end Get_JNI_Type;

   -------------------
   -- Java_JNI_Data --
   -------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Java_JNI_Data) return Dynamic_Expression
   is
   begin
      return D.Simple_Type_View.Ref.Full_Java_Name;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Java_JNI_Data) return Dynamic_Expression
   is
   begin
      return D.Simple_Type_View.Ref.Full_Java_Name;
   end To_JNI_Type_Name;

   --  Create_Converstion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Java_JNI_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties)
   is
      pragma Unreferenced (Prop);
   begin
      null;
   end Create_Conversion_To_JNI;

   --  Create_Conversion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Java_JNI_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_From_JNI;

   --  Get_JNI_Type  --

   function Get_JNI_Type (D : Java_JNI_Data) return Dynamic_Expression is
   begin
      return "L" & To_Dynamic_Expression
        (Replace_Dots_By_Slashes (To_Wide_String (D.To_Type_Name))) & ";";
   end Get_JNI_Type;

   --------------------
   -- Java_Enum_Data --
   --------------------

   function To_Type_Name (D : Java_Enum_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         return D.Simple_Type_View.Ref.Full_Java_Name & ".Ref";
      else
         return D.Simple_Type_View.Ref.Full_Java_Name;
      end if;
   end To_Type_Name;

   function To_JNI_Type_Name
     (D : Java_Enum_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         return D.Simple_Type_View.Ref.Full_Java_Name & ".Ref";
      else
         return To_Dynamic_Expression ("int");
      end if;
   end To_JNI_Type_Name;

   procedure Create_Conversion_From_JNI
     (D          : in out Java_Enum_Data;
      Expression : Wrapping_Expression) is
   begin
      --  Forces the creation of the bound element even if it's not in the
      --  explicit bound units.
      Add_Bound_Element
        (D.Kernel,
         Simple_Element_View_Access (D.Simple_Type_View.Ref));

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         null;
      else
         Expression.Expression := D.To_Type_Name
           & ".class.getEnumConstants() [" & Expression.Expression & "]";
      end if;
   end Create_Conversion_From_JNI;

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Enum_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties)
   is
      pragma Unreferenced (Prop);
   begin
      --  Forces the creation of the bound element even if it's not in the
      --  explicit bound units.
      Add_Bound_Element
        (D.Kernel,
         Simple_Element_View_Access (D.Simple_Type_View.Ref));

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         null;
      else
         Expression.Expression := Expression.Expression & ".ordinal ()";
      end if;
   end Create_Conversion_To_JNI;

   function Get_JNI_Type (D : Java_Enum_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         return To_Dynamic_Expression
           ("L" & Replace_Dots_By_Slashes
              (To_Wide_String (D.Simple_Type_View.Ref.Full_Java_Name))
            & "$Ref;");
      else
         return To_Dynamic_Expression ("I");
      end if;
   end Get_JNI_Type;

   procedure Adjust_Unit_For_Use
     (D : in out Java_Enum_Data; Unit : Bound_Unit)
   is
   begin
      null;
   end Adjust_Unit_For_Use;

   ---------------------
   -- Ada_Scalar_Data --
   ---------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Ada_Scalar_Data) return Dynamic_Expression is
   begin
      if To_Wide_String (D.Simple_Type_View.Ref.Full_Ada_Name) = "" then
         if D.Simple_Type_View.Ref.Size > 32 then
            return To_Dynamic_Expression ("Standard.Long_Long_Integer");
         else
            return To_Dynamic_Expression ("Standard.Integer");
         end if;
      else
         if Simple_Profile_Data_View
           (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
         then
            return D.Simple_Type_View.Initial_Subtype_Name;
         else
            Add_Dependency_On_Type
              (D.Kernel,
               D.Unit.Root_Unit.Ada_Spec_File,
               D.Simple_Type_View.Ref);

            return D.Simple_Type_View.Ref.Full_Ada_Name;
         end if;
      end if;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Ada_Scalar_Data) return Dynamic_Expression
   is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         return JNI_Pckg & ".J_Object";
      else
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Integer_Kind | Enumeration_Kind =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return JNI_Pckg & ".J_Long";
               else
                  return JNI_Pckg & ".J_Int";
               end if;
            when Generic_Float_Kind =>
               return JNI_Pckg & ".J_Double";
            when Standard_Boolean_Kind =>
               return JNI_Pckg & ".J_Boolean";
            when Standard_Character_Kind =>
               return JNI_Pckg & ".J_Char";
            when others =>
               null;
         end case;
      end if;

      return To_Dynamic_Expression ("");
   end To_JNI_Type_Name;

   --  Create_Conversion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Scalar_Data;
      Expression : Wrapping_Expression)
   is
      Input : Dynamic_Expression;
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         Input := Expression.Expression;
         D.Initial_Input := Expression.Expression;
         Expression.Expression := To_Dynamic_Expression (Get_Unique_Id);

         Append
           (Expression.Bloc.Declarations,
            New_Line & Expression.Expression & " : "
            & To_Type_Name (D));

         if Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode then
            Append
              (Expression.Bloc.Declarations, " := ");

            case D.Simple_Type_View.Ref.Kind is
               when Enumeration_Kind =>
                  Append
                    (Expression.Bloc.Declarations,
                     To_Type_Name (D)
                     & "'Val (" & AJIS_Pckg & ".Java_Get_"
                     & Get_Annot_Name (D.Simple_Type_View)
                     & "_Value (" & Env_Parameter_Name
                     & ", " & Input & "))");

               when Generic_Integer_Kind =>
                  Append
                    (Expression.Bloc.Declarations,
                     To_Type_Name (D)
                     & " (" & AJIS_Pckg & ".Java_Get_"
                     & Get_Annot_Name (D.Simple_Type_View)
                     & "_Value (" & Env_Parameter_Name
                     & ", " & Input & "))");

               when Standard_Boolean_Kind =>
                  Append
                    (Expression.Bloc.Declarations,
                     To_Type_Name (D)
                     & " (" & AJIS_Pckg & ".Java_Get_Boolean_Value ("
                     & Env_Parameter_Name
                     & ", " & Input & "))");

               when Generic_Float_Kind =>
                  Append
                    (Expression.Bloc.Declarations,
                     To_Type_Name (D)
                     & " (" & AJIS_Pckg & ".Java_Get_Double_Value ("
                     & Env_Parameter_Name
                     & ", " & Input & "))");

               when Standard_Character_Kind =>
                  Append
                    (Expression.Bloc.Declarations,
                     To_Type_Name (D)
                     & " (Ada.Characters.Conversions.To_Character"
                     & " (Wide_Character (" & AJIS_Pckg
                     & ".Java_Get_Character_Value ("
                     & Env_Parameter_Name & ", "
                     & Input & "))))");

               when others =>
                  null;

            end case;
         end if;

         Append (Expression.Bloc.Declarations, ";");

         case D.Simple_Type_View.Ref.Kind is
            when Enumeration_Kind =>
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & AJIS_Pckg & ".Java_Set_"
                  & Get_Annot_Name (D.Simple_Type_View)
                  & "_Value (" & Env_Parameter_Name
                  & ", "
                  & Input & ", " & Get_JNI_Name (D.Simple_Type_View) & " ("
                  & To_Type_Name (D) & "'Pos ("
                  & Expression.Expression & ")));");

            when Standard_Boolean_Kind =>
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & AJIS_Pckg
                  & ".Java_Set_Boolean_Value (" & Env_Parameter_Name
                  & ", "
                  & Input & ", " & JNI_Pckg & ".J_Boolean ("
                  & Expression.Expression & "));");

            when Generic_Integer_Kind =>
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line
                  & AJIS_Pckg & ".Java_Set_"
                  & Get_Annot_Name (D.Simple_Type_View)
                  & "_Value (" & Env_Parameter_Name & ", "
                  & Input & ", " & Get_JNI_Name (D.Simple_Type_View)
                  & " (" & Expression.Expression & "));");

            when Generic_Float_Kind =>
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line
                  & AJIS_Pckg & ".Java_Set_Double_Value ("
                  & Env_Parameter_Name & ", "
                  & Input & ", " & JNI_Pckg
                  & ".J_Double (" & Expression.Expression & "));");

            when Standard_Character_Kind =>
               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & AJIS_Pckg & ".Java_Set_Character_Value ("
                  & Env_Parameter_Name & ", "
                  & Input
                  & ", " & JNI_Pckg
                  & ".J_Char (Ada.Characters.Conversions.To_Wide_Character ("
                  & Expression.Expression & ")));");

            when others =>
               null;

         end case;
      else
         case D.Simple_Type_View.Ref.Kind is
            when Generic_Float_Kind
               | Generic_Integer_Kind
               | Standard_Boolean_Kind =>

               Expression.Expression :=
                 To_Type_Name (D) & " (" & Expression.Expression & ")";

            when Standard_Character_Kind =>

               Expression.Expression :=
                 To_Type_Name (D)
                 & " (Ada.Characters.Conversions.To_Character"
                 & " (Wide_Character ("
                 & Expression.Expression & ")))";

            when Enumeration_Kind  =>

               Expression.Expression :=
                 To_Type_Name (D) & "'Val (" & Expression.Expression & ")";

            when others =>
               null;

         end case;
      end if;
   end Create_Conversion_From_JNI;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Scalar_Data;
      Expression : Wrapping_Expression)
   is
      Tmp_Obj : constant Wide_String := Get_Unique_Id;
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode)
      then
         case D.Simple_Type_View.Ref.Kind is
            when Standard_Character_Kind =>
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Tmp_Obj & " : " & JNI_Pckg & ".J_Object := "
                  & AJIS_Pckg & ".New_Java_Character_Wrapper ("
                  & Env_Parameter_Name);

               if Parameter_Simple_View (D.Decl_View.all).Mode
                 = In_Out_Mode
               then
                  Append (Expression.Bloc.Declarations, ", "
                          & JNI_Pckg & ".J_Char ("
                          & "Ada.Characters.Conversions.To_Wide_Character"
                          & " (" & Expression.Expression & ")));");
               else
                  Append (Expression.Bloc.Declarations, ");");
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Expression.Expression & " := "
                  & "Ada.Characters.Conversions.To_Character"
                  & " (Wide_Character (" & AJIS_Pckg
                  & ".Java_Get_Character_Value ("
                  & Env_Parameter_Name & ", "
                  & Tmp_Obj & ")));");

            when Enumeration_Kind =>
               if Ada2Java.Use_Java_1_5_Enums then
                  declare
                     Ref_Bound : constant Bound_Element := Get_Bound_Element
                       (D.Kernel,
                        Simple_Element_View_Access
                          (D.Simple_Type_View.Ref.Wrapper));

                     Ref_Id : constant Dynamic_Expression :=
                       Get_JNI_Constructor
                         (Bound_Reference (Ref_Bound.all));
                  begin
                     Append
                       (Expression.Bloc.Declarations,
                        New_Line & Tmp_Obj & "_Args : " & JNI_Pckg
                        & ".J_Value_Array (1 .. 0);"
                        & New_Line & Tmp_Obj & " : " & JNI_Pckg
                        & ".J_Object := "
                        & JNI_Pckg & ".New_Object_A ("
                        & Env_Parameter_Name
                        & ", " & AJIS_Pckg & ".Get_Class ("
                        & Ref_Id & "), "
                        & AJIS_Pckg & ".Get_Id (" & Ref_Id & "), "
                        & Tmp_Obj & "_Args);");

                     if Parameter_Simple_View (D.Decl_View.all).Mode
                       = In_Out_Mode
                     then
                        Append
                          (Expression.Bloc.Statements_Before,
                           New_Line
                           & AJIS_Pckg & ".Java_Set_Enum_Value ("
                           & Env_Parameter_Name & ", "
                           & Tmp_Obj & ", " & JNI_Pckg & ".J_Int ("
                           & D.To_Type_Name & "'Pos ("
                           & Expression.Expression & ")));");

                        Nest_Ada_Statements (Expression.Bloc);
                     end if;
                  end;
               else
                  Append
                    (Expression.Bloc.Declarations,
                     New_Line & Tmp_Obj & " : " & JNI_Pckg & ".J_Object := "
                     & AJIS_Pckg & ".New_Java_"
                     & Get_Annot_Name (D.Simple_Type_View)
                     & "_Wrapper (" & Env_Parameter_Name);

                  if Parameter_Simple_View (D.Decl_View.all).Mode
                    = In_Out_Mode
                  then
                     Append
                       (Expression.Bloc.Declarations,
                        ", " & Get_JNI_Name (D.Simple_Type_View)  & " ("
                        & To_Type_Name (D)
                        & "'Pos (" & Expression.Expression & ")));");
                  else
                     Append (Expression.Bloc.Declarations, ");");
                  end if;
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Expression.Expression & " := "
                  & D.To_Type_Name
                  & "'Val (" & AJIS_Pckg & ".Java_Get_"
                  & Get_Annot_Name (D.Simple_Type_View)
                  & "_Value (" & Env_Parameter_Name
                  &  ", "
                  & Tmp_Obj & "));");

            when Generic_Float_Kind =>
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Tmp_Obj & " : " & JNI_Pckg & ".J_Object := "
                  & AJIS_Pckg & ".New_Java_Double_Wrapper ("
                  & Env_Parameter_Name);

               if Parameter_Simple_View (D.Decl_View.all).Mode
                 = In_Out_Mode
               then
                  Append
                    (Expression.Bloc.Declarations,
                     ", " & JNI_Pckg & ".J_Double ("
                     & Expression.Expression & "));");
               else
                  Append (Expression.Bloc.Declarations, ");");
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Expression.Expression & " := "
                  & D.To_Type_Name
                  & " (" & AJIS_Pckg & ".Java_Get_Double_Value ("
                  & Env_Parameter_Name & ", "
                  & Tmp_Obj & "));");

            when Standard_Boolean_Kind =>
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Tmp_Obj & " : " & JNI_Pckg & ".J_Object := "
                  & AJIS_Pckg & ".New_Java_Boolean_Wrapper ("
                  & Env_Parameter_Name);

               if Parameter_Simple_View (D.Decl_View.all).Mode
                 = In_Out_Mode
               then
                  Append
                    (Expression.Bloc.Declarations,
                     ", " & JNI_Pckg & ".J_Boolean ("
                     & Expression.Expression & "));");
               else
                  Append (Expression.Bloc.Declarations, ");");
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Expression.Expression & " := "
                  & D.To_Type_Name
                  & " (" & AJIS_Pckg & ".Java_Get_Boolean_Value ("
                  & Env_Parameter_Name & ", "
                  & Tmp_Obj & "));");

            when Generic_Integer_Kind =>
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Tmp_Obj & " : " & JNI_Pckg & ".J_Object := "
                  & AJIS_Pckg & ".New_Java_"
                  & Get_Annot_Name (D.Simple_Type_View)
                  & "_Wrapper (" & Env_Parameter_Name);

               if Parameter_Simple_View (D.Decl_View.all).Mode
                 = In_Out_Mode
               then
                  Append
                    (Expression.Bloc.Declarations,
                     ", " & Get_JNI_Name (D.Simple_Type_View) & " ("
                     & Expression.Expression & "));");
               else
                  Append
                    (Expression.Bloc.Declarations, ");");
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Expression.Expression & " := "
                  & D.To_Type_Name
                  & " (" & AJIS_Pckg & ".Java_Get_"
                  & Get_Annot_Name (D.Simple_Type_View) & "_Value ("
                  & Env_Parameter_Name & ", "
                  & Tmp_Obj & "));");

            when others =>
               raise Not_Supported;

         end case;

         Expression.Expression := To_Dynamic_Expression (Tmp_Obj);
      else
         case D.Simple_Type_View.Ref.Kind is
            when Standard_Character_Kind =>
               Expression.Expression :=
                 To_JNI_Type_Name (D)
                 & " (Ada.Characters.Conversions.To_Wide_Character"
                 & " (" & Expression.Expression & "))";

            when Enumeration_Kind =>
               Expression.Expression :=
                 To_JNI_Type_Name (D)
                 & " (" & To_Type_Name (D)
                 & "'Pos (" & Expression.Expression & "))";

            when others =>
               Expression.Expression :=
                 To_JNI_Type_Name (D) & " (" & Expression.Expression & ")";
         end case;
      end if;
   end Create_Conversion_To_JNI;

   -- Get_JNI_Call --

   function Get_JNI_Call (D : Ada_Scalar_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         if  D.Simple_Type_View.Ref.Size > 32 then
            return JNI_Pckg & ".Call_Long_Method_A";
         else
            return JNI_Pckg & ".Call_Int_Method_A";
         end if;
      else
         case D.Simple_Type_View.Ref.Kind is
            when Standard_Character_Kind =>
               return JNI_Pckg & ".Call_Char_Method_A";

            when Generic_Float_Kind =>
               return JNI_Pckg & ".Call_Double_Method_A";

            when Standard_Boolean_Kind =>
               return JNI_Pckg & ".Call_Boolean_Method_A";

            when Generic_Integer_Kind | Enumeration_Kind  =>
               if D.Simple_Type_View.Ref.Size > 32 then
                  return JNI_Pckg & ".Call_Long_Method_A";
               else
                  return JNI_Pckg & ".Call_Int_Method_A";
               end if;

            when others =>
               return To_Dynamic_Expression ("");

         end case;
      end if;
   end Get_JNI_Call;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_Scalar_Data) return Dynamic_Expression is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
           or else Parameter_Simple_View (D.Decl_View.all).Mode = In_Out_Mode)
      then
         return Default_J_Object;
      else
         case D.Simple_Type_View.Ref.Kind is
            when Standard_Character_Kind =>
               return Default_J_Char;

            when Generic_Float_Kind =>
               return Default_J_Float;

            when Standard_Boolean_Kind =>
               return Default_J_Boolean;

            when Generic_Integer_Kind | Enumeration_Kind  =>
               return Default_J_Int;

            when others =>
               raise Not_Supported;

         end case;
      end if;
   end Get_Default_JNI_Value;

   --  Adjust_Unit_For_Use  --

   procedure Adjust_Unit_For_Use
     (D : in out Ada_Scalar_Data; Unit : Bound_Unit)
   is
   begin
      if D.Simple_Type_View.Ref.Kind = Enumeration_Kind then
         Add_Dependency_On_Type
           (D.Kernel, Unit.Ada_Body_File, D.Simple_Type_View.Ref);
      end if;
   end Adjust_Unit_For_Use;

   ---------------------
   -- Ada_Object_Data --
   ---------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Ada_Object_Data) return Dynamic_Expression is
   begin
      Add_Dependency_On_Type
        (D.Kernel,
         D.Unit.Ada_Spec_File,
         D.Simple_Type_View.Ref);

      if Should_Be_Class_Wide (D) then
         if Simple_Profile_Data_View
           (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
         then
            return D.Simple_Type_View.Initial_Subtype_Name & "'Class";
         else
            return D.Simple_Type_View.Ref.Full_Ada_Name & "'Class";
         end if;
      else
         if Simple_Profile_Data_View
           (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
         then
            return D.Simple_Type_View.Initial_Subtype_Name;
         else
            return D.Simple_Type_View.Ref.Full_Ada_Name;
         end if;
      end if;
   end To_Type_Name;

   --  To_Qualified_Type_Name

   function To_Qualified_Type_Name
     (D : Ada_Object_Data) return Dynamic_Expression is
   begin
      Add_Dependency_On_Type
        (D.Kernel,
         D.Unit.Ada_Spec_File,
         D.Simple_Type_View.Ref);

      if D.Simple_Type_View.Is_Class_Wide then
         if Simple_Profile_Data_View
           (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
         then
            return D.Simple_Type_View.Initial_Subtype_Name & "'Class'";
         else
            return D.Simple_Type_View.Ref.Full_Ada_Name & "'Class'";
         end if;
      else
         if Simple_Profile_Data_View
           (D.Decl_View.all).Enclosing_Sb.Is_Real_Primitive
         then
            return D.Simple_Type_View.Initial_Subtype_Name & "'";
         else
            return D.Simple_Type_View.Ref.Full_Ada_Name & "'";
         end if;
      end if;
   end To_Qualified_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name (D : Ada_Object_Data) return Dynamic_Expression is
   begin
      if D.Simple_Type_View.Pass_By_Address then
         return JNI_Pckg & ".J_Int_J_Array";
      elsif D.Simple_Type_View.Ref.Kind = Tagged_Record_Kind then
         return JNI_Pckg & ".J_Object";
      else
         return JNI_Pckg & ".J_Int_J_Array";
      end if;
   end To_JNI_Type_Name;

   --  Create_Conversion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Object_Data;
      Expression : Wrapping_Expression)
   is
      Input : constant Dynamic_Expression := Expression.Expression;
      Conversion : constant Type_Record :=
        Get_Or_Create_Conversion_Type
          (D.Kernel, D.Simple_Type_View.Ref);
   begin
      Expression.Expression := To_Dynamic_Expression (Get_Unique_Id);

      Append
        (Expression.Bloc.Declarations,
         New_Line & Expression.Expression & " : "
         & Conversion.Conversion_Package_Name.all
         & ".Object_Pointer := "
         & Conversion.Conversion_Package_Name.all);

      if D.Simple_Type_View.Ref.Kind = Tagged_Record_Kind then
         Append (Expression.Bloc.Declarations,
                 ".To_Pointer_From_Object ("
                 & Env_Parameter_Name & ", " & Input & ");");
      elsif Is_Unconstrained_Array (D.Simple_Type_View.Ref) then
         Append
           (Expression.Bloc.Declarations,
            ".To_Object_Pointer ("
            & Env_Parameter_Name & ", "
            & Input & ");");

         if D.Has_Out_Mode then
            Generate_Ada_Array_Access
              (D.Kernel,
               Expression,
               D.Simple_Type_View.Ref,
               To_Dynamic_Expression
                 (Conversion.Conversion_Package_Name.all
                  & ".Array_Pointer"),
               False);
         else
            Generate_Ada_Array_Access
              (D.Kernel,
               Expression,
               D.Simple_Type_View.Ref,
               To_Dynamic_Expression
                 (Conversion.Conversion_Package_Name.all
                  & ".Array_Pointer_Cst"),
               True);
         end if;
      else
         Append (Expression.Bloc.Declarations,
                 ".To_Pointer (" & Env_Parameter_Name & ", " & Input & ");");
      end if;

      Expression.Expression :=
        D.To_Type_Name & " (" & Expression.Expression & ".all" & ")";
   end Create_Conversion_From_JNI;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Object_Data;
      Expression : Wrapping_Expression)
   is
      Tmp_Name : constant Wide_String := Get_Unique_Id;
      Out_Var : Dynamic_Expression;
      Conversion : constant Type_Record :=
        Get_Or_Create_Conversion_Type
          (D.Kernel, D.Simple_Type_View.Ref);
      Convert_Out_Class : Boolean := False;
      pragma Unreferenced (Convert_Out_Class);
   begin
      if D.Simple_Type_View.Ref.Kind /= Tagged_Record_Kind
        or else D.Simple_Type_View.Pass_By_Address
      then
         Out_Var := Expression.Expression;

         Append (Expression.Bloc.Declarations, New_Line & Tmp_Name & " : "
                 & Conversion.Conversion_Package_Name.all
                 & ".Object_Pointer := ");

         if Is_Unconstrained_Array (D.Simple_Type_View.Ref) then
            if D.Simple_Type_View.Ref.Is_Constant then
               Append
                 (Expression.Bloc.Declarations,
                  Conversion.Conversion_Package_Name.all
                  & ".Create_Constant_Encapsulation (");
            else
               Append
                 (Expression.Bloc.Declarations,
                  Conversion.Conversion_Package_Name.all
                  & ".Create_General_Encapsulation (");
            end if;
         end if;

         if D.Decl_View.all in Parameter_Simple_View'Class then
            Append
              (Expression.Bloc.Declarations,
               Expression.Expression & "'Unrestricted_Access");
         else
            Append
              (Expression.Bloc.Declarations,
               "new "
               & To_Type_Name (D) & "'(" & Expression.Expression & ")");
         end if;

         if Is_Unconstrained_Array (D.Simple_Type_View.Ref) then
            Append (Expression.Bloc.Declarations, ");");

            Expression.Expression := To_Dynamic_Expression
              (Conversion.Conversion_Package_Name.all
               & ".To_J_Int_J_Array ("
               & Env_Parameter_Name & ", " & Tmp_Name & ")");
         else
            Append (Expression.Bloc.Declarations, ";");

            Expression.Expression := To_Dynamic_Expression
              (Conversion.Conversion_Package_Name.all
               & ".To_JintArray ("
               & Env_Parameter_Name & ", " & Tmp_Name & ")");
         end if;
      else
         declare
            Input : constant Dynamic_Expression := Expression.Expression;
         begin
            Convert_Out_Class := True;
            Expression.Expression := To_Dynamic_Expression (Get_Unique_Id);

            Append
              (Expression.Bloc.Declarations,
               New_Line
               & Tmp_Name & " : "
               & Conversion.Conversion_Package_Name.all
               & ".Object_Pointer_Cst;");

            Append
              (Expression.Bloc.Declarations,
               New_Line & Expression.Expression & " : " & JNI_Pckg
               & ".J_Object;");

            if D.Decl_View.all not in Parameter_Simple_View'Class then
               --  We know that we need an allocation in case of a returned
               --  object, and we don't want to do additional copies in case
               --  of limited types. So we initialize the object right away on
               --  the heap.

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line
                  & Tmp_Name & " := new "
                  & D.To_Type_Name & "'(" & Input &  ");");

               Out_Var := To_Dynamic_Expression (Tmp_Name & ".all");
            else
               --  Othewsie, in case of e.g. parameters, we want to stack
               --  the copy.

               Out_Var := To_Dynamic_Expression (Get_Unique_Id);

               Append
                 (Expression.Bloc.Declarations,
                  New_Line
                  & Out_Var & " : " & D.To_Type_Name & "'Class renames "
                  & D.To_Type_Name & "'Class ("
                  & Input & ");");

               if D.Decl_View.Is_Aliased then
                  Append
                    (Expression.Bloc.Statements_Before,
                     New_Line
                     & Tmp_Name & " := " & Out_Var & "'Unchecked_Access;");
               else
                  Append
                    (Expression.Bloc.Statements_Before,
                     New_Line
                     & Tmp_Name & " := " & Out_Var & "'Unrestricted_Access;");
               end if;
            end if;

            Append
              (Expression.Bloc.Statements_Before,
               Gen_Ada_Loc ("Create_Conversion_To_JNI (Ada_Object_Data)") &
               New_Line &
               "if " & Out_Var
               & " in " & AJIS_Pckg & ".Cross_Language_Class'Class"
               & New_Line & "  and then " & JNI_Pckg & ".""/="" (" & AJIS_Pckg
               & ".Cross_Language_Class'Class (" & Out_Var & ")'"
               & "Unchecked_Access.Get_Java_Ref, " & JNI_Pckg
               & ".J_Null_Object)"
               & New_Line & "then"
               & Indent (1));

            if D.Decl_View.all not in Parameter_Simple_View'Class then
               --  In case of a returned object, we force a copy. Otherwise,
               --  we try to retreive the already stored object.

               declare
                  Tmp : constant Wide_String := Get_Unique_Id;
               begin
                  --  We've got to get the ref using the Get_Java_Ref primitive
                  --  twice, because the Set_Owner will replace the old one
                  --  by a new one.

                  Append
                    (Expression.Bloc.Statements_Before,
                     New_Line & "declare"
                     & New_Line (1) & Tmp & " : " & JNI_Pckg & ".J_Object;"
                     & New_Line (-1) & "begin"
                     & New_Line (1) & Tmp
                     & " := " & AJIS_Pckg & ".Cross_Language_Class'Class ("
                     & Tmp_Name & ".all)'Unchecked_Access.Get_Java_Ref;"
                     & New_Line  & AJIS_Pckg & ".Set_Allocator ("
                     & Env_Parameter_Name
                     & ", " & Tmp & ", " & AJIS_Pckg
                     & ".Dynamic);"
                     & New_Line  & AJIS_Pckg & ".Set_Owner ("
                     & Env_Parameter_Name
                     & ", " & Tmp & ", " & AJIS_Pckg
                     & ".Proxy);"
                     & New_Line (-1) & "end;");
               end;
            end if;

            Append
              (Expression.Bloc.Statements_Before,
               New_Line & Expression.Expression
               & " := " & AJIS_Pckg & ".Cross_Language_Class'Class ("
               & Tmp_Name & ".all)'Unchecked_Access.Get_Java_Ref;"
               & New_Line (-1) & "else");

            Append
              (Expression.Bloc.Statements_Before,
               New_Line (1) & Expression.Expression
               & " := " & AJIS_Pckg & ".Create_Java_Object ("
               & Env_Parameter_Name & ", "
               & Out_Var & "'Tag, "
               & Conversion.Conversion_Package_Name.all
               & ".To_JintArray_Cst (" & Env_Parameter_Name
               & ", " & Tmp_Name & "));");

            if D.Decl_View.all not in Parameter_Simple_View'Class then
               --  If we're not on a parameter, then we've created a new object
               --  which needs to be owned by the proxy.

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line  & AJIS_Pckg & ".Set_Allocator ("
                  & Env_Parameter_Name
                  & ", " & Expression.Expression & ", " & AJIS_Pckg
                  & ".Dynamic);"
                  & New_Line  & AJIS_Pckg & ".Set_Owner (" & Env_Parameter_Name
                  & ", " & Expression.Expression & ", " & AJIS_Pckg
                  & ".Proxy);");
            else
               --  Otherwise, we're referencing a native object that is
               --  directly accessed.

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line  & AJIS_Pckg & ".Set_Allocator ("
                  & Env_Parameter_Name
                  & ", " & Expression.Expression & ", " & AJIS_Pckg
                  & ".Static);"
                  & New_Line  & AJIS_Pckg & ".Set_Owner (" & Env_Parameter_Name
                  & ", " & Expression.Expression & ", " & AJIS_Pckg
                  & ".Native);");
            end if;

            Append
              (Expression.Bloc.Statements_Before,
               New_Line (-1) & "end if;");

            Nest_Ada_Statements (Expression.Bloc);
         end;
      end if;
   end Create_Conversion_To_JNI;

   -- Get_JNI_Call --

   function Get_JNI_Call (D : Ada_Object_Data) return Dynamic_Expression is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".Call_Object_Method_A";
   end Get_JNI_Call;

   --  Adjust_Unit_For_Use --

   procedure Adjust_Unit_For_Use
     (D : in out Ada_Object_Data; Unit : Bound_Unit)
   is
      pragma Unreferenced (D, Unit);
   begin
      null;
   end Adjust_Unit_For_Use;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_Object_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return Default_J_Int_Array;
   end Get_Default_JNI_Value;

   --------------------------------
   -- Ada_Pointer_To_Object_Data --
   --------------------------------

   --  To_Type_Name  --

   function To_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression
   is
      Name : Dynamic_Expression;
   begin
      Add_Dependency_On_Type
        (D.Kernel,
         D.Unit.Ada_Spec_File,
         D.Simple_Type_View.Ref);

      if D.Simple_Type_View.Ref.Full_Ada_Name /= Empty_Dynamic_Expression
        and then not D.Simple_Type_View.Ref.Is_Anonymous
      then
         Name := D.Simple_Type_View.Initial_Subtype_Name;
         if  D.Simple_Type_View.Is_Not_Null then
            return "not null " & Name;
         end if;
         return Name;
      else
         Name := D.Simple_Type_View.Ref.Target_Type.Ref.Full_Ada_Name;

         if Should_Be_Class_Wide (D) then
            Name := Name & "'Class";
         end if;

         if D.Simple_Type_View.Ref.Is_Constant then
            Name := "constant " & Name;
         end if;

         Name := "access " & Name;

         if  D.Simple_Type_View.Is_Not_Null then
            Name := "not null " & Name;
         end if;

         return Name;
      end if;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression is
   begin
      if D.Simple_Type_View.Pass_By_Address then
         return JNI_Pckg & ".J_Int_J_Array";
      elsif D.Simple_Type_View.Ref.Target_Type.Ref.Kind = Tagged_Record_Kind
        or else
          (D.Decl_View.all in Parameter_Simple_View'Class
           and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode)
      then
         return JNI_Pckg & ".J_Object";
      else
         return JNI_Pckg & ".J_Int_J_Array";
      end if;
   end To_JNI_Type_Name;

   -- To_Ada_Internal_Type_Name --

   function To_Ada_Internal_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression
   is
   begin
      Add_Dependency_On_Type
        (D.Kernel,
         D.Unit.Ada_Spec_File,
         D.Simple_Type_View.Ref);

      return D.Simple_Type_View.Initial_Subtype_Name;
   end To_Ada_Internal_Type_Name;

   --  Create_Conversion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Pointer_To_Object_Data;
      Expression : Wrapping_Expression)
   is
      Input : Dynamic_Expression := Expression.Expression;

      Conv : constant Type_Record :=
        Get_Or_Create_Conversion_Type
          (D.Kernel, D.Simple_Type_View.Ref.Target_Type.Ref);
   begin
      if D.Call_Convention = Java_To_Ada then
         D.Initial_Input := Expression.Expression;
      end if;

      Expression.Expression := To_Dynamic_Expression (Get_Unique_Id);

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         Input := AJIS_Pckg & ".Get_Serialized_Address ("
           & Env_Parameter_Name & ", " & AJIS_Pckg
           & ".Java_Get_Object_Value ("
           & Env_Parameter_Name & ", " & Input & "))";
      elsif D.Simple_Type_View.Ref.Target_Type.Ref.Kind
        = Tagged_Record_Kind
      then
         Input := AJIS_Pckg & ".Get_Serialized_Address ("
           & Env_Parameter_Name & ", " & Input & ")";
      end if;

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
          ((Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
            and then D.Call_Convention = Java_To_Ada)
           or else
             (Parameter_Simple_View (D.Decl_View.all).Mode = In_Mode
              and then D.Call_Convention = Ada_To_Java))
      then
         Append
           (Expression.Bloc.Declarations,
            New_Line & Expression.Expression & " : "
            & D.To_Ada_Internal_Type_Name & " := null;");
      else
         if D.Decl_View.all in Parameter_Simple_View'Class
           and then Parameter_Simple_View (D.Decl_View.all).Mode /= Out_Mode
           and then Is_Unconstrained_Array
             (D.Simple_Type_View.Ref.Target_Type.Ref)
         then
            Append
              (Expression.Bloc.Declarations,
               New_Line & Expression.Expression & " : constant "
               & Conv.Conversion_Package_Name.all & ".Object_Pointer := "
               & Conv.Conversion_Package_Name.all
               & ".To_Object_Pointer ("
               & Env_Parameter_Name & ", "
               & Input & ");");

            Generate_Ada_Array_Access
              (D.Kernel,
               Expression,
               D.Simple_Type_View.Ref.Target_Type.Ref,
               D.To_Ada_Internal_Type_Name,
               D.Simple_Type_View.Ref.Is_Constant);
         else
            Append
              (Expression.Bloc.Declarations,
               New_Line & Expression.Expression & " : "
               & D.To_Ada_Internal_Type_Name & " := ");

            Append
              (Expression.Bloc.Declarations,
               D.To_Ada_Internal_Type_Name & " (");

            if D.Simple_Type_View.Ref.Full_Ada_Name /= Empty_Dynamic_Expression
              and then not D.Simple_Type_View.Ref.Is_Anonymous
            then
               Expression.Expression :=
                 D.Simple_Type_View.Ref.Full_Ada_Name
                   & " (" & Expression.Expression & ")";
            else
               Expression.Expression := "" & Expression.Expression;
            end if;

            if Is_Unconstrained_Array
              (D.Simple_Type_View.Ref.Target_Type.Ref)
            then
               Append
                 (Expression.Bloc.Declarations,
                  D.Conversion_Package_Name.all
                  & ".To_Object_Pointer (" & Env_Parameter_Name & ", "
                  & Input & ").General_Pointer");
            else
               Append
                 (Expression.Bloc.Declarations,
                  D.Conversion_Package_Name.all
                  & ".To_Pointer (" & Env_Parameter_Name & ", "
                  & Input & ")");
            end if;

            Append (Expression.Bloc.Declarations, ");");
         end if;
      end if;

      if D.Has_Out_Mode
        and then D.Call_Convention /= Ada_To_Java
      then
         declare
            After_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record;
         begin
            After_Expression.Expression := Expression.Expression;
            After_Expression.Bloc := new Wrapping_Bloc_Record;
            After_Expression.Bloc.Code_Node := New_Dynamic_Expression;
            After_Expression.Bloc.Statements_Before := New_Dynamic_Expression;
            After_Expression.Bloc.Statements_After := New_Dynamic_Expression;
            After_Expression.Bloc.Declarations := New_Dynamic_Expression;

            Append
              (Expression.Bloc.Statements_After,
               New_Line
               & "declare"
               & Indent (1) & After_Expression.Bloc.Declarations
               & New_Line (-1) & "begin"
               & Indent (1)
               & After_Expression.Bloc.Statements_Before
               & After_Expression.Bloc.Code_Node
               & After_Expression.Bloc.Statements_After
               & New_Line (-1) & "end;");

            Create_Conversion_To_JNI (D, After_Expression);

            --  A nesting is likely to have happen here, add a null statement
            --  in case nothing else has to be done.

            Append
              (After_Expression.Bloc.Code_Node,
               New_Line & "null;");
         end;
      end if;
   end Create_Conversion_From_JNI;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Pointer_To_Object_Data;
      Expression : Wrapping_Expression)
   is
      Input : constant Dynamic_Expression := Expression.Expression;
      Pointer_Type : Dynamic_Expression;
      To_Jint_Conv : Dynamic_Expression;
      The_Type : constant Bound_Element :=
        Get_Bound_Element
          (D.Kernel,
           Simple_Element_View_Access
             (D.Simple_Type_View.Ref.Target_Type.Ref));
      Conversion : constant Type_Record :=
        Get_Or_Create_Conversion_Type
          (D.Kernel, D.Simple_Type_View.Ref.Target_Type.Ref);

      Wrapper_Id : constant Dynamic_Expression
        := To_Dynamic_Expression (Get_Unique_Id) & "_Wrapper";
      Proxy_Id   : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id) & "_Proxy";
      Ada_Ptr    : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id) & "_Ptr";
      Proxy_Addr_Id : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id) & "_Addr";

      Generate_Encapsulation : Dynamic_Expression;
      Array_Pointer          : Dynamic_Expression;

      Bound_Wrapper : Bound_Element;
      Null_Pointer : Dynamic_Expression;
   begin
      if D.Call_Convention = Ada_To_Java then
         D.Initial_Input := Expression.Expression;
      end if;

      --  Prepare global types of data

      if Is_Unconstrained_Array (D.Simple_Type_View.Ref.Target_Type.Ref) then
         Pointer_Type := To_Dynamic_Expression ("Object_Pointer");

         To_Jint_Conv := To_Dynamic_Expression
           ("To_J_Int_J_Array");
         Null_Pointer := To_Dynamic_Expression
           (Conversion.Conversion_Package_Name.all & ".Null_Pointer");

         if Get_Type_View (D).Ref.Is_Constant then
            Generate_Encapsulation := To_Dynamic_Expression
              ("Create_Constant_Encapsulation");
            Array_Pointer := To_Dynamic_Expression ("Array_Pointer_Cst");
         else
            Generate_Encapsulation := To_Dynamic_Expression
              ("Create_General_Encapsulation");
            Array_Pointer := To_Dynamic_Expression ("Array_Pointer");
         end if;
      else
         Null_Pointer := To_Dynamic_Expression ("null");

         if Get_Type_View (D).Ref.Is_Constant then
            Pointer_Type := To_Dynamic_Expression ("Object_Pointer_Cst");
            To_Jint_Conv := To_Dynamic_Expression ("To_JintArray_Cst");
         else
            Pointer_Type := To_Dynamic_Expression ("Object_Pointer");
            To_Jint_Conv := To_Dynamic_Expression ("To_JintArray");
         end if;
      end if;

      Add_Extra_Unit_Dependency
        (D.Unit.Ada_Body_File, The_Type.Enclosing_Unit);

      Append
        (Expression.Bloc.Declarations,
         New_Line & Proxy_Id & " : " & JNI_Pckg & ".J_Object;");

      --  Check wether a Java wrapper needs to be created

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
      then
         --  In this case, we're either on an out or in out parameter
         --  We need to build the reference wrapper to pass the information
         --  to the JNI side if it's not already created

         if D.Call_Convention = Ada_To_Java then
            Bound_Wrapper := Get_Bound_Element
              (D.Kernel,
               Simple_Element_View_Access
                 (D.Simple_Type_View.Ref.
                    Target_Type.Ref.Wrapper));

            declare
               Construct_Name : constant Dynamic_Expression :=
                 Get_JNI_Constructor
                   (Bound_Reference (Bound_Wrapper.all));
            begin
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Wrapper_Id & " : "
                  & JNI_Pckg & ".J_Object := " & JNI_Pckg &
                  ".New_Object_A (" & Env_Parameter_Name & ", "
                  & AJIS_Pckg & ".Get_Class (" & Construct_Name & ", "
                  & Env_Parameter_Name & "), " & AJIS_Pckg & ".Get_Id ("
                  & Construct_Name & "), "
                  & "(1 .. 0 => <>));");
            end;
         else
            Append
              (Expression.Bloc.Declarations,
               New_Line & Wrapper_Id & " : "
               & JNI_Pckg & ".J_Object := " & D.Initial_Input & ";");
         end if;
      end if;

      --  Do the actual translation if needed

      if D.Decl_View.all in Parameter_Simple_View'Class
        and then
         ((Parameter_Simple_View (D.Decl_View.all).Mode = In_Mode
          and then D.Call_Convention = Java_To_Ada)
          or else
            (Parameter_Simple_View (D.Decl_View.all).Mode = Out_Mode
          and then D.Call_Convention = Ada_To_Java))
      then
         --  Case 1, the whole call is Java to Ada, so we ignore in parameters
         --  when doing translations back to Java

         --  Case 2, the whole call is Ada to Java, so we ignore out parameters
         --  when doing translations to Java

         --  We'll just reuse the previously set wrapper, or null.

         if D.Decl_View.all in Parameter_Simple_View'Class
           and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
         then
            Expression.Expression := Wrapper_Id;
         elsif D.Simple_Type_View.Ref.Target_Type.Ref.Kind
           /= Tagged_Record_Kind
           or else D.Simple_Type_View.Pass_By_Address
         then
            Expression.Expression := D.Conversion_Package_Name.all
              & "." & To_Jint_Conv & " (" & Env_Parameter_Name
              & ", null)";
         else
            Expression.Expression := JNI_Pckg & ".J_Null_Access";
         end if;
      else
         --  Otherwise, the parameters needs to be actually translated from
         --  Ada to Java (JNI)

         --  First, retreive the Ada raw access and place it in Ada_Ptr. Some
         --  extra processing is needed in case of arrays

         Append
           (Expression.Bloc.Declarations, New_Line & Ada_Ptr & " : "
            & D.Conversion_Package_Name.all & "." & Pointer_Type & " := "
            & D.Conversion_Package_Name.all & "." & Pointer_Type & " (");

         if Get_Array_Manipulation_Kind
           (D.Simple_Type_View) = Direct_Array_Access
         then
            Append
              (Expression.Bloc.Declarations,
               D.Conversion_Package_Name.all
               & "." & Generate_Encapsulation & " ("
               & D.Conversion_Package_Name.all & "." & Array_Pointer & " ("
               & Expression.Expression
               & ")));");
         elsif D.Simple_Type_View.Ref.Target_Type.Ref.Kind
           /= Tagged_Record_Kind
           or else D.Simple_Type_View.Pass_By_Address
         then
            if D.Simple_Type_View.Ref.Full_Ada_Name
              /= Empty_Dynamic_Expression
            then
               Append
                 (Expression.Bloc.Declarations,
                  D.Simple_Type_View.Initial_Subtype_Name
                  & "'(" & Expression.Expression & "));");
            else
               Append
                 (Expression.Bloc.Declarations,
                  Expression.Expression & ");");
            end if;
         elsif D.Simple_Type_View.Ref.Is_Constant then
            Append
              (Expression.Bloc.Declarations,
               Conversion.Conversion_Package_Name.all
               & ".To_Object_Pointer_Cst (" & Input & "));");
         else
            Append
              (Expression.Bloc.Declarations,
               Conversion.Conversion_Package_Name.all
               & ".To_Object_Pointer (" & Input & "));");
         end if;

         --  Second, create or retreive the corresponding JNI proxy, null
         --  if the Ada pointer is null.

         if D.Simple_Type_View.Ref.Target_Type.Ref.Kind /= Tagged_Record_Kind
           or else D.Simple_Type_View.Pass_By_Address
         then
            Expression.Expression := D.Conversion_Package_Name.all
              & "." & To_Jint_Conv & " (" & Env_Parameter_Name
              & ", " & Ada_Ptr & ")";

            if D.Decl_View.all in Parameter_Simple_View'Class
              and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
            then
               --  If we're on an "out" or "in out" case, we need to create the
               --  proxy manually here, depending on the fact that the object
               --  is null or not

               Append
                 (Expression.Bloc.Statements_Before,
                  New_Line & "if "
                  & Conversion.Conversion_Package_Name.all
                  & ".""="" ("
                  & Ada_Ptr & ", " & Null_Pointer & ") then"
                  & New_Line (1) & Proxy_Id
                  & " := " & JNI_Pckg & ".J_Null_Object;"
                  & New_Line (-1)
                  & "else "
                  & New_Line (1) & "declare"
                  & New_Line (1)
                  & Proxy_Addr_Id & " : " & JNI_Pckg & ".J_Object := "
                  & AJIS_Pckg
                  & ".New_Java_Access_Constructor (" & Env_Parameter_Name
                  & ", "  & Expression.Expression & ");"
                  & New_Line (-1) & "begin"
                  & New_Line (1) & Proxy_Id
                  & " := " & JNI_Pckg & ".New_Object_A ("
                  & Env_Parameter_Name & ", " & AJIS_Pckg & ".Get_Class ("
                  & Get_Constructor_Id (Bound_Type (The_Type.all))
                  & "), "
                  & AJIS_Pckg & ".Get_Id ("
                  & Get_Constructor_Id (Bound_Type (The_Type.all))
                  & "), (1 .. 1 => ("
                  & JNI_Pckg & ".jobject, " & Proxy_Addr_Id & ")));"
                  & New_Line (-1)
                  & "end;"
                  & New_Line (-1)
                  & "end if;");

               Expression.Expression := Proxy_Id;
            else
               --  Othewise, just keep the expression as an int [], the
               --  Java side will take care of the proxy conversion
               --  ??? Perhaps it would be cleaner to have all the proxy
               --  creations consistently managed by Ada.

               null;
            end if;
         else
            Append
              (Expression.Bloc.Statements_Before,
               New_Line & "if "
               & Conversion.Conversion_Package_Name.all
               & ".""="" (" & Ada_Ptr & ", null) then"
               & New_Line (1) & Proxy_Id
               & " := " & JNI_Pckg & ".J_Null_Object;"
               & New_Line (-1)
               & "elsif " & Ada_Ptr
               & ".all in " & AJIS_Pckg & ".Cross_Language_Class'Class then"
               & New_Line (1) & Proxy_Id
               & " := " & AJIS_Pckg & ".Cross_Language_Class'Class ("
               & Ada_Ptr & ".all)'Unchecked_Access.Get_Java_Ref;"
               & New_Line (-1)
               & "else"
               & New_Line (1) & Proxy_Id
               & " := " & AJIS_Pckg & ".Create_Java_Object ("
               & Env_Parameter_Name & ", "
               & Ada_Ptr & ".all'Tag, "
               & Conversion.Conversion_Package_Name.all
               & "." & To_Jint_Conv & " (" & Env_Parameter_Name & ", "
               & Ada_Ptr & "));"
               & New_Line  & AJIS_Pckg & ".Set_Allocator ("
               & Env_Parameter_Name
               & ", " & Proxy_Id & ", " & AJIS_Pckg
               & ".Dynamic);"
               & New_Line (-1) & "end if;");

            Expression.Expression := Proxy_Id;
         end if;

         --  If we're on either an "out" or "in out" parameter, then the
         --  value needs to be set to the wrapper

         if D.Decl_View.all in Parameter_Simple_View'Class
           and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode
         then
            Append
              (Expression.Bloc.Statements_Before,
               New_Line
               & AJIS_Pckg & ".Java_Set_Object_Value ("
               & Env_Parameter_Name & ", "
               & Wrapper_Id & ", "
               & Proxy_Id & ");");

            Expression.Expression := Wrapper_Id;
         end if;

         --  In case we need to rely on the statements before /
         --  declarations afterwards, e.g. when managing returned values.
         --  Objects have been assigned after their declaration so we need to
         --  nest further statements.

         Nest_Ada_Statements (Expression.Bloc);
      end if;

      if D.Has_Out_Mode and then D.Call_Convention /= Java_To_Ada then
         declare
            After_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record;
         begin
            After_Expression.Expression := Expression.Expression;
            After_Expression.Bloc := new Wrapping_Bloc_Record;
            After_Expression.Bloc.Code_Node := New_Dynamic_Expression;
            After_Expression.Bloc.Statements_Before := New_Dynamic_Expression;
            After_Expression.Bloc.Statements_After := New_Dynamic_Expression;
            After_Expression.Bloc.Declarations := New_Dynamic_Expression;

            Append
              (Expression.Bloc.Statements_After,
               New_Line
               & "declare"
               & Indent (1) & After_Expression.Bloc.Declarations
               & New_Line (-1) & "begin"
               & Indent (1)
               & After_Expression.Bloc.Statements_Before
               & After_Expression.Bloc.Code_Node
               & After_Expression.Bloc.Statements_After
               & New_Line (-1) & "end;");

            Create_Conversion_From_JNI (D, After_Expression);

            Append
              (After_Expression.Bloc.Code_Node,
               New_Line & Input & " := " & After_Expression.Expression & ";");
         end;
      end if;
   end Create_Conversion_To_JNI;

   -- Get_JNI_Call --

   function Get_JNI_Call
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".Call_Object_Method_A";
   end Get_JNI_Call;

   --  Adjust_Unit_For_Use --

   procedure Adjust_Unit_For_Use
     (D : in out Ada_Pointer_To_Object_Data; Unit : Bound_Unit)
   is
      pragma Unreferenced (Unit);

      Conversion : Type_Record;
   begin
      Conversion :=
        Get_Or_Create_Conversion_Type
          (D.Kernel, D.Simple_Type_View.Ref.Target_Type.Ref);
      D.Conversion_Package_Name := Conversion.Conversion_Package_Name;
   end Adjust_Unit_For_Use;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression is
   begin
      if (D.Decl_View.all in Parameter_Simple_View'Class
          and then Parameter_Simple_View (D.Decl_View.all).Mode /= In_Mode)
        or else D.Decl_View.Type_Of.Ref.Target_Type.Ref.Kind
          = Tagged_Record_Kind
      then
         return Default_J_Object;
      else
         return Default_J_Int_Array;
      end if;
   end Get_Default_JNI_Value;

   --------------------------------
   -- Ada_Pointer_To_Scalar_Data --
   -------------------------------

   --  To_Type_Name  --

   function To_Type_Name
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      raise Not_Supported;
      return Empty_Dynamic_Expression;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      raise Not_Supported;
      return Empty_Dynamic_Expression;
   end To_JNI_Type_Name;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Pointer_To_Scalar_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_To_JNI;

   -- Get_JNI_Call --

   function Get_JNI_Call
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".Call_Object_Method_A";
   end Get_JNI_Call;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".J_Null_Object";
   end Get_Default_JNI_Value;

   -----------------------
   -- Ada_Callback_Data --
   -----------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Ada_Callback_Data) return Dynamic_Expression is
   begin
      if not D.Simple_Type_View.Ref.Is_Anonymous then
         return D.Simple_Type_View.Ref.Full_Ada_Name;
      else
         return "access " & To_Original_Expanded_Profile
           (D.Kernel,
              D.Simple_Type_View.Ref.Target_Type.Ref.Target_Subprogram);
      end if;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Ada_Callback_Data) return Dynamic_Expression
   is
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class and then
        Parameter_Simple_View_Access (D.Decl_View).Is_Controlling
      then
         return JNI_Pckg & ".J_Int_J_Array";
      else
         return JNI_Pckg & ".J_Object";
      end if;
   end To_JNI_Type_Name;

   --  Create_Initialization_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Callback_Data;
      Expression : Wrapping_Expression)
   is
      Input : constant Dynamic_Expression := Expression.Expression;

      Bound_Sb : constant Bound_Element := Get_Bound_Element
        (D.Kernel, Simple_Element_View_Access
           (D.Simple_Type_View.Ref));

      Id_Callback : constant Wide_String := Get_Unique_Id;
   begin
      if D.Decl_View.all in Parameter_Simple_View'Class and then
        Parameter_Simple_View_Access (D.Decl_View).Is_Controlling
      then
         declare
            Conv_Id : constant Wide_String := Get_Unique_Id;
         begin
            Append
              (Expression.Bloc.Declarations,
               New_Line & "function " & Conv_Id
               & " is new Ada.Unchecked_Conversion (System.Address, "
               & D.Simple_Type_View.Ref.Full_Ada_Name & ");");

            Expression.Expression :=
              Conv_Id & " (" & AJIS_Pckg & ".To_Address ("
                & Env_Parameter_Name & ", "
                & Expression.Expression & "))";
         end;
      else
         Expression.Expression := To_Dynamic_Expression (Get_Unique_Id);

         if D.Simple_Type_View.Ref.Target_Type.Ref.Target_Subprogram
           .Returned_Type /= null
         then
            Append (Expression.Bloc.Declarations, New_Line & "function ");
         else
            Append (Expression.Bloc.Declarations, New_Line & "procedure ");
         end if;

         Append
           (Expression.Bloc.Declarations,
            Expression.Expression
            & " is new "
            & Bound_Sb.Enclosing_Unit.Ada_Pckg_Name & "."
            & Get_Ada_High_Name (Bound_Callback (Bound_Sb.all)'Access)
            & " (" & Env_Parameter_Name & ", " & Input & ");"
            & New_Line & Id_Callback & " : " & D.To_Type_Name & ";");

         Append
           (Expression.Bloc.Statements_Before,
            New_Line & "if " & JNI_Pckg & ".""="" (" & Input & ", "
            & JNI_Pckg & ".J_Null_Object) then"
            & New_Line (1) & Id_Callback & " := null;"
            & New_Line (-1) & "else"
            & New_Line (1) & Id_Callback & " := " & Expression.Expression
            & "'Unrestricted_Access;"
            & New_Line (-1) & "end if;");

         Expression.Expression := To_Dynamic_Expression (Id_Callback);

         Nest_Ada_Statements (Expression.Bloc);
      end if;
   end Create_Conversion_From_JNI;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Callback_Data;
      Expression : Wrapping_Expression)
   is
      Args_1_Id : constant Wide_String := Get_Unique_Id;
      Args_2_Id : constant Wide_String := Get_Unique_Id;

      Conv_Id : constant Wide_String := Get_Unique_Id;

      Bound_Sb : constant Bound_Element := Get_Bound_Element
        (D.Kernel, Simple_Element_View_Access
           (D.Simple_Type_View.Ref));
      Default_Implementation_Unit_Name : constant Dynamic_Expression :=
        Get_Default_Implementation_Unit_Name
          (Bound_Callback (Bound_Sb.all)'Access);
      Constructor_Name : constant Dynamic_Expression :=
        Bound_Sb.Enclosing_Unit.Ada_Pckg_Name & "."
        & Default_Implementation_Unit_Name & "_Constructor";
      Id_Callback : constant Wide_String := Get_Unique_Id;
   begin
      --  ??? This mechanism should be factorized somewhere in Ada2Java_Runtime

      Append
        (Expression.Bloc.Declarations,
         New_Line & "function " & Conv_Id
         & " is new Ada.Unchecked_Conversion ("
         & D.Simple_Type_View.Ref.Full_Ada_Name & ", System.Address);");

      Append
        (Expression.Bloc.Declarations,
         New_Line & Args_1_Id
         & " : " & JNI_Pckg & ".J_Value_Array (1 .. 1) := (1 => ("
         & JNI_Pckg & ".Jobject, "
         & AJIS_Pckg & ".To_J_Int_J_Array (" & Env_Parameter_Name & ", "
         & Conv_Id & " (" & Expression.Expression & "))));");

      Append
        (Expression.Bloc.Declarations,
         New_Line & Args_2_Id
         & " : " & JNI_Pckg & ".J_Value_Array (1 .. 1) := (1 => ("
         & JNI_Pckg & ".Jobject, "
         & JNI_Pckg & ".New_Object_A (" & Env_Parameter_Name & ", "
         & AJIS_Pckg & ".Get_Class ("
         & AJIS_Pckg & ".Access_Meth_Constructor), "
         & AJIS_Pckg & ".Get_Id ("
         & AJIS_Pckg & ".Access_Meth_Constructor), "
         & Args_1_Id & ")));"
         & New_Line & Id_Callback & " : " & JNI_Pckg & ".J_Object;");

      if D.Simple_Type_View.Ref.Is_Anonymous then
         Append
           (Expression.Bloc.Statements_Before,
            New_Line
            & "if " & Expression.Expression & " = null then");
      else
         Append
           (Expression.Bloc.Statements_Before,
            New_Line
            & "if " & Get_Bound_Package_Name
              (Bound_Sb.Enclosing_Unit.Base_Pckg)
            & ".""="" (" & Expression.Expression & ", null) then");
      end if;

      Append
        (Expression.Bloc.Statements_Before,
         New_Line (1) & Id_Callback & " := " & JNI_Pckg & ".J_Null_Object;"
         & New_Line (-1) & "else"
         & New_Line (1) & Id_Callback & " := " & JNI_Pckg & ".New_Object_A ("
         & Env_Parameter_Name & ", " & AJIS_Pckg & ".Get_Class ("
         & Constructor_Name & "), " & AJIS_Pckg
         & ".Get_Id ("
         & Constructor_Name &  "), "
         & Args_2_Id & ");"
         & New_Line (-1) & "end if;");

      Expression.Expression := To_Dynamic_Expression (Id_Callback);

      Nest_Ada_Statements (Expression.Bloc);
   end Create_Conversion_To_JNI;

   -- Get_JNI_Call --

   function Get_JNI_Call (D : Ada_Callback_Data) return Dynamic_Expression is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".Call_Object_Method_A";
   end Get_JNI_Call;

   --  Adjust_Unit_For_Use  --

   procedure Adjust_Unit_For_Use
     (D : in out Ada_Callback_Data; Unit : Bound_Unit)
   is
      Callback_Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (D.Kernel, Simple_Element_View_Access (D.Simple_Type_View.Ref));
   begin
      Add_Extra_Unit_Dependency
        (Unit.Ada_Body_File,
         Callback_Unit);
   end Adjust_Unit_For_Use;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_Callback_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return Default_J_Object;
   end Get_Default_JNI_Value;

   ------------------
   -- Ada_JNI_Data --
   ------------------

   --  To_Type_Name  --

   function To_Type_Name (D : Ada_JNI_Data) return Dynamic_Expression is
   begin
      return D.Simple_Type_View.Ref.Full_Ada_Name;
   end To_Type_Name;

   --  To_JNI_Type_Name  --

   function To_JNI_Type_Name
     (D : Ada_JNI_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".J_Object";
   end To_JNI_Type_Name;

   --  Create_Conversion_From_JNI  --

   procedure Create_Conversion_From_JNI
     (D          : in out Ada_JNI_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_From_JNI;

   --  Create_Conversion_To_JNI  --

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_JNI_Data;
      Expression : Wrapping_Expression)
   is
      pragma Unreferenced (D, Expression);
   begin
      null;
   end Create_Conversion_To_JNI;

   --  Get_JNI_Call  --

   function Get_JNI_Call (D : Ada_JNI_Data) return Dynamic_Expression is
      pragma Unreferenced (D);
   begin
      return To_Dynamic_Expression ("");
   end Get_JNI_Call;

   --  Get_Default_JNI_Value  --

   function Get_Default_JNI_Value
     (D : Ada_JNI_Data) return Dynamic_Expression
   is
      pragma Unreferenced (D);
   begin
      return JNI_Pckg & ".J_Null_Object";
   end Get_Default_JNI_Value;

end Ada2Java.Bound_Elements.Subprograms.Data;
