------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--                     A S I S _ U L . P R O J E C T S                      --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                       Copyright (C) 2013, AdaCore                        --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING.  If  not,  write  to  the Free Software Foundation, 51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains high-level interfaces to project files. These
--  interfaces are supposed to be enough to extract all the information an
--  ASIS-based tool may need from the argument project file.
--
--  A part of project file processing is tool-specific: there are tool-specific
--  packages and parameters in the project file, so this package defines only
--  the common steps needed for extracting the information from the project
--  file by providing the Arg_Project_Type type. A tool has to extend this
--  type by providing operations for dealing with tool-specific attributes.
--
--  The recommended way of project file processing is:
--
--  1. Scanning project-specific arguments as a part of tool command line
--     processing
--
--     - for -P option: call Store_Project_Source procedure with
--       Projet_Source_File set to option parameter;
--         * check if this is the first -P option provided;
--         * check if the parameter denotes an existing file;
--       raise ASIS_UL.Common.Parameter_Error if any of these check fails,
--       store the name of the project file otherwise. Th
--
--     - for -X options: call Store_External_Variable with the option parameter
--       and Is_From_Command_Line => True
--
--     - for -U [main_unit] - set U_Option_Set ON and store main_unit in
--       Main_Unit if given.
--
--  2.  Call Initialize_Environment to get prepared for loading the project.
--
--  3.  Call Register_Tool_Attributes to register attributes that are specific
--      for a tool. Most probably you will have to redefine this procedure for
--      your tool.
--
--  4.  Load the project (call to Load_Tool_Project);
--
--  5.  Extract and store tool-specific attributes and set values of
--      external variables (call to Extract_Tool_Attributes).
--
--  6.  Extract and store the path information and the list of sources to
--      process

with GNAT.OS_Lib;       use GNAT.OS_Lib;

with GNATCOLL.Projects; use GNATCOLL.Projects;

package ASIS_UL.Projects is

   U_Option_Set : Boolean := False;
   --  Inducates if -U option is specified for the project (to process all the
   --  units of the closure of the argument project or to process the closure
   --  of the main unit if the main unit is set)

   Main_Unit : String_Access;
   --  If the tool is called with "... Pproj -U main_unit" stores main_unit
   --  here

   procedure Initialize_Environment;
   --  Initializes the environment for extracting the information from the
   --  project file. This includes setting the parameters specific for the
   --  given tool version assuming that the tools for cross environment are
   --  named in a standard way (that is, <cross-prefix>-<tool_name>.

   type Arg_Project_Type is new Project_Tree with private;

   procedure Store_Project_Source
     (My_Project         : in out Arg_Project_Type;
      Projet_Source_File : String);
   --  Checks that:
   --    - check if this is the first -P option provided as a tool parameter;
   --    - Projet_Source_File denotes an existing file;
   --  Raises ASIS_UL.Common.Parameter_Error if any of these check fails,
   --  stores the name of the project file My_Project otherwise.

   function Is_Specified (My_Project : Arg_Project_Type) return Boolean;
   --  Checks if the argument has been defined as a tool parameter.

   procedure Register_Tool_Attributes (My_Project : in out Arg_Project_Type);
   --  Registers attributes that are specific for a given tool. The default
   --  version registers the attribute "default_switches" in package
   --  <toolname>.

   procedure Load_Tool_Project (My_Project : in out Arg_Project_Type);
   --  Does the same as GNATCOLL.Projects.Load, the only difference is that
   --  all the parameters except the project are hidden.

   procedure Extract_Tool_Xvars (My_Project : in out Arg_Project_Type);
   --  Extracts from the project and stores '-X' options that are specific
   --  for a tool. The default version look for '-X'  options in the list of
   --  values of "<default_switches" attribute in the ""<toolname>" package.
   --  All the other attributes from this list are also stored.
   --
   --  This procedure is not supposed to be used by the tool directly, but it
   --  is used by Extract_Tool_Attributes, and you may want to redefine it for
   --  your tool.

   procedure Extract_Tool_Attributes (My_Project : in out Arg_Project_Type);
   --  First of all, tries to extract from the projects all the '-X' attributes
   --  that are specific to a tool. This is done by calling to
   --  Extract_Tool_Xvars. Then sets the values of all the external variables
   --  using the information stored in external variables table. Raises
   --  Fatal_Error if any problem is detected (such as the name of the
   --  non-existent variable or a wrong value). Then reads and stores all the
   --  tool-specific options and parameters (this part is tool-specific, the
   --  default version does nothing).

   procedure Get_Paths_And_Sources_From_Project
     (My_Project : Arg_Project_Type);
   --  Extracts and stores the project source path (as a set of -I option for
   --  gcc call and as ???), and the list of sources to process.

   procedure Process_Project_File (My_Project : in out Arg_Project_Type'Class);
   --  Combines all the actions desctibed in paragraphs 2..6 in the

   -------------------------------
   --  External variables table --
   -------------------------------

   --  An external variable is the parameter of a '-X<par>' tool option, it
   --  is guaranteed that it has a structure <variable_name>=<value>

   procedure Init_External_Variables_Table;
   --  Cleans up and (re)initialize the storage for external variables. You do
   --  not need to call this procedure before the  first usage of this storage.

   procedure Store_External_Variable
     (Var                  : String;
      Is_From_Command_Line : Boolean);
   --  Checks if the value for the given external variable is already stored
   --  (the check is case-sensitive), and if it does not, stores the value
   --  setting for the given variable. If it does, overrides the stored value
   --  in case if both the stored and new value are given in the tool command
   --  line or if both of them are from the project file.

private
   type Arg_Project_Type is new Project_Tree with record
      Source_Prj : String_Access;
      --  Stores the name of the project file that is a tool argument
   end record;

end ASIS_UL.Projects;
