------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--                     A S I S _ U L . P R O J E C T S                      --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2013, AdaCore                        --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING.  If  not,  write  to  the Free Software Foundation, 51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------
pragma Ada_2012;

with Ada.Characters.Handling;  use Ada.Characters.Handling;
with Ada.Containers.Ordered_Sets;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;

with Hostparm;

with GNATCOLL.VFS;             use GNATCOLL.VFS;

with ASIS_UL.Common;           use ASIS_UL.Common;
with ASIS_UL.Compiler_Options; use ASIS_UL.Compiler_Options;
with ASIS_UL.Options;
with ASIS_UL.Output;           use ASIS_UL.Output;
with ASIS_UL.Source_Table;

package body ASIS_UL.Projects is

   Project_Env      : Project_Environment_Access;
   Project_File_Set : Boolean := False;

   -------------------------------
   --  External variables table --
   -------------------------------

   type X_Var_Record is record
      Var_Name          : String_Access;
      Var_Value         : String_Access;
      From_Command_Line : Boolean;
   end record;

   function "<" (Left, Right : X_Var_Record) return Boolean is
     (To_Lower (Left.Var_Name.all) < To_Lower (Right.Var_Name.all));

   function "=" (Left, Right : X_Var_Record)  return Boolean is
     (To_Lower (Left.Var_Name.all) = To_Lower (Right.Var_Name.all));

   package X_Vars_Sets is new Ada.Containers.Ordered_Sets
     (Element_Type => X_Var_Record);

   X_Vars : X_Vars_Sets.Set;

   ------------------------
   --  Local subprograms --
   ------------------------

   function Is_Ada_File
     (File :       Virtual_File;
      My_Project : Arg_Project_Type)
      return Boolean;
   --  Checks if the given source file is an Ada file.
   --  NOT IMPLEMENTED AT THE MOMENT!!!

   function Is_Externally_Built
     (File :       Virtual_File;
      My_Project : Arg_Project_Type)
      return Boolean;
   --  Checks if the given source file belongs to an externally build library.

   procedure Store_Path_And_I_Options (Dirs : GNATCOLL.VFS.File_Array);
   --  Stores the directories from the argument directore list as a part of
   --  the rool source search path and as -I options used when a tool calls
   --  the compiler to build the tree.

   -----------------------------
   -- Extract_Tool_Attributes --
   -----------------------------

   procedure Extract_Tool_Attributes (My_Project : in out Arg_Project_Type) is
      Vars : Scenario_Variable_Array := My_Project.Scenario_Variables;
      use X_Vars_Sets;
      C        : Cursor;
      Next_Var : X_Var_Record;
   begin

      Extract_Tool_Xvars (My_Project);

      C := First (X_Vars);

      while Has_Element (C) loop

         Next_Var := Element (C);

         for I in Vars'Range loop

            if External_Name (Vars (I)) = Next_Var.Var_Name.all then

               declare
                  Pos_Vals : constant String_List :=
                    My_Project.Possible_Values_Of (Vars (I));
                  Present : Boolean := False;
               begin

                  for J in Pos_Vals'Range loop
                     if Pos_Vals (J).all = Next_Var.Var_Value.all then
                        Present  := True;
                        exit;
                     end if;
                  end loop;

                  if not Present then
                     Error
                       ("value " & Next_Var.Var_Value.all &
                        " is illegal for " & Next_Var.Var_Name.all);
                     raise Parameter_Error;
                  end if;

               end;

               Set_Value (Vars (I), Next_Var.Var_Value.all);
               exit;
            end if;

         end loop;

         C := Next (C);

      end loop;

      My_Project.Change_Environment (Vars);
      My_Project.Recompute_View;

   end Extract_Tool_Attributes;

   ------------------------
   -- Extract_Tool_Xvars --
   ------------------------

   procedure Extract_Tool_Xvars (My_Project : in out Arg_Project_Type) is
      Proj         : constant Project_Type := My_Project.Root_Project;
      Tool_Package : constant String       := Tool_Name.all;

      Attr : constant Attribute_Pkg_List :=
        Build (Tool_Package, "Default_Switches");
      Tool_Switches : String_List_Access;

      Idx : Natural;

   begin
      if Has_Attribute (Proj, Attr) then
         Tool_Switches := Attribute_Value (Proj, Attr);

         for I in Tool_Switches'Range loop
            Idx := Tool_Switches (I)'First;

            if Tool_Switches (I)'Length > 2
              and then
               Tool_Switches (I) (Idx .. Idx + 1) = "-X"
            then
               Store_External_Variable
                 (Var => Tool_Switches (I) (Idx + 2 .. Tool_Switches (I)'Last),
                  Is_From_Command_Line => False);
            end if;

         end loop;

      end if;
   end Extract_Tool_Xvars;

   ----------------------------------------
   -- Get_Paths_And_Sources_From_Project --
   ----------------------------------------

   procedure Get_Paths_And_Sources_From_Project
     (My_Project : Arg_Project_Type)
   is
      Files    : File_Array_Access;
      Not_Used : Boolean;
   begin
      --  Get source search path:
      Store_Path_And_I_Options
        (My_Project.Root_Project.Source_Dirs (Recursive => True));

      --  Get sources:

      declare
         Extensions : constant String_List :=
           Registered_Extensions (Project_Env.all, "Ada");
      begin
         for J in Extensions'Range loop
            Info (Extensions (J).all);
         end loop;
      end;

      if ASIS_UL.Options.No_Argument_File_Specified then
         if Main_Unit = null then
            Files := My_Project.Root_Project.Source_Files
              (Recursive => U_Option_Set);

            if Files'Length = 0 then
               Error (My_Project.Source_Prj.all &
                      "does not contain source files");
               return;
            end if;

            for F in Files'Range loop
               if not Is_Externally_Built (Files (F), My_Project)
                 and then
                  Is_Ada_File (Files (F), My_Project)
               then
                  ASIS_UL.Source_Table.Store_Sources_To_Process
                    (Files (F).Display_Base_Name,
                     Not_Used);
               end if;
            end loop;
         end if;
      end if;
   end Get_Paths_And_Sources_From_Project;

   -----------------------------------
   -- Init_External_Variables_Table --
   -----------------------------------

   procedure Init_External_Variables_Table is
   begin
      X_Vars_Sets.Clear (X_Vars);
   end Init_External_Variables_Table;

   ----------------------------
   -- Initialize_Environment --
   ----------------------------

   procedure Initialize_Environment is
      Gnatls_To_Call : String_Access;
      Tmp            : String_Access;

      Firts_Idx : constant Natural := Tool_Name'First;
      Last_Idx  : constant Natural :=
        Index (Tool_Name.all, "-", Ada.Strings.Backward);
   begin
      if Last_Idx > 0 and then not Hostparm.OpenVMS then
         Gnatls_To_Call :=
           new String'(Tool_Name (Firts_Idx .. Firts_Idx) & "gnatls");
      else
         Gnatls_To_Call := new String'("gnatls");
      end if;

      Initialize (Project_Env);
      Project_Env.Set_Path_From_Gnatls (Gnatls_To_Call.all, Tmp);

      Free (Gnatls_To_Call);
      Free (Tmp);
   end Initialize_Environment;

   -----------------
   -- Is_Ada_File --
   -----------------

   function Is_Ada_File
     (File :       Virtual_File;
      My_Project : Arg_Project_Type)
      return Boolean
   is
      pragma Unreferenced (File, My_Project);
   begin
      return True;
   end Is_Ada_File;

   -------------------------
   -- Is_Externally_Built --
   -------------------------

   function Is_Externally_Built
     (File :       Virtual_File;
      My_Project : Arg_Project_Type)
      return Boolean
   is
      F_Info : constant File_Info    := Info (My_Project, File);
      Proj   : constant Project_Type := Project (F_Info);
      Attr   : constant Attribute_Pkg_String := Build ("", "externally_built");
   begin
      if Has_Attribute (Proj, Attr) then
         if Attribute_Value (Proj, Attr) = "true" then
            return True;
         end if;
      end if;
      return False;
   end Is_Externally_Built;

   ------------------
   -- Is_Specified --
   ------------------

   function Is_Specified (My_Project : Arg_Project_Type) return Boolean is
   begin
      return My_Project.Source_Prj /= null;
   end Is_Specified;

   -----------------------
   -- Load_Tool_Project --
   -----------------------

   procedure Load_Tool_Project (My_Project : in out Arg_Project_Type) is
   begin
      My_Project.Load
        (GNATCOLL.VFS.Create (+My_Project.Source_Prj.all),
         Project_Env);
   end Load_Tool_Project;

   --------------------------
   -- Process_Project_File --
   --------------------------

   procedure Process_Project_File
     (My_Project : in out Arg_Project_Type'Class)
   is
   begin
      if not My_Project.Is_Specified then
         return;
      end if;

      Initialize_Environment;
      Register_Tool_Attributes           (My_Project);
      Load_Tool_Project                  (My_Project);
      Extract_Tool_Attributes            (My_Project);
      Get_Paths_And_Sources_From_Project (My_Project);
   end Process_Project_File;

   ------------------------------
   -- Register_Tool_Attributes --
   ------------------------------

   procedure Register_Tool_Attributes (My_Project : in out Arg_Project_Type) is
      pragma Unreferenced (My_Project);

      First_Idx : Natural := Index (Tool_Name.all, "-", Ada.Strings.Backward);
      Last_Idx  : constant Natural := Tool_Name'Last;
      Dummy     : String_Access;

   begin
      if First_Idx = 0 then
         First_Idx := Tool_Name'First;
      end if;

      Dummy := new String'
        (Register_New_Attribute
           (Name    => "Default_Switches",
            Pkg     => Tool_Name (First_Idx .. Last_Idx),
            Is_List => True));

      if Dummy.all /= "" then
         Error ("cannot parse project file: " & Dummy.all);
         raise Fatal_Error;
      end if;

      Free (Dummy);

   end Register_Tool_Attributes;

   -----------------------------
   -- Store_External_Variable --
   -----------------------------

   procedure Store_External_Variable
     (Var                  : String;
      Is_From_Command_Line : Boolean)
   is
      Var_Name_Start  : constant Natural := Var'First;
      Var_Name_End    :          Natural := Index (Var, "=");
      Var_Value_Start :          Natural;
      Var_Value_End   : constant Natural := Var'Last;

      New_Var_Rec : X_Var_Record;

      use X_Vars_Sets;
      C : Cursor;
   begin
      if Var_Name_End <= Var_Name_Start then
         Error ("wrong parameter of -X option: " & Var);
         raise Parameter_Error;
      else
         Var_Name_End    := Var_Name_End - 1;
         Var_Value_Start := Var_Name_End + 2;
         New_Var_Rec    :=
           (Var_Name  => new String'(Var (Var_Name_Start .. Var_Name_End)),
            Var_Value => new String'(Var (Var_Value_Start .. Var_Value_End)),
            From_Command_Line => Is_From_Command_Line);
      end if;

      C := Find (X_Vars, New_Var_Rec);

      if Has_Element (C) then
         if not Element (C).From_Command_Line
           or else
            Element (C).From_Command_Line = Is_From_Command_Line
         then
            Replace_Element (Container => X_Vars,
                             Position  => C,
                             New_Item  => New_Var_Rec);
         else
            Free (New_Var_Rec.Var_Name);
            Free (New_Var_Rec.Var_Value);
         end if;
      else
         Insert (X_Vars, New_Var_Rec);
      end if;
   end Store_External_Variable;

   ------------------------------
   -- Store_Path_And_I_Options --
   ------------------------------

   procedure Store_Path_And_I_Options (Dirs : GNATCOLL.VFS.File_Array)
   is
   begin
      for F in Dirs'Range loop
         Store_I_Option (Dirs (F).Display_Full_Name);
         Add_To_Source_Search_Path (Dirs (F).Display_Full_Name);
      end loop;

      if Dirs'Length > 0 then
         I_Options_Specified := True;
      end if;
   end Store_Path_And_I_Options;

   --------------------------
   -- Store_Project_Source --
   --------------------------

   procedure Store_Project_Source
     (My_Project         : in out Arg_Project_Type;
      Projet_Source_File : String)
   is
   begin
      if Project_File_Set then
         Error ("cannot have several project files specified");
         raise Parameter_Error;
      else
         Project_File_Set := True;
      end if;

      if Is_Regular_File (Projet_Source_File) then
         My_Project.Source_Prj :=
           new String'(Normalize_Pathname (Projet_Source_File));
      else
         Error ("project file " & Projet_Source_File & " not found");
         raise Parameter_Error;
      end if;
   end Store_Project_Source;

end ASIS_UL.Projects;
