------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2008-2013, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters.Conversions;

with Asis.Elements; use Asis.Elements;
with Asis.Expressions; use Asis.Expressions;

with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Simplifications;     use Ada2Java.Simplifications;
with Ada2Java.Kernel;              use Ada2Java.Kernel;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;

package body Ada2Java.Pragmas is

   Annotation_Name : constant Wide_String := "AJIS";

   Pragma_Annotation_Renaming : constant Wide_String := "ANNOTATION_RENAMING";
   Pragma_Assumed_Escaped     : constant Wide_String := "ASSUME_ESCAPED";
   Pragma_Attached            : constant Wide_String := "ATTACHED";
   Pragma_Lock                : constant Wide_String := "LOCK";
   Pragma_Locking             : constant Wide_String := "LOCKING";
   Pragma_Rename              : constant Wide_String := "RENAME";

   -------------------
   -- Handle_Pragma --
   -------------------

   procedure Handle_Pragma
     (Kernel     : not null access Ada2Java.Kernel.Kernel_Record;
      The_Pragma : Pragma_Element)
   is
      Args : constant Association_List :=
        Pragma_Argument_Associations (The_Pragma);
      Name : Program_Text := Pragma_Name_Image (The_Pragma);

      function Extract_Declaration_From_Arg
        (Arg : Asis.Element) return Asis.Element;

      function Extract_String_From_Arg
        (Arg : Asis.Element) return Wide_String;

      ----------------------------------
      -- Extract_Declaration_From_Arg --
      ----------------------------------

      function Extract_Declaration_From_Arg
        (Arg : Asis.Element) return Asis.Element
      is
         Param  : Asis.Element;
         Result : Asis.Element := Nil_Element;
         Loc    : Location_Handle;
      begin
         Loc := Push_Location (Get_Source_Location (Arg));

         Param := Actual_Parameter (Arg);

         if Expression_Kind (Param) = A_Selected_Component then
            Param := Selector (Param);
         end if;

         if Expression_Kind (Param) = An_Identifier then
            Result := Corresponding_Name_Declaration (Param);
         else
            Trace_With_Location ("identifier expected", Errors_Only);
         end if;

         Pop_Location (Loc);

         return Result;
      end Extract_Declaration_From_Arg;

      -----------------------------
      -- Extract_String_From_Arg --
      -----------------------------

      function Extract_String_From_Arg
        (Arg : Asis.Element) return Wide_String
      is
         Param  : Asis.Element;
         Loc    : Location_Handle;
      begin
         Loc := Push_Location (Get_Source_Location (Arg));

         Param := Actual_Parameter (Arg);

         if Expression_Kind (Param) = A_String_Literal then
            declare
               Name : constant Wide_String := Get_First_Name (Param);
            begin
               Pop_Location (Loc);
               return Name (Name'First + 1 .. Name'Last - 1);
            end;
         else
            Trace_With_Location ("string literal expected", Errors_Only);
         end if;

         Pop_Location (Loc);

         return "";
      end Extract_String_From_Arg;

   begin
      To_Upper (Name);

      if Args'Length = 0 then
         return;
      end if;

      declare
         Tool_Name : Wide_String := Get_First_Name (Args (1));
      begin
         To_Upper (Tool_Name);

         if Tool_Name /= Annotation_Name then
            return;
         end if;
      end;

      if Name = "ANNOTATE" then
         declare
            AJIS_Name : Wide_String := Get_First_Name (Args (2));
         begin
            To_Upper (AJIS_Name);

            if AJIS_Name = Pragma_Assumed_Escaped then

               declare
                  Enablement : Boolean := False;
                  Subprogram : Simple_Subprogram_View_Access;
               begin
                  declare
                     Param : Wide_String := Get_First_Name (Args (3));
                  begin
                     To_Upper (Param);

                     if Param = "TRUE" then
                        Enablement := True;
                     elsif Param = "FALSE" then
                        Enablement := False;
                     else
                        Trace_With_Location
                          ("boolean expected for enablement parameter",
                           Errors_Only);

                        return;
                     end if;

                  end;

                  declare
                     Asis_Element : constant Asis.Element :=
                       Extract_Declaration_From_Arg (Args (4));
                     Element : Simple_Element_View_Access;
                  begin
                     Element := Get_Simple_Element
                       (Kernel, Asis_Element);

                     if Element = null then
                        Trace_With_Location
                          (Get_First_Name (Asis_Element)
                           & " can't be bound.",
                           Errors_And_Warnings);

                        return;
                     elsif Element.all
                       not in Simple_Subprogram_View'Class
                     then
                        Trace_With_Location
                          (Get_First_Name (Asis_Element)
                           & " should be a subprogram",
                           Errors_Only);

                        return;
                     end if;

                     Subprogram := Simple_Subprogram_View_Access (Element);

                     if Subprogram.Can_Be_Bound = No then
                        Trace_With_Location
                          (Get_First_Name (Asis_Element)
                           & " can't be bound", Errors_Only);

                        return;
                     end if;
                  end;

                  declare
                     Param : Wide_String := Extract_String_From_Arg (Args (5));
                  begin
                     To_Upper (Param);

                     for J in Subprogram.Parameters'Range loop
                        declare
                           Param_Name : Wide_String := To_Wide_String
                             (Subprogram.Parameters (J).Name);
                        begin
                           To_Upper (Param_Name);

                           if Param_Name = Param then
                              if not Subprogram.Annotation_Renaming then
                                 Subprogram.Parameters (J).Assume_Stored :=
                                   Enablement;
                              else
                                 Simple_Subprogram_View_Access
                                   (Subprogram.Renaming).Parameters (J).
                                   Assume_Stored := Enablement;
                              end if;

                              return;
                           end if;
                        end;
                     end loop;

                     Trace_With_Location
                       ("can't find parameter: """ & Param & """",
                        Errors_Only);
                  end;
               end;

            elsif AJIS_Name = Pragma_Rename then

               declare
                  Sb : constant Simple_Subprogram_View_Access :=
                    Simple_Subprogram_View_Access
                      (Get_Simple_Element
                           (Kernel, Extract_Declaration_From_Arg (Args (3))));
                  Sb_To_Rename : Simple_Subprogram_View_Access;
                  New_Name     : constant Dynamic_Expression :=
                    To_Dynamic_Expression
                      (Extract_String_From_Arg (Args (4)));

                  procedure Recursive_Rename
                    (Subprogram : Simple_Subprogram_View_Access);

                  procedure Recursive_Rename
                    (Subprogram : Simple_Subprogram_View_Access)
                  is
                     use Simplified_Elements_List;
                     Cur : Simplified_Elements_List.Cursor :=
                       Subprogram.Overriding_Sbs.First;
                  begin
                     Subprogram.Name := New_Name;

                     while Cur /= Simplified_Elements_List.No_Element loop
                        Recursive_Rename
                          (Simple_Subprogram_View_Access
                             (Simplified_Elements_List.Element (Cur)));

                        Cur := Next (Cur);
                     end loop;
                  end Recursive_Rename;
               begin
                  if not Sb.Annotation_Renaming then
                     Sb_To_Rename := Sb;
                  else
                     Sb_To_Rename :=
                       Simple_Subprogram_View_Access (Sb.Renaming);
                  end if;

                  if Sb_To_Rename.Overridden_Sb /= null then
                     Trace_With_Location
                       ("cannot rename an overriding primitive", Errors_Only);
                  else
                     Recursive_Rename (Sb_To_Rename);
                  end if;
               end;

            elsif AJIS_Name = Pragma_Annotation_Renaming then

               declare
                  Asis_Element : constant Asis.Element :=
                    Extract_Declaration_From_Arg (Args (3));
                  Element : constant Simple_Element_View_Access :=
                    Get_Simple_Element (Kernel, Asis_Element);
               begin
                  if Element = null then
                     Trace_With_Location
                       ("bound entity not found: """
                        & Get_First_Name (Asis_Element) & """",
                        Errors_Only);

                     return;
                  elsif Element.Renaming = null then
                     Trace_With_Location
                       ("entity """ & Get_First_Name (Asis_Element) & """"
                        & " is not a renaming",
                        Errors_Only);

                     return;
                  else
                     Element.Annotation_Renaming := True;
                  end if;
               end;

            elsif AJIS_Name = Pragma_Lock then

               declare
                  Asis_Element : constant Asis.Element :=
                    Extract_Declaration_From_Arg (Args (3));
                  Element : Simple_Element_View_Access :=
                    Get_Simple_Element (Kernel, Asis_Element);
                  Param      : constant Wide_String :=
                    Extract_String_From_Arg (Args (4));
               begin
                  if Element = null then
                     Trace_With_Location
                       ("bound entity not found: """
                        & Get_First_Name (Asis_Element) & """",
                        Errors_Only);

                     return;
                  elsif Element.Annotation_Renaming then
                     Element := Element.Renaming;
                  end if;

                  Element.Lock_Var := To_Dynamic_Expression (Param);
               end;

            elsif AJIS_Name = Pragma_Locking then

               declare
                  Asis_Element : constant Asis.Element :=
                    Extract_Declaration_From_Arg (Args (4));
                  Element : Simple_Element_View_Access :=
                    Get_Simple_Element (Kernel, Asis_Element);
                  Param      : Wide_String := Get_First_Name (Args (3));
               begin
                  if Element = null then
                     Trace_With_Location
                       ("bound entity not found: """
                        & Get_First_Name (Asis_Element) & """",
                        Errors_Only);

                     return;
                  elsif Element.Annotation_Renaming then
                     Element := Element.Renaming;
                  end if;

                  To_Upper (Param);
                  Element.Locking_State := Locking_State_Type'Value
                    (To_String (Param));

               end;

            elsif AJIS_Name = Pragma_Attached then

               declare
                  Enablement : Boolean := False;
                  Subprogram : Simple_Subprogram_View_Access;
               begin
                  declare
                     Param : Wide_String := Get_First_Name (Args (3));
                  begin
                     To_Upper (Param);

                     if Param = "TRUE" then
                        Enablement := True;
                     elsif Param = "FALSE" then
                        Enablement := False;
                     else
                        Trace_With_Location
                          ("boolean expected for enablement parameter",
                           Errors_Only);

                        return;
                     end if;

                  end;

                  declare
                     Asis_Element : constant Asis.Element :=
                       Extract_Declaration_From_Arg (Args (4));
                     Element : Simple_Element_View_Access;
                  begin
                     Element := Get_Simple_Element
                       (Kernel, Asis_Element);

                     if Element.all
                     not in Simple_Subprogram_View'Class
                     then
                        Trace_With_Location
                          (Get_First_Name (Asis_Element)
                           & " should be a subprogram",
                           Errors_Only);

                        return;
                     end if;

                     Subprogram := Simple_Subprogram_View_Access (Element);
                  end;

                  --  ??? We should perform some check on the viability of the
                  --  attachement here.

                  if not Subprogram.Annotation_Renaming then
                     Subprogram.Parameters (1).Attached :=
                       Enablement;
                  else
                     Simple_Subprogram_View_Access
                       (Subprogram.Renaming).Parameters (1).
                       Attached := Enablement;
                  end if;
               end;

            else
               Trace_With_Location
                 ("unknown Proxy annotation: """
                  & Get_First_Name (Args (2)) & """",
                  Errors_And_Warnings);
            end if;

         end;
      end if;
   end Handle_Pragma;

end Ada2Java.Pragmas;
