------------------------------------------------------------------------------
--                                Ada2Java                                  --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Address_To_Access_Conversions;

with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with Ada.Exceptions; use Ada.Exceptions;

with Interfaces.C; use Interfaces.C;
with Interfaces.C.Strings; use Interfaces.C.Strings;

with Ada.Tags; use Ada.Tags;

with Ada.Finalization; use Ada.Finalization;

with Ada.Text_IO; use Ada.Text_IO;

with AJIS.Java;

with Ada.Environment_Variables; use Ada.Environment_Variables;

package body AJIS.Internal.Java is

   function To_String (Env : JNI_Env_Access; Str : J_String) return String;
   --  Extract the actual string from the J_String object.

   ---------------
   -- To_String --
   ---------------

   function To_String (Env : JNI_Env_Access; Str : J_String) return String is
      Is_Copy : aliased J_Boolean;

      C_Str : constant Interfaces.C.Strings.chars_ptr :=
        Get_String_UTF_Chars (Env, Str, Is_Copy'Access);
      Ada_Str : constant Unchecked_String_Access :=
        To_Unchecked_String (Convert (C_Str));
      pragma Suppress (Access_Check, Ada_Str);

      Result : constant String := Ada_Str
        (1 .. Integer (Interfaces.C.Strings.Strlen (C_Str)));
   begin
      Release_String_UTF_Chars (Env, Str, C_Str);

      return Result;
   end To_String;

   ---------------------
   -- Get_Java_Method --
   ---------------------

   function Get_Java_Method
     (Class_Name     : String;
      Method_Name    : String;
      Method_Profile : String;
      Is_Static      : Boolean := False) return Java_Method_Access is
   begin
      return new Java_Method'
        (new String'(Class_Name),
         new String'(Method_Name),
         new String'(Method_Profile),
         J_Null_Class,
         J_Null_Method_ID,
         Is_Static);
   end Get_Java_Method;

   --------------------
   -- Get_Java_Class --
   --------------------

   function Get_Java_Class (Class_Name : String) return Java_Class_Access is
   begin
      return new Java_Class'
        (Class_Name => new String'(Class_Name),
         Class => J_Null_Class);
   end Get_Java_Class;

   Clone_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/internal/ada/AdaProxy",
        "cloneNoCopy",
        "()Lcom/adacore/ajis/internal/ada/AdaProxy;");

   Proxy_Cloned_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/IProxy",
        "proxyCloned",
        "(Lcom/adacore/ajis/IProxy;)V");

   Set_Address_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/internal/ada/AdaProxy", "setAddress", "([I)V");

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (This : in out Backlinked_Java_Reference) is
   begin
      This.Ref := J_Null_Object;
      This.Enclosing := new Cross_Lg_Container;
   end Initialize;

   ------------
   -- Adjust --
   ------------

   procedure Adjust (This : in out Backlinked_Java_Reference) is
   begin
      This.Enclosing := new Cross_Lg_Container'(This.Enclosing.all);

      if This.Ref /= J_Null_Object then
         declare
            Env    : aliased JNI_Env_Access;
            Rest   : J_Int := Attach_Current_Thread
              (This.VM, Env'Access, System.Null_Address);

            Garbage : J_Object;

            Clone_Method : J_Method_ID := Get_Id (Clone_Access, Env);
            Clone_Args   : aliased J_Value_Array (1 .. 0);

            Proxy_Cloned_Method : J_Method_ID := Get_Id
              (Proxy_Cloned_Access, Env);
            Proxy_Cloned_Args : aliased J_Value_Array :=
              (1 => To_J_Value (This.Ref));
         begin
            Garbage := This.Ref;

            --  Creates a new Java object and initialize the backlink

            This.Ref :=
              New_Global_Ref
                (Env, Call_Object_Method_A
                     (Env, This.Ref, Clone_Method, Clone_Args));
            This.Is_Global_Ref := True;
            Set_Enclosing_Address (This, null);

            --  Tells the proxy that the object is managed by the native object

            declare
               Owner_Class : J_Class := Get_Class (IProxy_Owner_Class);

               NATIVE_Owner_Id : J_Field_ID := Get_Static_Field_ID
                 (Env,
                  Owner_Class,
                  String'("NATIVE"),
                  "Lcom/adacore/ajis/IProxy$Owner;");

               NATIVE_Owner_Obj : J_Object := Get_Static_Object_Field
                 (Env,
                  Owner_Class,
                  NATIVE_Owner_Id);

               myOwner_Field_Id : J_Field_ID := Get_Field_ID
                 (Env,
                  Get_Class (Ada_Proxy_Class),
                  String'("myOwner"),
                  "Lcom/adacore/ajis/IProxy$Owner;");
            begin
               Set_Object_Field
                 (Env, This.Ref, myOwner_Field_Id, NATIVE_Owner_Obj);
            end;

            --  The newly created object has been created from Ada - it's the
            --  responsibility of the Ada programmer to free it, so the owner
            --  is set to native. In addition, the new proxy must points to
            --  the new object. It does not yet - but will as soon as a native
            --  method is called.

            Call_Void_Method_A
              (Env, This.Ref, Proxy_Cloned_Method, Proxy_Cloned_Args);
         end;
      end if;
   end Adjust;

   --------------
   -- Finalize --
   --------------

   procedure Finalize (This : in out Backlinked_Java_Reference) is
      procedure Free is new Ada.Unchecked_Deallocation
        (Cross_Lg_Container, Cross_Lg_Container_Acc);
   begin
      if This.Ref /= J_Null_Object then
         declare
            No_Arg : aliased J_Value_Array (1 .. 0);
            Env    : aliased JNI_Env_Access;
            Rest   : J_Int :=  Attach_Current_Thread
              (This.VM, Env'Access, System.Null_Address);
         begin
            --  ??? If the enclosing object is Finalized, then there will be
            --  a call done in Java after this delete ref. Need to ensure
            --  that the GC is not called in between. Alternatively, the
            --  Delete_Global_Ref could be done in the glue finalize in that
            --  case instead of here (and we would do only this is enclosing
            --  is not Controlled).

            if This.Is_Global_Ref then
               Delete_Global_Ref (Env, This.Ref);
            else
               Delete_Weak_Global_Ref (Env, This.Ref);
            end if;
         end;
      end if;

      Free (This.Enclosing);
   end Finalize;

   --------------------
   -- Set_Global_Ref --
   --------------------

   procedure Set_Global_Ref
     (This      : in out Backlinked_Java_Reference;
      Is_Global : Boolean)
   is
   begin
      if This.Is_Global_Ref /= Is_Global then
         declare
            Old_Ref : J_Object;
            No_Arg : aliased J_Value_Array (1 .. 0);
            Env    : aliased JNI_Env_Access;
            Rest   : J_Int :=  Attach_Current_Thread
              (This.VM, Env'Access, System.Null_Address);
         begin
            This.Is_Global_Ref := Is_Global;
            Old_Ref := This.Ref;

            if Is_Global then
               This.Ref := New_Global_Ref (Env, Old_Ref);
               Delete_Weak_Global_Ref (Env, Old_Ref);
            else
               This.Ref := New_Weak_Global_Ref (Env, Old_Ref);
               Delete_Global_Ref (Env, Old_Ref);
            end if;
         end;
      end if;
   end Set_Global_Ref;

   ---------------------------
   -- Set_Enclosing_Address --
   ---------------------------

   procedure Set_Enclosing_Address
     (This      : Backlinked_Java_Reference;
      Enclosing : access constant Cross_Language_Class'Class)
   is
   begin
      if This.Ref /= J_Null_Object then
         declare
            Env    : aliased JNI_Env_Access;
            Rest   : J_Int := Attach_Current_Thread
              (This.VM, Env'Access, System.Null_Address);

            Set_Address_Method : J_Method_ID :=
              Get_Id (Set_Address_Access, Env);
            Set_Address_Args : aliased J_Value_Array (1 .. 1);
         begin
            if Enclosing /= null then
               Set_Address_Args :=
                 (1 => To_J_Value (Get_Address_For_Proxy (Env, Enclosing)));
            else
               Set_Address_Args :=
                 (1 => To_J_Value (J_Null_Int_J_Array));
            end if;

            This.Enclosing.Ptr := Enclosing;

            Call_Void_Method_A
              (Env, This.Ref, Set_Address_Method, Set_Address_Args);
         end;
      end if;
   end Set_Enclosing_Address;

   ------------
   -- Get_Id --
   ------------

   function Get_Id
     (This   : Java_Method_Access;
      Env    : JNI_Env_Access) return J_Method_ID is
   begin
      if This.Method_Id = J_Null_Method_ID then
         Initialize (This, Env);
      end if;

      return This.Method_ID;
   end Get_Id;

   ------------
   -- Get_Id --
   ------------

   function Get_Id (This : Java_Method_Access) return J_Method_Id is
   begin
      if This.Method_Id = J_Null_Method_ID then
         raise Internal_Error with "Method not loaded: {class="
           & This.Class_Name.all
           & ", name="
           & This.Method_Name.all
           & ", profile=" & This.Method_Profile.all & "}";
      end if;

      return This.Method_Id;
   end Get_Id;

   ---------------
   -- Get_Class --
   ---------------

   function Get_Class
     (This   : Java_Method_Access;
      Env    : JNI_Env_Access) return J_Class is
   begin
      if This.Class = J_Null_Class then
         Initialize (This, Env);
      end if;

      return This.Class;
   end Get_Class;

   ---------------
   -- Get_Class --
   ---------------

   function Get_Class (This : Java_Method_Access) return J_Class is
   begin
      if This.Class = J_Null_Class then
         raise Internal_Error with "Class not loaded: {name="
           & This.Class_Name.all & "}";
      end if;

      return This.Class;
   end Get_Class;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (This : Java_Method_Access; Env : JNI_Env_Access) is
   begin
      if This.Class = J_Null_Class then
         This.Class := New_Global_Ref
           (Env, Find_Class (Env, This.Class_Name.all));

         if This.Class = J_Null_Class then
            raise Internal_Error with "Class not found: {name="
              & This.Class_Name.all & "}";
         end if;
      end if;

      if This.Method_Id = J_Null_Method_ID then
         declare
            Arg1 : char_array := To_C (This.Method_Name.all);
            Arg2 : char_array := To_C (This.Method_Profile.all);
         begin
            if This.Is_Static then
               This.Method_Id := Get_Static_Method_ID
                 (Env,
                  This.Class,
                  Arg1,
                  Arg2);
            else
               This.Method_Id := Get_Method_ID
                 (Env,
                  This.Class,
                  Arg1,
                  Arg2);
            end if;
         end;

         if This.Method_Id = J_Null_Method_ID then
            raise Internal_Error with "Method not found: {class="
              & This.Class_Name.all
              & ", name="
              & This.Method_Name.all
              & ", profile=" & This.Method_Profile.all & "}";
         end if;
      end if;
   end Initialize;

   ---------------
   -- Get_Class --
   ---------------

   function Get_Class
     (This   : Java_Class_Access;
      Env    : JNI_Env_Access) return J_Class
   is
   begin
      if This.Class = J_Null_Class then
         Initialize (This, Env);
      end if;

      return This.Class;
   end Get_Class;

   ---------------
   -- Get_Class --
   ---------------

   function Get_Class (This   : Java_Class_Access) return J_Class is
   begin
      if This.Class = J_Null_Class then
         raise Internal_Error with "Class not loaded: {name="
           & This.Class_Name.all & "}";
      end if;

      return This.Class;
   end Get_Class;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (This : Java_Class_Access; Env : JNI_Env_Access) is
   begin
      if This.Class = J_Null_Class then
         This.Class := New_Global_Ref
           (Env, Find_Class (Env, This.Class_Name.all));

         if This.Class = J_Null_Class then
            raise Internal_Error with "Class not found: {name="
              & This.Class_Name.all & "}";
         end if;
      end if;
   end Initialize;

   ------------------------------
   -- New_Java_Integer_Wrapper --
   ------------------------------

   New_Java_Integer_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/IntegerRef", "<init>", "(I)V");
   New_Java_Integer_Void_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/IntegerRef", "<init>", "()V");

   function New_Java_Integer_Wrapper
     (Env : JNI_Env_Access; Val : J_Int) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jint, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Integer_Access, Env),
         Get_Id (New_Java_Integer_Access, Env),
         Args);
   end New_Java_Integer_Wrapper;

   function New_Java_Integer_Wrapper
     (Env : JNI_Env_Access) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Integer_Void_Access, Env),
         Get_Id (New_Java_Integer_Void_Access, Env),
         Args);
   end New_Java_Integer_Wrapper;

   ---------------------------
   -- New_Java_Long_Wrapper --
   ---------------------------

   New_Java_Long_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/LongRef", "<init>", "(J)V");
   New_Java_Long_Void_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/LongRef", "<init>", "()V");

   function New_Java_Long_Wrapper
     (Env : JNI_Env_Access; Val : J_Long) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jlong, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Long_Access, Env),
         Get_Id (New_Java_Long_Access, Env),
         Args);
   end New_Java_Long_Wrapper;

   function New_Java_Long_Wrapper
     (Env : JNI_Env_Access) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Long_Void_Access, Env),
         Get_Id (New_Java_Long_Void_Access, Env),
         Args);
   end New_Java_Long_Wrapper;

   ------------------------------
   -- New_Java_Boolean_Wrapper --
   ------------------------------

   New_Java_Boolean_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/BooleanRef", "<init>", "(Z)V");
   New_Java_Boolean_Void_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/BooleanRef", "<init>", "()V");

   function New_Java_Boolean_Wrapper
     (Env : JNI_Env_Access; Val : J_Boolean) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jboolean, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Boolean_Access, Env),
         Get_Id (New_Java_Boolean_Access, Env),
         Args);
   end New_Java_Boolean_Wrapper;

   function New_Java_Boolean_Wrapper
     (Env : JNI_Env_Access) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Boolean_Void_Access, Env),
         Get_Id (New_Java_Boolean_Void_Access, Env),
         Args);
   end New_Java_Boolean_Wrapper;

   -----------------------------
   -- New_Java_Double_Wrapper --
   -----------------------------

   New_Java_Double_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/DoubleRef", "<init>", "(D)V");
   New_Java_Double_Void_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/DoubleRef", "<init>", "()V");

   function New_Java_Double_Wrapper
     (Env : JNI_Env_Access; Val : J_Double) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jdouble, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Double_Access, Env),
         Get_Id (New_Java_Double_Access, Env),
         Args);
   end New_Java_Double_Wrapper;

   function New_Java_Double_Wrapper
     (Env : JNI_Env_Access) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Double_Void_Access, Env),
         Get_Id (New_Java_Double_Void_Access, Env),
         Args);
   end New_Java_Double_Wrapper;

   --------------------------------
   -- New_Java_Character_Wrapper --
   --------------------------------

   New_Java_Character_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/CharacterRef", "<init>", "(C)V");
   New_Java_Character_Void_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/CharacterRef", "<init>", "()V");

   function New_Java_Character_Wrapper
     (Env : JNI_Env_Access; Val : J_Char) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jchar, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Character_Access, Env),
         Get_Id (New_Java_Character_Access, Env),
         Args);
   end New_Java_Character_Wrapper;

   function New_Java_Character_Wrapper
     (Env : JNI_Env_Access) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return New_Object_A
        (Env,
         Get_Class (New_Java_Character_Void_Access, Env),
         Get_Id (New_Java_Character_Void_Access, Env),
         Args);
   end New_Java_Character_Wrapper;

   -----------------------------
   -- New_Java_Object_Wrapper --
   -----------------------------

   New_Java_Object_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/IProxy",
        "NewProxyRef",
        "()Lcom/adacore/ajis/IProxyRef;");

   function New_Java_Object_Wrapper
     (Env : JNI_Env_Access; Val : J_Object) return J_Object
   is
      Args : J_Value_Array (1 .. 0);
   begin
      return Call_Object_Method_A
        (Env,
         Val,
         Get_Id (New_Java_Object_Access, Env),
         Args);
   end New_Java_Object_Wrapper;

   function New_Java_Access_Constructor
     (Env : JNI_Env_Access; Val : J_Object) return J_Object
   is
      Args : J_Value_Array (1 .. 1) := (1 => (Jobject, Val));
   begin
      return New_Object_A
        (Env,
         Get_Class (Access_Meth_Constructor, Env),
         Get_Id (Access_Meth_Constructor, Env),
         Args);
   end New_Java_Access_Constructor;

   ----------------------------
   -- Java_Set_Integer_Value --
   ----------------------------

   Set_Integer_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/IntegerRef", "setValue", "(I)V");

   procedure Java_Set_Integer_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Int)
   is
      Method : J_Method_ID := Get_Id (Set_Integer_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jint, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Integer_Value;

   -------------------------
   -- Java_Set_Long_Value --
   -------------------------

   Set_Long_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/LongRef", "setValue", "(J)V");

   procedure Java_Set_Long_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Long)
   is
      Method : J_Method_ID := Get_Id (Set_Long_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jlong, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Long_Value;

   ----------------------------
   -- Java_Set_Boolean_Value --
   ----------------------------

   Set_Boolean_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/BooleanRef", "setValue", "(Z)V");

   procedure Java_Set_Boolean_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Boolean)
   is
      Method : J_Method_ID := Get_Id (Set_Boolean_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jboolean, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Boolean_Value;

   ---------------------------
   -- Java_Set_Double_Value --
   ---------------------------

   Set_Double_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/DoubleRef", "setValue", "(D)V");

   procedure Java_Set_Double_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Double)
   is
      Method : J_Method_ID := Get_Id (Set_Double_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jdouble, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Double_Value;

   ------------------------------
   -- Java_Set_Character_Value --
   ------------------------------

   Set_Character_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/CharacterRef", "setValue", "(C)V");

   procedure Java_Set_Character_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Char)
   is
      Method : J_Method_ID := Get_Id (Set_Character_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jchar, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Character_Value;

   ------------------------------
   -- Java_Set_Object_Value --
   ------------------------------

   Set_Object_Value_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/IProxyRef",
        "setValue",
        "(Lcom/adacore/ajis/IProxy;)V");

   procedure Java_Set_Object_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Object)
   is
      Method : J_Method_ID := Get_Id (Set_Object_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jobject, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Object_Value;

   -------------------------
   -- Java_Set_Enum_Value --
   -------------------------

   Set_Enum_Value_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/internal/ada/IEnumRef", "setValueInt", "(I)V");

   procedure Java_Set_Enum_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Int)
   is
      Method : J_Method_ID := Get_Id (Set_Enum_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 1) := (1 => (Jint, Val));
   begin
      Call_Void_Method_A (Env, Obj, Method, Values);
   end Java_Set_Enum_Value;

   -----------------------------
   --  Java_Get_Integer_Value --
   -----------------------------

   Get_Integer_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/IntegerRef", "getValue", "()I");

   function Java_Get_Integer_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Int
   is
      Method : J_Method_ID := Get_Id (Get_Integer_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Int_Method_A (Env, Obj, Method, Values);
   end Java_Get_Integer_Value;

   --------------------------
   --  Java_Get_Long_Value --
   --------------------------

   Get_Long_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/LongRef", "getValue", "()J");

   function Java_Get_Long_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Long
   is
      Method : J_Method_ID := Get_Id (Get_Long_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Long_Method_A (Env, Obj, Method, Values);
   end Java_Get_Long_Value;

   -----------------------------
   --  Java_Get_Boolean_Value --
   -----------------------------

   Get_Boolean_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/BooleanRef", "getValue", "()Z");

   function Java_Get_Boolean_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Boolean
   is
      Method : J_Method_ID := Get_Id (Get_Boolean_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Boolean_Method_A (Env, Obj, Method, Values);
   end Java_Get_Boolean_Value;

   ----------------------------
   --  Java_Get_Double_Value --
   ----------------------------

   Get_Double_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/DoubleRef", "getValue", "()D");

   function Java_Get_Double_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Double
   is
      Method : J_Method_ID := Get_Id (Get_Double_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Double_Method_A (Env, Obj, Method, Values);
   end Java_Get_Double_Value;

   ------------------------------
   -- Java_Get_Character_Value --
   ------------------------------

   Get_Character_Value_Access : Java_Method_Access :=
     Get_Java_Method ("com/adacore/ajis/CharacterRef", "getValue", "()C");

   function Java_Get_Character_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Char
   is
      Method : J_Method_ID := Get_Id (Get_Character_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Char_Method_A (Env, Obj, Method, Values);
   end Java_Get_Character_Value;

   ------------------------------
   -- Java_Get_Object_Value --
   ------------------------------

   Get_Object_Value_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/IProxyRef",
        "getValue",
        "()Lcom/adacore/ajis/IProxy;");

   function Java_Get_Object_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Object
   is
      Method : J_Method_ID := Get_Id (Get_Object_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Object_Method_A (Env, Obj, Method, Values);
   end Java_Get_Object_Value;

   --------------------------
   --  Java_Get_Enum_Value --
   --------------------------

   Get_Enum_Value_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/internal/ada/IEnumRef", "getValueInt", "()I");

   function Java_Get_Enum_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Int
   is
      Method : J_Method_ID := Get_Id (Get_Enum_Value_Access, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      return Call_Int_Method_A (Env, Obj, Method, Values);
   end Java_Get_Enum_Value;

   ----------
   -- Dump --
   ----------

   procedure Dump (Val : J_Int_Array) is
   begin
      for J in Val'Range loop
         if J > Val'First then
            Put (", ");
         end if;

         Put (Val (J)'Img);
      end loop;

      New_Line;
   end Dump;

   ----------------
   -- Set_Values --
   ----------------

   procedure Set_Values
     (Env : JNI_Env_Access; Dest : J_Int_J_Array; Src : J_Int_Array) is
   begin
      Set_Int_Array_Region (Env, Dest, 0, Src'Length, Src);
   end Set_Values;

   ----------------
   -- Get_Values --
   ----------------

   type J_Int_Array_Access is access all J_Int_Array;

   function Get_Values
     (Env  : JNI_Env_Access;
      Src  : J_Int_J_Array) return J_Int_Array
   is
      Is_Copy : aliased J_Boolean;

      Result_J_Int_Star : J_Int_Star :=
        Get_Int_Array_Elements (Env, Src, Is_Copy'Access);

      Result_Array : J_Int_Array := Int_Pointers.Value
        (Result_J_Int_Star, ptrdiff_t (Get_Array_Length (Env, Src)));
   begin
      Release_Int_Array_Elements (Env, Src, Result_J_Int_Star, 0);

      return Result_Array;
   end Get_Values;

   --------------------------------
   -- Access_To_Jint_Conversions --
   --------------------------------

   function To_J_Int_J_Array
     (Env : JNI_Env_Access; V : System.Address) return J_Int_J_Array
   is
      Size : J_Size := V'Size / J_Int'Size;
   begin
      if V'Size mod J_Int'Size /= 0 then
         Size := Size + 1;
      end if;

      declare
         subtype Result_Subtype is J_Int_Array (1 .. Integer (Size));
         function Conv is new Ada.Unchecked_Conversion
           (System.Address, Result_Subtype);

         Contents : Result_Subtype := Conv (V);

         Result : J_Int_J_Array := New_Int_Array (Env, Size);
      begin
         Set_Values (Env, Result, Contents);

         return Result;
      end;
   end To_J_Int_J_Array;

   function To_Address
     (Env : JNI_Env_Access; V : J_Int_J_Array) return System.Address
   is
      subtype Contents_Subtype is J_Int_Array
        (1 .. Integer (Get_Array_Length (Env, V)));
      function Conv is new Ada.Unchecked_Conversion
        (Contents_Subtype, System.Address);

      Contents : Contents_Subtype;
   begin
      Contents := Get_Values (Env, V);
      return Conv (Contents);
   end To_Address;

   Get_Access_Method : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/internal/ada/AdaProxy", "getAccess", "()[I");

   function Get_Serialized_Address
     (Env : JNI_Env_Access; V : J_Object) return J_Int_J_Array
   is
      Method : J_Method_ID := Get_Id (Get_Access_Method, Env);
      Values : aliased J_Value_Array (1 .. 0);
   begin
      if V = J_Null_Object then
         return To_J_Int_J_Array (Env, System.Null_Address);
      else
         return Call_Object_Method_A (Env, V, Method, Values);
      end if;
   end Get_Serialized_Address;

   -------------------
   -- Serialization --
   -------------------

   package body Serialization is

      ------------------
      -- To_JintArray --
      ------------------

      function To_J_Int_J_Array
        (Env : JNI_Env_Access; V : Object_Type) return J_Int_J_Array
      is
         Size : J_Size := V'Size / J_Int'Size;
      begin
         if V'Size mod J_Int'Size /= 0 then
            Size := Size + 1;
         end if;

         declare
            subtype Result_Subtype is J_Int_Array (1 .. Integer (Size));

            function Conv is new Ada.Unchecked_Conversion
              (Object_Type, Result_Subtype);

            Result : J_Int_J_Array;
         begin
            Result := New_Int_Array (Env, Size);
            Set_Values (Env, Result, Conv (V));

            return Result;
         end;
      end To_J_Int_J_Array;

      --------------------
      -- To_Object_Type --
      --------------------

      function To_Object_Type
        (Env : JNI_Env_Access; V : J_Int_J_Array) return Object_Type
      is
         subtype Contents_Subtype is J_Int_Array
           (1 .. Integer (Get_Array_Length (Env, V)));
         Contents : Contents_Subtype;

         function Conv is new Ada.Unchecked_Conversion
           (Contents_Subtype, Object_Type);
      begin
         Contents := Get_Values (Env, V);

         return Conv (Contents);
      end To_Object_Type;

   end Serialization;

   --------------------------------
   -- Access_To_Jint_Conversions --
   --------------------------------

   package body Access_To_Jint_Conversions is

      package Access_Serialization is new Serialization (Object_Pointer);

      package Access_Serialization_Cst is new
        Serialization (Object_Pointer_Cst);

      -----------------------
      -- To_Object_Pointer --
      -----------------------

      function To_Object_Pointer
        (V : access The_Type) return Object_Pointer is
      begin
         if V = null then
            return null;
         else
            return V.all'Unchecked_Access;
         end if;
      end To_Object_Pointer;

      ---------------------------
      -- To_Object_Pointer_Cst --
      ---------------------------

      function To_Object_Pointer_Cst
        (V : access constant The_Type) return Object_Pointer_Cst is
      begin
         if V = null then
            return null;
         else
            return V.all'Unchecked_Access;
         end if;
      end To_Object_Pointer_Cst;

      ------------------
      -- To_JintArray --
      ------------------

      function To_JintArray
        (Env : JNI_Env_Access; V : Object_Pointer) return J_Int_J_Array
      is
      begin
         return Access_Serialization.To_J_Int_J_Array (Env, V);
      end To_JintArray;

      ----------------------
      -- To_JintArray_Cst --
      ----------------------

      function To_JintArray_Cst
        (Env : JNI_Env_Access; V : Object_Pointer_Cst) return J_Int_J_Array
      is
      begin
         return Access_Serialization_Cst.To_J_Int_J_Array (Env, V);
      end To_JintArray_Cst;

      ----------------
      -- To_Pointer --
      ----------------

      function To_Pointer
        (Env : JNI_Env_Access; V : J_Int_J_Array) return Object_Pointer
      is
      begin
         return Access_Serialization.To_Object_Type (Env, V);
      end To_Pointer;

      -----------------------------
      --  To_Pointer_From_Object --
      -----------------------------

      function To_Pointer_From_Object
        (Env : JNI_Env_Access; V : J_Object) return Object_Pointer
      is
      begin
         return To_Pointer (Env, Get_Serialized_Address (Env, V));
      end To_Pointer_From_Object;

   end Access_To_Jint_Conversions;

   ----------------
   -- To_J_Value --
   ----------------

   function To_J_Value (Val : J_Boolean) return J_Value is
   begin
      return (Jboolean, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Byte) return J_Value is
   begin
      return (Jbyte, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Char) return J_Value is
   begin
      return (Jchar, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Short) return J_Value is
   begin
      return (Jshort, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Int) return J_Value is
   begin
      return (Jint, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Long) return J_Value is
   begin
      return (Jlong, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Float) return J_Value is
   begin
      return (Jfloat, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Double) return J_Value is
   begin
      return (Jdouble, Val);
   end To_J_Value;

   function To_J_Value (Val : J_Object) return J_Value is
   begin
      return (Jobject, Val);
   end To_J_Value;

   --------------------------------
   -- Get_Env_For_Current_Thread --
   --------------------------------

   function Get_Env_For_Current_Thread
     (VM : Java_VM_Access) return JNI_Env_Access
   is
      Env : aliased JNI_Env_Access;
      Result : J_Int;
   begin
      Result := Attach_Current_Thread (VM, Env'Access, System.Null_Address);

      if Result /= JNI_Ok then
         raise Internal_Error;
      end if;

      return Env;
   end Get_Env_For_Current_Thread;

   ----------------------------------
   -- Get_Closest_Type_Tree_Record --
   ----------------------------------

   function Get_Closest_Type_Tree_Record
     (The_Tag : Ada.Tags.Tag) return Type_Tree_Record_Access
   is
      function Internal
        (Rec : Type_Tree_Record_Access) return Type_Tree_Record_Access
      is
         C : Types_List.Cursor := First (Rec.Children);
      begin
         while C /= Types_List.No_Element loop
            if The_Tag = Element (C).The_Tag then
               return Element (C);
            elsif Is_Descendant_At_Same_Level
              (The_Tag, Element (C).The_Tag)
            then
               return Internal (Element (C));
            end if;

            C := Next (C);
         end loop;

         return Rec;
      end Internal;
   begin
      return Internal (Root_Type'Access);
   end Get_Closest_Type_Tree_Record;

   ------------------------
   -- Create_Java_Object --
   ------------------------

   function Create_Java_Object
     (Env          : JNI_Env_Access;
      The_Tag      : Ada.Tags.Tag;
      Access_Array : J_Int_J_Array) return J_Object
   is
      Rec    : Type_Tree_Record_Access :=
         Get_Closest_Type_Tree_Record (The_Tag);
      Object : J_Object;
   begin
      if Rec.Constructor = null then
         raise Internal_Error with "Type "
           & Ada.Tags.Expanded_Name (The_Tag) & " is not in a bound hierarchy";
      else
         declare
            Meth_Id : J_Method_Id :=
              Get_Id (Rec.Constructor, Env);

            Access_Meth_Id : J_Method_Id :=
              Get_Id (Access_Meth_Constructor, Env);

            Access_Args : J_Value_Array (1 .. 1);
            Args : J_Value_Array (1 .. 1);
         begin
            Access_Args (1) := To_J_Value (Access_Array);

            Args (1) := To_J_Value (New_Object_A
              (Env       => Env,
               Class     => Access_Meth_Constructor.Class,
               Method_Id => Access_Meth_Id,
               Args      => Access_Args));

            Object := New_Object_A
              (Env       => Env,
               Class     => Rec.Constructor.Class,
               Method_ID => Meth_Id,
               Args      => Args);

            Handle_Java_Exception (Env);

            return Object;
         end;
      end if;
   end Create_Java_Object;

   -------------------------------------
   -- Reference_Java_Object_Generator --
   -------------------------------------

   procedure Reference_Java_Object_Constructor
     (The_Tag : Ada.Tags.Tag; Constructor : Java_Method_Access)
   is
      Rec : Type_Tree_Record_Access := Get_Closest_Type_Tree_Record (The_Tag);
      New_Rec : Type_Tree_Record_Access := new Type_Tree_Record;
   begin
      New_Rec.The_Tag := The_Tag;
      New_Rec.Constructor := Constructor;

      Append
        (Rec.Children,
         New_Rec);
      --  ??? The following code generates strange references to tasking...
--           new Type_Tree_Record'
--             (The_Tag   => The_Tag,
--              Generator => Generator,
--              others    => <>));
   end Reference_Java_Object_Constructor;

   ---------------------
   -- Throwable_Image --
   ---------------------

   type J_Throwable_Int is mod System.Memory_Size;

   Get_Throwable_Message_Access : Java_Method_Access :=
     Get_Java_Method
       ("java/lang/Throwable", "getMessage", "()Ljava/lang/String;");

   Get_Class_Access : Java_Method_Access :=
     Get_Java_Method
       ("java/lang/Object", "getClass", "()Ljava/lang/Class;");

   Get_Class_Name_Access : Java_Method_Access :=
     Get_Java_Method
       ("java/lang/Class", "getName", "()Ljava/lang/String;");

   function Throwable_Image
     (Env : JNI_Env_Access; Exc : J_Throwable) return String
   is
      Get_Message      : J_Method_ID := Get_Id
        (Get_Throwable_Message_Access, Env);
      Get_Class      : J_Method_ID := Get_Id
        (Get_Class_Access, Env);
      Get_Class_Name      : J_Method_ID := Get_Id
        (Get_Class_Name_Access, Env);

      Args : J_Value_Array (1 .. 0);

      function To_J_Throwable_Int is new Ada.Unchecked_Conversion
        (J_Throwable, J_Throwable_Int);

      Address : constant String :=
        J_Throwable_Int'Image (To_J_Throwable_Int (Exc));

      Message_Head : constant String := "@["
        & Address (Address'First + 1 .. Address'Last) & "]"
        & ASCII.LF
        & To_String
        (Env,
         Call_Object_Method_A
           (Env,
            Call_Object_Method_A (Env, Exc, Get_Class, Args),
            Get_Class_Name,
            Args));

      J_Message : J_String;

   begin
      J_Message := Call_Object_Method_A (Env, Exc, Get_Message, Args);

      if J_Message = J_Null_Object then
         return Message_Head;
      else
         return Message_Head & ": "
           & To_String (Env, J_Message);
      end if;
   end Throwable_Image;

   ---------------------
   -- Throwable_Value --
   ---------------------

   function Throwable_Value (Exc : String) return J_Throwable is
      function To_J_Throwable is new Ada.Unchecked_Conversion
        (J_Throwable_Int, J_Throwable);

      Start_Pos, End_Pos : Integer := Exc'First;
   begin
      for J in Exc'Range loop
         if Exc (J) = '[' then
            Start_Pos := J + 1;

            exit;
         end if;
      end loop;

      for J in Start_Pos + 1 .. Exc'Last loop
         if Exc (J) = ']' then
            End_Pos := J - 1;

            exit;
         end if;
      end loop;

      if Start_Pos < End_Pos then
         return To_J_Throwable
           (J_Throwable_Int'Value (Exc (Start_Pos .. End_Pos)));
      else
         return J_Null_Throwable;
      end if;
   end Throwable_Value;

   ---------------
   -- Set_Owner --
   ---------------

   Set_Owner_Access : Java_Method_Access :=
     Get_Java_Method
       ("com/adacore/ajis/IProxy",
        "setOwner",
        "(Lcom/adacore/ajis/IProxy$Owner;)V");

   procedure Set_Owner
     (Env       : JNI_Env_Access;
      Obj       : J_Object;
      The_Owner : Owner)
   is
      Owner_Class : J_Class := Get_Class (IProxy_Owner_Class);

      NATIVE_Owner_Id : J_Field_ID := Get_Static_Field_ID
        (Env,
         Owner_Class,
         String'("NATIVE"),
         "Lcom/adacore/ajis/IProxy$Owner;");

      NATIVE_Owner_Obj : J_Object := Get_Static_Object_Field
        (Env,
         Owner_Class,
         NATIVE_Owner_Id);

      PROXY_Owner_Id : J_Field_ID := Get_Static_Field_ID
        (Env,
         Owner_Class,
         String'("PROXY"),
         "Lcom/adacore/ajis/IProxy$Owner;");

      PROXY_Owner_Obj : J_Object := Get_Static_Object_Field
        (Env,
         Owner_Class,
         PROXY_Owner_Id);

      Set_Owner_Args : J_Value_Array (1 .. 1);
   begin
      case The_Owner is
         when Proxy =>
            Set_Owner_Args (1) := To_J_Value (PROXY_Owner_Obj);
         when Native =>
            Set_Owner_Args (1) := To_J_Value (NATIVE_Owner_Obj);
      end case;

      Call_Void_Method_A
        (Env, Obj, Get_Id (Set_Owner_Access, Env), Set_Owner_Args);
   end Set_Owner;

   -------------------
   -- Set_Allocator --
   -------------------

   procedure Set_Allocator
     (Env           : JNI_Env_Access;
      Obj           : J_Object;
      The_Allocator : Allocator)
   is
      Allocator_Class : J_Class := Get_Class (IProxy_Allocator_Class);

      STATIC_Allocator_Id : J_Field_ID := Get_Static_Field_ID
        (Env,
         Allocator_Class,
         String'("STATIC"),
         "Lcom/adacore/ajis/IProxy$Allocator;");

      STATIC_Allocator_Obj : J_Object := Get_Static_Object_Field
        (Env,
         Allocator_Class,
         STATIC_Allocator_Id);

      DYNAMIC_Allocator_Id : J_Field_ID := Get_Static_Field_ID
        (Env,
         Allocator_Class,
         String'("DYNAMIC"),
         "Lcom/adacore/ajis/IProxy$Allocator;");

      DYNAMIC_Allocator_Obj : J_Object := Get_Static_Object_Field
        (Env,
         Allocator_Class,
         DYNAMIC_Allocator_Id);

      UNKNOWN_Allocator_Id : J_Field_ID := Get_Static_Field_ID
        (Env,
         Allocator_Class,
         String'("UNKNOWN"),
         "Lcom/adacore/ajis/IProxy$Allocator;");

      UNKNOWN_Allocator_Obj : J_Object := Get_Static_Object_Field
        (Env,
         Allocator_Class,
         UNKNOWN_Allocator_Id);

      myAllocator_Id : J_Field_ID := Get_Field_ID
        (Env,
         Get_Class (Ada_Proxy_Class),
         String'("myAllocator"),
         "Lcom/adacore/ajis/IProxy$Allocator;");

      Val : J_Object;
   begin
      case The_Allocator is
         when Static =>
            Val := STATIC_Allocator_Obj;
         when Dynamic =>
            Val := DYNAMIC_Allocator_Obj;
         when Unknown =>
            Val := UNKNOWN_Allocator_Obj;
      end case;

      Set_Object_Field (Env, Obj, myAllocator_Id, Val);

      Handle_Java_Exception (Env);
   end Set_Allocator;

   ----------------------------
   -- Get_Simple_Null_Access --
   ----------------------------

   function Get_Simple_Null_Access
     (Env : JNI_Env_Access; Class : J_Class) return J_Int_J_Array;
   pragma Export
     (C,
      Get_Simple_Null_Access,
      "Java_com_adacore_ajis_internal_ada_AdaAccess_getSimpleNullAccess");

   function Get_Simple_Null_Access
     (Env : JNI_Env_Access; Class : J_Class) return J_Int_J_Array
   is
      package Conv is new Access_To_Jint_Conversions (Integer);
   begin
      return Conv.To_JintArray (Env, null);
   end Get_Simple_Null_Access;

   -------------------------
   -- Get_Fat_Null_Access --
   -------------------------

   function Get_Fat_Null_Access
     (Env : JNI_Env_Access; Class : J_Class) return J_Int_J_Array;
   pragma Export
     (C,
      Get_Fat_Null_Access,
      "Java_com_adacore_ajis_internal_ada_AdaAccess_getFatNullAccess");

   function Get_Fat_Null_Access
     (Env : JNI_Env_Access; Class : J_Class) return J_Int_J_Array
   is
      type Arr is array (Integer range <>) of Integer;

      package Conv is new Access_To_Jint_Conversions (Arr);
   begin
      return Conv.To_JintArray (Env, null);
   end Get_Fat_Null_Access;

   -------------------------
   -- Reference_Exception --
   -------------------------

   procedure Reference_Exception
     (Except : Ada.Exceptions.Exception_Id; Java_Class : String) is
   begin
      Ada_To_Java_Exceptions.Insert
        (Except,
         Get_Java_Method
           (Java_Class, "<init>", "(Ljava/lang/String;[I)V"));

      Java_To_Ada_Exceptions.Insert
        (Get_Java_Class (Java_Class),
         Except);
   end Reference_Exception;

   ---------------------------
   -- Create_Java_Exception --
   ---------------------------

   function Create_Java_Exception
     (Env    : JNI_Env_Access;
      Except : Ada.Exceptions.Exception_Occurrence)
      return J_Object
   is
   begin
      if Ada_To_Java_Exceptions.Contains (Exception_Identity (Except)) then
         declare
            Constructor : Java_Method_Access :=
              Ada_To_Java_Exceptions.Element (Exception_Identity (Except));

            Meth_Id : J_Method_Id :=
              Get_Id (Constructor, Env);

            Args : J_Value_Array (1 .. 2);

            Result : J_Object;

            package Exception_Conv is new
              Access_To_Jint_Conversions (Exception_Occurrence);

            Exc : Exception_Conv.Object_Pointer :=
              Exception_Conv.Object_Pointer (Save_Occurrence (Except));

            C_Str : chars_ptr;
            J_Str : J_String;
         begin
            C_Str := New_String (Exception_Message (Except));
            J_Str := New_String_UTF (Env, C_Str);
            Free (C_Str);

            Args (1) := To_J_Value (J_Str);
            Args (2) := To_J_Value (Exception_Conv.To_JintArray (Env, Exc));

            Result := New_Object_A
              (Env       => Env,
               Class     => Constructor.Class,
               Method_ID => Meth_Id,
               Args      => Args);

            return Result;
         end;
      else
         return J_Null_Object;
      end if;
   end Create_Java_Exception;

   ---------------------------
   -- Handle_Java_Exception --
   ---------------------------

   Get_Exception_Addr_Method : Java_Method_Access := Get_Java_Method
     ("com/adacore/ajis/internal/ada/AdaException",
      "getExceptionAddr", "()[I");

   procedure Handle_Java_Exception (Env : JNI_Env_Access) is
      Exc : Interfaces.Java.JNI.J_Throwable :=
        Interfaces.Java.JNI.Exception_Occurred (Env);
   begin
      if Interfaces.Java.JNI."/="
        (Exc, Interfaces.Java.JNI.J_Null_Throwable)
      then
         Interfaces.Java.JNI.Exception_Clear (Env);

         if Is_Instance_Of
           (Env, Exc, Get_Class (Get_Exception_Addr_Method, Env))
         then
            declare
               Args : J_Value_Array (1 .. 0);
               J_Ada_Except : J_Int_J_Array;

               package Exception_Conv is new
                 Access_To_Jint_Conversions (Exception_Occurrence);

               Ada_Except : Exception_Conv.Object_Pointer;
            begin
               J_Ada_Except := Call_Object_Method_A
                 (Env, Exc, Get_Id (Get_Exception_Addr_Method, Env), Args);

               Ada_Except := Exception_Conv.To_Pointer (Env, J_Ada_Except);

               Ada.Exceptions.Reraise_Occurrence (Ada_Except.all);
            end;
         else
            raise AJIS.Java.Java_Exception
            with AJIS.Internal.Java.Throwable_Image (Env, Exc);
         end if;
      end if;
   end Handle_Java_Exception;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Ada.Exceptions.Exception_Id) return Boolean is
   begin
      return Exception_Name (Left) < Exception_Name (Right);
   end "<";

   ---------
   -- "=" --
   ---------

   function "=" (Left, Right : Ada.Exceptions.Exception_Id) return Boolean is
   begin
      return Exception_Name (Left) = Exception_Name (Right);
   end "=";

   ---------
   -- "+" --
   ---------

   function "<" (Left, Right : Java_Class_Access) return Boolean is
   begin
      return Left.Class_Name.all < Right.Class_Name.all;
   end "<";

   ------------------------------
   --  Create_Java_Environment --
   ------------------------------

   procedure Initialize_JVM_Parameters (Params : in out Java_VM_Init_Args)
   is
      Args : Java_VM_Option_Array_Access := new Java_VM_Option_Array (1 .. 1);
      pragma Convention (C, Args);
   begin
      if Exists ("CLASSPATH") then
         Args (1).Option_String :=
           New_String ("-Djava.class.path=" & Value ("CLASSPATH"));
      end if;

      Params := (Options => Args, N_Options => 1, others => <>);
   end Initialize_JVM_Parameters;

   --------------------
   -- Initialize_JNI --
   --------------------

   procedure Initialize_JNI (Env : JNI_Env_Access) is
   begin
      --  Initialize AJIS methods

      Initialize (Clone_Access, Env);
      Initialize (Proxy_Cloned_Access, Env);
      Initialize (Set_Address_Access, Env);
      Initialize (New_Java_Integer_Access, Env);
      Initialize (New_Java_Integer_Void_Access, Env);
      Initialize (New_Java_Long_Access, Env);
      Initialize (New_Java_Long_Void_Access, Env);
      Initialize (New_Java_Boolean_Access, Env);
      Initialize (New_Java_Boolean_Void_Access, Env);
      Initialize (New_Java_Double_Access, Env);
      Initialize (New_Java_Double_Void_Access, Env);
      Initialize (New_Java_Character_Access, Env);
      Initialize (New_Java_Character_Void_Access, Env);
      Initialize (New_Java_Object_Access, Env);
      Initialize (Set_Integer_Value_Access, Env);
      Initialize (Set_Long_Value_Access, Env);
      Initialize (Set_Boolean_Value_Access, Env);
      Initialize (Set_Double_Value_Access, Env);
      Initialize (Set_Character_Value_Access, Env);
      Initialize (Set_Object_Value_Access, Env);
      Initialize (Set_Enum_Value_Access, Env);
      Initialize (Get_Integer_Value_Access, Env);
      Initialize (Get_Long_Value_Access, Env);
      Initialize (Get_Boolean_Value_Access, Env);
      Initialize (Get_Double_Value_Access, Env);
      Initialize (Get_Character_Value_Access, Env);
      Initialize (Get_Object_Value_Access, Env);
      Initialize (Get_Enum_Value_Access, Env);
      Initialize (Get_Access_Method, Env);
      Initialize (Get_Throwable_Message_Access, Env);
      Initialize (Get_Class_Access, Env);
      Initialize (Get_Class_Name_Access, Env);
      Initialize (Set_Owner_Access, Env);
      Initialize (Get_Exception_Addr_Method, Env);
      Initialize (Access_Meth_Constructor, Env);

      --  Initialize AJIS classes

      Initialize (Native_Exception_Class, Env);
      Initialize (IProxy_Owner_Class, Env);
      Initialize (Ada_Proxy_Class, Env);
      Initialize (IProxy_Allocator_Class, Env);
   end Initialize_JNI;

begin
   Access_Meth_Constructor := Get_Java_Method
     ("com/adacore/ajis/internal/ada/AdaAccess", "<init>", "([I)V");
   Native_Exception_Class := Get_Java_Class
     ("com/adacore/ajis/NativeException");
   IProxy_Owner_Class := Get_Java_Class
     ("com/adacore/ajis/IProxy$Owner");
   Ada_Proxy_Class := Get_Java_Class
     ("com/adacore/ajis/internal/ada/AdaProxy");
   IProxy_Allocator_Class := Get_Java_Class
     ("com/adacore/ajis/IProxy$Allocator");
end AJIS.Internal.Java;
