------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--                    A S I S _ U L . U T I L I T I E S                     --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2007-2009, AdaCore                     --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains ASIS utilities (both ASIS secondary queries and ASIS
--  extensions that directly operates on the GNAT tree) that are used more
--  then in one place (either in two or more tools or in some tool and in the
--  ASIS UL itself). This package can be considered as a starting point for
--  adding this or that utility to Asis.Extensions.

with Asis;                       use Asis;
with Asis.Extensions.Flat_Kinds; use Asis.Extensions.Flat_Kinds;

package ASIS_UL.Utilities is

   function First_Name (Dcl : Asis.Element) return Asis.Element;
   --  Returns the first defining name from an argument declaration.
   --
   --  Appropriate Element_Kinds:
   --     A_Declaration
   --
   --  ??? Move to Asis.Extensions???

   function Get_Called_Element (Call : Asis.Element) return Asis.Element;
   --  Supposing that the argument Is_Call, tries to compute the called
   --  element, depending on the kind of the call. If the called element
   --  can be detected, returns Corresponding_Element for it, otherwise
   --  returns Nil_Element.
   --
   --  What about calls to attribute subprograms in case when the corresponding
   --  attribute is predefined???

   function Get_Call_Parameters
     (Call       : Asis.Element;
      Normalized : Boolean := False)
      return       Asis.Element_List;
   --  Returns the parameter list from the call. Combines the functionality of
   --  Asis.Statements.Call_Statement_Parameters and
   --  Asis.Expressions.Function_Call_Parameters
   --
   --  Appropriate Expression_Kinds:
   --    A_Function_Call
   --
   --  Appropriate Statement_Kinds:
   --    An_Entry_Call_Statement
   --    A_Procedure_Call_Statement
   --
   --  Returns Element_Kinds:
   --    A_Parameter_Association
   --
   --  ??? Move to Asis.Extensions???

   function Get_Type_Structure (Decl : Asis.Element) return Asis.Element;
   --  This function gets to the type declaration that defines the actual
   --  structure of the type represented by argument type or subtype
   --  declaration. It unwinds subtyping and derivations and goes from private
   --  to full type declarations till it stops either at formal type
   --  declaration or at derived type with non-empty extension part (this
   --  gives the possibility to analyze extension components, this is the
   --  difference of this query from the standard
   --  Asis.Definitions.Corresponding_Type_Structure query) or at the
   --  declaration of non-private non-derived type. (If the argument is
   --  An_Incomplete_Type_Declaration kind, it is returned unchanged.
   --
   --  Appropriate Declaration_Kinds:
   --    An_Ordinary_Type_Declaration
   --    A_Task_Type_Declaration
   --    A_Protected_Type_Declaration
   --    An_Incomplete_Type_Declaration
   --    A_Tagged_Incomplete_Type_Declaration
   --    A_Private_Type_Declaration
   --    A_Private_Extension_Declaration
   --    A_Subtype_Declaration
   --    A_Formal_Type_Declaration
   --
   --  Returns Declaration_Kinds:
   --    An_Ordinary_Type_Declaration
   --    A_Task_Type_Declaration
   --    A_Protected_Type_Declaration
   --    An_Incomplete_Type_Declaration
   --    A_Tagged_Incomplete_Type_Declaration
   --    A_Formal_Type_Declaration

   function Get_Subtype_Structure (Def : Asis.Element) return Asis.Element;
   --  Similar to Get_Type_Structure, but works on A_Subtype_Indication
   --  and An_Access_Definition (An_Anonymous_Access_To_Variable and
   --  An_Anonymous_Access_To_Constant) Elements.

   function Inheritance_Depth (Type_Def : Asis.Element) return Natural;
   --  Assuming that Type_Def is A_Type_Definition element, computes the
   --  inheritance depth for the type. For tagged derived type, private
   --  extension and interface type the inheritance depth is the length of the
   --  longest path in the inheritance tree from the root to a leaf, for any
   --  other type the inheritance depth is 0.

   function Interface_List (Decl : Asis.Element) return Asis.Element_List;
   --  If May_Have_Interface_List (Decl), returns the list of names from the
   --  interface list. Otherwise raises Asis_Inappropriate_Element.

   function Normalize_Reference (Ref : Asis.Element) return Asis.Element;
   --  This function is supposed to be called for the ASIS Elements
   --  representing a subtype mark. A subtype mark can be represented by
   --  an Element of one of the tree following kinds:
   --
   --       An_Identifier
   --       A_Selected_Component
   --       An_Attribute_Reference
   --
   --  This function "prepares" its argument for applying the ASIS
   --  Corresponding_Name_Definition and Corresponding_Name_Declaration
   --  queries, that is, returns its argument if it is of An_Identifier kind,
   --  returns the selector of the argument if it is of A_Selected_Component
   --  kind, and applies itself to the attribute prefix in case of
   --  An_Attribute_Reference
   --
   --  ??? Move to Asis.Extensions???

   function Primitive_Of_Type (Op : Asis.Element) return Asis.Element;
   --  Providing that Op is a declaration of a dispatching operation of some
   --  tagged type, returns the declaration of this type. This function is
   --  equivalent to Asis.Extensions.Primitive_Owner (except that it goes one
   --  step up the ASIS tree - from type definition to type declaration) in
   --  case if the operand is an explicit declaration, but differs in case of
   --  implicit declarations. This function does not check its argument, the
   --  caller is responsible for providing only the subprogram declaration
   --  Elements that denote the dispatching primitive operations

   function Unwind_Exception_Renamings
     (Exc  : Asis.Element)
      return Asis.Element;
   --  Provided that Exc is a defining name denoting an exception, checks if
   --  this name is defined by exception renamimg declaration, and if it is,
   --  unwinds the renamings and returns the corresponding defining name from
   --  the exception declaration. Otherwise returns the argument unchanged.

   --------------------
   -- Test functions --
   --------------------

   function Adds_New_Nesting_Level
     (El_Kind : Flat_Element_Kinds)
      return    Boolean;
   --  Checks if the element of the given kind adds a new level pf construct
   --  nesting, according to the GNAT -gnatyLnnn option.

   function Can_Create_Return_Object (SM : Asis.Element) return Boolean;
   --  Provided that SM is a subtype mark represented by a simple name, checks
   --  if in a function body it is possible to create a local object of the
   --  denoted subtype and to return it as a function result. Currently we
   --  check that the type denoted by SM is not limited, and that SM does not
   --  denote an unconstrained array subtype or unconstrained subtype with
   --  nodefault discriminant values.

   function Can_Have_Elaboration_Calls (El : Asis.Element) return Boolean;
   --  Checks if the argument ELement can issue a call as a part of its
   --  evaluation or elaboration. (For example, as a part of computing a
   --  default parameter or component value).

   function Does_Not_Add_New_Components (El : Asis.Element) return Boolean;
   --  Checks if the argument is a A_Derived_Record_Extension_Definition that
   --  does not add any new component to a type. Returns False for any
   --  unexpected element.
   --
   --  Expected Type_Kinds:
   --     A_Derived_Record_Extension_Definition

   function Contains_Raise_Stmt (C : Asis.Element) return Boolean;
   --  Checks if the argument construct contains a raise statement.

   function Has_Pragma_Inline (Subpr : Element) return Boolean;
   --  Checks if Subpr is a subprogram declaration or a subprogram body to
   --  which the Inline pragma is applied, or a subprogram instantiation that
   --  is inlined because pragma Inline is applied to the corresponding
   --  template. Returns False for any unexpected element.
   --
   --  Expected Declaration_Kinds:
   --
   --    A_Procedure_Declaration
   --    A_Function_Declaration
   --    A_Procedure_Body_Declaration
   --    A_Function_Body_Declaration
   --    A_Procedure_Instantiation
   --    A_Function_Instantiation

   function Is_Call_To_Attribute_Subprogram
     (El   : Asis.Element)
      return Boolean;
   --  Checks if the argument element is a call to an attribute subprogram,
   --  this check is used if ASIS returns Nil_Element as the corresponding
   --  called subprogram

   function Is_Derived_From
     (Descendant : Element;
      Ancestor   : Element)
      return       Boolean;
   --  Checks if Descendant is the declaration of a type that is derived
   --  (directly or indirectly) from Ancestor (that is also supposed to be
   --  a type declaration Element). Returns False if any of arguments are not
   --  type declaration Elements or if they represent the same type.

   function Is_Executable_Body (El : Element) return Boolean;
   --  Checks if its argument represents the executable body, that is, the
   --  body for which it makes sense to compute the complexity metrics. At
   --  the moment, we compute complexity for subprogram, task and entry bodies.
   --  We do not compute the complexity for protected bodies (they just do not
   --  contain statements on their own). For a package body this function
   --  returns true only if the body contains statements of its own.

   function Is_Handled
     (Exc  : Asis.Element;
      H    : Asis.Element)
      return Boolean;
   --  Provided that Exc is a defining name from an exception declaration, and
   --  H is of An_Exception_Handler type, checks if the exception is handled
   --  by the handler in such a way that no exception is raised as the result
   --  of handling

   function Is_Imported_Subprogram (El : Asis.Element) return Boolean;
   --  Checks if its argument is a subprogram declaration completed by the
   --  pragma Import or an instantiation of a generic subprogram that is
   --  completed by the pragma Import.
   --
   --  Returns False for any unexpected Element.
   --
   --  Expected Declaration_Kinds:
   --
   --    A_Procedure_Declaration
   --    A_Function_Declaration
   --    A_Procedure_Instantiation
   --    A_Function_Instantiation,

   function Is_In_Visible_Part
     (Decl  : Element;
      Scope : Element)
      return  Boolean;
   function Is_In_Private_Part
     (Decl  : Element;
      Scope : Element)
      return  Boolean;
   --  Check if Decl is declared in the visible (private) part of Scope.
   --  Currently this function is implemented only for the case when Scope is
   --  either a package or a generic package, it may be extended for task and
   --  protected definitions as scopes. Returns False for any unexpected
   --  elements. This function supposes that for both Decl and Scope
   --  Is_Part_Of_Instance and Is_Part_Of_Implicit is False, but it does not
   --  check this itself.
   --
   --  Expected Element_Kinds for Decl:
   --     A_Declaration
   --
   --  Expected Declaration_Kinds for Scope:
   --     A_Package_Declaration
   --     A_Generic_Package_Declaration

   function Is_Indefinite_Subtype (SM : Asis.Element) return Boolean;
   --  If SM is a subtype mark represented by a simple or expanded name, check
   --  that the denoted subtype is indefinite. Returns False for any unexpected
   --  element (that is, returns False for an attribute reference used as a
   --  subtype mark). A caller is respondible for the fact that SN indeed is a
   --  subtype mark

   function Is_Non_Structural_Statement
     (Stmt         : Element;
      Exit_Is_Goto : Boolean := True)
      return         Boolean;
   --  Check if this statement argument is a non-structural control statement.
   --  If Stmt is not a control statement (see the list of expected statement
   --  kinds), the result is always False.
   --
   --  A control statement is considered as non-structural, if it contains some
   --  statement which transfers the control outside this control statement,
   --  and this does not allow to treat this control statement as one-entry and
   --  one-exit control structure.
   --
   --  The following ways of transferring the control are possible:
   --  - goto statement (if it transfers the control outside the control
   --    statement in question)
   --  - return statement
   --  - raise statement
   --  - terminate alternative in a selective accept
   --  - exit statement is considered as (non-structural) control transfer if
   --    Exit_Is_Goto is set ON)
   --
   --  Different forms of a select statement are considered as non-structural
   --  only if they contain any of the transfer control statements listed above
   --  (that is, asynchronous select is not considered as non-structural
   --  statement on its own)
   --
   --  Expected statement kinds:
   --    An_If_Statement
   --    A_Case_Statement
   --    A_Loop_Statement
   --    A_While_Loop_Statement
   --    A_For_Loop_Statement
   --    A_Selective_Accept_Statement
   --    A_Timed_Entry_Call_Statement
   --    A_Conditional_Entry_Call_Statement
   --    An_Asynchronous_Select_Statement

   function Is_Program_Unit (El : Element) return Boolean;
   --  Checks if its argument represents a program unit for which it makes
   --  makes sense to compute some metrics. See the body for the details.
   --  Note, that renamings, instantiations and (generic) subprogram
   --  declarations are considered as program units only if they are library
   --  items.

   function Is_Publically_Accessible (Decl : Element) return Boolean;
   --  Checks if Decl is A_Declaration Element that can be accessed from
   --  another (non-private) compilation unit. That is, returns True a
   --  declaration located in the visible part of a library package or a
   --  library generic package, except for the case when the declaration is
   --  located in the private part of a local package. Returns False for any
   --  declaration located in a private library unit.
   --
   --  Returns False for any ubexpected element.
   --
   --  Expected Element_Kinds:
   --     A_Declaration

   function Is_RM_Program_Unit (El : Element) return Boolean;
   --  Checks if the argument corresponds to the notion of a program unit as
   --  it is defioned in RM95 10.1(1) ("A program unit is either a package,
   --  a task unit, a protected unit, a protected entry, a generic unit, or
   --  an explicitly declared subprogram other than an enumeration literal.")

   function Is_Static_Loop (Loop_Stmt : Element) return Boolean;
   --  Checks if its argument is a static for-loop (that is, the discrete
   --  subtype indication in the loop parameter specification is static).
   --  Returns False for any unexpected element

   function May_Have_Interface_List (Decl : Element) return Boolean;
   --  Checks if Decl represents a declaration that may have an interface list
   --  as a component thereof. Returns False for any unexpected element.
   --
   --  Expected Declaration_Kinds:
   --     An_Ordinary_Type_Declaration
   --     A_Formal_Type_Declaration
   --     A_Private_Extension_Declaration
   --     A_Task_Type_Declaration
   --     A_Protected_Type_Declaration
   --     A_Single_Task_Declaration
   --     A_Single_Protected_Declaration

end ASIS_UL.Utilities;
