------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--             A S I S _ U L . C O M P I L E R _ O P T I O N S              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                     Copyright (C) 2004-2009, AdaCore                     --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING.  If not,  write  to the  Free Software Foundation,  51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

--  This package provides simple utilities and data structures for processing
--  and storing the ASIS tool arguments that have to be passed to the compiler
--  when the compiler is called to create the tree file for ASIS

with GNAT.OS_Lib; use GNAT.OS_Lib;

package ASIS_UL.Compiler_Options is

   --  To use the resources provided by this package, the tool is supposed to
   --  use the following way of parameter processing and creating the trees
   --  for ASIS:
   --
   --  ..._Environment.Initialize.Scan_Parameters should contain the following
   --  sequence of statements (provided that GNAT.Command_Line is used):
   --
   --       Initialize_Option_Scan
   --        (Stop_At_First_Non_Switch => False,
   --         Section_Delimiters       => "cargs");
   --      loop
   --         case
   --            GNAT.Command_Line.Getopt
   --              (<tool-specific options, may include compiler options>)
   --         is
   --            ...
   --            when 'I' | 'g' =>
   --               if Full_Switch = "gnatec" then
   --                  Store_gnatec_Option (Parameter);
   --               elsif Full_Switch = "I"
   --                  Store_I_Option (Parameter);
   --               ...
   --               end if;
   --            ...
   --         end case;
   --      end loop;
   --
   --      ...
   --
   --      Read_Tool_Arguments; -- Any procedure or operator sequence that
   --                           --  reads tool argument(s)
   --
   --      ...
   --
   --      Process_cargs_Section;
   --
   --  ..._Environment.Initialize.Check_Parameters should contain a call to
   --  Set_Arg_List.
   --
   --  Calls to Asis.Extensions.Compile should use
   --  ASIS_UL.Compiler_Options.Arg_List as an actual for Args parameter.
   --
   --  The call to Initialize_Option_Scan may define other option sections,
   --  if so, the processing for these sections should be provided along with
   --  the call to Process_cargs_Section.

   Arg_List : Argument_List_Access;
   --  This variable should contain a full list of compilation options to be
   --  passed to gcc when gcc is called from the tool to create the tree for
   --  ASIS. This list does not have to contain the following options
   --  '-c -gnatc -gnatt -gnatws -x ada', they are always set by the
   --  Asis.Extensions.Compile routine. The options that are supposed to be
   --  represented by this variable include search path options, reference
   --  to configuration file etc.

   Tool_I_Options : Argument_List_Access;
   --  This variable should contain a full list of -I options that are set as
   --  tool arguments or are the result of parsing the ADA_PRJ_INCLUDE_FILE
   --  file. May be used to call some GNAT tools needed by the tool (such as
   --  gnatbind).

   Source_Search_Path : String_Access;
   --  This variable should contain a path (in the format suitable for passing
   --  to Locate_Regular_File procesure) to look for the source files specified
   --  as tool agruments or searched as sources that are needed by the tool to
   --  complete the processing.

   Custom_RTS : String_Access;
   --  If --RTS option is specified, contains the paraneter of this option

   Look_Into_Current_Dir : Boolean := True;
   --  Flag indicating if we have to look into the current dir when looking
   --  for needed sources. Is turned of if '-I-' option is set. Is ignored
   --  if a tool is called with a project file

   procedure Store_Path_Option
     (Switch : String;
      Path   : String);
   --  Stores the switch that contains a path as a parameter. A switch can be
   --  any switch other then -I and -gnatec. (For these options two specific
   --  procedures - Store_I_Option and Store_gnatec_Option should be used
   --  respectively. A path is converted into absolute normalized form. The
   --  actual for Switch should represent the needed switch in the form that
   --  allows to get the right form of the switch by concatenating the path

   procedure Store_I_Option (Path : String);
   --  Similar to Store_Path_Option, but takes into account -I- option as well.
   --  In case if -I- it is stored as is, because if "-" is passed as a
   --  parameter of path normalization, the result is an empty string.

   procedure Store_GNAT_Option_With_Path (Option : String; Path : String);
   --  Similar to Store_Path_Option, but takes into account that for the
   --  Option the path parameter may or may not be separated by the space
   --  character or '='

   procedure Store_Option (Switch : String);
   --  Stores the option as is.

   procedure Set_Arg_List;
   --  Assigns the value to Arg_List variable.

   procedure Process_cargs_Section (No_Preprocessing : Boolean := False);
   --  Implements the processing of all the options in -cargs section. Note
   --  that this procedure does NOT call Initialize_Option_Scan to define
   --  'cargs' section.
   --  If No_Preprocessing is set ON, then this procedure raises
   --  Parameter_Error with the corresponding error message for the
   --  preprocessor options ('-gnatep', '-gnateD'). This prevents a tool
   --  from calling the proprocessor for an argument source before processing
   --  the source.
   --  If -cargs section contains '-gnata' option, this option is just skipped.
   --  '-gnata' should not be pacced to the compiler call that creates the tree
   --  because it may cause tree transformations that are not good for ASIS.

   procedure Process_ADA_PRJ_INCLUDE_FILE;
   --  Scans the temporary file with source directories created by the GNAT
   --  driver and stores directories listed in this file as '-I' options to be
   --  passed to the calls to compiler. The name of this file is supposed to be
   --  stored as the value of ADA_PRJ_INCLUDE_FILE environment variable. If
   --  this variable is not set, does nothing

   procedure Set_Source_Search_Path;
   --  On the base of compiler -I options stored as a result of processing tool
   --  arguments and/or ADA_PRJ_INCLUDE_FILE file, forms the values of
   --  Source_Search_Path and Tool_I_Options

   ---------------------------------
   -- Source search path iterator --
   ---------------------------------

   --  This iterator allows to go through the source search path set by
   --  '-I' options

   procedure Reset_Search_Path_Iterator;
   --  Resets the iterator into initial state

   function No_More_Source_Dir return Boolean;
   --  Checks if there are more source dir(s) to iterate through. As a side
   --  effect, makes the iterator prepared to return the next source dir

   function Next_Source_Dir return String;
   --  Returns the (full normalized) name of the next source dir. Should be
   --  called after the call to No_More_Source_Dir

   --  The correct way of using this iterator is:

   --  Reset_Search_Path_Iterator;
   --
   --  while not No_More_Source_Dir loop
   --     Do_Something_With (Next_Source_Dir);
   --  end loop;

end ASIS_UL.Compiler_Options;
