------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--     A S I S _ U L . S O U R C E _ T A B L E . P R O C E S S I N G .      --
--                      A S I S _ P R O C E S S I N G                       --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                   Copyright (C) 2007-2009, AdaCore                       --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not, write to the Free Software Foundation, 59 Temple Place  --
--  - Suite 330, Boston,                                                    --
--                                                                          --
-- ASIS UL is maintained by ACT Europe (http://www.act-europe.fr).          --
--                                                                          --
------------------------------------------------------------------------------

--  This is a placeholder for the routine that implements a tool-specific
--  actions performed for a given compilation unit. Because this is
--  tool-specific, this placeholder does nothing except changing the source
--  status to avoid cycling in source processing.

with Ada.Directories;             use Ada.Directories;
with Ada.Strings.Wide_Unbounded;  use Ada.Strings.Wide_Unbounded;

with Asis.Implementation;
with Asis.Iterator;
with Asis.Ada_Environments;
with Asis.Elements;          use Asis.Elements;

with GNAT.Directory_Operations; use GNAT.Directory_Operations;

with Ada2Java;                      use Ada2Java;
with Ada2Java.Bound_Elements;       use Ada2Java.Bound_Elements;
with Ada2Java.Bound_Units;          use Ada2Java.Bound_Units;
with Ada2Java.Config;               use Ada2Java.Config;
with Ada2Java.Kernel;               use Ada2Java.Kernel;
with Ada2Java.Packages;             use Ada2Java.Packages;
with Ada2Java.Utils;                use Ada2Java.Utils;
with Ada2Java.Simplifications;      use Ada2Java.Simplifications;
with Ada2Java.Pragmas;              use Ada2Java.Pragmas;

with Ada.Exceptions; use Ada.Exceptions;
with Ada.Characters.Conversions; use Ada.Characters;

with Ada.Text_IO; use Ada.Text_IO;
with Ada.Wide_Text_IO; use Ada.Wide_Text_IO;

separate (ASIS_UL.Source_Table.Processing)
procedure ASIS_Processing (CU : Asis.Compilation_Unit; SF : SF_Id) is

   Current_Package : Package_Handle;
   Declarations_To_Bind : Simplified_Elements_List.List;

   procedure Pre_Procedure
     (Element :     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer);

   procedure Post_Procedure
     (Element :     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer);

   -------------------
   -- Pre_Procedure --
   -------------------

   procedure Pre_Procedure
     (Element :     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer)
   is
      pragma Unreferenced (State);
      Loc : Location_Handle;
      Simple_Element : Simple_Element_View_Access;
   begin
      if Is_Private (Element) then
         Control := Abandon_Children;
      elsif Declaration_Kind (Element) = A_Package_Declaration then
         Control := Continue;
      elsif Element_Kind (Element) = A_Declaration
        or else Element_Kind (Element) = A_Defining_Name
      then
         if Declaration_Kind (Element) in An_Object_Declaration
           or else Declaration_Kind (Element) = An_Object_Renaming_Declaration
         then
            Control := Continue;
         else
            Control := Abandon_Children;
            Loc := Push_Location (Get_Source_Location (Element));
            Simple_Element := Get_Simple_Element (The_Kernel, Element);

            if Simple_Element /= null and then Simple_Element.Can_Be_Bound then
               Declarations_To_Bind.Append (Simple_Element);
            end if;

            Pop_Location (Loc);
         end if;
      elsif Element_Kind (Element) = A_Pragma then
         Loc := Push_Location (Get_Source_Location (Element));
         Handle_Pragma (The_Kernel, Element);
         Pop_Location (Loc);
      end if;
   exception
      when Silent_Not_Supported =>
         Pop_Location (Loc);

      when E : others =>
         Trace_With_Location
           ("Unexpected exception - "
            & Conversions.To_Wide_String (Exception_Information (E)),
            Level => Errors_Only);

         Pop_Location (Loc);
   end Pre_Procedure;

   --------------------
   -- Post_Procedure --
   --------------------

   procedure Post_Procedure
     (Element :     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer) is
      pragma Unreferenced (Element, Control, State);
   begin
      null;
   exception
      when others =>
         null;
   end Post_Procedure;

   procedure Traverse_Node is new Asis.Iterator.Traverse_Element
     (Integer, Pre_Procedure, Post_Procedure);

   Unit_Spec : Asis.Compilation_Unit;

   The_Declaration : Asis.Declaration;

   State   : Integer := 0;
   Control : Asis.Traverse_Control := Asis.Continue;

begin
   Unit_Spec := Corresponding_Declaration (CU);

   if Unit_Kind (Unit_Spec) = A_Package then
      The_Declaration := Asis.Elements.Unit_Declaration (Unit_Spec);

      Current_Package := Get_Or_Create_Package (The_Kernel, The_Declaration);

      Traverse_Node (The_Declaration, Control, State);

      declare
         use Simplified_Elements_List;

         Cur : Simplified_Elements_List.Cursor :=
           First (Declarations_To_Bind);
         Element : Simple_Element_View_Access;
         Dummy : Bound_Element;
         Loc : Location_Handle;

         pragma Unreferenced (Dummy);
      begin
         while Cur /= Simplified_Elements_List.No_Element loop
            Element := Simplified_Elements_List.Element (Cur);

            if not Element.Annotation_Renaming then
               Loc := Push_Location (Element.Location);

               Dummy := Get_Bound_Element
                 (The_Kernel, Simplified_Elements_List.Element (Cur));

               Pop_Location (Loc);
            end if;

            Cur := Next (Cur);
         end loop;
      end;

      declare
         Base_Unit : Bound_Unit;
      begin
         Base_Unit := Get_Or_Create_Bound_Unit
           (Handle       => The_Kernel,
            Base_Package => Current_Package,
            Unit_Suffix  => "");

         Base_Unit.Create_Java_File := False;
      end;
   else
      Trace
        ("Warning: no binding generated for " & Text_Name (Unit_Spec),
         Errors_And_Warnings);
   end if;

   Set_Source_Status (SF, Processed);

exception
   when E : others =>
      Trace
        ("Unexpected exception: " &
         Conversions.To_Wide_String (Exception_Information (E)),
         Errors_Only);
      Set_Source_Status (SF, Processed);

end ASIS_Processing;
