-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions;
with Ada.Strings.Wide_Unbounded; use Ada.Strings.Wide_Unbounded;
with Ada.Directories;            use Ada.Directories;

with GNAT.Strings; use GNAT.Strings;

limited with Ada2Java.Kernel;

package Ada2Java is

   Not_Supported : exception;
   --  This exception is raised when we find an element that we cannot bind.
   --  Normally, it should issue only a warning and the binding should
   --  continue.

   Silent_Not_Supported : exception;
   --  Same as above, but shouldn't display any message. Maybe because the
   --  message has already been displayed.

   Ada2Java_Error : exception;

   type Source_Location is record
      File   : Wide_String_Access;
      Line   : Integer;
      Column : Integer;
   end record;

   Null_Location : constant Source_Location := (null, 0, 0);

   function Dump (Loc : Source_Location) return Wide_String;
   --  Return a string representation of the source location given in
   --  parameter.

   type Location_Handle is private;

   function Push_Location (Loc : Source_Location) return Location_Handle;

   procedure Pop_Location (Up_To : Location_Handle);

   function Dump_Current_Location return Wide_String;

   The_Kernel : access Ada2Java.Kernel.Kernel_Record;

   Default_Base_Package    : Unbounded_Wide_String;
   Default_Class_Path      : Unbounded_Wide_String :=
     To_Unbounded_Wide_String
       (Ada.Characters.Conversions.To_Wide_String (Current_Directory));
   Default_Ada_Base_Folder : Unbounded_Wide_String :=
     To_Unbounded_Wide_String
       (Ada.Characters.Conversions.To_Wide_String (Current_Directory));
   --  ??? These tree variables should be added in the kernel and managed from
   --  there.

   type Trace_Level is
     (Silent,
      Errors_Only,
      Errors_And_Warnings,
      Limited_Verbosity,
      Full_Verbosity,
      Debug);

   Current_Trace_Level : Trace_Level := Errors_And_Warnings;

   procedure Trace_With_Location
     (Text  : Wide_String;
      Level : Trace_Level;
      Loc   : Source_Location := Null_Location);
   --  Send a message with the location given in parameter if the given level
   --  of trace is accepted. If Loc is null, then the current location will
   --  be taken.

   procedure Trace (Text : Wide_String; Level : Trace_Level);
   --  Send a message on the output if the given level of trace is accepted

   Ada_JNI_Suffix  : constant Wide_String := "_JNI";
   Java_JNI_Suffix : constant Wide_String := "_JNI";

   Native_Library_Name : String_Access := null;
   --  When this is not null, it gives the name of the library to be build and
   --  will generate the subsequent build files.

   Main_Unit_Name : String_Access := null;
   --  When this is not null, it gives the name of the main Ada unit to be
   --  generated, embeding the JVM

   Java_Main_Class : String_Access := null;
   --  When set, specify the name of the class to use for the main call, in
   --  case of a main Ada subprogram.

   Default_Assumed_Stored : Boolean := True;
   --  Global default value for assuming the fact that an access parameter may
   --  be stored.

   type Locking_State_Type is (Disable, Check, Protect);
   --  Different states for the monitor ensuring thread-safety.

   Default_Locking_State : Locking_State_Type := Protect;
   Default_Finalize_Locking_State : Locking_State_Type := Protect;
   --  Global default value for allowing or not the global monitor for thread
   --  safety.

   Default_Lock : Wide_String_Access := null;
   --  Name for an user-defined default lock, null if none.

   Initial_Dir : constant String_Access := new String'(Current_Directory);
   --  Initial directory from which the tool has been called.

   Default_Attachement : Boolean := False;
   --  Default value for general attachement.

   Default_Access_Attachement : Boolean := False;
   --  Default value for attachement of access types.

   Default_Controlling_Attachement : Boolean := True;
   --  Default value of attachement of controlling primitives. Unattached
   --  controlling primitives will lead to underivable java classes.

   Default_Ada_2005_Attachement : Boolean := False;
   --  Implies attachement of controlling primitives + non controlling
   --  subprogram whose first parameter is a class wide type declared in the
   --  current scope, for both in, out and access modes.

   Allow_Unrestricted_Access : Boolean := True;
   --  When this flag is false, unrestricted access will not be generated by
   --  the tool and situations where it should be will lead to warnings
   --  messages.

   type Link_Method_Type is (Export, Register_Natives);

   Link_Method : Link_Method_Type := Export;
   --  This flag gives the method used to link the Java to the Ada symbols,
   --  either using simple export statements or calls to Register_Native.

   Use_Java_1_5_Enums : Boolean := True;
   --  Flag specifing wether we should use Java 1.5 enums or generate
   --  static integer constants.

private

   type Location_Handle is new Integer range 0 .. 2024;
   type Location_Array is array (Location_Handle) of Source_Location;

   Current_Location : Location_Handle := 1;
   Location_Stack   : Location_Array;

end Ada2Java;
