-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters;
with Ada.Text_IO;                use Ada.Text_IO;

with Asis.Clauses;           use Asis.Clauses;
with Asis.Compilation_Units; use Asis.Compilation_Units;
with Asis.Declarations;      use Asis.Declarations;
with Asis.Elements;          use Asis.Elements;
with Asis.Expressions;       use Asis.Expressions;
with Asis.Set_Get;           use Asis.Set_Get;

with GNAT.Directory_Operations; use GNAT.Directory_Operations;

with Ada2Java.Kernel;              use Ada2Java.Kernel;
with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Simplifications;     use Ada2Java.Simplifications;

package body Ada2Java.Packages is

   ----------
   -- Free --
   ----------

   procedure Free (This : in out Package_Handle) is
   pragma Unreferenced (This);
   begin
      null;
   end Free;

   --------------------
   -- Create_Package --
   --------------------

   function Get_Or_Create_Package
     (Handle : not null access Kernel.Kernel_Record;
      Unit   : Asis.Element) return Package_Handle
   is
      function Remove_File_Name (Str : Wide_String) return Wide_String;

      function Remove_File_Name (Str : Wide_String) return Wide_String is
      begin
         for J in reverse Str'Range loop
            if Ada.Characters.Conversions.To_Character (Str (J))
              = Dir_Separator
            then
               return Str (Str'First .. J - 1);
            end if;
         end loop;

         return "";
      end Remove_File_Name;

      File_Path : constant Wide_String := Remove_File_Name
        (Text_Name (Enclosing_Compilation_Unit (Unit)));

      Binding_Config : Configuration := Get_Default_Configuration (Handle);
      File_Config    : Ada.Text_IO.File_Type;

      Unit_Name : Dynamic_Expression;

      Result : Package_Handle;
      Parent_Unit : Compilation_Unit;
   begin
      if Is_Standard_Package (Unit) then
         Unit_Name := To_Dynamic_Expression ("Standard");
      else
         Unit_Name := To_Dynamic_Expression (Get_First_Name (Unit));
      end if;

      if Get_Bound_Packages_DB (Handle).Contains
        (To_Wide_String (Unit_Name)) then
         return Get_Bound_Packages_DB (Handle).Element
           (To_Wide_String (Unit_Name));
      end if;

      if Unit_Kind (Enclosing_Compilation_Unit (Unit))
        in A_Generic_Unit_Declaration
      then
         Trace_With_Location
           ("generic packages are not supported", Errors_And_Warnings);
         raise Silent_Not_Supported;
      end if;

      if File_Path /= "" then
         if Exists (Ada.Characters.Conversions.To_String (File_Path)
                    & Dir_Separator
                    & "ada2java.conf")
         then
            Ada.Text_IO.Open
              (File_Config,
               Ada.Text_IO.In_File,
               Ada.Characters.Conversions.To_String (File_Path)
               & Dir_Separator
               & "ada2java.conf");
            Parse_File (File_Config, Binding_Config);
            Set_Ada_Output_Directory
              (Binding_Config,
               File_Path
               & Ada.Characters.Conversions.To_Wide_Character (Dir_Separator)
               & Get_Ada_Output_Directory (Binding_Config));
            Set_Java_Class_Path
              (Binding_Config,
               File_Path
               & Ada.Characters.Conversions.To_Wide_Character (Dir_Separator)
               & Get_Java_Class_Path (Binding_Config));
            Ada.Text_IO.Close (File_Config);
         end if;
      end if;

      Result := Get_Or_Create_Package
        (Handle, To_Wide_String (Unit_Name), Binding_Config);

      Result.From_Ada_Package := True;

      declare
         Clauses : constant Asis.Context_Clause_List :=
           Context_Clause_Elements (Enclosing_Compilation_Unit (Unit));
      begin
         for J in Clauses'Range loop
            declare
               Names : constant Asis.Name_List := Clause_Names (Clauses (J));
               Pckg_Name        : Asis.Element;
               Pckg_Declaration : Asis.Element;
            begin
               for K in Names'Range loop
                  if (Clause_Kind (Clauses (J)) = A_Use_Package_Clause
                      or else Clause_Kind (Clauses (J)) = A_With_Clause)
                    and then Trait_Kind (Clauses (J)) /= A_Private_Trait
                  then
                     Pckg_Name := Names (K);

                     if Expression_Kind (Pckg_Name) = A_Selected_Component then
                        Pckg_Name := Selector (Pckg_Name);
                     end if;

                     Pckg_Declaration := Enclosing_Element
                       (Corresponding_Name_Definition (Pckg_Name));

                     while Declaration_Kind (Pckg_Declaration)
                       = A_Package_Renaming_Declaration
                     loop
                        Pckg_Declaration := Renamed_Entity (Pckg_Declaration);
                     end loop;

                     if Declaration_Kind (Pckg_Declaration)
                       /= A_Package_Instantiation
                     then
                        --  We bind only non-generic entries. This may change
                        --  in the future, when generics are supported.

                        if Clause_Kind (Clauses (J))
                          = A_Use_Package_Clause
                        then
                           Add_Clause
                             (Result,
                              Get_First_Name (Pckg_Declaration),
                              Use_Clause);
                        elsif Clause_Kind (Clauses (J)) = A_With_Clause then
                           Add_Clause
                             (Result,
                              Get_First_Name (Pckg_Declaration),
                              With_Clause);
                        end if;
                     end if;
                  end if;
               end loop;
            end;
         end loop;
      end;

      Parent_Unit :=
        Corresponding_Parent_Declaration (Enclosing_Compilation_Unit (Unit));

      if not Is_Nil (Parent_Unit) and then not Is_Standard (Parent_Unit) then
         Result.Parent_Package := Get_Or_Create_Package
           (Handle,
            Asis.Elements.Unit_Declaration (Parent_Unit));
      end if;

      return Result;
   end Get_Or_Create_Package;

   --------------------
   -- Create_Package --
   --------------------

   function Get_Or_Create_Package
     (Handle             : not null access Kernel.Kernel_Record;
      Name               : Wide_String;
      Binding_Config     : Configuration) return Package_Handle
   is
      Pckg          : constant Package_Handle := new Package_Record;

      Unit_Name : Wide_String renames Name;

      Ada_Package_Name : constant Wide_String := Unit_Name & "_JNI";
   begin
      if Contains (Get_Bound_Packages_DB (Handle).all, Unit_Name) then
         return Packages_Container.Element
           (Get_Bound_Packages_DB (Handle).all, Unit_Name);
      end if;

      Pckg.Binding_Config := Binding_Config;

      Pckg.Ada_Binding_Name :=
        To_Unbounded_Wide_String (Ada_Package_Name);

      Pckg.Bound_Pckg := To_Unbounded_Wide_String (Unit_Name);

      if Get_Java_Base_Package (Binding_Config) /= "" then
         if To_Unbounded_Wide_String (Unit_Name) /= "" then
            Pckg.Java_Name := Get_Java_Base_Package (Binding_Config)
              & "." & To_Unbounded_Wide_String (Unit_Name);
         else
            Pckg.Java_Name := To_Unbounded_Wide_String
              (Get_Java_Base_Package (Binding_Config));
         end if;
      else
         Pckg.Java_Name := To_Unbounded_Wide_String (Unit_Name);
      end if;

      Insert (Get_Bound_Packages_DB (Handle).all, Unit_Name, Pckg);

      Pckg.Ada_JNI_Name :=
        To_Unbounded_Wide_String
          (Mangle_ID ("Java." & To_Wide_String (Pckg.Java_Name)));

      Pckg.From_Ada_Package := False;

      return Pckg;
   end Get_Or_Create_Package;

   -----------------------------
   -- Get_Bound_Package_Name --
   -----------------------------

   function Get_Bound_Package_Name
     (Pckg : not null access Package_Record) return Wide_String
   is
   begin
      return To_Wide_String (Pckg.Bound_Pckg);
   end Get_Bound_Package_Name;

   -------------------
   -- Get_Java_Name --
   -------------------

   function Get_Java_Name
     (Pckg : not null access Package_Record) return Wide_String is
   begin
      return To_Wide_String (Pckg.Java_Name);
   end Get_Java_Name;

   ------------------
   -- Get_Ada_Name --
   ------------------

   function Get_Binding_Package_Name
     (Pckg : not null access Package_Record) return Wide_String is
   begin
      return To_Wide_String (Pckg.Ada_Binding_Name);
   end Get_Binding_Package_Name;

   --------------------------
   -- Get_Ada_Binding_Name --
   --------------------------

   function Get_JNI_Name
     (Pckg : not null access Package_Record) return Wide_String is
   begin
      return To_Wide_String (Pckg.Ada_JNI_Name);
   end Get_JNI_Name;

   -----------------------
   -- Get_Configuration --
   -----------------------

   function Get_Configuration
     (Pckg : not null access Package_Record) return Configuration
   is
   begin
      return Pckg.Binding_Config;
   end Get_Configuration;

   function Get_Last_Child_Name
     (Pckg : not null access Package_Record) return Wide_String
   is
      Bound_Name : constant Wide_String := To_Wide_String (Pckg.Bound_Pckg);
   begin
      for J in reverse Bound_Name'Range loop
         if Bound_Name (J) = '.' then
            return Bound_Name (J + 1 .. Bound_Name'Last);
         end if;
      end loop;

      return Bound_Name;
   end Get_Last_Child_Name;

   ----------------
   -- Add_Clause --
   ----------------

   procedure Add_Clause
     (Pckg : not null access Package_Record;
      Name : Wide_String;
      Kind : Clause_Kinds)
   is
      Tab : Clause_Set;
   begin
      if not Contains (Pckg.Clauses, Name) then
         Tab := (others => False);

         if Kind = Use_Clause then
            Tab.Use_Clause := True;
         else
            Tab.With_Clause := True;
         end if;

         Insert (Pckg.Clauses, Name, Tab);
      else
         Tab := Clause_Container.Element (Pckg.Clauses, Name);

         if Kind = Use_Clause then
            Tab.Use_Clause := True;
         else
            Tab.With_Clause := True;
         end if;

         Replace (Pckg.Clauses, Name, Tab);
      end if;
   end Add_Clause;

   -----------------
   -- Get_Clauses --
   -----------------

   function Get_Clauses
     (Pckg : not null access Package_Record) return Wide_String
   is
      It          : Clause_Container.Cursor := First (Pckg.Clauses);
      Result      : Unbounded_Wide_String;
      Bound_Name : constant Wide_String := To_Wide_String (Pckg.Bound_Pckg);
   begin
      while It /= Clause_Container.No_Element loop
         Append (Result, Conversions.To_Wide_Character (ASCII.LF));

         if Clause_Container.Element (It).With_Clause
           or else Clause_Container.Element (It).Use_Clause
         then
            --  As soon as we have a use, we want to ensure that the
            --  corresponding with is there.
            Append (Result, "with " & Key (It) & "; ");
         end if;

         if Clause_Container.Element (It).Use_Clause then
            Append (Result, "use " & Key (It) & ";");
         end if;

         It := Next (It);
      end loop;

      if Pckg.From_Ada_Package then
         declare
            Upper_Case_Name : Wide_String := Bound_Name;
         begin
            To_Upper (Upper_Case_Name);

            for J in Bound_Name'Range loop
               if Bound_Name (J) = '.'
                 and then Upper_Case_Name
                   (Bound_Name'First .. J - 1) /= "STANDARD"
               then
                  Append
                    (Result,
                     Conversions.To_Wide_Character (ASCII.LF)
                     & "with " & Bound_Name (Bound_Name'First .. J - 1)
                     & "; use " & Bound_Name
                       (Bound_Name'First .. J - 1) & ";");
               end if;
            end loop;

            if Upper_Case_Name /= "STANDARD" then
               Append
                 (Result,
                  Conversions.To_Wide_Character (ASCII.LF)
                  & "with " & Bound_Name & "; use " & Bound_Name & ";");
            end if;
         end;
      end if;

      return To_Wide_String (Result);
   end Get_Clauses;

   ------------------------
   -- Get_Parent_Package --
   ------------------------

   function Get_Parent_Package
     (Pckg : not null access Package_Record)
      return access Package_Record
   is
   begin
      return Pckg.Parent_Package;
   end Get_Parent_Package;

   -----------------
   -- Is_From_Ada --
   -----------------

   function Is_From_Ada
     (Pckg : not null access Package_Record) return Boolean
   is
   begin
      return Pckg.From_Ada_Package;
   end Is_From_Ada;

end Ada2Java.Packages;
