-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2008, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters.Conversions;

package body Ada2Java.Kernel is

   ---------------------------
   -- Get_Bound_Elements_DB --
   ---------------------------

   function Get_Bound_Elements_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Elements_DB
   is
   begin
      return Handle.Elements'Access;
   end Get_Bound_Elements_DB;

   --------------------------
   -- Get_Binding_Files_DB --
   --------------------------

   function Get_Binding_Units_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Units_DB
   is
   begin
      return Handle.Units'Access;
   end Get_Binding_Units_DB;

   ---------------------------
   -- Get_Bound_Packages_DB --
   ---------------------------

   function Get_Bound_Packages_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Packages_DB is
   begin
      return Handle.Packages'Access;
   end Get_Bound_Packages_DB;

   -------------------------------
   -- Set_Default_Configuration --
   -------------------------------

   procedure Set_Default_Configuration
     (Handle : not null access Kernel_Record;
      Conf   : Configuration) is
   begin
      Handle.Default_Conf := Conf;
   end Set_Default_Configuration;

   -------------------------------
   -- Get_Default_Configuration --
   -------------------------------

   function Get_Default_Configuration
     (Handle : not null access Kernel_Record) return Configuration is
   begin
      return Handle.Default_Conf;
   end Get_Default_Configuration;

   -------------------------------
   -- Add_Bound_Element_Creator --
   -------------------------------

   procedure Add_Bound_Element_Creator
     (Handle : not null access Kernel_Record;
      Sb     : Create_Bound_Element) is
   begin
      Append (Handle.Bound_Elements_Fcts, Sb);
   end Add_Bound_Element_Creator;

   ----------------------------------------
   -- Create_Corresponding_Bound_Element --
   ----------------------------------------

   function Create_Corresponding_Bound_Element
     (Handle  : not null access Kernel_Record;
      Element : Simple_Element_View_Access)
      return Bound_Element
   is
      Result : Bound_Element;
      Cur    : Create_Bound_Elements_Lists.Cursor :=
        First (Handle.Bound_Elements_Fcts);
   begin
      while Cur /= Create_Bound_Elements_Lists.No_Element loop
         Result := Create_Bound_Elements_Lists.Element (Cur) (Element);

         if Result /= null then
            return Result;
         end if;

         Cur := Next (Cur);
      end loop;

      return null;
   end Create_Corresponding_Bound_Element;

   --------------------------------
   -- Get_Simplified_Elements_DB --
   --------------------------------

   function Get_Simple_Elements_DB
     (Handle : not null access Kernel_Record)
      return not null access Simple_Elements_DB_Pckg.Map is
   begin
      return Handle.Simplified_Elements'Access;
   end Get_Simple_Elements_DB;

   ------------------------------
   -- Get_Library_Binding_Unit --
   ------------------------------

   function Get_Library_Binding_Unit
     (Handle : not null access Kernel_Record)
      return Bound_Unit
   is
   begin
      if Handle.Library_Binding_Unit = null then
         Handle.Library_Binding_Unit := Get_Or_Create_Bound_Unit
           (Handle       => Handle,
            Base_Package => Get_Or_Create_Package
              (Handle,
               "Ada2Java",
               Get_Default_Configuration (Handle)),
            Unit_Suffix  => "Library");

         Handle.Library_Binding_Unit.Create_Java_File := True;
         Handle.Library_Binding_Unit.Create_Ada_Body_File := True;
         Handle.Library_Binding_Unit.Create_Ada_Spec_File := True;

         if Ada2Java.Native_Library_Name /= null then
            Append
              (Handle.Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
               New_Line & New_Line &
               "procedure Load (Env : "
               & JNI_Pckg & ".JNI_Env_Access; Class : "
               & JNI_Pckg & ".J_Class);"
               & New_Line
               & "pragma Export (C, Load, ""Java_"
               & Mangle_ID
                 (To_Wide_String
                    (Handle.Library_Binding_Unit.Java_File.Full_Class_Name))
               & "_startLibrary"");"
               & New_Line & New_Line
               & "procedure Load (Env : " & JNI_Pckg
               & ".JNI_Env_Access; Class : " & JNI_Pckg & ".J_Class) is"
               & New_Line (1) & "procedure Internal;"
               & New_Line & "pragma Import (C, Internal, """
               & To_Wide_String (Ada2Java.Native_Library_Name.all)
               & "init"");"
               & New_Line (-1) & "begin " & New_Line (1) & "Internal;");

            if Ada2Java.Link_Method = Register_Natives then
               Append
                 (Handle.Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
                  New_Line & "Register_All_Natives (Env);");
            end if;

            Append
              (Handle.Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
               New_Line (-1)
               & "end Load;");
         end if;
      end if;

      return Handle.Library_Binding_Unit;
   end Get_Library_Binding_Unit;

   ------------------------------------
   -- Generate_Register_Native_Calls --
   ------------------------------------

   function Generate_Register_Native_Calls
     (Handle : not null access Kernel_Record) return Dynamic_Expression
   is
      Result : Dynamic_Expression := New_Dynamic_Expression;
      It     : Bound_Units_Cursor;

      use Bound_Unit_List_Pckg;
   begin
      It := Bound_Unit_List_Pckg.First
        (Get_Binding_Units_DB (Handle).all);

      while It /= Bound_Unit_List_Pckg.No_Element loop
         declare
            Package_Name : constant Wide_String := Key (It);
            Unit         : constant Bound_Unit := Element (It);
         begin
            if Unit.Ada_Body_File.Native_Number > 0 then
               Append
                 (Result,
                  New_Line & Package_Name & ".Register_Natives ("
                  & Env_Parameter_Name & ");");
            end if;

            It := Next (It);
         end;
      end loop;

      return Result;
   end Generate_Register_Native_Calls;

end Ada2Java.Kernel;
