-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2008-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Text_IO; use Ada.Text_IO;
with OS_Utils; use OS_Utils;
with GNAT.OS_Lib; use GNAT.OS_Lib;
with Ada.Characters.Conversions; use Ada.Characters.Conversions;

package body Ada2Java.Code_Templates is

   -----------------------------
   -- Load_Template_From_Path --
   -----------------------------

   function Load_Template_From_Path
     (File_Path : String) return Code_Template
   is
      File : File_Type;

      Result : constant Code_Template := new Template_Record;
   begin
      Open (File, In_File, File_Path);

      while not End_Of_File (File) loop
         declare
            Contents : constant String := Get_Line (File);

            Last_Mark : Integer := Contents'First - 1;

            Kind : Section_Kind := Raw_Text;
         begin
            for J in Contents'Range loop
               if Contents (J) = '%' then
                  Result.Sections.Append
                    ((Length   => J - Last_Mark - 1,
                      Kind     => Kind,
                      Contents => Contents (Last_Mark + 1 .. J - 1)));

                  if Kind = Raw_Text then
                     Kind := Parameter;
                  else
                     Kind := Raw_Text;
                  end if;

                  Last_Mark := J;
               elsif J = Contents'Last then
                  Result.Sections.Append
                    ((Length   => J - Last_Mark,
                      Kind     => Kind,
                      Contents => Contents (Last_Mark + 1 .. J)));
               end if;
            end loop;

            Result.Sections.Append
              ((Length   => 3,
                Kind     => Parameter,
                Contents => "eol"));
         end;
      end loop;

      Close (File);

      return Result;
   end Load_Template_From_Path;

   -----------------
   -- Instanciate --
   -----------------

   function Instanciate
     (Template : Code_Template;
      Parameters : Template_Parameter_Association.Map)
      return Dynamic_Expression
   is
      use Section_List;

      Cursor : Section_List.Cursor := Template.Sections.First;

      Result : Dynamic_Expression := New_Dynamic_Expression;
   begin
      while Cursor /= Section_List.No_Element loop
         declare
            Section : constant Section_Record := Section_List.Element (Cursor);
         begin
            if Section.Kind = Raw_Text then
               Append (Result, To_Wide_String (Section.Contents));
            else
               if Section.Contents = "eol" then
                  Append (Result, New_Line);
               else
                  Append
                    (Result,
                     Parameters.Element (To_Wide_String (Section.Contents)));
               end if;
            end if;
         end;

         Cursor := Section_List.Next (Cursor);
      end loop;

      return Result;
   end Instanciate;

   function Load_Template (Name : String) return Code_Template is
   begin
      return Load_Template_From_Path
        (Executable_Location & Directory_Separator & "share"
         & Directory_Separator & "ajis" & Directory_Separator & "templates"
         & Directory_Separator & Name & ".tpl");
   end Load_Template;

end Ada2Java.Code_Templates;
