-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2008, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers; use Ada.Containers;

with Ada.Characters.Conversions; use Ada.Characters;

with Ada2Java.Bound_Elements; use Ada2Java.Bound_Elements;
with Ada2Java.Bound_Units;    use Ada2Java.Bound_Units;
with Ada2Java.Kernel;         use Ada2Java.Kernel;

with Ada2Java.Code_Templates; use Ada2Java.Code_Templates;

package body Ada2Java.Bound_Units is

   Array_Conversion_Spec_Template : constant Code_Template := Load_Template
     ("array_conversion_spec");

   Array_Conversion_Body_Template : constant Code_Template := Load_Template
     ("array_conversion_body");

   function Append_Suffix_To_Packages
     (Name : Wide_String) return Wide_String;
   --  Appends the JNI Suffix to all the elements of the package name given
   --  in parameter.

   procedure Generate_Conversion_Package
     (Unit              : Bound_Unit;
      Conversion_Record : Type_Record;
      Name              : Wide_String);

   ------------------------------
   -- Get_Or_Create_Java_Class --
   ------------------------------

   function Create_Java_Class
     (Unit        : access Bound_Unit_Record;
      Handle      : access Kernel.Kernel_Record;
      Pckg        : access Packages.Package_Record;
      Name        : Wide_String) return Binding_File
   is
      pragma Unreferenced (Handle);

      File : constant Binding_File :=
        new Binding_File_Record (Java_Class_Kind);
   begin
      File.Unit           := Unit;
      File.Binding_Config := Get_Configuration (Pckg);
      File.Class_Name     := To_Dynamic_Expression (Name);

      Append
        (File.Dependencies_Part, "package " & Get_Java_Name (Pckg) & ";");

      File.Package_Name := To_Dynamic_Expression (Get_Java_Name (Pckg));

      Append
        (File.Close_Part, New_Line & New_Line (-1) & "} // " & Name);

      if Get_Java_Base_Package (File.Binding_Config) /= "" then
         File.Full_Class_Name :=
           To_Dynamic_Expression
             (Get_Java_Base_Package (File.Binding_Config)
              & "." & Get_Bound_Package_Name (Pckg)
              & "." & Name);
      else
         File.Full_Class_Name :=
           To_Dynamic_Expression
             (Get_Bound_Package_Name (Pckg)
              & "." & Name);
      end if;

      return File;
   end Create_Java_Class;

   ---------------------------------
   -- Generate_Conversion_Package --
   ---------------------------------

   procedure Generate_Conversion_Package
     (Unit              : Bound_Unit;
      Conversion_Record : Type_Record;
      Name              : Wide_String)
   is
   begin
      if Conversion_Record.Type_Of.Kind = Tagged_Record_Kind then
         Append
           (Unit.Ada_Spec_File.Public_Body_Part,
            New_Line & New_Line & "package "
            & Name
            & " is new "
            & AJIS_Pckg & ".Access_To_Jint_Conversions ("
            & Conversion_Record.Type_Of.Full_Name & "'Class);");
      elsif Is_Unconstrained_Array (Conversion_Record.Type_Of) then
         Add_Extra_Unit_Dependency (Unit.Ada_Spec_File, "System");
         Add_Extra_Unit_Dependency
           (Unit.Ada_Spec_File, "Ada.Unchecked_Deallocation");
         Add_Extra_Unit_Dependency
           (Unit.Ada_Body_File, "System.Address_To_Access_Conversions");

         declare
            Params : Template_Parameter_Association.Map;
         begin
            Params.Insert
              ("package_name", To_Dynamic_Expression (Name));
            Params.Insert
              ("array_type", Conversion_Record.Type_Of.Full_Name);

            declare
               Declarations : Dynamic_Expression :=
                 New_Dynamic_Expression;

               Defaults     : Dynamic_Expression :=
                 New_Dynamic_Expression;
            begin
               for J in Conversion_Record.Type_Of.Indexes'Range loop
                  declare
                     Bound_Number : constant Wide_String :=
                       Ada.Characters.Conversions.To_Wide_String
                         (Integer'Image (J));
                     Trimed_Bound_Number : constant Wide_String :=
                       Bound_Number
                         (Bound_Number'First + 1 ..  Bound_Number'Last);
                     Bound_Name   : constant Wide_String :=
                       "Bound_" & Trimed_Bound_Number;
                  begin
                     Append
                       (Declarations,
                        New_Line & Bound_Name
                        & "_Low, " & Bound_Name & "_High : "
                        &  Conversion_Record.Type_Of.Indexes
                          (J).Type_Of.Ref.Full_Name
                        & ";");

                     Append
                       (Defaults,
                        Conversion_Record.Type_Of.Indexes
                          (J).Type_Of.Initial_Subtype_Name
                        & "'First, "
                        & Conversion_Record.Type_Of.Indexes
                          (J).Type_Of.Initial_Subtype_Name
                        & "'First, ");
                  end;
               end loop;

               Params.Insert
                 ("bounds", Declarations);

               Params.Insert
                 ("bounds_default",
                  Defaults);

               Params.Insert
                 ("static_bounds_extraction",
                  Get_Array_Bounds
                    (Conversion_Record.Type_Of,
                     To_Dynamic_Expression ("V")));
            end;

            Append
              (Unit.Ada_Spec_File.Public_Body_Part,
               New_Line & New_Line
               & Instanciate
                 (Array_Conversion_Spec_Template, Params));

            Append
              (Unit.Ada_Body_File.Public_Body_Part,
               New_Line & New_Line
               & Instanciate
                 (Array_Conversion_Body_Template, Params));
         end;
      else
         Append
           (Unit.Ada_Spec_File.Public_Body_Part,
            New_Line & New_Line & "package "
            & Name
            & " is new " & AJIS_Pckg & ".Access_To_Jint_Conversions ("
            & Conversion_Record.Type_Of.Full_Name & ");");
      end if;
   end Generate_Conversion_Package;

   --------------
   -- Assemble --
   --------------

   function Assemble
     (Parts : Binding_File_Record) return Dynamic_Expression
   is
      Result        : Dynamic_Expression := Parts.Dependencies_Part;
      Import_Cursor : String_Lists.Cursor;
   begin
      if Parts.Kind = Java_Class_Kind then
         Import_Cursor := First (Parts.Import_List);

         while Import_Cursor /= String_Lists.No_Element loop
            Append
              (Result,
               New_Line & "import "
               & String_Lists.Element (Import_Cursor) & ";");

            Import_Cursor := Next (Import_Cursor);
         end loop;
      end if;

      if Parts.Kind in Ada_Spec_Kind .. Ada_Body_Kind
        and then Parts.Base_Pckg /= null
      then
         Append (Result, Get_Clauses (Parts.Base_Pckg));
      end if;

      declare
         use Dependecy_List;

         C : Dependecy_List.Cursor :=
           Parts.Extra_Unit_Dependencies.First;
      begin
         while C /= Dependecy_List.No_Element loop
            declare
               Name : constant Wide_String := Element (C);
               Upper_Name : Wide_String := Name;
            begin
               To_Upper (Upper_Name);

               if Upper_Name /= "STANDARD" then
                  Append
                    (Result,
                     Conversions.To_Wide_Character (ASCII.LF)
                     & "with " & Name & ";");
               end if;
            end;
            C := Next (C);
         end loop;
      end;

      if Parts.Kind = Java_Class_Kind then
         Append
           (Result, New_Line & New_Line & "@SuppressWarnings(""unused"")");

         if Parts.Is_Enum and then Ada2Java.Use_Java_1_5_Enums then
            Append (Result, New_Line & "public enum ");
         elsif Parts.Is_Final then
            Append (Result, New_Line & "public final class ");
         elsif Parts.Is_Abstract then
            Append (Result, New_Line & "public abstract class ");
         else
            Append (Result, New_Line & "public class ");
         end if;

         Append (Result, Parts.Class_Name);

         if To_Wide_String (Parts.Java_Parent_Class) /= "" then
            Append
              (Result,
               " extends " & Parts.Java_Parent_Class);
         end if;

         if Parts.Interfaces_List.Length > 0 then
            Append
              (Result,
               To_Dynamic_Expression (" implements "));

            declare
               Int_It : String_Lists.Cursor := First (Parts.Interfaces_List);
               Comma  : Dynamic_Expression := Empty_Dynamic_Expression;
            begin
               while Int_It /= String_Lists.No_Element loop
                  Append (Result, Comma & Element (Int_It));
                  Comma := To_Dynamic_Expression (", ");
                  Int_It := Next (Int_It);
               end loop;
            end;
         end if;

         Append
           (Result,
            " {"
            & Indent (1));
      end if;

      Append (Result, Parts.Open_Part);

      if Parts.Kind = Ada_Body_Kind
        and then Ada2Java.Link_Method = Register_Natives
      then
         if Parts.Native_Number > 0 then
            declare
               Array_Id : constant Wide_String := Get_Unique_Id;
            begin
               Append
                 (Result,
                  New_Line & New_Line &
                  "procedure Register_Natives (" & Env_Parameter_Name
                  & " : " & JNI_Pckg & ".JNI_Env_Access) is"
                  & New_Line (1) & Array_Id & " : "
                  & JNI_Pckg & ".JNI_Native_Method_Arr := ("
                  & Parts.Reg_Natives & ");"
                  & New_Line & "Result : " & JNI_Pckg & ".J_Int;"
                  & New_Line & "Class : " & AJIS_Pckg & ".Java_Class_Access"
                  & " := " & AJIS_Pckg & ".Get_Java_Class (""L"
                  & Replace_Dots_By_Slashes
                    (To_Wide_String
                       (Parts.Unit.Java_File.Full_Class_Name)) & ";"");"
                  & New_Line (-1) & "begin"
                  & New_Line (1)
                  & "Result := " & JNI_Pckg & ".Register_Natives ("
                  & Env_Parameter_Name
                  & ", " & AJIS_Pckg & ".Get_Class (Class, "
                  & Env_Parameter_Name & "), "
                  & Array_Id & ", " & Array_Id & "'Length);"
                  & New_Line (-1) & "end Register_Natives;");
            end;
         else
            Append
              (Result,
               New_Line & New_Line &
               "procedure Register_Natives (" & Env_Parameter_Name
               & " : " & JNI_Pckg & ".JNI_Env_Access) is"
              & New_Line & "begin"
              & New_Line (1) & "null;"
              & New_Line (-1) & "end Register_Natives;");
         end if;
      end if;

      Append (Result, Parts.Public_Body_Part & Parts.Private_Body_Part);

      if Parts.Kind = Ada_Body_Kind then
         Append (Result, Parts.Elab_Part);
      end if;

      if Parts.Kind = Java_Class_Kind then
         declare
            Ada2Java_Library_Pckg : Dynamic_Expression :=
              New_Dynamic_Expression;
            Base_Package : constant Wide_String :=
              Get_Java_Base_Package (Parts.Binding_Config);
         begin
            if Base_Package /= "" then
               Append (Ada2Java_Library_Pckg, Base_Package & ".Ada2Java");
            else
               Append (Ada2Java_Library_Pckg, "Ada2Java");
            end if;

            Append
              (Result,
               New_Line & New_Line & "static {"
               & New_Line (1) & Ada2Java_Library_Pckg & ".Library.load ();"
               & New_Line (-1) & "}");
         end;
      end if;

      return Result & Parts.Close_Part;
   end Assemble;

   ------------------------
   -- Add_Import_Element --
   ------------------------

   procedure Add_Import_Element (File : Binding_File; Element : Wide_String) is
   begin
      if File.Kind = Java_Class_Kind
        or else To_Wide_String (File.Full_Class_Name) /= Element
      then
         if not Contains (File.Import_List, Element) then
            Insert (File.Import_List, Element);
         end if;
      end if;
   end Add_Import_Element;

   -------------------------------
   -- Add_Extra_Unit_Dependency --
   -------------------------------

   procedure Add_Extra_Unit_Dependency
     (File : Binding_File; Element : Wide_String)
   is
   begin
      if not File.Extra_Unit_Dependencies.Contains (Element) then
         File.Extra_Unit_Dependencies.Insert (Element);
      end if;
   end Add_Extra_Unit_Dependency;

   ----------------------------
   -- Add_Dependency_On_Type --
   ----------------------------

   procedure Add_Dependency_On_Type
     (Handle   : not null access Kernel.Kernel_Record;
      File     : Binding_File;
      The_Type : Simple_Type_View_Access)
   is
      Enclosing_Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (Handle, Simple_Element_View_Access (The_Type));
   begin
      if Enclosing_Unit /= null then
         Add_Extra_Unit_Dependency
           (File, Get_Bound_Package_Name (Enclosing_Unit.Base_Pckg));

         if Enclosing_Unit.Root_Unit /= File.Unit.Root_Unit then
            --  We want to avoid cirular dependencies between sub units
            --  generated from a single Ada unit - all type that subunits needs
            --  to share should be declared in the root unit. So we don't
            --  create dependencies between sub units when they share the same
            --  root.

            Add_Extra_Unit_Dependency
              (File, To_Wide_String (Enclosing_Unit.Ada_Pckg_Name));
         end if;
      end if;
   end Add_Dependency_On_Type;

   -------------------------------
   -- Append_Suffix_To_Packages --
   -------------------------------

   function Append_Suffix_To_Packages
     (Name : Wide_String) return Wide_String
   is
      Suffix   : constant Wide_String := "_JNI";
      New_Size : Integer := Name'Length + Suffix'Length;
   begin
      for J in Name'Range loop
         if Name (J) = '.' then
            New_Size := New_Size + Suffix'Length;
         end if;
      end loop;

      declare
         New_Name : Wide_String (1 .. New_Size);
         New_Index : Integer := 1;
      begin
         for J in Name'Range loop
            if Name (J) = '.' then
               New_Name (New_Index .. New_Index + Suffix'Length) :=
                 Suffix & ".";
               New_Index := New_Index + Suffix'Length + 1;
            else
               New_Name (New_Index) := Name (J);
               New_Index := New_Index + 1;
            end if;
         end loop;

         New_Name (New_Index .. New_Name'Last) := Suffix;

         return New_Name;
      end;
   end Append_Suffix_To_Packages;

   ------------------------------
   -- Get_Or_Create_Bound_Unit --
   ------------------------------

   function Get_Or_Create_Bound_Unit
     (Handle : not null access Kernel.Kernel_Record;
      View   : Simple_Element_View_Access) return Bound_Unit
   is
      Unit : Bound_Unit;
   begin
      if View.all in Simple_Type_View'Class then
         declare
            Type_Of : Simple_Type_View_Access :=
              Simple_Type_View_Access (View);
         begin
            if Type_Of.Kind = Access_Kind
              and then Type_Of.Target_Type.Ref.Kind /= Subprogram_Kind
            then
               Type_Of := Type_Of.Target_Type.Ref;
            end if;

            if Type_Of.Kind = Standard_Boolean_Kind
              or else Type_Of.Kind = Standard_Character_Kind
            then
               return null;
            end if;

            if Type_Of.Kind in
              Generic_Float_Kind .. Standard_Character_Kind
            then
               Unit := Get_Or_Create_Bound_Unit
                 (Handle       => Handle,
                  Base_Package => Type_Of.Base_Package,
                  Unit_Suffix  => "");
            else
               Unit := Get_Or_Create_Bound_Unit
                 (Handle       => Handle,
                  Base_Package => Type_Of.Base_Package,
                  Unit_Suffix  => To_Wide_String (Type_Of.Type_Name));
            end if;

            Unit.Java_File.Is_Abstract := Type_Of.Is_Abstract;

            if Type_Of.Kind = Access_Kind
              and then Type_Of.Target_Type.Ref.Kind = Subprogram_Kind
            then
               Unit.Java_File.Is_Abstract := True;
            end if;
         end;
      elsif View.all in Simple_Subprogram_View'Class then
         declare
            Simple_View : constant Simple_Subprogram_View_Access :=
              Simple_Subprogram_View_Access (View);
            Type_Of : Simple_Type_View_Access;
         begin
            if Simple_View.Is_Attached then
               Type_Of := Simple_View.Parameters
                 (Simple_View.Parameters'First).Type_Of.Ref;

               if Type_Of.Kind = Access_Kind then
                  Type_Of := Type_Of.Target_Type.Ref;
               end if;

               Add_Bound_Element
                 (Handle, Simple_Element_View_Access (Type_Of));

               Unit := Get_Or_Create_Bound_Unit
                 (Handle,
                  Simple_Element_View_Access (Type_Of));
            else
               Unit := Get_Or_Create_Bound_Unit
                 (Handle,
                  Simple_View.Base_Package,
                  Get_Last_Child_Name (Simple_View.Base_Package) & "_Package");
            end if;
         end;
      elsif View.all in Simple_Object_View'Class then
         Unit := Get_Or_Create_Bound_Unit
           (Handle,
            View.Base_Package,
            Get_Last_Child_Name (View.Base_Package) & "_Package");
      elsif View.all in Simple_Exception_View'Class then
         Unit := Get_Or_Create_Bound_Unit
           (Handle,
            View.Base_Package,
            To_Wide_String (Simple_Exception_View (View.all).Name));
      else
         raise Not_Supported;
      end if;

      return Unit;
   end Get_Or_Create_Bound_Unit;

   ------------------------------
   -- Get_Or_Create_Bound_Unit --
   ------------------------------

   function Get_Or_Create_Bound_Unit
     (Handle       : not null access Kernel.Kernel_Record;
      Base_Package : Package_Handle;
      Unit_Suffix  : Wide_String) return Bound_Unit
   is
      function Get_Full_Name return Wide_String;

      function Get_Full_Name return Wide_String is
      begin
         if Unit_Suffix /= "" then
            return "JNI_Binding." & Append_Suffix_To_Packages
              (Get_Bound_Package_Name (Base_Package)
               & "."
               & Unit_Suffix);
         else
            return  "JNI_Binding." & Append_Suffix_To_Packages
              (Get_Bound_Package_Name (Base_Package));
         end if;
      end Get_Full_Name;

      Full_Name : constant Wide_String := Get_Full_Name;

      Unit : Bound_Unit;
   begin
      if Contains (Get_Binding_Units_DB (Handle).all, Full_Name) then
         Unit := Bound_Unit_List_Pckg.Element
           (Get_Binding_Units_DB (Handle).all, Full_Name);
      else
         Unit := new Bound_Unit_Record;
         Unit.Ada_Pckg_Name := To_Dynamic_Expression (Full_Name);
         Unit.Java_Class_Name := To_Dynamic_Expression (Unit_Suffix);

         Insert
           (Get_Binding_Units_DB (Handle).all, Full_Name, Unit);

         Unit.Root_Unit := Get_Or_Create_Bound_Unit (Handle, Base_Package, "");
      end if;

      Unit.Base_Pckg := Base_Package;

      if Unit.Ada_Spec_File = null then
         Unit.Ada_Spec_File := New_Ada_Spec_File
           (Unit           => Unit,
            Name           => To_Wide_String (Unit.Ada_Pckg_Name),
            Binding_Config => Get_Configuration (Base_Package));

         Unit.Ada_Spec_File.Base_Pckg := Base_Package;

         Unit.Base_Pckg := Base_Package;
      end if;

      if Unit.Java_File = null then
         Unit.Java_File := Create_Java_Class
           (Unit        => Unit,
            Handle      => Handle,
            Pckg        => Base_Package,
            Name        => To_Wide_String (Unit.Java_Class_Name));

         Unit.Java_File.Base_Pckg := Base_Package;
      end if;

      if Unit.Ada_Body_File = null then
         Unit.Ada_Body_File := New_Ada_Body_File
           (Unit           => Unit,
            Name           => To_Wide_String (Unit.Ada_Pckg_Name),
            Binding_Config => Get_Configuration (Base_Package));

         Unit.Ada_Body_File.Base_Pckg := Base_Package;
      end if;

      return Unit;
   end Get_Or_Create_Bound_Unit;

   -----------------------------------
   -- Generate_Missing_Bound_Units --
   -----------------------------------

   procedure Generate_Missing_Bound_Units
     (Handle : not null access Kernel.Kernel_Record)
   is
      C : Bound_Units_Cursor := First (Get_Binding_Units_DB (Handle).all);
      Parent : Package_Handle;
   begin
      while C /= Bound_Unit_List_Pckg.No_Element loop
         if Bound_Unit_List_Pckg.Element (C).Ada_Spec_File /= null
           and then
             (Bound_Unit_List_Pckg.Element (C).Create_Ada_Spec_File
              or else Bound_Unit_List_Pckg.Element (C).Create_Ada_Body_File)
         then
            Parent := Package_Handle
              (Bound_Unit_List_Pckg.Element (C).Base_Pckg);

            while Parent /= null loop
               declare
                  Name : constant Wide_String :=
                    Get_Binding_Package_Name (Parent);
                  New_Unit : Bound_Unit;
               begin
                  if not Contains
                    (Get_Binding_Units_DB  (Handle).all, Name)
                  then
                     New_Unit :=
                       Get_Or_Create_Bound_Unit (Handle, Parent, "");
                     New_Unit.Create_Java_File := False;
                  end if;
               end;

               Parent := Package_Handle (Get_Parent_Package (Parent));
            end loop;
         end if;

         C := Next (C);
      end loop;

      declare
         Base_Unit : constant Bound_Unit := new Bound_Unit_Record;
         Base_Package_Name : constant Wide_String := "JNI_Binding";
      begin
         Base_Unit.Ada_Spec_File :=
           New_Ada_Spec_File
             (Base_Unit,
              Base_Package_Name,
              Get_Default_Configuration (Handle));
         Base_Unit.Ada_Body_File :=
           New_Ada_Body_File
             (Base_Unit,
              Base_Package_Name,
              Get_Default_Configuration (Handle));

         Base_Unit.Base_Pckg := Get_Or_Create_Package
           (Handle, Base_Package_Name, Get_Default_Configuration (Handle));

         Base_Unit.Ada_Spec_File.Base_Pckg := null;
         Base_Unit.Ada_Body_File.Base_Pckg := null;
         Insert
           (Get_Binding_Units_DB  (Handle).all,
            Base_Package_Name, Base_Unit);
      end;
   end Generate_Missing_Bound_Units;

   -----------------------
   -- New_Ada_Spec_File --
   -----------------------

   function New_Ada_Spec_File
     (Unit           : access Bound_Unit_Record;
      Name           : Wide_String;
      Binding_Config : Configuration) return Binding_File
   is
      File : constant Binding_File := new Binding_File_Record (Ada_Spec_Kind);
   begin
      File.Unit           := Unit;
      File.Binding_Config := Binding_Config;
      Append
        (File.Open_Part,
         New_Line & New_Line
         & "package " & Name & " is "
         & New_Line (1) & "pragma Elaborate_Body;");
      Append
        (File.Close_Part, New_Line & New_Line (-1) & "end " & Name & ";");
      Append
        (File.Dependencies_Part, "pragma Warnings (Off);"
         & New_Line & "pragma Style_Checks (""NM32766"");");
      Append
        (File.Dependencies_Part, New_Line & New_Line
         & "with " & JNI_Pckg & ";");
      Append
        (File.Dependencies_Part,
         New_Line & "with AJIS;");
      Append
        (File.Dependencies_Part,
         New_Line & "with AJIS.Internal;");
      Append
        (File.Dependencies_Part,
         New_Line & "with AJIS.Java;");
      Append
        (File.Dependencies_Part,
         New_Line & "with " & AJIS_Pckg & ";");
      Append
        (File.Dependencies_Part,
         New_Line & "with Ada.Characters.Conversions;");
      Append
        (File.Dependencies_Part,
         New_Line & "with Ada.Finalization;");
      File.Package_Name := To_Dynamic_Expression (Name);

      if Ada2Java.Link_Method = Register_Natives then
         Append
           (File.Public_Body_Part,
            New_Line & New_Line
            & "procedure Register_Natives (" & Env_Parameter_Name
            & " : " & JNI_Pckg & ".JNI_Env_Access);");
      end if;

      return File;
   end New_Ada_Spec_File;

   -----------------------
   -- New_Ada_Body_File --
   -----------------------

   function New_Ada_Body_File
     (Unit           : access Bound_Unit_Record;
      Name           : Wide_String;
      Binding_Config : Configuration) return Binding_File
   is
      File : constant Binding_File := new Binding_File_Record (Ada_Body_Kind);
   begin
      File.Unit              := Unit;
      File.Binding_Config    := Binding_Config;
      File.Dependencies_Part := New_Dynamic_Expression;
      File.Public_Body_Part  := New_Dynamic_Expression;
      File.Private_Body_Part := New_Dynamic_Expression;
      File.Open_Part         := New_Dynamic_Expression;
      File.Close_Part        := New_Dynamic_Expression;
      File.Package_Name := To_Dynamic_Expression (Name);

      Append
        (File.Open_Part,
         New_Line & New_Line
         & "package body " & Name & " is " & Indent (1));
      Append
        (File.Close_Part, New_Line (-1) & "end " & Name & ";");

      Append
        (File.Dependencies_Part,
         "pragma Warnings (Off);"
         & New_Line & "pragma Style_Checks (""NM32766"");");
      Append
        (File.Dependencies_Part,
         New_Line & New_Line
         & "with Ada.Unchecked_Deallocation;" & New_Line
         & "with System.Address_To_Access_Conversions;" & New_Line
         & "with Ada.Unchecked_Conversion;" & New_Line
         & "with Ada.Tags;" & New_Line
         & "with Ada.Exceptions;");

      Append
        (File.Dependencies_Part,
         New_Line & "with Interfaces.C;"
         & New_Line & "with Interfaces.C.Strings;");

      Append
        (File.Elab_Part,
         New_Line & New_Line (-1) & "begin" & New_Line (1) & "null;");

      return File;
   end New_Ada_Body_File;

   -----------------------------------
   -- Get_Or_Create_Conversion_Type --
   -----------------------------------

   function Get_Or_Create_Conversion_Type
     (Handle  : not null access Kernel.Kernel_Record;
      Element : Simple_Type_View_Access) return Type_Record
   is
      Result : Type_Record;

      Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (Handle, Simple_Element_View_Access (Element));
   begin
      if not Types_Map.Contains
        (Unit.Root_Unit.Ada_Spec_File.Access_To_Address_Conversions,
         Element.Index)
      then
         declare
            Name : constant Wide_String := Get_Unique_Id;
         begin
            Result :=
              (new Wide_String'(To_Wide_String (Unit.Root_Unit.Ada_Pckg_Name)
               & "." & Name),
               Element);

            Insert_Type_In_Type_Map (Unit.Root_Unit, Element.Index, Result);

            Generate_Conversion_Package (Unit.Root_Unit, Result, Name);
         end;
      else
         Result := Types_Map.Element
           (Unit.Root_Unit.Ada_Spec_File.Access_To_Address_Conversions,
            Element.Index);
      end if;

      return Result;
   end Get_Or_Create_Conversion_Type;

   -----------------------------
   -- Insert_Type_In_Type_Map --
   -----------------------------

   procedure Insert_Type_In_Type_Map
     (Unit    : Bound_Unit;
      Index   : Element_Index;
      Element : Type_Record) is
   begin
      if Index.Location = null then
         raise Ada2Java_Error
           with "Internal error: location shouldn't be null";
      end if;

      Types_Map.Insert
        (Unit.Ada_Spec_File.Access_To_Address_Conversions,
         Index,
         Element);
   end Insert_Type_In_Type_Map;

end Ada2Java.Bound_Units;
