-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2008, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions;  use Ada.Characters.Conversions;

with Asis.Ids;               use Asis.Ids;

with Ada2Java.Kernel; use Ada2Java.Kernel;
with Ada2Java.Bound_Elements.Subprograms;
use Ada2Java.Bound_Elements.Subprograms;
with Ada2Java.Bound_Elements.Objects;
use Ada2Java.Bound_Elements.Objects;
with Ada2Java.Bound_Elements.Exceptions;
use Ada2Java.Bound_Elements.Exceptions;

with Ada2Java.Bound_Elements.Types; use Ada2Java.Bound_Elements.Types;
with Ada2Java.Bound_Elements.Enums; use Ada2Java.Bound_Elements.Enums;

package body Ada2Java.Bound_Elements is

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Asis.Element) return Boolean is
      Left_Id    : constant Asis.Ids.Id := Create_Id (Left);
      Right_Id   : constant Asis.Ids.Id := Create_Id (Right);
   begin
      return Left_Id < Right_Id;
   end "<";

   -----------------------
   -- Add_Bound_Element --
   -----------------------

   procedure Add_Bound_Element
     (Kernel  : not null access Ada2Java.Kernel.Kernel_Record;
      Element : Simple_Element_View_Access)
   is
      E : Bound_Element := null;
      DB : constant access Bound_Elements_DB :=
        Get_Bound_Elements_DB (The_Kernel.all'Access);

   begin
      if Element.Index.Location = null then
         return;
      end if;

      if Bound_Elements_DB_Pckg.Contains (DB.all, Element.Index) then
         return;
      end if;

      E := Create_Corresponding_Bound_Element (Kernel, Element);

      if E = null then
         if Element.all in Simple_Type_View'Class then
            declare
               Simple_View : Simple_Type_View_Access :=
                 Simple_Type_View_Access (Element);
            begin
               if Simple_View.Kind = Access_Kind then
                  Simple_View  := Simple_View.Target_Type.Ref;
               end if;

               if Simple_View.Kind = Record_Kind
                 or else Simple_View.Kind = Array_Kind
                 or else Simple_View.Kind = Private_Kind
               then
                  E := new Bound_Type;
               elsif Simple_View.Kind = Enumeration_Kind then
                  E := new Bound_Enum;
               end if;
            end;
         elsif Element.all in Simple_Subprogram_View'Class then
            E := new Bound_Subprogram;
         elsif Element.all in Simple_Object_View'Class then
            E := new Bound_Object;
         elsif Element.all in Simple_Exception_View'Class then
            E := new Bound_Exception;
         else
            raise Not_Supported;
         end if;
      end if;

      if E /= null then
         if Element.all in Simple_Type_View'Class then
            declare
               Simple_View : constant Simple_Type_View_Access :=
                 Simple_Type_View_Access (Element);
            begin
               if Simple_View.Kind = Tagged_Record_Kind
                 and then Simple_View.Target_Type /= Null_Type_Reference
               then
                  --  If this type has a parent, we need to bind it as well.

                  Add_Bound_Element
                    (Kernel,
                     Simple_Element_View_Access (Simple_View.Target_Type.Ref));

                  --  This parent is not necessary in a bound unit itself - so
                  --  we must at least bind its primitives, if any

                  if Simple_View.Target_Type.Ref.Primitives /= null then
                     for J in Simple_View.Target_Type.Ref.Primitives'Range loop
                        declare
                           Subprogram : constant Simple_Subprogram_View_Access
                             := Simple_Subprogram_View_Access
                               (Copy (Simple_View.Target_Type.Ref.
                                  Primitives (J)));
                        begin
                           Add_Bound_Element
                             (Kernel, Simple_Element_View_Access (Subprogram));
                        end;
                     end loop;
                  end if;
               end if;
            end;
         end if;

         E.Kernel := Kernel;
         E.Enclosing_Unit := Get_Or_Create_Bound_Unit (Kernel, Element);
         Insert (DB.all, Element.Index, E);
         Bind (Kernel, Element, E);
      else
         Trace_With_Location
           ("Element not bound.", Limited_Verbosity);
      end if;
   end Add_Bound_Element;

   -----------------------
   -- Get_Bound_Element --
   -----------------------

   function Get_Bound_Element
     (Kernel  : access Ada2Java.Kernel.Kernel_Record;
      Element : Simple_Element_View_Access) return Bound_Element
   is
      Result : Bound_Element;
   begin
      Add_Bound_Element (Kernel, Element);

      if not Bound_Elements_DB_Pckg.Contains
        (Get_Bound_Elements_DB (Kernel).all, Element.Index)
      then
         return null;
      else
         Result := Bound_Elements_DB_Pckg.Element
           (Get_Bound_Elements_DB (Kernel).all,
            Element.Index);

         return Result;
      end if;
   end Get_Bound_Element;

   -----------------------
   -- Add_Bound_Element --
   -----------------------

   procedure Add_Bound_Element
     (Kernel  : access Ada2Java.Kernel.Kernel_Record;
      Element : Bound_Element;
      Index   : Element_Index)  is
   begin
      if not Bound_Elements_DB_Pckg.Contains
        (Get_Bound_Elements_DB (Kernel).all, Index)
      then
         Bound_Elements_DB_Pckg.Insert
           (Container => Get_Bound_Elements_DB (Kernel).all,
            Key       => Index,
            New_Item  => Element);
      else
         raise Ada2Java_Error with "Element already in map.";
      end if;
   end Add_Bound_Element;

   -----------------------
   -- Get_Bound_Element --
   -----------------------

   function Get_Bound_Element
     (Kernel : access Ada2Java.Kernel.Kernel_Record;
      Index  : Element_Index) return Bound_Element is
   begin
      if Bound_Elements_DB_Pckg.Contains
        (Get_Bound_Elements_DB (Kernel).all, Index)
      then
         return Bound_Elements_DB_Pckg.Element
           (Get_Bound_Elements_DB (Kernel).all,
            Index);
      else
         return null;
      end if;
   end Get_Bound_Element;

   -----------------
   -- Get_Package --
   -----------------

   function Get_Package
     (Element : access Bound_Element_Record'Class) return Package_Handle is
   begin
      return Element.Enclosing_Unit.Base_Pckg;
   end Get_Package;

   --------------------------------
   -- Create_Unrestricted_Access --
   --------------------------------

   procedure Create_Unrestricted_Access
     (Handle       : not null access Kernel.Kernel_Record;
      Unit         : Bound_Unit;
      Initial_Type : Simple_Type_Reference;
      New_Type     : out Simple_Type_Reference;
      Created      : out Boolean)
   is
      Bound_Type : Bound_Element;
      Conversion : Type_Record;
   begin
      case Initial_Type.Ref.Kind is
         when Record_Kind | Private_Kind | Array_Kind =>
            New_Type := Create_Access_To (Initial_Type);

            New_Type.Ref.Full_Name := Initial_Type.Ref.Named_Access;

            Created := True;
            New_Type.Is_From_Unrestricted_Access := True;

            if Get_Array_Manipulation_Kind  (New_Type) = Wrapped_Access then
               Conversion := Get_Or_Create_Conversion_Type
                 (Handle, Initial_Type.Ref);

               New_Type.Initial_Subtype_Name :=
                 To_Dynamic_Expression
                   (Conversion.Conversion_Package_Name.all
                    & ".Object_Pointer");
            end if;
         when Tagged_Record_Kind =>
            New_Type  := Create_Access_To (Initial_Type);

            Bound_Type := Get_Bound_Element
              (Handle, Simple_Element_View_Access (Initial_Type.Ref));

            if Initial_Type.Is_Class_Wide then
               New_Type.Ref.Full_Name := Initial_Type.Ref.Named_Class_Access;
            else
               New_Type.Ref.Full_Name := Initial_Type.Ref.Named_Access;
            end if;

            if not Unit.Ada_Spec_File.Extra_Unit_Dependencies.Contains
              (To_Wide_String (Bound_Type.Enclosing_Unit.Ada_Pckg_Name))
            then
               Unit.Ada_Spec_File.Extra_Unit_Dependencies.Insert
                 (To_Wide_String
                    (Bound_Type.Enclosing_Unit.Ada_Pckg_Name));
            end if;

            Created := True;
            New_Type.Is_From_Unrestricted_Access := True;

         when others =>
            New_Type := Initial_Type;
            Created := False;

      end case;
   end Create_Unrestricted_Access;

   ----------------------
   -- Get_Array_Bounds --
   ----------------------

   function Get_Array_Bounds
     (Element : Simple_Type_View_Access;
      Variable : Dynamic_Expression) return Dynamic_Expression
   is
      Result : Dynamic_Expression := New_Dynamic_Expression;
   begin
      for J in Element.Indexes'Range loop
         declare
            Bound_Number : constant Wide_String :=
              Ada.Characters.Conversions.To_Wide_String
                (Integer'Image (J));
            Trimed_Bound_Number : constant Wide_String :=
              Bound_Number
                (Bound_Number'First + 1 ..  Bound_Number'Last);
         begin
            if Element.Indexes'Last = 1 then
               Append
                 (Result, Variable & "'First, " & Variable & "'Last, ");
            else
               Append
                 (Result,
                  Variable & "'First (" & Trimed_Bound_Number
                  & "), " & Variable & "'Last ("
                  & Trimed_Bound_Number & "), ");
            end if;
         end;
      end loop;

      return Result;
   end Get_Array_Bounds;

end Ada2Java.Bound_Elements;
