-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2008, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Wide_Text_IO; use Ada.Wide_Text_IO;

with Ada2Java.Utils; use Ada2Java.Utils;

with Ada2Java.Bound_Elements.Subprograms;
use Ada2Java.Bound_Elements.Subprograms;
with Ada2Java.Code_Wrappers;       use Ada2Java.Code_Wrappers;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Simplifications; use Ada2Java.Simplifications;
with Ada2Java.Bound_Units;     use Ada2Java.Bound_Units;

with Ada.Exceptions; use Ada.Exceptions;
with Ada.Characters.Conversions; use Ada.Characters;

package body Ada2Java.Bound_Elements.Objects is

   type Modifier_Mode is (Setter, Getter);

   type Ada_Modifiers_Handler is new Code_Wrapper with record
      Mode     : Modifier_Mode;
      Var_Name : Dynamic_Expression;

      Object : Simple_Object_View_Access;
      Is_Unrestricted_Access : Boolean := False;
      --  This field is true when the value of the field is carried trough
      --  an unrestricted access.
   end record;

   overriding
   procedure Wrapp
     (Handler    : in out Ada_Modifiers_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   procedure Bind
     (Handle        : not null access Kernel.Kernel_Record;
      Element       : Simple_Element_View_Access;
      Element_Bound : access Bound_Object)
   is
      pragma Unreferenced (Element_Bound);

      Simple_View : constant Simple_Object_View_Access :=
        Simple_Object_View_Access (Element);

      Getter_View : constant Simple_Subprogram_View_Access :=
        Create (Element);
      Setter_View : constant Simple_Subprogram_View_Access :=
        Create (Element);

      Getter_Ada_Call_Handler : aliased Ada_Modifiers_Handler;
      Setter_Ada_Call_Handler : aliased Ada_Modifiers_Handler;

      Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit (Handle, Element);

      Getter_Map : Wrapping_Map;
      Setter_Map : Wrapping_Map;

      Used_Object_Type : Simple_Type_Reference;
   begin
      Create_Unrestricted_Access
        (Handle       => Handle,
         Unit         => Unit,
         Initial_Type => Simple_View.Type_Of,
         New_Type     => Used_Object_Type,
         Created      => Getter_Ada_Call_Handler.Is_Unrestricted_Access);

      Setter_Ada_Call_Handler.Object := Simple_View;
      Getter_Ada_Call_Handler.Object := Simple_View;

      Setter_Ada_Call_Handler.Is_Unrestricted_Access :=
        Getter_Ada_Call_Handler.Is_Unrestricted_Access;

      Getter_View.Is_Dispatching := False;
      Getter_View.Name           := Simple_View.Name;
      Getter_View.Returned_Type  := Create (Getter_View);
      Getter_View.Returned_Type.Index   := Element.Index;
      Getter_View.Returned_Type.Type_Of := Used_Object_Type;

      Getter_Ada_Call_Handler.Mode := Getter;
      Getter_Ada_Call_Handler.Var_Name := Getter_View.Name;

      Add_Wrapper
        (Map     => Getter_Map,
         Wrapper => Getter_Ada_Call_Handler'Unchecked_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Parameter_Resolution);

      Bind_Subprogram
        (Handle   => Handle,
         View     => Getter_View,
         Unit     => Unit,
         Wrappers => Getter_Map);

      if not Simple_View.Is_Constant
        and then not Simple_View.Type_Of.Ref.Is_Limited
        and then not
          (Simple_View.Type_Of.Ref.Kind = Access_Kind
           and then
             Simple_View.Type_Of.Ref.Target_Type.
               Ref.Target_Subprogram /= null)
      then
         --  We do not support setters to subprograms accesses. Could be lifted
         --  in the long run in case of accesses coming from Ada (which are not
         --  completely handled yet).

         Setter_View.Is_Dispatching := False;
         Setter_View.Name           := Simple_View.Name;
         Setter_View.Parameters     := new Parameter_Array (1 .. 1);
         Setter_View.Parameters (1) := Create (Setter_View);
         Setter_View.Parameters (1).Type_Of        := Simple_View.Type_Of;
         Setter_View.Parameters (1).Name           :=
           To_Dynamic_Expression ("Value");
         Setter_View.Parameters (1).Is_Controlling := True;

         Setter_Ada_Call_Handler.Mode := Setter;
         Setter_Ada_Call_Handler.Var_Name := Setter_View.Name;

         Add_Wrapper
           (Map     => Setter_Map,
            Wrapper => Setter_Ada_Call_Handler'Unchecked_Access,
            Lang    => Ada_Lang,
            Target  => High_Sb,
            Context => Parameter_Resolution);

         Bind_Subprogram
           (Handle   => Handle,
            View     => Setter_View,
            Unit     => Unit,
            Wrappers => Setter_Map);
      end if;
   exception
      when Silent_Not_Supported =>
         raise;

      when E : others =>
         Ada.Wide_Text_IO.Put_Line
           ("EXCEPTION !!! "
            & Conversions.To_Wide_String (Exception_Information (E)));

         raise;
   end Bind;

   -----------
   -- Wrapp --
   -----------

   procedure Wrapp
     (Handler    : in out Ada_Modifiers_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Parameters);
   begin
      if Expression.Expression /= Empty_Dynamic_Expression then
         Append
           (Expression.Enclosing_Expression.Expression,
            Handler.Var_Name & " := "
            & Handler.Object.Type_Of.Initial_Subtype_Name & " ("
            & Expression.Expression & ")");
      else
         if Handler.Mode = Getter and then Handler.Is_Unrestricted_Access then
            if Is_Unconstrained_Array (Handler.Object.Type_Of.Ref) then
               if Ada2Java.Allow_Unrestricted_Access then
                  Append
                    (Expression.Enclosing_Expression.Expression,
                     "(" & AJIS_Pckg & ".Static, "
                      & Get_Array_Bounds
                         (Handler.Object.Type_Of.Ref, Handler.Var_Name)
                     & Handler.Var_Name & "'Address)");
               else
                  --  This should have been detected earlier

                  raise Not_Supported;
               end if;
            elsif Handler.Object.Is_Aliased then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Handler.Var_Name & "'Access");
            elsif Ada2Java.Allow_Unrestricted_Access then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Handler.Var_Name & "'Unrestricted_Access");
            else
               --  This should have been detected earlier.

               raise Not_Supported;
            end if;
         else
            Append
              (Expression.Enclosing_Expression.Expression,
               Handler.Var_Name);
         end if;
      end if;
   end Wrapp;

end Ada2Java.Bound_Elements.Objects;
