-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                     Copyright (C) 2008, AdaCore                   --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Config; use Ada2Java.Config;
with Ada2Java.Kernel; use Ada2Java.Kernel;

package body Ada2Java.Bound_Elements.Common is

   -------------------------------------
   -- Create_Exception_Occurence_Type --
   -------------------------------------

   function Get_Exception_Occurence
     (Handle : not null access Ada2Java.Kernel.Kernel_Record)
      return Simple_Type_View_Access
   is
      Exception_Occurence_Type : Simple_Type_View_Access := null;
      Exceptions_Package : Package_Handle;
      Ada_Package        : Package_Handle;
      Ada_Unit           : Bound_Unit;
      pragma Unreferenced (Ada_Unit);
      Exceptions_Unit    : Bound_Unit;
   begin
      if not Get_Simple_Elements_DB (Handle).Contains
        (Exception_Occurence_Index)
      then
         Ada_Package := Get_Or_Create_Package
           (Handle, "Ada", Get_Default_Configuration (Handle));
         Ada_Unit := Get_Or_Create_Bound_Unit
           (Handle, Ada_Package, "");
         Exceptions_Package := Get_Or_Create_Package
           (Handle,
            "Ada.Exceptions",
            Get_Default_Configuration (Handle));

         Exception_Occurence_Type := Create (null);
         Initialize_Configurable_Properties (Handle, Exception_Occurence_Type);
         Exception_Occurence_Type.Base_Package := Exceptions_Package;
         Exception_Occurence_Type.Kind := Private_Kind;
         Exception_Occurence_Type.Full_Name := To_Dynamic_Expression
           ("Ada.Exceptions.Exception_Occurrence");
         Exception_Occurence_Type.Full_Java_Name :=
           To_Dynamic_Expression
             (Get_Java_Base_Package (Get_Default_Configuration (Handle)));

         if To_Wide_String (Exception_Occurence_Type.Full_Java_Name) /= "" then
            Exception_Occurence_Type.Full_Java_Name :=
              Exception_Occurence_Type.Full_Java_Name & ".";
         end if;

         Exception_Occurence_Type.Full_Java_Name :=
           Exception_Occurence_Type.Full_Java_Name
             & "Ada.Exceptions.Exception_Occurrence";

         Exception_Occurence_Type.Type_Name :=
           To_Dynamic_Expression ("Exception_Occurrence");
         Exception_Occurence_Type.Index := Create_Dummy_Unique_Location;
         --  This is a dummy location
         Exception_Occurence_Type.Is_Limited := True;

         Exceptions_Unit := Get_Or_Create_Bound_Unit
           (Handle, Simple_Element_View_Access (Exception_Occurence_Type));

         Append
           (Exceptions_Unit.Ada_Spec_File.Dependencies_Part,
            New_Line & "with Ada.Exceptions; use Ada.Exceptions;");

         Add_Bound_Element
           (Handle, Simple_Element_View_Access (Exception_Occurence_Type));

         Simple_Elements_DB_Pckg.Insert
           (Container => Get_Simple_Elements_DB (Handle).all,
            Key       => Exception_Occurence_Index,
            New_Item  => Simple_Element_View_Access
              (Exception_Occurence_Type));
      end if;

      return Simple_Type_View_Access
        (Get_Simple_Elements_DB (Handle).Element (Exception_Occurence_Index));
   end Get_Exception_Occurence;

   ----------------
   -- Get_String --
   ----------------

   function Get_String
     (Handle : not null access Ada2Java.Kernel.Kernel_Record)
      return Simple_Type_Reference
   is
      Standard_Package : constant Package_Handle :=
        Get_Standard_Package (Handle);
      Standard_Unit    : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (Handle, Standard_Package, "AdaString");

      String_Type : Simple_Type_View_Access;
   begin
      if not Get_Simple_Elements_DB (Handle).Contains (String_Index) then
         String_Type := Create (null);

         Initialize_Configurable_Properties (Handle, String_Type);
         String_Type.Kind := Array_Kind;
         String_Type.Index := Create_Dummy_Unique_Location;
         String_Type.Type_Name := To_Dynamic_Expression ("AdaString");
         String_Type.Full_Name := To_Dynamic_Expression ("Standard.String");
         String_Type.Full_Java_Name := Standard_Unit.Java_File.Package_Name
           & ".AdaString";
         String_Type.Base_Package := Standard_Package;

         String_Type.Target_Type.Ref := Create (String_Type);
         String_Type.Target_Type.Ref.Kind := Standard_Character_Kind;
         String_Type.Target_Type.Ref.Full_Name := To_Dynamic_Expression
           ("Standard.Character");
         String_Type.Indexes := new Simple_Object_View_Array (1 .. 1);
         String_Type.Indexes (1) := Create (String_Type);
         String_Type.Indexes (1).Type_Of.Ref := Create (String_Type);
         String_Type.Indexes (1).Type_Of.Ref.Kind := Generic_Integer_Kind;
         String_Type.Indexes (1).Type_Of.Ref.Type_Name :=
           To_Dynamic_Expression ("Standard.Integer'Base");
         String_Type.Indexes (1).Type_Of.Ref.Full_Name :=
           To_Dynamic_Expression ("Standard.Integer'Base");
         String_Type.Indexes (1).Type_Of.Initial_Subtype_Name :=
           To_Dynamic_Expression ("Standard.Integer");
         String_Type.Indexes (1).Type_Of.Ref.Is_Constant := False;

         Simple_Elements_DB_Pckg.Insert
           (Container => Get_Simple_Elements_DB (Handle).all,
            Key       => String_Index,
            New_Item  => Simple_Element_View_Access (String_Type));
      end if;

      return
        (Ref => Simple_Type_View_Access
           (Get_Simple_Elements_DB (Handle).Element (String_Index)),
         Initial_Subtype_Name => To_Dynamic_Expression ("Standard.String"),
         others => <>);
   end Get_String;

   -----------------
   -- Get_Boolean --
   -----------------

   Boolean_Type : Simple_Type_View_Access := null;
   --  ??? This should go in the kernel

   function Get_Boolean
     (Handle : not null access Ada2Java.Kernel.Kernel_Record)
      return Simple_Type_View_Access
   is
   begin
      if Boolean_Type = null then
         Boolean_Type := Create (null);

         Initialize_Configurable_Properties (Handle, Boolean_Type);
         Boolean_Type.Kind := Standard_Boolean_Kind;
         Boolean_Type.Index := Create_Dummy_Unique_Location;
         Boolean_Type.Type_Name := To_Dynamic_Expression ("Standard.Boolean");
         Boolean_Type.Full_Name := To_Dynamic_Expression ("Standard.Boolean");
      end if;

      return Boolean_Type;
   end Get_Boolean;

   --------------------------
   -- Get_Standard_Package --
   --------------------------

   function Get_Standard_Package
     (Handle : not null access Ada2Java.Kernel.Kernel_Record)
      return Ada2Java.Packages.Package_Handle
   is
      Standard_Package : Package_Handle;
      Standard_Unit    : Bound_Unit;
      pragma Unreferenced (Standard_Unit);
   begin
      Standard_Package := Get_Or_Create_Package
        (Handle, "Standard", Get_Default_Configuration (Handle));
      Standard_Unit := Get_Or_Create_Bound_Unit
        (Handle, Standard_Package, "");

      return Standard_Package;
   end Get_Standard_Package;

end Ada2Java.Bound_Elements.Common;
