-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2008, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada2Java.Code_Wrappers;       use Ada2Java.Code_Wrappers;
with Ada2Java.Simplifications;     use Ada2Java.Simplifications;

with Ada2Java.Bound_Elements.Subprograms;
use Ada2Java.Bound_Elements.Subprograms;

package body Ada2Java.Bound_Elements.Callbacks is

   type Callback_Wrapper is
     new Code_Wrapper with null record;
   --  This wrapper has to be applied on ada internal subprogram  wrapping the
   --  primitives of an extensible type.

   overriding
   procedure Wrapp
     (Wrapper    : in out Callback_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);
   --  See inherited documentation.

   procedure Generate_Child_Type
     (Handle        : access Kernel.Kernel_Record;
      Unit          : Bound_Unit;
      Element       : access Bound_Callback;
      Type_Of       : Simple_Type_View_Access);

   ---------------------------
   -- Create_Bound_Callback --
   ---------------------------

   function Create_Bound_Callback
     (Element : Simple_Element_View_Access) return Bound_Element is
   begin
      if Element.all in Simple_Type_View'Class then
         declare
            Type_Of : Simple_Type_View renames
              Simple_Type_View (Element.all);
         begin
            if Type_Of.Kind = Access_Kind
              and then Type_Of.Target_Type.Ref.Kind = Subprogram_Kind
            then
               return new Bound_Callback;
            end if;
         end;
      end if;

      return null;
   end Create_Bound_Callback;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Handle        : not null access Kernel.Kernel_Record;
      Element       : Simple_Element_View_Access;
      Element_Bound : access Bound_Callback)
   is
      Type_Of : Simple_Type_View renames Simple_Type_View
        (Element.all);

      Simple_View      : Simple_Subprogram_View_Access;
      Simple_Impl_View : Simple_Subprogram_View_Access;
      Unit : Bound_Unit;
      Tmp_Array : Parameter_Array_Access;

      Simple_Impl_Map : Wrapping_Map;
      Simple_Impl_Map_Wrapper : aliased Callback_Wrapper;

      Callback_Name : constant Wide_String :=
        To_Wide_String (Type_Of.Type_Name) & "_Body";
   begin
      Unit := Get_Or_Create_Bound_Unit (Handle, Element);
      Unit.Java_File.Is_Final := False;

      Unit.Java_File.Interfaces_List.Insert ("com.adacore.ajis.IEscapable");

      if Type_Of.Is_Anonymous then
         --  If the type is anonymous, then the first stage is to create an
         --  actual type for it.

         Append
           (Unit.Ada_Spec_File.Public_Body_Part,
            New_Line & New_Line & "type " & Type_Of.Type_Name
            & "_Anon is access " & To_Original_Expanded_Profile
              (Handle, Type_Of.Target_Type.Ref.Target_Subprogram) &
            ";");

         --  ??? Not sure that it's wise to change a structure property at this
         --  stage... Would be better to use some data stored in the bound
         --  structure !
         Type_Of.Full_Name :=
           Unit.Ada_Pckg_Name & "." & Type_Of.Type_Name & "_Anon";
      end if;

      Generate_Child_Type
        (Handle, Unit, Element_Bound, Simple_Type_View_Access (Element));

      Simple_View                 := Simple_Subprogram_View_Access
        (Copy (Type_Of.Target_Type.Ref.Target_Subprogram));
      Simple_View.Call_Convention := Ada_To_Java;
      Simple_View.Name            := To_Dynamic_Expression (Callback_Name);
      Simple_View.Force_Attach    := Simple_Type_View_Access (Element);
      Simple_View.Is_Abstract     := True;
      Simple_View.Is_Final        := False;
      Simple_View.Is_Real_Primitive := True;

      Bind_Subprogram (Handle, Simple_View, Unit, Empty_Wrapping_Map);

      Element_Bound.Ada_High_Name := Simple_View.Ada_High_Sb_Name;

      Simple_Impl_View := Create (Element);
      Simple_Impl_View := Simple_Subprogram_View_Access
        (Copy (Type_Of.Target_Type.Ref.Target_Subprogram));

      Simple_Impl_View.Call_Convention := Java_To_Ada;
      Tmp_Array := Simple_Impl_View.Parameters;
      Simple_Impl_View.Parameters := new Parameter_Array
        (1 .. Tmp_Array'Length + 1);
      Simple_Impl_View.Parameters (2 .. Tmp_Array'Length + 1) :=
        Tmp_Array.all;

      Simple_Impl_View.Parameters (1)                := Create
        (Simple_Impl_View);
      Simple_Impl_View.Parameters (1).Type_Of        :=
        (Simple_Type_View_Access (Element),
         others         => <>);
      Simple_Impl_View.Parameters (1).Name           := To_Dynamic_Expression
        ("This");
      Simple_Impl_View.Parameters (1).Is_Controlling := True;
      Simple_Impl_View.Parameters (1).Is_Primitive   := True;
      Simple_Impl_View.Parameters (1).Attached       := True;
      Simple_Impl_View.Parameters (1).Assume_Stored  := False;

      Simple_Impl_View.Name := To_Dynamic_Expression (Callback_Name);
      Simple_Impl_View.Is_Abstract := False;

      Add_Wrapper
        (Map     => Simple_Impl_Map,
         Wrapper => Simple_Impl_Map_Wrapper'Unrestricted_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Parameter_Resolution);

      Bind_Subprogram
        (Handle, Simple_Impl_View,
         Get_Default_Implementation_Unit (Element_Bound),
         Simple_Impl_Map);

      Append
        (Unit.Java_File.Public_Body_Part,
         New_Line & New_Line & "public boolean isEscapable () {"
         & New_Line (1)
         & "return false;"
         & New_Line (-1) & "}");
   end Bind;

   -------------------------
   -- Generate_Child_Type --
   -------------------------

   procedure Generate_Child_Type
     (Handle        : access Kernel.Kernel_Record;
      Unit          : Bound_Unit;
      Element       : access Bound_Callback;
      Type_Of       : Simple_Type_View_Access)
   is
      pragma Unreferenced (Handle, Type_Of);

      Child_Name : constant Wide_String_Access :=
        new Wide_String'(Get_Unique_Id);

      Java_Class_Name : Dynamic_Expression := New_Dynamic_Expression;

      Java_Class : constant Binding_File :=
        Get_Default_Implementation_Unit (Element).Java_File;
   begin
      Element.Child_Wrapper_Type := Child_Name;

      Append
        (Java_Class_Name,
         Replace_Dots_By_Slashes (To_Wide_String
           (Java_Class.Full_Class_Name)));

      Append
        (Java_Class.Public_Body_Part,
         New_Line & New_Line
         & "int [] adaAccess;"
         & New_Line & New_Line
         & "public " & To_Wide_String (Java_Class.Class_Name)
         & " (com.adacore.ajis.internal.ada.AdaAccess access) {" & New_Line (1)
         &  "this.adaAccess = access.fAcc;" & New_Line (-1)
         & "}");

      Append
        (Java_Class.Public_Body_Part,
         New_Line & New_Line & "public boolean isEscapable () {"
         & New_Line (1) & "return true;"
         & New_Line (-1) & "}");

      Append
        (Unit.Java_File.Dependencies_Part,
         New_Line & "import " & To_Wide_String
           (Java_Class.Full_Class_Name) & ";");

      Append
        (Unit.Ada_Spec_File.Public_Body_Part,
         New_Line & Element.Default_Implementation.Java_File.Class_Name
         & "_Constructor : " & AJIS_Pckg & ".Java_Method_Access := "
         & AJIS_Pckg & ".Get_Java_Method ("""
         & "L" & Replace_Dots_By_Slashes
           (To_Wide_String (Java_Class.Full_Class_Name))
         & ";"", ""<init>"", "
         & """(Lcom/adacore/ajis/internal/ada/AdaAccess;)V"");");
   end Generate_Child_Type;

   -------------------------------------
   -- Get_Default_Implementation_Unit --
   -------------------------------------

   function Get_Default_Implementation_Unit
     (Element_Bound : access Bound_Callback) return Bound_Unit
   is
      Id   : constant Wide_String := Get_Unique_Id;
      Unit : Bound_Unit;
   begin
      if Element_Bound.Default_Implementation = null then
         Unit := Get_Or_Create_Bound_Unit
           (Element_Bound.Kernel,
            Element_Bound.Enclosing_Unit.Base_Pckg,
            Id);
         Element_Bound.Default_Implementation := Unit;

         Append
           (Unit.Java_File.Java_Parent_Class,
            To_Wide_String
              (Element_Bound.Enclosing_Unit.Java_File.Class_Name));
      end if;

      return Element_Bound.Default_Implementation;
   end Get_Default_Implementation_Unit;

   -----------------------
   -- Get_Ada_High_Name --
   -----------------------

   function Get_Ada_High_Name
     (Element_Bound : access Bound_Callback) return Dynamic_Expression is
   begin
      return Element_Bound.Ada_High_Name;
   end Get_Ada_High_Name;

   -----------
   -- Wrapp --
   -----------

   procedure Wrapp
     (Wrapper    : in out Callback_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : Parameter_Handlings_Params renames Parameter_Handlings_Params
        (Parameters.all);
   begin
      if Params.Is_First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & ".all");

         if not Params.Is_Last then
            Append (Expression.Enclosing_Expression.Expression, " (");
         end if;
      end if;

      if Params.Param_Number > 2 then
         Append (Expression.Enclosing_Expression.Expression, ", ");
      end if;

      if not Params.Is_First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression);
      end if;

      if Params.Is_Last and then not Params.Is_First then
         Append (Expression.Enclosing_Expression.Expression, ")");
      end if;
   end Wrapp;

end Ada2Java.Bound_Elements.Callbacks;
