------------------------------------------------------------------------------
--                                                                          --
--                            GNAT2XML COMPONENTS                           --
--                                                                          --
--                    G N A T 2 X M L . A D A _ T R E E S                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                  Copyright (C) 2012-2013, AdaCore, Inc.                  --
--                                                                          --
-- Gnat2xml is free software; you can redistribute it and/or modify it      --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. Gnat2xml is distributed  in the hope  that it will be useful,   --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MER-      --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License distributed with GNAT; see file COPYING. If   --
-- not, write to the Free Software Foundation, 59 Temple Place Suite 330,   --
-- Boston, MA 02111-1307, USA.                                              --
-- The gnat2xml tool was derived from the Avatox sources.                   --
------------------------------------------------------------------------------

pragma Ada_2012;

with Ada.Wide_Text_IO;

with Ada_Trees;                use Ada_Trees;
with Ada_Trees.Self_Rep;
with Ada_Trees.Asis_To_Tree;   use Ada_Trees.Asis_To_Tree;
with Ada_Trees.Buffers;        use Ada_Trees.Buffers;
with ASIS_UL.Formatted_Output; use ASIS_UL.Formatted_Output;
with ASIS_UL.Common;           use ASIS_UL;

package body Ada_Trees.PP is

   Source_Buf : Buffer;
   --  Buffer containing the text of the original source file

   Write_BOM : Boolean;
   --  True if BOM should be written to the output

   procedure To_Ada (CU : Asis.Compilation_Unit; Options : Formatting_Options);
   --  Helper for Asis_To_Ada

   procedure To_Ada
     (CU      : Asis.Compilation_Unit;
      Options : Formatting_Options)
   is

      Tree : constant Ada_Tree_Base := Compilation_Unit_To_Tree (CU);
   --  In case Free_Tree is changed to null-out its parameter, we don't want to
   --  check the predicate of Tree afterward.
   begin
      if Ada_Trees.Debug_Mode then
         Ada_Trees.Self_Rep.Put_Ada_Tree (Tree);
         Put ("\n");
      end if;

      Tree_To_Ada (Tree, Source_Buf, Write_BOM, Options);
      Free_Tree (Tree);
   end To_Ada;

   procedure Asis_To_Ada
     (CU          : Asis.Compilation_Unit;
      Source_Name : String;
      Options     : Formatting_Options)
   is
   begin
      Read_Ada_File (Source_Buf, Source_Name, Write_BOM, Expand_Tabs => True);
      --  Expand tabs unconditionally. This differs from the behavior of the
      --  old gnatpp, which has an option for that (but only for comments).

      if Assert_Enabled then
         To_Ada (CU, Options);
      else
         begin
            To_Ada (CU, Options);
         exception
            --  If we got an error, don't produce output

            when Common.Fatal_Error =>
               raise;

            when others =>
               --  In order to avoid damaging the user's source code, if there
               --  is a bug (like a token mismatch in Final_Check), we write
               --  the input to the output unchanged.

               if not At_Beginning (Source_Buf) then
                  while not At_End (Source_Buf) loop
                     Move_Forward (Source_Buf);
                  end loop;
                  Reset (Source_Buf);
               end if;

               declare
                  Elems :
                    W_Str renames
                    Elements (Source_Buf)
                      (1 .. Last_Position (Source_Buf) - 1);
                  use Ada.Wide_Text_IO;
               begin
                  if Write_BOM then
                     Put (W_Char'Val (16#FEFF#)); -- BOM as a wide character
                  end if;
                  Put (Elems);
               end;
         end;
      end if;
   end Asis_To_Ada;

end Ada_Trees.PP;
