------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2013, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Containers.Doubly_Linked_Lists;
with Ada.Containers.Ordered_Maps;

with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Packages;            use Ada2Java.Packages;

with Asis; use Asis;
with Asis.Extensions.Flat_Kinds; use Asis.Extensions.Flat_Kinds;

package Ada2Java.Simplifications is

   type Simple_Element_View;

   type Simple_Element_View_Access is access all Simple_Element_View'Class;

   type Can_Be_Bound_Status is (Unknown, Yes, No);

   type Simple_Element_View is abstract tagged record
      Index                : Element_Index := Anonymous_Element_Index;
      Base_Package         : Package_Handle;

      Annotation_Renaming : Boolean := False;
      --  If this flag is true, then all annotations will go to the renamed
      --  entity (if any), and the entity itself won't be bound.

      Renaming            : Simple_Element_View_Access := null;
      --  If this subprogram is a renaming of an other subprogram, the
      --  renamed subprogram is access through this field - null otherwise.

      Location            : Source_Location;

      Locking_State       : Locking_State_Type :=
        Ada2Java.Default_Locking_State;
      --  If this is true, subprograms generated out of this element will be
      --  thread safe.
      --  ??? Perform the relevant copy when needed.

      Lock_Var            : Dynamic_Expression;
      --  The full access to the lock.

      Can_Be_Bound        : Can_Be_Bound_Status := Unknown;
      --  If false, then the element can't be bound

      Resolve_Ambiguous_Expression : Boolean := False;
      --  If true, uses qualified expression in binding

   end record;

   function Copy
     (This : access Simple_Element_View)
      return access Simple_Element_View is abstract;

   function Get_Simple_Element
     (Handle  : not null access Kernel.Kernel_Record;
      Element : Asis.Element) return Simple_Element_View_Access;
   --  Return the simplified element corresponding to this Asis element. If
   --  the element has already been computed previously, it will be returned,
   --  otherwise it will be computed. This procedure may raise Not_Supported
   --  or Silent_Not_Supported if the element cannot be simplified.

   package Simplified_Elements_List is new Ada.Containers.Doubly_Linked_Lists
     (Simple_Element_View_Access);

   function Get_Source_Location
     (Element : Asis.Element) return Source_Location;
   --  Return the source location associated to this element.

   function Get_Source_Location_Of_Name
     (Element : Asis.Element) return Source_Location;
   --  Return the source location of the first name associated to this element.

   type Simple_Type_Kind is
     (Unknown_Kind,

      --  Kinds bound to primitive types:

      Generic_Float_Kind,
      Generic_Integer_Kind,
      Standard_Boolean_Kind,
      Standard_Character_Kind,
      Enumeration_Kind,

      Access_Kind,

      --  Kinds leading to access issues (need 'Access or object allocation):

      Array_Kind,
      Subprogram_Kind,
      Private_Kind,
      Record_Kind,
      Tagged_Record_Kind,

      --  Kind of data already managed through JNI types (no conversion needed)

      JNI_Kind);

   subtype Access_Issue_Kinds is Simple_Type_Kind
   range Array_Kind .. Tagged_Record_Kind;

   type Simple_Type_View;
   type Simple_Type_View_Access is access all Simple_Type_View'Class;

   type Simple_Type_Reference is record
      Ref            : Simple_Type_View_Access;
      Is_Class_Wide  : Boolean := False;

      Pass_By_Address : Boolean := False;

      --  Type references passing may lead to Java wrappers creations
      --  / retrievals. When this is true, only the typeless address
      --  of the object will be managed by the chain of objects.

      Initial_Subtype_Name : Dynamic_Expression;
      --  The initial subtype name - replaced by the actual type name in the
      --  Ref attribute. Needed for e.g. discriminant types.

      Is_From_Unrestricted_Access : Boolean := False;
      --  When this flag is set to true, then the object managed by this
      --  reference is known to have been created from an unrestricted access
      --  ??? perhaps we should extend this to anything access through a
      --  'Access ?

      Is_Constrained_St_From_Unconstrained : Boolean := False;
      --  True if the subtype is a constrained subtype of an unconstrained
      --  type.

      Is_Not_Null : Boolean := False;
      --  If the reference has a null exclusion
   end record;

   Null_Type_Reference : constant Simple_Type_Reference :=
     (null, False, False, Empty_Dynamic_Expression, False, False, False);

   type Simple_Type_Wrapper_View;
   type Simple_Type_Wrapper_Access is access all
     Simple_Type_Wrapper_View'Class;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Type_Wrapper_Access;

   type Simple_Object_View;
   type Simple_Object_View_Access is access all Simple_Object_View'Class;
   type Simple_Object_View_Array is array
     (Positive range <>) of Simple_Object_View_Access;
   type Simple_Object_View_Array_Access is access all Simple_Object_View_Array;

   type Simple_Subprogram_View;
   type Simple_Subprogram_View_Access
     is access all Simple_Subprogram_View'Class;
   type Simple_Subprogram_View_Array is array
     (Positive range <>) of Simple_Subprogram_View_Access;
   type Simple_Subprogram_View_Array_Access is access all
     Simple_Subprogram_View_Array;

   type Values_Array is array (Positive range <>) of Dynamic_Expression;
   type Values_Array_Access is access all Values_Array;

   type Simple_Type_View is abstract new Simple_Element_View with record
      Kind              : Simple_Type_Kind := Unknown_Kind;
      Target_Type       : aliased Simple_Type_Reference := Null_Type_Reference;
      Target_Subprogram : Simple_Subprogram_View_Access := null;

      Full_Ada_Name     : Dynamic_Expression;
      Full_Java_Name    : Dynamic_Expression;
      Exported_Name     : Dynamic_Expression;

      Is_Abstract       : Boolean := False;
      Is_Limited        : Boolean := False;
      Is_Constant       : Boolean := False;

      Indexes           : Simple_Object_View_Array_Access := null;
      --  Contains list of array indexes or type discriminants.

      Components        : Simple_Object_View_Array_Access := null;
      Discriminants     : Simple_Object_View_Array_Access := null;
      Enum_Values       : Values_Array_Access := null;
      Primitives        : Simple_Subprogram_View_Array_Access := null;

      Allow_Java_Creation : Boolean := True;
      --  When this flag is false, then the contructor creating the object from
      --  java will not be generated.

      Allow_Java_Child_Types : Boolean := True;
      --  Certain types, e.g. tagged types, requires the creation of child
      --  types for extension. When this flag is false, the child type is not
      --  generated, and the java class is not derivable (should be marked as
      --  "final").

      Is_Anonymous : Boolean := False;
      --  If this is the type of an anonymous object, this is true. The name
      --  set is then a arbitrary created name.

      Size : Natural := 0;
      --  The size needed to encode the type, in number of bytes.

      Named_Access          : Dynamic_Expression := New_Dynamic_Expression;
      Named_Constant_Access : Dynamic_Expression := New_Dynamic_Expression;
      Named_Class_Access    : Dynamic_Expression := New_Dynamic_Expression;
      --  These classes are set during the bound process to the full names
      --  of a named access and a named class access. May be used for
      --  declaring local variables or returned types.
      --  ??? We should use them in more situation, in order to limit cases
      --  where we have to retreive e.g. conversion types.

      Wrapper : Simple_Type_Wrapper_Access;
      --  If needed, a wrapper may be generated for this type and handle
      --  operations needed for e.g. some cases of in out parameters.
   end record;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Type_View_Access;

   function Is_Type_Consistent
     (Ref : Simple_Type_Reference) return Boolean;
   --  Return true if the type reference given in parameter is consistent,
   --  false if it's malformed.

   function Simplify_Type_Of
     (Handle  : access Kernel.Kernel_Record;
      Element : Asis.Element) return Simple_Type_Reference;
   pragma Postcondition
     (Is_Type_Consistent (Simplify_Type_Of'Result),
      "simplified type inconsistent");
   --  This function take an object declaration / definition in parameter, and
   --  return a simplified view of the type associated to it. It works with
   --  type declarations, variables / parameters / constants declarations,
   --  and will return the data of the returned type of a function.

   procedure Simplify_Type_Of
     (Handle  : access Kernel.Kernel_Record;
      Element : Asis.Element;
      Ref     : access Simple_Type_Reference);
   pragma Postcondition
     (Is_Type_Consistent (Ref.all),
      "simplified type inconsistent");
   --  Same as above, using a procedure. Using this subprogram may be required
   --  when the reference has to be assigned prior to type definition analyis,
   --  in order to ensuire type consistency in certain circular dependencies.

   procedure Free (This : in out Simple_Type_View_Access);

   function Dump (This : Simple_Type_View) return String;
   --  Return a string based on the contents of the simplified type view.

   type Simple_Type_Wrapper_View is abstract new Simple_Element_View
   with record
      Wrapped_Type : Simple_Type_View_Access;
   end record;

   type Simple_Object_View is abstract new Simple_Element_View with record
      Type_Of     : Simple_Type_Reference;

      Name        : Dynamic_Expression;
      --  The original name of the object.

      Java_Name : Dynamic_Expression := Empty_Dynamic_Expression;
      --  This is the Java name of the entity, may differ than the Ada name,
      --  e.g. if the Ada name is a Java reserver word.

      Glue_Name  : Dynamic_Expression := Empty_Dynamic_Expression;
      --  On some occasion, in order to avoid name clashes, we need a glue
      --  Name that differs from the original name - this is being used
      --  e.g. on Ada glue generated code. When no special glue name is
      --  required, this is equals to Empty_Expression. This should not
      --  be accessed directly, but using Get_Glue_Name instead, in order
      --  to get Name by default if Glue_Name is empty.

      Is_Constant : Boolean := False;
      Is_Aliased  : Boolean := False;
   end record;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Object_View_Access;

   function Get_Glue_Name
     (Element : Simple_Object_View'Class) return Dynamic_Expression;
   --  Return the name to be used in the glue Ada code, possibly differs from
   --  the original name in order to avoid certain cases of name clashes.

   function Get_Java_Name
     (Element : Simple_Object_View'Class) return Dynamic_Expression;
   --  Return the name to be used for this object on the Java side.

   type Parameter_Mode is (In_Mode, Out_Mode, In_Out_Mode);

   type Simple_Profile_Data_View is abstract new Simple_Object_View with record
      Is_Primitive : Boolean := False;

      Is_Controlling : Boolean := False;
      --  This is likely to represent the same information as the
      --  Is_Dispatching in the type reference.

      Enclosing_Sb : Simple_Subprogram_View_Access;

      Assume_Stored : Boolean := Ada2Java.Default_Assumed_Stored;
      --  If this flag is true and the type of the parameter is an access
      --  type, then checks will be generated to avoid parameter that should
      --  not be stored (e.g. owned objects).
   end record;

   type Simple_Profile_Data_View_Access is access all
     Simple_Profile_Data_View'Class;

   function Create
     (Enclosing : not null access Simple_Subprogram_View'Class)
      return Simple_Profile_Data_View_Access;

   type Parameter_Simple_View is abstract new Simple_Profile_Data_View
   with record
      Mode          : Parameter_Mode := In_Mode;

      Attached : Boolean := False;
      --  When this is true, the subprogram will be attached to the class of
      --  this parameter.
   end record;

   type Parameter_Simple_View_Access is access all Parameter_Simple_View'Class;

   function Create
     (Enclosing : not null access Simple_Subprogram_View'Class)
      return Parameter_Simple_View_Access;

   function To_Parameter_Simple_View
     (Handle       : access Kernel.Kernel_Record;
      Enclosing_Sb : Simple_Subprogram_View_Access;
      Element      : Asis.Element;
      Name         : Asis.Element)
      return Parameter_Simple_View_Access;
   --  Computes the simple view of the given parameter.

   type Parameter_Array is array
     (Positive range <>) of Parameter_Simple_View_Access;

   type Parameter_Array_Access is access all Parameter_Array;

   type Call_Conventions is (Java_To_Ada, Ada_To_Java);

   type JNI_Env_Origins is (Generic_Parameter, Object_Field);

   type Simple_Subprogram_View is abstract new Simple_Element_View with record
      Returned_Type  : Simple_Profile_Data_View_Access;
      Parameters     : Parameter_Array_Access;
      Is_Dispatching : Boolean := False;

      Is_Final       : Boolean := True;
      Is_Constructor : Boolean := False;
      Is_Abstract    : Boolean := False;

      Bind_Java       : Boolean := True;
      --  If Bind_Java is false, then no binding will be generated in the java
      --  side. This is typically used when we bind calls from ada to java.

      Bind_Ada       : Boolean := True;
      --  If Bind_Ada is false, then no binding glue will be generated in the
      --  ada side. No Java JNI subprogram will be generated either.

      Name            : Dynamic_Expression := Empty_Dynamic_Expression;
      --  Name of the subprogram as known by the generated binding. This is the
      --  name that will be used by the final user of the binding, may be
      --  different from the name initially given in Ada (hold by the
      --  Original_Name field).

      Java_Name       : Dynamic_Expression := Empty_Dynamic_Expression;
      --  Name to use on the Java side, which may be different from the Ada
      --  name.

      Ada_High_Sb_Name : Dynamic_Expression := Empty_Dynamic_Expression;
      --  This variable is set during the binding process, to the name of the
      --  actual High wrapping subprogram.

      Original_Name : Dynamic_Expression := Empty_Dynamic_Expression;
      --  If the subprogram is coming from a subprogram entity analyzis, this
      --  holds the original name of this subprogram, which might be renamed
      --  afterwards.

      Call_Convention : Call_Conventions := Java_To_Ada;

      JNI_Env_Origin  : JNI_Env_Origins := Generic_Parameter;
      --  In the case Ada_To_Java, we have to know where the
      --  JNI_Env_Access & J_Class parameter are coming from. It's either from
      --  a generic parameter (callback case) or an object field
      --  (overriding case).

      Is_Real_Primitive : Boolean := False;
      --  If we are in the process of generating a "real" primitive, this is
      --  set to true. In this case, parameters and return types can't be
      --  changed arbitrary.

      High_Sb_With_Env : Boolean := False;
      --  If the High subprogram needs to have access to the java environment,
      --  this flag has to be set to true. For example, this is required by the
      --  constructor.

      Overridden_Sb : Simple_Subprogram_View_Access;
      --  If not null, this subprogram if overriding the subprogram given
      --  in parameter.

      Overriding_Sbs : Simplified_Elements_List.List;
      --  This list constains the list of subprograms overriding this one.

      Force_Attach : Simple_Type_View_Access := null;
      --  If this is not null, and if the subprogram is not naturally attached,
      --  this will force the attachement to a given bound type, but the java
      --  "this" parameter will simply be ignored.
   end record;

   function Create
     (Enclosing : not null access Simple_Element_View'Class)
      return Simple_Subprogram_View_Access;

   function Is_Attached (This : Simple_Subprogram_View) return Boolean;
   --  Return true if this subprogram is attached to its first parameter,
   --  false otherwise.

   function Get_Attached_Type
     (This : Simple_Subprogram_View) return Simple_Type_View_Access;
   --  Returned the type on which this subprogram is attached to, null if none.

   function Get_Controlling_Param
     (This : Simple_Subprogram_View) return Parameter_Simple_View_Access;
   --  Return the first controlling parameter of the subprogram, null if none.

   function Get_Java_Name
     (This : Simple_Subprogram_View) return Dynamic_Expression;
   --  Return the name to be used to the java side for this subprogram

   function Is_Standard_Package (E : Asis.Element) return Boolean;
   --  Return True if the element given in parameter is the package standard.

   function Is_Standard_Boolean (E : Asis.Element) return Boolean;
   --  Return True if the element given in parameter is the standard type
   --  Boolean.

   function Is_Standard_Character (E : Asis.Element) return Boolean;
   --  Return True if the element given in parameter is the standard type
   --  Character.

   function Is_Standard_String (E : Asis.Element) return Boolean;
   --  Return True if the element given in parameter is the standard type
   --  String.

   function "<" (Left, Right : Asis.Element) return Boolean;
   --  Needed to instantiate Bound_Element_DB

   type Simple_Exception_View is abstract new Simple_Element_View with record
      Name : Dynamic_Expression;
   end record;

   type Simple_Exception_View_Access is access all Simple_Exception_View'Class;

   function Create
     (Enclosing : access Simple_Element_View'Class)
      return Simple_Exception_View_Access;

   type Simple_Element_Index is record
      Elem_Index        : Element_Index;
      Instantiation_Loc : String_Access;
   end record;

   Exception_Occurence_Index : constant Simple_Element_Index :=
     ((Not_An_Element, new String'("exception_occurence")), new String'(""));
   --  This index should be used of the Exception_Occurence type.

   String_Index : constant Simple_Element_Index :=
     ((Not_An_Element, new String'("string")), new String'(""));

   function To_Simple_Element_Index
     (Element : Asis.Element) return Simple_Element_Index;
   --  Create a simple element index out of this asis element. Each ASIS
   --  element is associated to a unique simplement element index, even
   --  implicit elements.

   function "<" (Left, Right : Simple_Element_Index) return Boolean;

   package Simple_Elements_DB_Pckg is new Ada.Containers.Ordered_Maps
     (Simple_Element_Index, Simple_Element_View_Access, "<");

   use Simple_Elements_DB_Pckg;

   function Create_Access_To
     (Base_Type : Simple_Type_Reference; Is_Constant : Boolean)
      return Simple_Type_Reference;
   --  Generate a dummy type reference accessing an object of the type given
   --  in parameter.

   procedure Initialize_Configurable_Properties
     (Handle : not null access Kernel.Kernel_Record;
      Object : access Simple_Element_View'Class);

   function Is_Unconstrained_Array
     (Type_Of : Simple_Type_View_Access) return Boolean;
   --  Return true if the type given in parameter in an unconstrained array
   --  type.

end Ada2Java.Simplifications;
