------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2013, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada2Java.Bound_Elements.Types;  use Ada2Java.Bound_Elements.Types;
with Ada2Java.Bound_Elements.Common; use Ada2Java.Bound_Elements.Common;
with Ada2Java.Kernel;               use Ada2Java.Kernel;
with Ada2Java.Simplifications;      use Ada2Java.Simplifications;
with Ada2Java.Utils;                use Ada2Java.Utils;

with Ada2Java.Bound_Elements.Subprograms;
use Ada2Java.Bound_Elements.Subprograms;

package body Ada2Java.Bound_Elements.Tagged_Types is

   procedure Generate_Child_Type
     (Handle  : access Kernel.Kernel_Record;
      Element : access Bound_Tagged_Type;
      Type_Of : Simple_Type_View_Access);
   --  Generate the default java implementation for abstract record types, and
   --  the shadow class if allowed.

   type Tagged_Type_Ada_Primitive_Wrapper is new Code_Wrapper
   with record
      Ada_Dispatching_Param : Dynamic_Expression;
   end record;

   procedure Wrap
     (Wrapper    : in out Tagged_Type_Ada_Primitive_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Tagged_Type_Java_Primitive_Wrapper is new Code_Wrapper
   with record
      Ada_Dispatching_Param : Dynamic_Expression;
   end record;

   procedure Wrap
     (Wrapper    : in out Tagged_Type_Java_Primitive_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type JNI_Set_Global_Ref_Wrapper is new Code_Wrapper
   with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out JNI_Set_Global_Ref_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Tagged_Discriminants_Handler is new Ada_Constructor_Handler
   with null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Tagged_Discriminants_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Tagged_Clone_Handler is new Ada_Constructor_Handler
   with null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Tagged_Clone_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   ------------------------------
   -- Create_Bound_Tagged_Type --
   ------------------------------

   function Create_Bound_Tagged_Type
     (Element : Simple_Element_View_Access) return Bound_Element is
   begin
      if Element.all in Simple_Type_View'Class then
         declare
            Type_Of : Simple_Type_View renames Simple_Type_View (Element.all);
         begin
            if Type_Of.Kind = Tagged_Record_Kind then
               return new Bound_Tagged_Type'
                 (Child_Wrapper_Type => To_Dynamic_Expression (Get_Unique_Id),
                  others => <>);
            end if;
         end;
      end if;

      return null;
   end Create_Bound_Tagged_Type;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Element_Bound : access Bound_Tagged_Type;
      Handle        : not null access Kernel.Kernel_Record)
   is
      Simple_View : constant Simple_Type_View_Access :=
        Simple_Type_View_Access (Element_Bound.Simple_Element);
      Unit           : Bound_Unit;

      Constructor_Wrapper_Map : Wrapping_Map;

      Anonymous_Access          : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Anonymous_Constant_Access : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
      Anonymous_Class_Access    : constant Dynamic_Expression :=
        To_Dynamic_Expression (Get_Unique_Id);
   begin
      Unit := Get_Or_Create_Bound_Unit (Handle, Element_Bound.Simple_Element);
      Unit.Java_File.Is_Final := False;

      Append
        (Simple_View.Named_Access, Unit.Root_Unit.Ada_Pckg_Name
         & "." & Anonymous_Access);
      Append
        (Simple_View.Named_Constant_Access, Unit.Root_Unit.Ada_Pckg_Name
         & "." & Anonymous_Constant_Access);
      Append
        (Simple_View.Named_Class_Access, Unit.Root_Unit.Ada_Pckg_Name
         & "." & Anonymous_Class_Access);

      Append
        (Unit.Root_Unit.Ada_Spec_File.Public_Body_Part,
         New_Line & New_Line & "type " & Anonymous_Access
         & " is access all " & Simple_View.Full_Ada_Name & ";"
         & New_Line & "type " & Anonymous_Constant_Access
         & " is access constant " & Simple_View.Full_Ada_Name & ";"
         & New_Line & "type " & Anonymous_Class_Access
         & " is access all " & Simple_View.Full_Ada_Name
         & "'Class;");

      Generate_Child_Type
        (Handle,
         Element_Bound,
         Simple_View);

      Bind_Equals (Handle, Simple_View, Unit);

      if Simple_View.Target_Type /= Null_Type_Reference then
         Append
           (Unit.Java_File.Java_Parent_Class,
            Simple_View.Target_Type.Ref.Full_Java_Name);

         Add_Import_Element
           (Unit.Java_File,
            To_Wide_String (Simple_View.Full_Java_Name));
      else
         Append
           (Unit.Java_File.Java_Parent_Class,
            "com.adacore.ajis.internal.ada.AdaProxy");
      end if;

      if Simple_View.Allow_Java_Creation then

         -- Constructor --

         declare
            Constructor_View : Simple_Subprogram_View_Access;
         begin
            if Simple_View.Allow_Java_Child_Types then
               Initialize_Constructor_View
                 (Type_View        => Simple_View,
                  Binded_Type_Name =>
                  new Wide_String'(To_Wide_String
                    (Element_Bound.Child_Wrapper_Type)),
                  Constructor_View => Constructor_View,
                  Constructor_Map  => Constructor_Wrapper_Map,
                  Ada_Parameter_Hander =>
                    new Ada_Tagged_Discriminants_Handler);
            else
               Initialize_Constructor_View
                 (Type_View        => Simple_View,
                  Binded_Type_Name =>
                  new Wide_String'(To_Wide_String (Simple_View.Full_Ada_Name)),
                  Constructor_View => Constructor_View,
                  Constructor_Map  => Constructor_Wrapper_Map);
            end if;

            Initialize_Constructor_Parameters
              (Constructor_View => Constructor_View);

            Bind_Subprogram
              (Handle   => Handle,
               View     => Constructor_View,
               Unit     => Unit,
               Wrappers => Constructor_Wrapper_Map);
         end;

         --  setGlobalRef  --

         declare
            Set_Global_Ref_View    : Simple_Subprogram_View_Access;
            This_Parameter         : Parameter_Simple_View_Access;
            Is_Global_Parameter    : Parameter_Simple_View_Access;
            Map                    : Wrapping_Map;
            Set_Global_Ref_Wrapper : aliased JNI_Set_Global_Ref_Wrapper;
         begin
            Set_Global_Ref_View := Create (Element_Bound.Simple_Element);

            This_Parameter                := Create (Set_Global_Ref_View);
            This_Parameter.Type_Of        := Create_Access_To
              ((Ref           => Simple_View,
                Is_Class_Wide => True,
                others        => <>),
               False);
            This_Parameter.Name           := To_Dynamic_Expression ("This");
            This_Parameter.Is_Primitive   := True;
            This_Parameter.Is_Controlling := True;
            This_Parameter.Attached       := True;
            This_Parameter.Assume_Stored  := False;

            Is_Global_Parameter := Create (Set_Global_Ref_View);
            Is_Global_Parameter.Type_Of        :=
              (Ref => Get_Boolean (Handle), others => <>);
            Is_Global_Parameter.Name           := To_Dynamic_Expression
              ("isGlobal");

            Set_Global_Ref_View.Parameters       := new
              Parameter_Array'
                (1 => This_Parameter, 2 => Is_Global_Parameter);
            Set_Global_Ref_View.Is_Final         := False;
            Set_Global_Ref_View.Name             := To_Dynamic_Expression
              ("setGlobalRef");
            Set_Global_Ref_View.High_Sb_With_Env := True;
            Set_Global_Ref_View.Location         := Simple_View.Location;

            Add_Wrapper
              (Map     => Map,
               Wrapper => Set_Global_Ref_Wrapper'Unrestricted_Access,
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Code_Node_Resolution);
            Bind_Subprogram
              (Handle   => Handle,
               View     => Set_Global_Ref_View,
               Unit     => Unit,
               Wrappers => Map);

            --  Override setOwner so that it sets the global reference

            Append
              (Unit.Java_File.Public_Body_Part,
               New_Line & New_Line
               & "@Override"
               & New_Line
               & "public void setOwner "
               & "(com.adacore.ajis.IProxy.Owner theOwner) {"
               & New_Line (1) &  "super.setOwner (theOwner);"
               & New_Line & "if (theOwner == "
               & "com.adacore.ajis.IProxy.Owner.PROXY) {"
               & New_Line (1) & "this.setGlobalRef (false);"
               & New_Line (-1) & "} else {"
               & New_Line (1) & "this.setGlobalRef (true);"
               & New_Line (-1) & "}"
               & New_Line (-1)
               & "}");

         end;
      end if;

      if Simple_View.Target_Type /= Null_Type_Reference then
         Append
           (Unit.Java_File.Public_Body_Part,
            New_Line & New_Line
            & "public " & Simple_View.Exported_Name
            & " (com.adacore.ajis.internal.ada.AdaAccess access) {"
            & New_Line (1)
            &  "super (access);" & New_Line (-1)
            & "}");
      else
         Append
           (Unit.Java_File.Public_Body_Part,
            New_Line & New_Line
            & "public " & Simple_View.Exported_Name
            & " (com.adacore.ajis.internal.ada.AdaAccess access) {"
            & New_Line (1)
            &  "this.adaAccess = access.fAcc;" & New_Line (-1)
            & "}");
      end if;

      Bind_Destructor
        (Handle   => Handle,
         Unit     => Unit,
         The_Type => Simple_View);

      Bind_Fields
        (Handle    => Handle,
         Type_View => Simple_View,
         Unit      => Unit);

      -- Clone --

      declare
         Clone_Wrapper : aliased Ada_Tagged_Clone_Handler;
      begin
         Bind_Clone
           (Handle, Simple_View, Unit, Clone_Wrapper'Unrestricted_Access);
      end;
   end Bind;

   -------------------------
   -- Generate_Child_Type --
   -------------------------

   procedure Generate_Child_Type
     (Handle  : access Kernel.Kernel_Record;
      Element : access Bound_Tagged_Type;
      Type_Of : Simple_Type_View_Access)
   is
      Type_Declaration : Dynamic_Expression := New_Dynamic_Expression;
      Unit : constant Bound_Unit :=
        Get_Or_Create_Bound_Unit
          (Handle, Simple_Element_View_Access (Type_Of));

      Child_Type : Simple_Type_View_Access;

      Conversion : constant Type_Record := Get_Or_Create_Conversion_Type
        (Handle, Type_Of);
   begin
      Child_Type := Simple_Type_View_Access (Copy (Type_Of));
      Child_Type.Exported_Name := Element.Child_Wrapper_Type;
      Child_Type.Full_Ada_Name := Child_Type.Exported_Name;
      Child_Type.Named_Access := To_Dynamic_Expression (Get_Unique_Id);
      Child_Type.Named_Constant_Access :=
        To_Dynamic_Expression (Get_Unique_Id);
      Child_Type.Named_Class_Access := To_Dynamic_Expression (Get_Unique_Id);

      if Type_Of.Is_Abstract then
         declare
            Java_Class : constant Binding_File :=
              Get_Default_Implementation_Unit (Element).Java_File;
         begin
            Append
              (Java_Class.Public_Body_Part,
               New_Line & New_Line
               & "public " & To_Wide_String (Java_Class.Class_Name)
               & " (com.adacore.ajis.internal.ada.AdaAccess access) {"
               & New_Line (1)
               &  "super (access);" & New_Line (-1)
               & "}");

            Append
              (Unit.Java_File.Dependencies_Part,
               New_Line & "import " & To_Wide_String
                 (Java_Class.Full_Class_Name) & ";");

            for J in Type_Of.Primitives'Range loop
               if Type_Of.Primitives (J).Is_Abstract
                 and then Type_Of.Primitives (J).Can_Be_Bound /= No
               then
                  Bind_Subprogram
                   (Handle,
                    Type_Of.Primitives (J),
                    Get_Default_Implementation_Unit (Element),
                    Empty_Wrapping_Map,
                    True);
               end if;
            end loop;

            Append
              (Unit.Ada_Body_File.Elab_Part,
               New_Line
               & AJIS_Pckg & ".Reference_Java_Object_Constructor ("
               & Type_Of.Full_Ada_Name
               & "'Tag, " & AJIS_Pckg & ".Get_Java_Method ("
               & """"
               & Replace_Dots_By_Slashes (To_Wide_String
                 (Java_Class.Full_Class_Name))
               & """, ""<init>"", "
               & """(Lcom/adacore/ajis/internal/ada/AdaAccess;)V""));");
         end;
      else
         Append
           (Unit.Ada_Body_File.Elab_Part,
            New_Line
            & AJIS_Pckg & ".Reference_Java_Object_Constructor ("
            & Type_Of.Full_Ada_Name
            & "'Tag, " & AJIS_Pckg & ".Get_Java_Method ("
            & """"
            & Replace_Dots_By_Slashes
              (To_Wide_String (Unit.Java_File.Full_Class_Name))
            & """, ""<init>"", "
            & """(Lcom/adacore/ajis/internal/ada/AdaAccess;)V""));");
      end if;

      if Type_Of.Allow_Java_Child_Types then
         Append
           (Type_Declaration, New_Line & New_Line & "type "
            & Child_Type.Exported_Name);

         if Type_Of.Indexes /= null and then Type_Of.Indexes'Length > 0 then
            Element.Discriminant_Renames := new
              Dynamic_Expression_Array (Type_Of.Indexes'Range);
            Append (Type_Declaration, " (");

            for J in Type_Of.Indexes'Range loop
               if J > Type_Of.Indexes'First then
                  Append (Type_Declaration, "; ");
               end if;

               Element.Discriminant_Renames (J) :=
                 To_Dynamic_Expression (Get_Unique_Id);

               Append
                 (Type_Declaration,
                  Element.Discriminant_Renames (J)
                  & " : " & Type_Of.Indexes (J).Type_Of.Initial_Subtype_Name);
            end loop;

            Append (Type_Declaration, ") is new "
                    & Type_Of.Full_Ada_Name & " (");

            for J in Element.Discriminant_Renames'Range loop
               if J > Element.Discriminant_Renames'First then
                  Append (Type_Declaration, ", ");
               end if;

               Append (Type_Declaration, Element.Discriminant_Renames (J));
            end loop;

            Append (Type_Declaration, ")");
         else
            Append
              (Type_Declaration,
               " is new " & Type_Of.Full_Ada_Name);
         end if;

         Append
           (Type_Declaration,
            " and " & AJIS_Pckg &".Cross_Language_Class with record "
            & New_Line (1)
            & "Backlink : " & AJIS_Pckg & ".Backlinked_Java_Reference := "
            & "(Ada.Finalization.Controlled with Enclosing => "
            & "new " & AJIS_Pckg & ".Cross_Lg_Container'(Ptr => null), "
            & "others => <>);"
            & New_Line (-1)
            & "end record;"
            & New_Line & New_Line
            & "procedure Set_Java_Ref (Object : in out "
            & Child_Type.Exported_Name & "; Ref : " & JNI_Pckg & ".J_Object);"
            & New_Line & New_Line
            & "procedure Set_Java_VM (Object : in out "
            & Child_Type.Exported_Name &
               "; VM : " & JNI_Pckg & ".Java_VM_Access);"
            & New_Line & New_Line
            & "function Get_Java_Ref (Object : access constant "
            & Child_Type.Exported_Name & ") return " & JNI_Pckg & ".J_Object;"
            & New_Line & New_Line
            & "function Get_Java_VM (Object : "
            & Child_Type.Exported_Name & ") return " & JNI_Pckg
            & ".Java_VM_Access;"
            & New_Line & New_Line
            & "function Copy_No_Clone (Object : "
            & Child_Type.Exported_Name & ") return access "
            & Child_Type.Exported_Name & ";"
            & New_Line & New_Line
            & "function Get_Address_For_Proxy "
            & " (Env : " & JNI_Pckg & ".JNI_Env_Access; Object : access "
            & Child_Type.Exported_Name
            & ") return " & JNI_Pckg & ".J_Int_J_Array;"
            & New_Line & New_Line
            & "procedure Set_Global_Ref " &
            "(This : in out " & Child_Type.Exported_Name
            & "; Is_Global : Boolean);"
            & New_Line & New_Line
            & "function Create_Instance (Base_Object : "
            & Child_Type.Exported_Name
            & "; Parent : " & Type_Of.Full_Ada_Name
            & ") return " & Child_Type.Exported_Name & ";"
            & New_Line & New_Line
            & "type " & Child_Type.Named_Class_Access & " is access all "
            & Child_Type.Exported_Name
            & "'Class;"
            & New_Line
            & "type " & Child_Type.Named_Access & " is access all "
            & Child_Type.Exported_Name & ";"
            & "type " & Child_Type.Named_Constant_Access
            & " is access constant " & Child_Type.Exported_Name & ";");

         Append (Unit.Ada_Spec_File.Open_Part, Type_Declaration);

         Append
           (Unit.Ada_Body_File.Public_Body_Part,
            New_Line & New_Line
            & "procedure Set_Java_Ref (Object : in out "
            & Child_Type.Exported_Name &
            "; Ref : " & JNI_Pckg & ".J_Object) is"
            & New_Line & "begin"
            & New_Line (1) & "Object.Backlink.Ref := Ref;"
            & New_Line (-1) & "end Set_Java_Ref;"
            & New_Line & New_Line
            & "procedure Set_Java_VM (Object : in out "
            & Child_Type.Exported_Name & "; VM : " & JNI_Pckg
            & ".Java_VM_Access) is"
            & New_Line & "begin"
            & New_Line (1) & "Object.Backlink.VM := VM;"
            & New_Line (-1) & "end Set_Java_VM;"
            & New_Line & New_Line
            & "function Get_Java_Ref (Object : access constant "
            & Child_Type.Exported_Name & ") return "
            & JNI_Pckg & ".J_Object is"
            & New_Line & "begin"
            & New_Line (1) & "if Object.Backlink.Enclosing.Ptr = null then"
            & New_Line (1) & "Object.Backlink.Set_Enclosing_Address "
            & "(Object);"
            & New_Line (-1) & "end if;"
            & New_Line & "return Object.Backlink.Ref;"
            & New_Line (-1) & "end Get_Java_Ref;"
            & New_Line & New_Line
            & "function Get_Java_VM (Object : "
            & Child_Type.Exported_Name & ") return " & JNI_Pckg
            & ".Java_VM_Access is"
            & New_Line & "begin"
            & New_Line (1) & "return Object.Backlink.VM;"
            & New_Line (-1) & "end Get_Java_VM;"
            & New_Line & New_Line
            & "function Copy_No_Clone (Object : "
            & Child_Type.Exported_Name & ") return access "
            & Child_Type.Exported_Name & " is"
            & New_Line (1) & "Res : " & Child_Type.Named_Class_Access
            & " := " & "new " & Child_Type.Exported_Name
            & "'(" & Type_Of.Full_Ada_Name
            & " (Object) with ");

         if Element.Discriminant_Renames /= null then
            for J in Element.Discriminant_Renames'Range loop
               Append
                 (Unit.Ada_Body_File.Public_Body_Part,
                  Element.Discriminant_Renames (J)
                  & " => Object." & Element.Discriminant_Renames (J) & ", ");
            end loop;
         end if;

         Append
           (Unit.Ada_Body_File.Public_Body_Part,
            "Backlink => (Ada.Finalization.Controlled with "
            & "Enclosing => new " & AJIS_Pckg
            & ".Cross_Lg_Container'(Ptr => null),"
            & " Ref => Object.Backlink.Ref, "
            & "VM => Object.Backlink.VM, "
            & "Is_Global_Ref => Object.Backlink.Is_Global_Ref));"
            & New_Line (-1) & "begin"
            & New_Line (1) & "Res.Backlink.Enclosing.Ptr := Res;"
            & New_Line & "return Res;"
            & New_Line (-1) & "end Copy_No_Clone;"
            & New_Line & New_Line
            & "function Get_Address_For_Proxy "
            & "(Env : " & JNI_Pckg & ".JNI_Env_Access; Object : access "
            & Child_Type.Exported_Name
            & ") return " & JNI_Pckg & ".J_Int_J_Array is"
            & New_Line (1)
            & "Ptr : "  & Conversion.Conversion_Package_Name.all
            & ".Object_Pointer := " & Conversion.Conversion_Package_Name.all
            & ".Object_Pointer (Object);"
            & New_Line (-1) & "begin"
            & New_Line (1) & "return "
            & Conversion.Conversion_Package_Name.all
            & ".To_JintArray (Env, Ptr);"
            & New_Line (-1) & "end Get_Address_For_Proxy;"
            & New_Line & New_Line
            & "procedure Set_Global_Ref "
            & "(This : in out " & Child_Type.Exported_Name
            & "; Is_Global : Boolean) is"
            & New_Line & "begin"
            & New_Line (1) & AJIS_Pckg
            & ".Set_Global_Ref (This.Backlink, Is_Global);"
            & New_Line (-1) & "end Set_Global_Ref;"
            & New_Line & New_Line
            & "function Create_Instance (Base_Object : "
            & Child_Type.Exported_Name & "; Parent : " & Type_Of.Full_Ada_Name
            & ") return " & Child_Type.Exported_Name & " is"
            &  New_Line (1) & Env_Parameter_Name & " : aliased " & JNI_Pckg
            & ".JNI_Env_Access;"
            & New_Line & Get_Unique_Id & " : " & JNI_Pckg & ".J_Int := "
            & JNI_Pckg & ".Attach_Current_Thread (Get_Java_VM (Base_Object), "
            & Env_Parameter_Name & "'Access, System.Null_Address);"
            & New_Line & "Class : " & JNI_Pckg & ".J_Class;"
            & New_Line & "Constr : " & JNI_Pckg & ".J_Method_ID;"
            & New_Line (-1) & "begin"
            & New_Line (1) & "Class := " & JNI_Pckg & ".Get_Object_Class ("
            & Env_Parameter_Name & ", Base_Object.Get_Java_Ref);"
            & New_Line & "Constr := " & JNI_Pckg & ".Get_Method_ID ("
            & Env_Parameter_Name & ", Class, String'(""<init>""), ""()V"");"
            & New_Line
            & "if " & JNI_Pckg & ".""="" (Constr, " & JNI_Pckg
            & ".J_Null_Method_ID) then"
            & New_Line (1) & "raise AJIS.Java.Java_Exception with "
            & """derived class has no default constructor"";"
            & New_Line (-1) & "end if;"
            & New_Line & "return Ret : " & Child_Type.Exported_Name
            & " := " & Child_Type.Exported_Name & "'(Parent"
            & " with ");

         if Element.Discriminant_Renames /= null then
            for J in Element.Discriminant_Renames'Range loop
               Append
                 (Unit.Ada_Body_File.Public_Body_Part,
                  Element.Discriminant_Renames (J)
                  & " => Parent." & Type_Of.Indexes (J).Name & ", ");
            end loop;
         end if;

         Append
           (Unit.Ada_Body_File.Public_Body_Part,
            "others => <>) do"
            & New_Line (1) & "Ret.Set_Java_VM (Base_Object.Get_Java_VM);"
            & New_Line  & "Ret.Set_Java_Ref (" & JNI_Pckg & ".New_Object_A ("
            & Env_Parameter_Name & ", Class, Constr, (1 .. 0 => <>)));"
            & New_Line (-1) & "end return;"
            & New_Line (-1) & "end Create_Instance;");

         if Type_Of.Primitives /= null then
            for J in Type_Of.Primitives'Range loop
               --  If the file where the type has been analyzed is not part of
               --  the files explicitely passed to the tool then the primitives
               --  are not yet added to the list of elements to be bound - even
               --  if they need to be for type completeness. This ensures that
               --  the primitive will get bound.

               if Type_Of.Primitives (J).Can_Be_Bound /= No then
                  --  Avoid binding operators since they are not supported, and
                  --  don't bind primitives that can't be bound.

                  Add_Bound_Element
                    (Handle,
                     Simple_Element_View_Access (Type_Of.Primitives (J)));

                  declare
                     Primitive_Wrapper_Map : Wrapping_Map;
                     Simple_Primitive_View : constant
                       Simple_Subprogram_View_Access :=
                         Simple_Subprogram_View_Access
                           (Copy (Type_Of.Primitives (J)));
                     Primitive_Wrapper : constant Code_Wrapper_Access :=
                       new Tagged_Type_Ada_Primitive_Wrapper;
                     Controlling_Type : Simple_Type_View_Access;

                     Controlling_Target_Type : Simple_Type_View_Access;
                     Data_Target_Type        : Simple_Type_View_Access;
                  begin
                     Controlling_Type := Simple_Primitive_View.Parameters (1)
                       .Type_Of.Ref;

                     if Controlling_Type.Kind = Access_Kind then
                        Controlling_Target_Type :=
                          Controlling_Type.Target_Type.Ref;
                     else
                        Controlling_Target_Type := Controlling_Type;
                     end if;

                     Simple_Primitive_View.Call_Convention := Ada_To_Java;
                     Simple_Primitive_View.JNI_Env_Origin := Object_Field;

                     for K in Simple_Primitive_View.Parameters'Range loop
                        Data_Target_Type :=
                          Simple_Primitive_View.Parameters (K).Type_Of.Ref;

                        if Data_Target_Type.Kind = Access_Kind
                          and then Data_Target_Type.Is_Anonymous
                        then
                           Data_Target_Type :=
                             Data_Target_Type.Target_Type.Ref;
                        end if;

                        if Simple_Primitive_View.Parameters (K).Is_Primitive
                          and then Controlling_Target_Type = Data_Target_Type
                          and then not Simple_Primitive_View.Parameters (K)
                          .Type_Of.Is_Class_Wide
                        then
                           if Simple_Primitive_View.Parameters (K).Type_Of.Ref
                             .Kind = Access_Kind
                           then
                              Simple_Primitive_View.Parameters (K).Type_Of.Ref
                                :=
                                Simple_Type_View_Access
                                  (Copy (Simple_Primitive_View.Parameters (K)
                                   .Type_Of.Ref));
                              Simple_Primitive_View.Parameters (K).Type_Of.Ref.
                                Target_Type.Ref := Child_Type;
                              Simple_Primitive_View.Parameters (K).Type_Of.
                                Initial_Subtype_Name :=
                                  Child_Type.Named_Access;
                           else
                              Simple_Primitive_View.Parameters (K).Type_Of.Ref
                                := Child_Type;
                              Simple_Primitive_View.Parameters (K).Type_Of.
                                Initial_Subtype_Name :=
                                  Child_Type.Exported_Name;
                           end if;
                        end if;
                     end loop;

                     Simple_Primitive_View.Bind_Java := False;

                     if Simple_Primitive_View.Returned_Type /= null then
                        Data_Target_Type :=
                          Simple_Primitive_View.Returned_Type.Type_Of.Ref;

                        if Data_Target_Type.Kind = Access_Kind
                          and then Data_Target_Type.Is_Anonymous
                        then
                           Data_Target_Type :=
                             Data_Target_Type.Target_Type.Ref;
                        end if;

                        if not Simple_Primitive_View.Returned_Type.Type_Of
                          .Is_Class_Wide
                          and then Controlling_Target_Type = Data_Target_Type
                        then
                           if Simple_Primitive_View.Returned_Type.Type_Of.
                             Ref.Kind = Access_Kind
                           then
                              Simple_Primitive_View.Returned_Type.Type_Of.Ref
                                :=
                                Simple_Type_View_Access
                                  (Copy
                                       (Simple_Primitive_View.
                                            Returned_Type.Type_Of.Ref));
                              Simple_Primitive_View.Returned_Type.Type_Of.Ref.
                                Target_Type.Ref := Child_Type;
                              Simple_Primitive_View.Returned_Type.Type_Of.
                                Initial_Subtype_Name :=
                                  Child_Type.Named_Access;
                           else
                              Simple_Primitive_View.Returned_Type.Type_Of.Ref
                                := Child_Type;
                              Simple_Primitive_View.Returned_Type.Type_Of.
                                Initial_Subtype_Name :=
                                  Child_Type.Exported_Name;
                           end if;
                        end if;
                     end if;

                     Add_Wrapper
                       (Map     => Primitive_Wrapper_Map,
                        Wrapper => Primitive_Wrapper,
                        Lang    => Ada_Lang,
                        Target  => High_Sb,
                        Context => Start);

                     Add_Wrapper
                       (Map     => Primitive_Wrapper_Map,
                        Wrapper => Primitive_Wrapper,
                        Lang    => Ada_Lang,
                        Target  => High_Sb,
                        Context => Add_Parameter_Original);

                     Add_Wrapper
                       (Map     => Primitive_Wrapper_Map,
                        Wrapper => Code_Wrapper_Access'
                          (new Tagged_Type_Java_Primitive_Wrapper),
                        Lang    => Java_Lang,
                        Target  => JNI_Sb,
                        Context => Parameter_Resolution);

                     Simple_Primitive_View.Is_Real_Primitive := True;

                     Bind_Subprogram
                       (Handle           => Handle,
                        View             => Simple_Primitive_View,
                        Unit             => Unit,
                        Wrappers         => Primitive_Wrapper_Map,
                        Add_Java_High_Sb => False);
                  end;
               end if;
            end loop;
         end if;

         Append
           (Unit.Ada_Spec_File.Open_Part, Unit.Ada_Spec_File.Public_Body_Part);

         Unit.Ada_Spec_File.Public_Body_Part := New_Dynamic_Expression;
      end if;
   end Generate_Child_Type;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Tagged_Type_Ada_Primitive_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);

      Dummy : constant Wide_String := Get_Unique_Id;
   begin
      if Params.Context = Start then
         Wrapper.Ada_Dispatching_Param := New_Dynamic_Expression;

         Append
           (Expression.Bloc.Declarations,
            New_Line & Env_Parameter_Name & " : aliased "
            & JNI_Pckg & ".JNI_Env_Access;"
            & New_Line & Dummy
            & " : " & JNI_Pckg
            & ".J_Int := " & JNI_Pckg & ".Attach_Current_Thread (Get_Java_VM ("
            & Wrapper.Ada_Dispatching_Param & "), " & Env_Parameter_Name
            & "'Access, "
            & "System.Null_Address);"
            & New_Line & Object_Parameter_Name
            & " : " & JNI_Pckg & ".J_Object := "
            & Wrapper.Ada_Dispatching_Param
            & "'Unchecked_Access.Get_Java_Ref;");
      elsif Params.Context = Add_Parameter_Original then
         declare
            Params : Parameter_Handlings_Params renames
              Parameter_Handlings_Params (Parameters.all);
         begin
            if Params.Parameter.Is_Controlling
              and then To_Wide_String (Wrapper.Ada_Dispatching_Param) = ""
            then
               if Params.Parameter.Type_Of.Ref.Kind = Access_Kind then
                  Append
                    (Wrapper.Ada_Dispatching_Param,
                     Params.Parameter.Glue_Name & ".all");
               else
                  Append
                    (Wrapper.Ada_Dispatching_Param,
                     Params.Parameter.Glue_Name);
               end if;
            end if;
         end;
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Tagged_Type_Ada_Dispatching_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);
      Dereferenced_Param_Name : Dynamic_Expression;
      Dereferenced_Type       : Simple_Type_View_Access;
   begin
      if Params.Context = Start then
         declare
            New_Code_Node : Dynamic_Expression := New_Dynamic_Expression;
         begin
            New_Code_Node := Copy (Expression.Bloc.Code_Node);
            Empty (Expression.Bloc.Code_Node);

            Wrapper.Ada_Dispatching_Param := New_Dynamic_Expression;
            Wrapper.Ada_Dispatching_Type := New_Dynamic_Expression;
            Wrapper.Open_Upcast := New_Dynamic_Expression;
            Wrapper.Close_Upcast := New_Dynamic_Expression;
            Wrapper.Ada_Parent_Type := New_Dynamic_Expression;
            Wrapper.Open_Instanciation := New_Dynamic_Expression;
            Wrapper.Close_Instanciation := New_Dynamic_Expression;

            if Params.Subprogram_View.Is_Abstract then
               if Params.Subprogram_View.Get_Controlling_Param.Type_Of.Ref.Kind
                 = Access_Kind
               then
                  Append
                    (Expression.Bloc.Declarations,
                     Set_Expression
                       (Wrapper.Open_Upcast,
                        Wrapper.Ada_Parent_Type & "'Class (")
                     & Set_Expression
                       (Wrapper.Close_Upcast,
                        To_Dynamic_Expression (".all)'Unchecked_Access")));
               else
                  Append
                    (Expression.Bloc.Declarations,
                     Set_Expression
                       (Wrapper.Open_Upcast,
                        Wrapper.Ada_Parent_Type & "'Class (")
                     & Set_Expression (Wrapper.Close_Upcast, Exp_Close_Paren));
               end if;

               Append
                 (Expression.Bloc.Statements_After,
                  Set_Expression (Wrapper.Open_Upcast, Exp_Empty)
                  & Set_Expression (Wrapper.Close_Upcast, Exp_Empty)
                  & Set_Expression (Wrapper.Open_Instanciation, Exp_Empty)
                  & Set_Expression (Wrapper.Close_Instanciation, Exp_Empty));
            else
               --  If the object is a child of Cross_Language_Class, it means
               --  that we are on an object created from Java, in other words
               --  from a Java dispatching call. If we're there, it means that
               --  there have been no overriding in Java, so we don't dispatch
               --  to Java and call the Ada parent function instead. Otherwise,
               --  we're calling this from an Ada, and we have to do the actual
               --  dispatching in case the Ada object is hidden.

               Append
                 (Expression.Bloc.Code_Node,
                  Gen_Ada_Loc ("Wrap (Tagged_Type_Ada_Dispatching_Wrapper)")
                  & New_Line
                  & "if " & Wrapper.Ada_Parent_Type & "'Class ("
                  & Wrapper.Ada_Dispatching_Param & ") in "
                  & AJIS_Pckg & ".Cross_Language_Class'Class then"
                  & Indent (1));

               if Params.Subprogram_View.Get_Controlling_Param.Type_Of.Ref.Kind
                 = Access_Kind
               then
                  Append
                    (Expression.Bloc.Code_Node,
                     Set_Expression
                       (Wrapper.Open_Upcast, Wrapper.Ada_Parent_Type & " (")
                     & Set_Expression
                       (Wrapper.Close_Upcast,
                        To_Dynamic_Expression (".all)'Unchecked_Access")));
                  Dereferenced_Param_Name :=
                    Params.Subprogram_View.Get_Controlling_Param.Name & ".all";
                  Dereferenced_Type :=
                    Params.Subprogram_View.Get_Controlling_Param.Type_Of.Ref.
                      Target_Type.Ref;
               else
                  Append
                    (Expression.Bloc.Code_Node,
                     Set_Expression
                       (Wrapper.Open_Upcast, Wrapper.Ada_Parent_Type & " (")
                     & Set_Expression
                       (Wrapper.Close_Upcast, Exp_Close_Paren));

                  Dereferenced_Param_Name :=
                    Params.Subprogram_View.Get_Controlling_Param.Name;
                  Dereferenced_Type :=
                    Params.Subprogram_View.Get_Controlling_Param.Type_Of.Ref;
               end if;

               --  If the return type is controlling, then we need to create
               --  an instance of the appropriate child type even when using
               --  static dispatching.

               if Params.Subprogram_View.Returned_Type /= null
                 and then Params.Subprogram_View.Returned_Type.Is_Controlling
                 and then Params.Subprogram_View.Returned_Type.Type_Of.Ref.Kind
                   /= Access_Kind
               then
                  Append
                    (Expression.Bloc.Code_Node,
                     Set_Expression
                       (Wrapper.Open_Instanciation,
                        "Create_Instance ("
                        & Bound_Tagged_Type
                          (Get_Bound_Element (Params.Handle,
                           Simple_Element_View_Access
                             (Dereferenced_Type)).all)
                        .Child_Wrapper_Type
                        & "'Class ("
                        & Dereferenced_Param_Name
                        & "), "
                        & Set_Expression
                          (Wrapper.Close_Instanciation, Exp_Close_Paren)));
               end if;

               Append
                 (Expression.Bloc.Code_Node,
                  New_Code_Node
                  & New_Line (-1) & "else"
                  & Indent (1));

               Append
                 (Expression.Bloc.Code_Node,
                  Set_Expression (Wrapper.Open_Instanciation, Exp_Empty)
                  & Set_Expression (Wrapper.Close_Instanciation, Exp_Empty));

               if Params.Subprogram_View.Get_Controlling_Param.Type_Of.Ref.Kind
                 = Access_Kind
               then
                  Append
                    (Expression.Bloc.Code_Node,
                     Set_Expression
                       (Wrapper.Open_Upcast, Wrapper.Ada_Parent_Type
                        & "'Class (")
                     & Set_Expression
                       (Wrapper.Close_Upcast,
                        To_Dynamic_Expression (".all)'Unchecked_Access")));
               else
                  Append
                    (Expression.Bloc.Code_Node,
                     Set_Expression
                       (Wrapper.Open_Upcast, Wrapper.Ada_Parent_Type
                        & "'Class (")
                     & Set_Expression
                       (Wrapper.Close_Upcast, Exp_Close_Paren));
               end if;

               Append
                 (Expression.Bloc.Code_Node,
                  New_Code_Node
                  & New_Line (-1) & "end if;");

               Append
                 (Expression.Bloc.Statements_After,
                  Set_Expression (Wrapper.Open_Upcast, Exp_Empty)
                  & Set_Expression (Wrapper.Close_Upcast, Exp_Empty));

               Expression.Bloc.Code_Node := New_Code_Node;

               Nest_Ada_Statements (Expression.Bloc);
            end if;
         end;
      elsif Params.Context = Add_Parameter_Original then
         declare
            Params : Parameter_Handlings_Params renames
              Parameter_Handlings_Params (Parameters.all);
         begin
            if Params.Parameter.Is_Controlling then
               if To_Wide_String (Wrapper.Ada_Parent_Type) = "" then
                  if Params.Parameter.Type_Of.Ref.Kind = Access_Kind then
                     Append
                       (Wrapper.Ada_Parent_Type,
                        Params.Parameter
                        .Type_Of.Ref.Target_Type.Ref.Full_Ada_Name);
                     Append
                       (Wrapper.Ada_Dispatching_Type,
                        Bound_Tagged_Type
                          (Get_Bound_Element (Params.Handle,
                           Simple_Element_View_Access
                             (Params.Parameter
                              .Type_Of.Ref.Target_Type.Ref)).all)
                        .Child_Wrapper_Type);
                     Append
                       (Wrapper.Ada_Dispatching_Param,
                        Params.Parameter.Glue_Name & ".all");
                  else
                     Append
                       (Wrapper.Ada_Parent_Type,
                        Params.Parameter.Type_Of.Ref.Full_Ada_Name);
                     Append
                       (Wrapper.Ada_Dispatching_Type,
                        Bound_Tagged_Type (Get_Bound_Element (Params.Handle,
                          Simple_Element_View_Access
                            (Params.Parameter
                             .Type_Of.Ref)).all)
                        .Child_Wrapper_Type);
                     Append
                       (Wrapper.Ada_Dispatching_Param,
                        Params.Parameter.Glue_Name);
                  end if;
               end if;

               Prepend (Expression.Expression, Wrapper.Open_Upcast);
               Append (Expression.Expression, Wrapper.Close_Upcast);
            end if;
         end;
      elsif Params.Context in Return_Type_Original then
         Prepend (Expression.Expression, Wrapper.Open_Instanciation);
         Append (Expression.Expression, Wrapper.Close_Instanciation);
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Tagged_Type_Java_Primitive_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : Parameter_Handlings_Params renames
        Parameter_Handlings_Params (Parameters.all);
   begin
      if Params.Is_First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & "." & Params.Subprogram_View.Name
            & "(");
      else
         if Params.Param_Number > 2 then
            Append
              (Expression.Enclosing_Expression.Expression, ", ");
         end if;

         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression);
      end if;

      if Params.Is_Last then
         Append (Expression.Enclosing_Expression.Expression, ")");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out JNI_Set_Global_Ref_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Append
        (Expression.Bloc.Code_Node,
         Gen_Ada_Loc ("Wrap (JNI_Set_Global_Ref_Wrapper)")
         & New_Line
         & "if This.all in " & AJIS_Pckg & ".Cross_Language_Class'Class then"
         & New_Line (1) & AJIS_Pckg & ".Set_Global_Ref ("
         & AJIS_Pckg & ".Cross_Language_Class'Class "
         & "(This.all), isGlobal);"
         & New_Line (-1) & "end if;");
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Tagged_Discriminants_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Parameter_Handlings_Params renames
        Parameter_Handlings_Params (Parameters.all);

      Original_Expression : constant Dynamic_Expression :=
        Expression.Expression;
   begin
      if Params.Target = High_Sb then
         if Params.Is_First then
            --  This aggregate initialization is used to shortcut the call of
            --  Initialize on the type -- Initialize will be called only on the
            --  parent object. This is a known limitation of the generated
            --  binding.

            Append
              (Expression.Enclosing_Expression.Expression,
               "new " & Wrapper.Type_Name.all & "'("
               & Params.Subprogram_View.Returned_Type.Type_Of.Ref
               .Target_Type.Ref.Full_Ada_Name & " with ");
         end if;

         if Expression.Expression /= Empty_Dynamic_Expression then
            Append
              (Expression.Enclosing_Expression.Expression,
               Original_Expression & ", ");

            Wrapper.Params_Nb := Wrapper.Params_Nb + 1;
         end if;

         if Params.Is_Last then
            declare
               VM_Id : constant Wide_String := Get_Unique_Id;
               Dummy_Id : constant Wide_String := Get_Unique_Id;

               New_Id : constant Wide_String := Get_Unique_Id;
               Cross_Id : constant Wide_String := Get_Unique_Id;

               Returned_View : constant Simple_Type_View_Access :=
                 Params.Subprogram_View.Returned_Type.Type_Of.
                   Ref.Target_Type.Ref;

               Conversion : constant Type_Record :=
                 Get_Or_Create_Conversion_Type
                   (Params.Handle, Returned_View);

               Element : Bound_Tagged_Type renames
                 Bound_Tagged_Type
                   (Get_Bound_Element
                        (Params.Handle,
                         Simple_Element_View_Access (Returned_View)).all);
            begin
               Append
                 (Expression.Bloc.Declarations,
                  New_Line & VM_Id & " : aliased "
                  & JNI_Pckg & ".Java_VM_Access;"
                  & New_Line & Dummy_Id & " : "
                  & JNI_Pckg & ".J_Int := " & JNI_Pckg & ".Get_Java_VM ("
                  & Env_Parameter_Name & ", "
                  & VM_Id & "'Access);");

               Append
                 (Expression.Enclosing_Expression.Expression,
                  "Backlink => (Ada.Finalization.Controlled with "
                  & "Enclosing => new "
                  & AJIS_Pckg & ".Cross_Lg_Container'(Ptr => null), "
                  & "Ref => " & JNI_Pckg & ".New_Weak_Global_Ref ("
                  & Env_Parameter_Name & ", "
                  & Object_Parameter_Name & "), VM => " & VM_Id
                  & ", Is_Global_Ref => False))");

               Append
                 (Expression.Bloc.Declarations,
                  New_Line & New_Id & " : "
                  & Conversion.Conversion_Package_Name.all
                  & ".Object_Pointer := "
                  & Expression.Enclosing_Expression.Expression & ";");

               Append
                 (Expression.Bloc.Declarations,
                  New_Line & Cross_Id & " : access "
                  & Element.Child_Wrapper_Type & "'Class"
                  & " := " & Element.Child_Wrapper_Type & "'Class ("
                  & New_Id & ".all)'Unchecked_Access;");

               Append
                 (Expression.Bloc.Statements_After,
                  New_Line & Cross_Id & ".Backlink.Enclosing.Ptr := "
                  & Cross_Id & ";");

               Expression.Enclosing_Expression.Expression :=
                 To_Dynamic_Expression (New_Id);

               Expression.Enclosing_Expression.Kind := Variable_Kind;
            end;
         end if;
      else
         Wrap (Ada_Constructor_Handler (Wrapper), Expression, Parameters);
      end if;
   end Wrap;

   -------------------------------------
   -- Get_Default_Implementation_Unit --
   -------------------------------------

   function Get_Default_Implementation_Unit
     (Element_Bound : access Bound_Tagged_Type) return Bound_Unit
   is
      Id : constant Wide_String := Get_Unique_Id;
   begin
      if Element_Bound.Default_Implementation = null then
         Element_Bound.Default_Implementation :=
           Get_Or_Create_Bound_Unit
             (Element_Bound.Kernel,
              Element_Bound.Enclosing_Unit.Base_Pckg,
              Id);
         Element_Bound.Default_Implementation.Java_File := Create_Java_Class
           (Unit   => Element_Bound.Default_Implementation,
            Handle => Element_Bound.Kernel,
            Pckg   => Element_Bound.Enclosing_Unit.Base_Pckg,
            Name   => Id);
         Append
           (Element_Bound.Default_Implementation.Java_File.Java_Parent_Class,
            To_Wide_String
              (Element_Bound.Enclosing_Unit.Java_File.Full_Class_Name));
      end if;

      return Element_Bound.Default_Implementation;
   end Get_Default_Implementation_Unit;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Tagged_Clone_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Tmp    : constant Wide_String := Get_Unique_Id;
      Result : constant Wide_String := Get_Unique_Id;

      Params : Parameter_Handlings_Params renames
        Parameter_Handlings_Params (Parameters.all);
      Conversion : constant Type_Record := Get_Or_Create_Conversion_Type
        (Params.Handle,
         Params.Parameter.Type_Of.Ref.Target_Type.Ref);
   begin
      Append
        (Expression.Bloc.Declarations,
         New_Line & Tmp & " : "
         & Conversion.Conversion_Package_Name.all & ".Object_Pointer := "
         & Conversion.Conversion_Package_Name.all
         & ".Object_Pointer (" & Expression.Expression & ");");

      Append
        (Expression.Bloc.Declarations,
         New_Line & Result & " : "
         & Conversion.Conversion_Package_Name.all & ".Object_Pointer;");

      Append
        (Expression.Bloc.Statements_Before,
         New_Line & "if " & Tmp & ".all in "
         & AJIS_Pckg & ".Cross_Language_Class'Class then"
         & New_Line (1) & Result & " := "
         & Conversion.Conversion_Package_Name.all & ".Object_Pointer ("
         & AJIS_Pckg & ".Cross_Language_Class'Class ("
         & Tmp & ".all).Copy_No_Clone);"
         & New_Line (-1) & "else" & New_Line (1)
         & Result & " := new "
         & Params.Parameter.Type_Of.Ref.Target_Type.Ref.Full_Ada_Name
         & "'Class'(" & Tmp & ".all);"
         & New_Line (-1) & "end if;");

      Nest_Ada_Statements (Expression.Bloc);

      Append
        (Expression.Enclosing_Expression.Expression, Result);
   end Wrap;

end Ada2Java.Bound_Elements.Tagged_Types;
