------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2013, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;

package body Ada2Java.Bound_Elements.References is

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
     (Handle        : not null access Kernel.Kernel_Record;
      Element       : Simple_Element_View_Access;
      Element_Bound : access Bound_Reference)
   is
   begin
      Initialize
        (Handle,
         Element,
         Bound_Element_Record (Element_Bound.all)'Access);
      Element_Bound.JNI_Constructor := To_Dynamic_Expression (Get_Unique_Id);
   end Initialize;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Element_Bound : access Bound_Reference;
      Handle        : not null access Kernel.Kernel_Record)
   is
      Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (Handle => Handle,
         View   => Element_Bound.Simple_Element);

      Type_View : constant Simple_Type_View_Access :=
        Simple_Type_Wrapper_Access (Element_Bound.Simple_Element).Wrapped_Type;

      Java_Ref_Class : Dynamic_Expression;
      Ref_Class_Name : Dynamic_Expression;
   begin
      if Type_View.Kind = Access_Kind then
         Ref_Class_Name := Type_View.Target_Type.Ref.Full_Java_Name;
      else
         Ref_Class_Name := Type_View.Full_Java_Name;
      end if;

      if Type_View.Kind = Enumeration_Kind then
         Java_Ref_Class := New_Line & New_Line
           & "public static class Ref "
           & "implements com.adacore.ajis.internal.ada.IEnumRef {"
           & New_Line (1)
           & "public " & Ref_Class_Name & " ref;"
           & New_Line & New_Line
           & "public void setValue (" & Ref_Class_Name & " r) {"
           & New_Line (1) & "ref = r;"
           & New_Line (-1) & "}"
           & New_Line & New_Line
           & "public " & Ref_Class_Name & " getValue () {"
           & New_Line (1)
           & "return ref;"
           & New_Line (-1) & "}"
           & New_Line & New_Line
           & "public void setValueInt (int value) {"
           & New_Line (1) & "ref = "
           & Ref_Class_Name & ".class.getEnumConstants () [value];"
           & New_Line (-1) & "}"
           & New_Line & New_Line
           & "public int getValueInt () {"
           & New_Line (1) & "return ref.ordinal ();"
           & New_Line (-1) & "}"
           & New_Line (-1) & "}";
      else
         Java_Ref_Class := New_Line & New_Line
           & "public static class Ref implements com.adacore.ajis.IProxyRef {"
           & New_Line (1)
           & "public " & Ref_Class_Name & " ref;"
           & New_Line & New_Line
           & "public void setValue (com.adacore.ajis.IProxy r) {"
           & New_Line (1) & "ref = (" & Ref_Class_Name & ") r;"
           & New_Line (-1) & "}"
           & New_Line & New_Line
           & "public " & Ref_Class_Name & " getValue () {"
           & New_Line (1)
           & "return ref;"
           & New_Line (-1) & "}"
           & New_Line (-1) & "}"
           & New_Line & New_Line
           & "public Ref NewProxyRef () {"
           & New_Line (1) & "Ref ref = new Ref ();"
           & New_Line & "ref.setValue (this);"
           & New_Line & "return ref;"
           & New_Line (-1) & "}";
      end if;

      Append (Unit.Java_File.Public_Body_Part, Java_Ref_Class);

      Append
        (Unit.Root_Unit.Ada_Spec_File.Public_Body_Part,
         New_Line & New_Line
         & Element_Bound.JNI_Constructor & " : " & AJIS_Pckg
         & ".Java_Method_Access := " & AJIS_Pckg & ".Get_Java_Method ("""
         & Replace_Dots_By_Slashes
           (To_Wide_String (Ref_Class_Name))
         & "$Ref"
         & """, ""<init>"", ""()V"");");

      Append
        (Unit.Root_Unit.Ada_Body_File.Initialize_Body,
         New_Line & AJIS_Pckg & ".Initialize ("
         & Element_Bound.JNI_Constructor
         & ", " & Env_Parameter_Name & ");");
   end Bind;

   -------------------------
   -- Get_JNI_Constructor --
   -------------------------

   function Get_JNI_Constructor
     (Element : Bound_Reference) return Dynamic_Expression is
   begin
      return Element.Enclosing_Unit.Root_Unit.Ada_Pckg_Name
        & "." & Element.JNI_Constructor;
   end Get_JNI_Constructor;

end Ada2Java.Bound_Elements.References;
