------------------------------------------------------------------------------
--                                   JNI                                    --
--                                                                          --
--                     Copyright (C) 2005-2007, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
------------------------------------------------------------------------------

with Ada.Containers.Indefinite_Hashed_Maps;
with Ada.Strings.Hash;
with Ada.Tags.Generic_Dispatching_Constructor;
with Ada.Unchecked_Deallocation;
with Interfaces.C;         use Interfaces.C;
with Interfaces.C.Strings; use Interfaces.C.Strings;
with Java_Primitives;      use Java_Primitives;
with JNI_JVM;              use JNI_JVM;

package body JNI_Object is

   procedure Set_Object_Class (Env : JNI_Env_Access);
   procedure Set_Class_Class (Env : JNI_Env_Access);

   ----------------
   -- Deallocate --
   ----------------

   procedure Deallocate is new
     Ada.Unchecked_Deallocation (Root_Interface'Class, Root_Interface_Access);

   -----------------
   -- Hashed Maps --
   -----------------

   package String_Tag_Maps is new Ada.Containers.Indefinite_Hashed_Maps
     (String, Tag, Ada.Strings.Hash, "=");
   use String_Tag_Maps;

   Tag_Map : String_Tag_Maps.Map;
   --  Mapping between concrete Java class name and Ada.Tags.Tag.
   --  With this mapping, it is possible to create a corresponding
   --  Ada type from a Java class name, thus at runtime.

   -------------------------
   -- Secure_Get_J_Object --
   -------------------------

   function Secure_Get_J_Object (O : access Root_Interface'Class)
                                 return J_Object
   is
   begin
      if O = null then
         return J_Null_Object;
      else
         return O.Get_J_Object;
      end if;
   end Secure_Get_J_Object;

   ----------
   -- Free --
   ----------

   procedure Free (I : in out Root_Interface_Access)
   is
      J   : J_Int;
      Env : aliased JNI_Env_Access;
   begin
      if I = null then
         return;
      end if;

      --  Get the Environment
      J := Get_Env (VM => Current_VM, Penv => Env'Access);

      if J /= JNI_OK then
         raise Program_Error with
           "JNI_Object.Free: Get_Env has not returned JNI_OK.";
      end if;

      Delete_Global_Ref (Env, I.Get_J_Object);
      --  The instance referenced by I.J_Obj can now be garbage-collected by
      --  the JVM.

      Deallocate (I);
   end Free;

   ------------------
   -- Get_J_Object --
   ------------------

   function Get_J_Object (D : JNI_Data) return J_Object
   is
   begin
      return D.J_Obj;
   end Get_J_Object;

   ------------------
   -- Set_J_Object --
   ------------------

   procedure Set_J_Object (D : in out JNI_Data; O : J_Object)
   is
   begin
      D.J_Obj := O;
   end Set_J_Object;

   --------------
   -- Register --
   --------------

   procedure Register (Class_Name : String; T : Tag)
   is
   begin
      if Tag_Map.Find (Class_Name) = No_Element then
         Tag_Map.Insert (Class_Name, T);
      else
         Tag_Map.Replace (Class_Name, T);
      end if;
   end Register;

   -------------
   -- Factory --
   -------------

   function Factory is new
     Generic_Dispatching_Constructor (Root_Interface, Parameters,
                                      JNI_Constructor);
   --  Allows to create every concrete type in the Root_Interface'Class
   --  hierarchy from the appropriate tag value.

   ----------------------
   -- Set_Object_Class --
   ----------------------

   Object_Class : J_Class := J_Null_Class;

   procedure Set_Object_Class (Env : JNI_Env_Access) is
      C : constant Standard.String := "java/lang/Object";
   begin
      Object_Class := Find_Class (Env, C);
      if Object_Class = J_Null_Class then
         raise Program_Error with "JNI.Object.Set_Object_Class: " & C &
           ".class not found.";
      end if;
   end Set_Object_Class;

   ---------------------
   -- Set_Class_Class --
   ---------------------

   Class_Class : J_Class := J_Null_Class;

   procedure Set_Class_Class (Env : JNI_Env_Access) is
      C : constant Standard.String := "java/lang/Class";
   begin
      Class_Class := Find_Class (Env, C);
      if Class_Class = J_Null_Class then
         raise Program_Error with "JNI.Object.Set_Class_Class: " & C &
           ".class not found.";
      end if;
   end Set_Class_Class;

   -----------------------
   -- Create_Ada_Object --
   -----------------------

   GetClass_ID : J_Method_ID := J_Null_Method_ID;
   GetName_ID : J_Method_ID := J_Null_Method_ID;

   function Create_Ada_Object (J_Obj : J_Object)
                               return access Root_Interface'Class
   is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Result : access Root_Interface'Class;
      Args   : J_Value_Array (1 .. 0);
      T      : Tag;
      C      : Cursor;
      Class  : J_Object;
      Str    : J_String;
      Name   : chars_ptr;
      Class_Descriptor  : constant String := "()Ljava/lang/Class;";
      String_Descriptor : constant String := "()Ljava/lang/String;";
      Global_J_Obj      : J_Object;
   begin
      if J_Obj = J_Null_Object then
         return null;
      end if;

      --  Get the Environment
      I := Get_Env (VM => Current_VM, Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error with
           "JNI_Object.Create_Ada_Object: Get_Env has not returned JNI_OK.";
      end if;

      --  Prevent the object represented by J_Obj from being garbage-collected.
      Global_J_Obj := New_Global_Ref (Env, J_Obj);

      --  Then delete the local reference J_Obj since it is not useful anymore.
      Delete_Local_Ref (Env, J_Obj);

      --  Check that the Object Class is set
      if Object_Class = J_Null_Class then
         Set_Object_Class (Env);
      end if;

      --  Check that the method ID is set
      if GetClass_ID = J_Null_Method_ID then
         GetClass_ID := Get_Method_ID (Env, Object_Class,
                                       "getClass",
                                       Class_Descriptor);

         if GetClass_ID = J_Null_Method_ID then
            raise Program_Error with "JNI_Object.Create_Ada_Object: " &
              "Can't get the GetClass method ID.";
         end if;
      end if;

      Class := Call_Object_Method_A (Env, Global_J_Obj, GetClass_ID, Args);
      --  We now have the class of the object to create

      --  Check that the Class Class is set
      if Class_Class = J_Null_Class then
         Set_Class_Class (Env);
      end if;

      --  Check that the method ID is set
      if GetName_ID = J_Null_Method_ID then
         GetName_ID := Get_Method_ID (Env, Class_Class,
                                      "getName",
                                      String_Descriptor);

         --  TODO: Should throw a Java exception instead
         if GetName_ID = J_Null_Method_ID then
            raise Program_Error with "JNI_Object.Create_Ada_Object: " &
              "Can't get the GetName method ID.";
         end if;
      end if;

      Str := J_String (Call_Object_Method_A (Env, Class, GetName_ID, Args));
      --  We now have the String Java class name

      Delete_Local_Ref (Env, Class);
      --  Class is not useful anymore

      Name := Get_String_UTF_Chars (Env, Str, null);

      Delete_Local_Ref (Env, J_Object (Str));
      --  Str is not useful anymore

      C := Tag_Map.Find (To_Ada (Value (Name)));

      if C /= No_Element then
         --  Get the tag from the Java class name
         T := Element (C);
      else
         raise Program_Error with "JNI_Object.Create_Ada_Object: " &
           "Tag not found Tag_Map.";
      end if;

      declare
         R : Root_Interface'Class := Factory (T, No_Param'Access);
      begin
         Result := new Root_Interface'Class'(R);
         Result.Set_J_Object (Global_J_Obj);
         return Result;
      end;
   end Create_Ada_Object;

end JNI_Object;
