------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Containers.Indefinite_Doubly_Linked_Lists;

with Ada.Wide_Text_IO; use Ada.Wide_Text_IO;

package Ada2Java.Dynamic_Expressions is

   type Dynamic_Expression is private;

   type Dynamic_Expression_Array is array
     (Integer range <>) of Dynamic_Expression;

   type Dynamic_Expression_Array_Access is access all Dynamic_Expression_Array;

   Empty_Dynamic_Expression : constant Dynamic_Expression;

   Dynamic_Expression_Error : exception;

   function New_Dynamic_Expression return Dynamic_Expression;
   --  Creates a new dynamic expression.

   function "&" (Right, Left : Dynamic_Expression) return Dynamic_Expression;

   function "&"
     (Right : Wide_String; Left : Dynamic_Expression)
      return Dynamic_Expression;

   function "&"
     (Right : Dynamic_Expression; Left : Wide_String)
      return Dynamic_Expression;

   procedure Append
     (Exp : in out Dynamic_Expression; Content : Wide_String);

   procedure Append
     (Exp : in out Dynamic_Expression; Content : Dynamic_Expression);

   procedure Prepend
     (Exp : in out Dynamic_Expression; Content : Wide_String);

   procedure Prepend
     (Exp : in out Dynamic_Expression; Content : Dynamic_Expression);

   procedure Set_Indentation_Offset
     (Exp : in out Dynamic_Expression; Offset : Integer);
   --  Changes the indentation offset of the dynamic expression given in
   --  parameter.

   function New_Line (Offset : Integer := 0) return Dynamic_Expression;
   --  Return a dynamic expression representing a new line, indented by Offest
   --  from the previous one.

   function Indent (Offset : Integer) return Dynamic_Expression;
   --  Return a dynamic expression inserting an indentation.

   function Set_Expression
     (Src : Dynamic_Expression; Value : Dynamic_Expression)
      return Dynamic_Expression;
   --  This call returns a dynamic expression which purpose is to perform the
   --  required operation, which is the assignment of a given dynamic
   --  expression. Assignment will be done when the computations functions like
   --  Write and To_Wide_String are called. Be aware that the initial state of
   --  these modified dynamic expressions will not be retreived once the
   --  computation is done.

   function To_Wide_String
     (Exp : Dynamic_Expression; Base_Indentation : access Integer := null)
      return Wide_String;
   --  Return the wide string resulting of the computation of the dynamic
   --  expression.

   function To_Dynamic_Expression
     (Str        : Wide_String;
      Persistent : Boolean := False)
      return Dynamic_Expression;
   --  Creates a dynamic expression based on the string given in parameter
   --  If Cst is true then this expression will have an extra ref and will
   --  never be freed

   procedure Free (Exp : in out Dynamic_Expression);
   --  Free the dynamic expression given in parameter.
   --  ??? Should it just make ref - 1 ?

   procedure Write
     (Exp : Dynamic_Expression; File : Ada.Wide_Text_IO.File_Type);
   --  Write the dynamic expression given in parameter to a file.

   function Copy (Exp : Dynamic_Expression) return Dynamic_Expression;
   --  Copy the contents of Exp into a new dynamic expression. Future addition
   --  and delection on exp will not change the contents of the dynamic
   --  expression.

   function Deep_Copy (Exp : Dynamic_Expression) return Dynamic_Expression;
   --  Perform a deep copy of all the components of the expression and its
   --  subcomponents.

   procedure Empty (Exp : in out Dynamic_Expression);
   --  Empty the contents of the dynamic expression.

   procedure Check_Cycles (Exp : Dynamic_Expression);
   --  This subprogram raises Dynamic_Expression_Error if a cycle is detected
   --  within the dynamic expression

   procedure Debug_Print (Exp : access Dynamic_Expression);
   pragma Export (C, Debug_Print, "exp_debug_print");

   procedure Put_Line (Exp : Dynamic_Expression);

   function Indirect (Str : Dynamic_Expression) return Dynamic_Expression;
   --  This function encapsulate the dynamic expression so that adding
   --  something to the result doesn't change the parameter.

private

   type Dynamic_Expression_Record;

   type Dynamic_Expression is access all Dynamic_Expression_Record;

   type Item_Type is (Expression, Litteral, Set_Exp);

   type Expression_Item (The_Type : Item_Type; Size : Natural) is record
      case The_Type is
         when Expression =>
            Contents_Exp : Dynamic_Expression;
         when Litteral =>
            Contents_Str : Wide_String (1 .. Size);
         when Set_Exp =>
            Src   : Dynamic_Expression;
            Value : Dynamic_Expression;
      end case;
   end record;

   package Item_List is new Ada.Containers.Indefinite_Doubly_Linked_Lists
     (Expression_Item);

   use Item_List;

   type Dynamic_Expression_Record is tagged record
      List               : Item_List.List;
      Ref_Counter        : Natural := 0;
      Indentation_Offset : Integer := 0;
      Cycle_Tag          : Integer := 0;
   end record;

   Empty_Dynamic_Expression : constant Dynamic_Expression := null;

end Ada2Java.Dynamic_Expressions;
