------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;

package Ada2Java.Code_Wrappers is

   type Wrapping_Bloc_Record;

   type Wrapping_Bloc is access all Wrapping_Bloc_Record;

   type Wrapping_Expression_Record;

   type Wrapping_Expression is access all Wrapping_Expression_Record;

   type Wrapping_Bloc_Record is record
      Code_Node         : Dynamic_Expression;
      Declarations      : Dynamic_Expression;
      Statements_Before : Dynamic_Expression;
      Statements_After  : Dynamic_Expression;
   end record;
   --  This record holds the layout of a bloc of code, wrapping an expression
   --  known to be computable at Code_Node location. Here's the general layout,
   --  in the Ada case:
   --
   --    Declarations
   --  begin
   --    Statements_Before
   --    Code_Node
   --    Statements_After
   --
   --  When this bloc holds a Java bloc, the layout looks like:
   --
   --  Declarations
   --  Statements_Before
   --  Code_Node
   --  Statements_After
   --
   --  The wrapper functions are supposed to add data to the Declarations,
   --  Statement_Before and Statements_After lists.
   --
   --  Code_Node : The location where the expression is needed in the code. May
   --    be replaced by a code structure, for example a declare bloc or an if
   --    statement.
   --  Declarations : The current declarative part. Implementers should either
   --    add data to it, or write it to a nested declarative part and provide
   --    a new one.
   --  Statement_Before : Statements to be made before the expression is valid.
   --    When adding information to such a list, most of the time you need to
   --    create a nested declarative bloc afterwards, otherwise the expression
   --    may be used in a declarative part before the call of the statement
   --    before list. This can be done either manually, or by using
   --    Nest_<Lang>_Statements. In order to minimize such constructions,
   --    initializations should be performed as much as possible in the
   --    declarative part. Note that no data should be prepended to this list,
   --    only append data is garanteed to be valid.
   --  Statement_After : Statements to be made after the computation of the
   --    expression, such as data deallocation. Data can be either pepended or
   --    appened.

   type Expression_Kind is (Unknown_Kind, Variable_Kind, Call_Kind);

   type Wrapping_Expression_Record is record
      Expression           : Dynamic_Expression;
      Kind                 : Expression_Kind := Unknown_Kind;
      Bloc                 : Wrapping_Bloc;
      Enclosing_Expression : Wrapping_Expression;
   end record;
   --  This hold a built expression. Wrappers may append data, or use it in
   --  e.g. the initialization of a newly created variable. In such a case, the
   --  expression will be replaced by the newly created object. The expression
   --  is supposed to be valid at Code_Node location of the referenced bloc.
   --  Kind gives the kind of expression that is currently hold by Expression.
   --  This can lead to potential simplifications & optimisations of the
   --  generated code.

   type Wrapper_Parameters is interface;
   --  This type is used in order to pass parameters to the wrapping procedure.

   type Code_Wrapper is interface;
   --  This interface is a generic interface for code generation, giving a way
   --  to interatively generate a complex expression. Examples of
   --  implementations are given in Ada2Java.Code_Wrappers.Tests.

   type Code_Wrapper_Access is access all Code_Wrapper'Class;

   procedure Wrap
     (Wrapper    : in out Code_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class) is abstract;
   --  This subprogram is the general wrapping procedure. It's used by code
   --  generation subprogram, in order to build the various values needed for
   --  a given expression.

   procedure Nest_Ada_Statements
     (Bloc                   : Wrapping_Bloc;
      Group_After_Statements : Boolean := True);
   --  Create a nested declarative part inside the code node. The following:
   --
   --    Declarations (in out)
   --  begin
   --    Code_Before (in out)
   --    Code_Node (in out)
   --    Code_After (in out)
   --
   --  Will be changed into:
   --
   --    Declarations (in)
   --  begin
   --    Code_Before (in)
   --    declare
   --      Declarations (out)
   --    begin
   --      Code_Before (out)
   --      Code_Node (out)
   --      Code_After (in out)
   --    end;
   --
   --  Note that Code_After is moved inside the new declarative bloc, since
   --  it may be written based on data declared in the new declarative bloc.
   --
   --  If Group_After_Statements is False, then the new layout will look like:
   --
   --    Declarations (in)
   --  begin
   --    Code_Before (in)
   --    declare
   --      Declarations (out)
   --    begin
   --      Code_Before (out)
   --      Code_Node (out)
   --      Code_After (out)
   --    end;
   --    Code_After (in)

   procedure Nest_Java_Statements
     (Bloc : Wrapping_Bloc; Group_After_Statements : Boolean := True);
   --  Nesting Java statements. Same as above, except that without the "begin",
   --  "declare", "end" keywords.

   type Indentical_Code_Wrapper is new Code_Wrapper with null record;
   --  This wrapper does not change anything.

   procedure Wrap
     (Wrapper    : in out Indentical_Code_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);
   --  Do nothing.

end Ada2Java.Code_Wrappers;
