------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Characters.Conversions;  use Ada.Characters;
with Ada.Exceptions;              use Ada.Exceptions;

with Ada2Java.Kernel;              use Ada2Java.Kernel;
with Ada2Java.Utils;               use Ada2Java.Utils;

with Ada.Strings;       use Ada.Strings;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;

package body Ada2Java.Bound_Elements.Types is

   type Java_Constructor_Code_Node_Resolver is new Code_Wrapper with
     null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Constructor_Code_Node_Resolver;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Default_Super_Caller is new Code_Wrapper with
     null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Default_Super_Caller;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Equals_Wrapper is new Code_Wrapper with record
      Type_Of : Simple_Type_View_Access;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Equals_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Equals_Wrapper is new Code_Wrapper with record
      Condition : Dynamic_Expression;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Equals_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Finalize_Wrapper is new Code_Wrapper with null record;

   overriding
   procedure Wrap
     (Wrapper     : in out Ada_Finalize_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Clone_Wrapper is new Code_Wrapper with null record;

   overriding
   procedure Wrap
     (Wrapper     : in out Ada_Clone_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Limited_Clone_Wrapper is new Ada_Constructor_Handler
   with null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Limited_Clone_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Component_Wrapper is new Code_Wrapper with record
      Is_From_Static_Field : Boolean := False;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Component_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   procedure Bind_Extra_String_Primitives
     (Handle : not null access Kernel_Record;
      Unit   : Bound_Unit;
      View   : Simple_Type_View_Access);
   --  Binds the extra subprograms that needs to be present for types derived
   --  from String.

   type Ada_To_String_Wrapper is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_To_String_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_String_Constructor_Wrapper is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_String_Constructor_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
     (Handle        : not null access Kernel.Kernel_Record;
      Element       : Simple_Element_View_Access;
      Element_Bound : access Bound_Type)
   is
   begin
      Initialize
        (Handle, Element, Bound_Element_Record (Element_Bound.all)'Access);

      Element_Bound.Constructor_Id :=
        To_Dynamic_Expression (Get_Unique_Id);
   end Initialize;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Element_Bound : access Bound_Type;
      Handle        : not null access Kernel.Kernel_Record)
   is
      Simple_View : constant Simple_Type_View_Access :=
        Simple_Type_View_Access (Element_Bound.Simple_Element);

      Unit : Bound_Unit;
      Conversion : Type_Record;
   begin
      if Simple_View.Kind = Access_Kind
        and then Simple_View.Target_Type.Ref.Kind = Subprogram_Kind
      then
         Unit := Get_Or_Create_Bound_Unit
           (Handle, Element_Bound.Simple_Element);
         Add_Bound_Element
           (Handle,
            Simple_Element_View_Access (Simple_View.Target_Type.Ref));
      elsif Simple_View.Kind = Access_Kind
        and then
          (Simple_View.Target_Type.Ref.Kind = Record_Kind
           or else Simple_View.Target_Type.Ref.Kind = Tagged_Record_Kind
           or else Simple_View.Target_Type.Ref.Kind = Private_Kind
           or else Simple_View.Target_Type.Ref.Kind = Array_Kind)
      then
         null;
      elsif Simple_View.Kind /= Access_Kind
        and then Simple_View.Kind /= Generic_Integer_Kind
        and then Simple_View.Kind /= Generic_Float_Kind
      then
         Unit := Get_Or_Create_Bound_Unit
           (Handle, Element_Bound.Simple_Element);

         if Simple_View.Kind = Record_Kind
           or else Simple_View.Kind = Private_Kind
           or else Simple_View.Kind = Array_Kind
         then
            Conversion := Get_Or_Create_Conversion_Type
              (Handle, Simple_View);

            if Is_Unconstrained_Array (Simple_View) then
               Append
                 (Simple_View.Named_Access,
                  Conversion.Conversion_Package_Name.all & ".Array_Pointer");
               Append
                 (Simple_View.Named_Constant_Access,
                  Conversion.Conversion_Package_Name.all
                  & ".Array_Pointer_Cst");
            else
               Append
                 (Simple_View.Named_Access,
                  Conversion.Conversion_Package_Name.all & ".Object_Pointer");
               Append
                 (Simple_View.Named_Constant_Access,
                  Conversion.Conversion_Package_Name.all
                  & ".Object_Pointer_Cst");
            end if;

            Bind_Equals (Handle, Simple_View, Unit);

            Bind_Clone (Handle, Simple_View, Unit);
         end if;

         Append
           (Unit.Java_File.Java_Parent_Class,
            "com.adacore.ajis.internal.ada.AdaProxy");

         -- Constructor --

         if Simple_View.Allow_Java_Creation then
            declare
               Constructor_View : Simple_Subprogram_View_Access;
               Constructor_Map : Wrapping_Map;
               Indexes_Length : Integer := 0;
               Current_Index_Param : Integer := 1;
               Index_Type : Simple_Type_Reference;
            begin
               Initialize_Constructor_View
                 (Type_View        => Simple_View,
                  Binded_Type_Name => new Wide_String'
                    (To_Wide_String (Simple_View.Full_Ada_Name)),
                  Constructor_View => Constructor_View,
                  Constructor_Map  => Constructor_Map);

               if Simple_View.Kind = Array_Kind then
                  for J in Simple_View.Indexes'Range loop
                     if not Simple_View.Indexes (J).Is_Constant then
                        Indexes_Length := Indexes_Length + 1;
                     end if;
                  end loop;

                  Constructor_View.Parameters := new Parameter_Array
                    (1 .. Indexes_Length * 2);

                  for J in Simple_View.Indexes'Range loop
                     if not Simple_View.Indexes (J).Is_Constant then
                        Index_Type := Simple_View.Indexes (J).Type_Of;

                        Constructor_View.Parameters (Current_Index_Param) :=
                          Create (Constructor_View);
                        Constructor_View.Parameters
                          (Current_Index_Param).Name := To_Dynamic_Expression
                          ("First_"
                           & Conversions.To_Wide_String
                             (Trim (Integer'Image (J), Left)));
                        Constructor_View.Parameters (Current_Index_Param).
                          Mode := In_Mode;

                        Constructor_View.Parameters
                          (Current_Index_Param).Type_Of := Index_Type;

                        Constructor_View.Parameters (Current_Index_Param).
                          Enclosing_Sb := Constructor_View;

                        Constructor_View.Parameters
                          (Current_Index_Param + 1) :=
                         Create (Constructor_View);
                        Constructor_View.Parameters (Current_Index_Param + 1).
                          Name :=
                            To_Dynamic_Expression
                              ("Last_"
                               & Conversions.To_Wide_String
                                 (Trim (Integer'Image (J), Left)));
                        Constructor_View.Parameters (Current_Index_Param + 1).
                          Mode := In_Mode;

                        Constructor_View.Parameters (Current_Index_Param + 1).
                          Type_Of :=
                            Simple_View.Indexes (J).Type_Of;
                        Constructor_View.Parameters (Current_Index_Param + 1).
                          Enclosing_Sb := Constructor_View;

                        Current_Index_Param := Current_Index_Param + 2;
                     end if;
                  end loop;
               else
                  Initialize_Constructor_Parameters
                    (Constructor_View => Constructor_View);
               end if;

               Bind_Subprogram
                 (Handle            => Handle,
                  View              => Constructor_View,
                  Unit              => Unit,
                  Wrappers          => Constructor_Map);
            end;
         end if;

         Append
           (Unit.Java_File.Public_Body_Part,
            New_Line & New_Line
            & "public " & Simple_View.Exported_Name
            & " (com.adacore.ajis.internal.ada.AdaAccess access) {"
            & New_Line (1)
            &  "this.adaAccess = access.fAcc;" & New_Line (-1)
            & "}");

         Bind_Destructor
           (Handle   => Handle,
            Unit     => Unit,
            The_Type => Simple_View);

         if Simple_View.Kind = Array_Kind then
            declare
               Getter_View : Simple_Subprogram_View_Access;
               Getter_Ada_Call_Handler : aliased
                 Ada_Component_Modifiers_Handler;

               Setter_View : Simple_Subprogram_View_Access;
               Setter_Ada_Call_Handler : aliased
                 Ada_Component_Modifiers_Handler;
               Getter_Java_Call_Handler : aliased
                 Java_Component_Wrapper;

               Setter_Map : Wrapping_Map;
               Getter_Map : Wrapping_Map;

               Used_Component_Type : Simple_Type_Reference;
               Primitive_Type      : Simple_Type_Reference;

               Null_Constructor : Simple_Subprogram_View_Access;
               Null_Constructor_Map : Wrapping_Map;
               Null_Constructor_Handler : aliased
                 Null_Value_Handler;
            begin
               Create_Unrestricted_Access
                 (Handle       => Handle,
                  Unit         => Unit,
                  Initial_Type => Simple_View.Target_Type,
                  Is_Constant  => False,
                  New_Type     => Used_Component_Type,
                  Created      =>
                    Getter_Ada_Call_Handler.Is_Unrestricted_Access);

               Getter_Java_Call_Handler.Is_From_Static_Field :=
                 Getter_Ada_Call_Handler.Is_Unrestricted_Access;

               Setter_Ada_Call_Handler.Is_Unrestricted_Access :=
                 Getter_Ada_Call_Handler.Is_Unrestricted_Access;

               Primitive_Type := Create_Access_To
                 ((Simple_View, others => <>), False);

               Getter_View               := Create (Simple_View);
               Getter_View.Returned_Type := Create (Getter_View);
               Getter_View.Returned_Type.Index   :=
                 Element_Bound.Simple_Element.Index;
               Getter_View.Returned_Type.Type_Of := Used_Component_Type;
               Getter_View.Parameters := new Parameter_Array
                 (1 .. Simple_View.Indexes'Last + 1);
               Getter_View.Is_Final       := True;
               Getter_View.Name           := To_Dynamic_Expression
                 ("Get_Element_At");
               Getter_View.Location := Simple_View.Location;

               Setter_View := Create (Simple_View);
               Setter_View.Parameters := new Parameter_Array
                 (1 .. Simple_View.Indexes'Last + 2);
               Setter_View.Is_Final   := True;
               Setter_View.Name       :=
                 To_Dynamic_Expression ("Set_Element_At");
               Setter_View.Location := Simple_View.Location;

               Getter_Ada_Call_Handler.Mode := Getter;

               Setter_Ada_Call_Handler.Mode := Setter;

               Getter_View.Parameters (1) := Create (Getter_View);
               Getter_View.Parameters (1).Type_Of := Primitive_Type;
               Getter_View.Parameters (1).Name :=
                 To_Dynamic_Expression ("This");
               Getter_View.Parameters (1).Mode := In_Mode;
               Getter_View.Parameters (1).Attached := True;
               Getter_View.Parameters (1).Assume_Stored := False;

               Setter_View.Parameters (1) := Create (Setter_View);
               Setter_View.Parameters (1).Type_Of := Primitive_Type;
               Setter_View.Parameters (1).Name :=
                 To_Dynamic_Expression ("This");
               Setter_View.Parameters (1).Mode := In_Mode;
               Setter_View.Parameters (1).Attached := True;
               Setter_View.Parameters (1).Assume_Stored := False;

               for J in Simple_View.Indexes'Range loop
                  Getter_View.Parameters (J + 1) := Create (Getter_View);
                  Getter_View.Parameters (J + 1).Type_Of :=
                    Simple_View.Indexes (J).Type_Of;
                  Getter_View.Parameters (J + 1).Name := To_Dynamic_Expression
                    ("Index_"
                     & Conversions.To_Wide_String (Trim (J'Img, Left)));
                  Getter_View.Parameters (J + 1).Mode := In_Mode;
                  Getter_View.Parameters (J + 1).Enclosing_Sb := Getter_View;

                  Setter_View.Parameters (J + 1) := Create (Setter_View);
                  Setter_View.Parameters (J + 1).Type_Of :=
                    Simple_View.Indexes (J).Type_Of;
                  Setter_View.Parameters (J + 1).Name := To_Dynamic_Expression
                    ("Index_"
                     & Conversions.To_Wide_String (Trim (J'Img, Left)));
                  Setter_View.Parameters (J + 1).Mode := In_Mode;
                  Setter_View.Parameters (J + 1).Enclosing_Sb := Setter_View;
               end loop;

               Setter_View.Parameters (Simple_View.Indexes'Last + 2) :=
                 Create (Setter_View);
               Setter_View.Parameters (Simple_View.Indexes'Last + 2).Type_Of :=
                 Simple_View.Target_Type;
               Setter_View.Parameters (Simple_View.Indexes'Last + 2).Name :=
                 To_Dynamic_Expression ("Value");
               Setter_View.Parameters (Simple_View.Indexes'Last + 2).Mode :=
                 In_Mode;
               Setter_View.Parameters (Simple_View.Indexes'Last + 2).
                 Enclosing_Sb := Setter_View;

               Add_Wrapper
                 (Map     => Setter_Map,
                  Wrapper => Setter_Ada_Call_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Parameter_Resolution);

               Add_Wrapper
                 (Map     => Setter_Map,
                  Wrapper => Setter_Ada_Call_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Code_Node_Resolution);

               Add_Wrapper
                 (Map     => Getter_Map,
                  Wrapper => Getter_Ada_Call_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Parameter_Resolution);

               Add_Wrapper
                 (Map     => Getter_Map,
                  Wrapper => Getter_Java_Call_Handler'Unchecked_Access,
                  Lang    => Java_Lang,
                  Target  => High_Sb,
                  Context => Return_Type_Converted);

               Bind_Subprogram
                 (Handle            => Handle,
                  View              => Getter_View,
                  Unit              => Unit,
                  Wrappers          => Getter_Map);

               Bind_Subprogram
                 (Handle            => Handle,
                  View              => Setter_View,
                  Unit              => Unit,
                  Wrappers          => Setter_Map);

               for J in Simple_View.Indexes'Range loop
                  for K in Boundary_Type'Range loop
                     declare
                        Bound_Map              : Wrapping_Map;
                        Bound_Ada_Call_Handler : aliased Ada_Bounds_Handler;
                        Bound_View    : Simple_Subprogram_View_Access;
                        Returned_Type : Simple_Type_Reference;
                        Name          : Dynamic_Expression;
                     begin
                        if K = Length then
                           Returned_Type :=
                             (Ref    => Create (Simple_View),
                              others => <>);
                           Returned_Type.Ref.Kind := Generic_Integer_Kind;
                           Returned_Type.Ref.Full_Ada_Name :=
                             To_Dynamic_Expression ("Standard.Integer");
                        else
                           Returned_Type := Simple_View.Indexes (J).Type_Of;
                        end if;

                        case K is
                           when Length =>
                              Name := To_Dynamic_Expression ("Length");
                           when First =>
                              Name := To_Dynamic_Expression ("First");
                           when Last =>
                              Name := To_Dynamic_Expression ("Last");
                        end case;

                        Bound_View  := Create (Simple_View);
                        Bound_View.Returned_Type := Create (Bound_View);
                        Bound_View.Returned_Type.Index :=
                          Element_Bound.Simple_Element.Index;
                        Bound_View.Returned_Type.Type_Of := Returned_Type;
                        Bound_View.Parameters := new Parameter_Array (1 .. 1);
                        Bound_View.Is_Final := True;
                        Bound_View.Location := Simple_View.Location;

                        Bound_View.Parameters (1) := Create (Bound_View);
                        Bound_View.Parameters (1).Type_Of.Ref := Simple_View;
                        Bound_View.Parameters (1).Name :=
                          To_Dynamic_Expression ("This");
                        Bound_View.Parameters (1).Mode := In_Mode;
                        Bound_View.Parameters (1).Enclosing_Sb := Bound_View;
                        Bound_View.Parameters (1).Attached := True;
                        Bound_View.Parameters (1).Assume_Stored := False;

                        if Simple_View.Indexes'Length > 1 then
                           Bound_View.Name := Name & "_"
                             & Conversions.To_Wide_String
                             (Trim (J'Img, Left));
                           Bound_Ada_Call_Handler.Bound_Number := J;
                        else
                           Bound_View.Name := Name;
                           Bound_Ada_Call_Handler.Bound_Number := 0;
                        end if;

                        Bound_Ada_Call_Handler.Bound := K;
                        Bound_Ada_Call_Handler.Type_Name :=
                          Simple_View.Indexes (J).Name;

                        Add_Wrapper
                          (Map     => Bound_Map,
                           Wrapper =>
                             Bound_Ada_Call_Handler'Unrestricted_Access,
                           Lang    => Ada_Lang,
                           Target  => High_Sb,
                           Context => Parameter_Resolution);

                        Bind_Subprogram
                          (Handle   => Handle,
                           View     => Bound_View,
                           Unit     => Unit,
                           Wrappers => Bound_Map);
                     end;
                  end loop;
               end loop;

               Null_Constructor := Create (Simple_View);
               Null_Constructor.Returned_Type := Create (Null_Constructor);
               Null_Constructor.Returned_Type.Type_Of := Primitive_Type;
               Null_Constructor.Returned_Type.Type_Of.Pass_By_Address := True;
               Null_Constructor.Name := To_Dynamic_Expression
                 ("createNullAccess");

               Add_Wrapper
                 (Map     => Null_Constructor_Map,
                  Wrapper =>
                    Null_Constructor_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Return_Type_Converted);

               Bind_Subprogram
                 (Handle   => Handle,
                  View     => Null_Constructor,
                  Unit     => Unit,
                  Wrappers => Null_Constructor_Map);

               Append
                 (Unit.Java_File.Public_Body_Part,
                  New_Line & New_Line
                  & "private static int [] nullArrayAccessVal;"
                  & New_Line & New_Line
                  & "public static int [] nullArrayAccess () {"
                  & New_Line (1) & "if (nullArrayAccessVal == null) {"
                  & New_Line (1) & "nullArrayAccessVal = createNullAccess ();"
                  & New_Line (-1) & "}"
                  & New_Line & "return nullArrayAccessVal;"
                  & New_Line (-1) & "}");
            end;

            if Simple_View.Target_Type.Ref.Kind = Standard_Character_Kind
              and then Simple_View.Indexes'Length = 1
              and then Simple_View.Indexes (1).Type_Of.Ref.Kind
              = Generic_Integer_Kind
            then
               Bind_Extra_String_Primitives (Handle, Unit, Simple_View);
            end if;

         elsif Simple_View.Kind = Record_Kind then
            Bind_Fields
              (Handle    => Handle,
               Type_View => Simple_View,
               Unit      => Unit);
         end if;

         if Simple_View.Kind = Record_Kind
           or else Simple_View.Kind = Private_Kind
           or else Simple_View.Kind = Array_Kind
         then
            Append
              (Unit.Ada_Spec_File.Public_Body_Part,
               New_Line & New_Line & Element_Bound.Constructor_Id
               & " : " & AJIS_Pckg & ".Java_Method_Access := "
               & AJIS_Pckg & ".Get_Java_Method (" & """"
               & Replace_Dots_By_Slashes
                 (To_Wide_String (Unit.Java_File.Full_Class_Name))
               & """, ""<init>"", "
               & """(Lcom/adacore/ajis/internal/ada/AdaAccess;)V"");");

            Append
              (Unit.Ada_Body_File.Initialize_Body,
               New_Line & AJIS_Pckg & ".Initialize ("
               & Element_Bound.Constructor_Id
               & ", " & Env_Parameter_Name & ");");
         end if;
      else
         Trace_With_Location ("type not supported", Errors_And_Warnings);
      end if;
   exception
      when Silent_Not_Supported =>
         null;

      when E : Not_Supported =>
         Trace_With_Location
           (Conversions.To_Wide_String (Exception_Message (E)),
            Errors_And_Warnings);

      when E : others =>
         Trace_With_Location
           ("unexpected exception: "
            & Conversions.To_Wide_String (Exception_Information (E)),
            Errors_Only);
   end Bind;

   --------------
   -- Get_View --
   --------------

   function Get_View (This : Bound_Type) return Simple_Type_View_Access is
   begin
      return This.Type_View;
   end Get_View;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Handler    : in out Ada_Field_Modifiers_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);
   begin
      if Params.Context = Parameter_Resolution then
         if Handler.Mode = Setter then
            if Handler.This_Exp = Empty_Dynamic_Expression then
               Handler.This_Exp := Expression.Expression;
            else
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Expression.Expression);
            end if;
         else
            if Handler.Is_Unrestricted_Access then
               if Is_Unconstrained_Array (Handler.Object_View.Type_Of.Ref) then
                  if Handler.Object_View.Is_Aliased
                    or else Ada2Java.Allow_Unaliased_Access
                  then
                     if Handler.Object_View.Type_Of.Ref.Indexes'Last = 1 then
                        Append
                          (Expression.Enclosing_Expression.Expression,
                           Gen_Ada_Loc ("Wrap (Ada_Field_Modifiers_Handler)")
                           & New_Line
                           & "(" & AJIS_Pckg & ".Static, "
                           & Expression.Expression & "." & Handler.Var_Name
                           & "'First, "
                           & Expression.Expression & "." & Handler.Var_Name
                           & "'Last, "
                           & Expression.Expression & "." & Handler.Var_Name
                           & "'Address)");
                     else
                        Append
                          (Expression.Enclosing_Expression.Expression,
                           Gen_Ada_Loc ("Wrap (Ada_Field_Modifiers_Handler)")
                           & New_Line
                           & "(" & AJIS_Pckg & ".Static, ");

                        for J in
                          Handler.Object_View.Type_Of.Ref.Indexes'Range
                        loop
                           Append
                             (Expression.Enclosing_Expression.Expression,
                              Expression.Expression & "." & Handler.Var_Name
                              & "'First (" & Conversions.To_Wide_String
                                (Trim (J'Img, Both))
                              & "), "
                              & Expression.Expression & "." & Handler.Var_Name
                              & "'Last (" & Conversions.To_Wide_String
                                (Trim (J'Img, Both))
                              & "), ");
                        end loop;

                        Append
                          (Expression.Enclosing_Expression.Expression,
                           Expression.Expression & "." & Handler.Var_Name
                           & "'Address)");
                     end if;
                  else
                     --  This should have been detected earlier

                     raise Not_Supported;
                  end if;
               elsif Handler.Object_View.Is_Aliased then
                  Append
                    (Expression.Enclosing_Expression.Expression,
                     Expression.Expression & "."
                     & Handler.Var_Name & "'Unchecked_Access");
               elsif Ada2Java.Allow_Unaliased_Access then
                  Append
                    (Expression.Enclosing_Expression.Expression,
                     Expression.Expression & "." & Handler.Var_Name
                     & "'Unrestricted_Access");
               else
                  --  This should have been detected earlier

                  raise Not_Supported;
               end if;
            else
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Expression.Expression & "." & Handler.Var_Name);
            end if;
         end if;
      elsif Params.Context = Code_Node_Resolution then
         Append
           (Expression.Bloc.Code_Node,
            New_Line & Handler.This_Exp & "." & Handler.Var_Name
            & " := " & Expression.Expression & ";");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Handler    : in out Ada_Component_Modifiers_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);

      Original_Expression : constant Dynamic_Expression :=
        Expression.Expression;

      Type_Of : Simple_Type_Reference;
   begin
      if Params.Context = Parameter_Resolution then
         declare
            Params : Parameter_Handlings_Params renames
              Parameter_Handlings_Params (Parameters.all);
         begin
            if Params.Is_First then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Expression.Expression);
            else
               if Handler.Params_Nb = 1 then
                  Append
                    (Expression.Enclosing_Expression.Expression,
                     " (");
               elsif not Params.Is_Last or else Handler.Mode = Getter then
                  Append (Expression.Enclosing_Expression.Expression, ", ");
               end if;

               if Params.Is_Last then
                  --  Append (Expression.Enclosing_Expression.Expression, ")");

                  if Handler.Mode = Getter then
                     Handler.This_Exp :=
                       Copy (Expression.Enclosing_Expression.Expression)
                       & Original_Expression & ")";
                     Empty (Expression.Enclosing_Expression.Expression);
                  else
                     Handler.This_Exp :=
                       Copy (Expression.Enclosing_Expression.Expression)
                       & ")";
                     Empty (Expression.Enclosing_Expression.Expression);
                     Append (Expression.Enclosing_Expression.Expression,
                             Original_Expression);
                  end if;

                  if Handler.Mode = Getter then
                     if Handler.Is_Unrestricted_Access then
                        Type_Of := Params.Subprogram_View.
                          Returned_Type.Type_Of.Ref.Target_Type;

                        if Is_Unconstrained_Array (Type_Of.Ref) then
                           if Ada2Java.Allow_Unaliased_Access then
                              Append
                                (Expression.Enclosing_Expression.Expression,
                                 "(" & AJIS_Pckg & ".Static, "
                                 & Get_Array_Bounds
                                   (Type_Of.Ref, Handler.This_Exp)
                                 & Handler.This_Exp & "'Address)");
                           else
                              --  This should have been detected earlier

                              raise Not_Supported;
                           end if;
                        else
                           Append
                             (Expression.Enclosing_Expression.Expression,
                              Handler.This_Exp & "'Unrestricted_Access");
                        end if;
                     else
                        Append
                          (Expression.Enclosing_Expression.Expression,
                           Handler.This_Exp);
                     end if;
                  end if;
               else
                  Append
                    (Expression.Enclosing_Expression.Expression,
                     Original_Expression);
               end if;
            end if;

            Handler.Params_Nb := Handler.Params_Nb + 1;
         end;
      elsif Params.Context = Code_Node_Resolution then
         Append
           (Expression.Bloc.Code_Node,
            New_Line & Handler.This_Exp
            & " := " & Expression.Expression & ";");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   overriding
   procedure Wrap
     (Handler           : in out Null_Value_Handler;
      Expression        : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Handler);
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);
   begin
      Expression.Expression :=
        Params.Subprogram_View.Returned_Type
          .Type_Of.Ref.Target_Type.Ref.Named_Access
        & "'(null)";
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Handler    : in out Ada_Bounds_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Parameters);
   begin
      if Handler.Bound = First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & "'First");
      elsif Handler.Bound = Last then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & "'Last");
      else
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & "'Length");
      end if;

      if Handler.Bound_Number /= 0 then
         Append
           (Expression.Enclosing_Expression.Expression,
            " ("
            & Conversions.To_Wide_String
              (Trim (Handler.Bound_Number'Img, Left))
            & ")");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Handler    : in out Ada_Constructor_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Parameter_Handlings_Params renames
        Parameter_Handlings_Params (Parameters.all);

      Original_Expression : constant Dynamic_Expression :=
        Expression.Expression;
   begin
      if Params.Target = High_Sb then
         if Expression.Expression /= Empty_Dynamic_Expression then
            if Params.Is_First then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Params.Subprogram_View.Returned_Type.Type_Of
                  .Initial_Subtype_Name & "'("
                  & " new " & Handler.Type_Name.all & " (");
            else
               if Params.Subprogram_View.Get_Attached_Type.Kind
                 = Array_Kind
               then
                  if Handler.Params_Nb mod 2 /= 0 then
                     Append
                       (Expression.Enclosing_Expression.Expression, " .. ");
                  else
                     Append
                       (Expression.Enclosing_Expression.Expression, ", ");
                  end if;
               else
                  Append
                    (Expression.Enclosing_Expression.Expression, ", ");
               end if;
            end if;

            Append
              (Expression.Enclosing_Expression.Expression,
               Original_Expression);

            if Params.Is_Last then
               Append
                 (Expression.Enclosing_Expression.Expression, "))");
            end if;

            Handler.Params_Nb := Handler.Params_Nb + 1;
         else
            Append
              (Expression.Enclosing_Expression.Expression,
               Params.Subprogram_View.Returned_Type.Type_Of
               .Initial_Subtype_Name & "'("
               & "new " & Handler.Type_Name.all & ")");
         end if;
      else
         if Params.Is_First then
            Append
              (Expression.Enclosing_Expression.Expression,
               Params.Ada_High_Sb_Name & "(" & Env_Parameter_Name & ", "
               & Object_Parameter_Name);
         end if;

         if Expression.Expression /= Empty_Dynamic_Expression then
            Append
              (Expression.Enclosing_Expression.Expression,
               ", " & Expression.Expression);
         end if;

         if Params.Is_Last then
            Append
              (Expression.Enclosing_Expression.Expression, ")");
         end if;
      end if;

   end Wrap;

   ---------------------------------
   -- Initialize_Constructor_View --
   ---------------------------------

   procedure Initialize_Constructor_View
     (Type_View            : Simple_Type_View_Access;
      Binded_Type_Name     : Wide_String_Access;
      Constructor_View     : out Simple_Subprogram_View_Access;
      Constructor_Map      : in out Wrapping_Map;
      Ada_Parameter_Hander : Ada_Constructor_Handler_Access := null)
   is
      Constructor_Ada_Call_Handler_Access : Code_Wrapper_Access;
   begin
      if Ada_Parameter_Hander = null then
         Constructor_Ada_Call_Handler_Access := new Ada_Constructor_Handler;
      else
         Constructor_Ada_Call_Handler_Access :=
           Code_Wrapper_Access (Ada_Parameter_Hander);
      end if;

      Constructor_View := Create (Type_View);
      Constructor_View.Returned_Type := Create (Constructor_View);
      Constructor_View.Returned_Type.Type_Of := Create_Access_To
        ((Type_View,
         Is_Class_Wide => Type_View.Kind = Tagged_Record_Kind,
         others => <>),
         False);

      Constructor_View.Returned_Type.Index := Type_View.Index;

      Constructor_View.Is_Final         := False;
      Constructor_View.Is_Constructor   := True;
      Constructor_View.Name             := Type_View.Exported_Name;
      Constructor_View.High_Sb_With_Env := True;
      Constructor_View.Force_Attach     := Type_View;
      Constructor_View.Location         := Type_View.Location;

      --  At this point, we want to send the address of the object and not the
      --  reference to the stored object in the case of a tagged type.

      if Constructor_View.Returned_Type.Type_Of.Ref.Target_Type.Ref.Kind
        = Tagged_Record_Kind
      then
         Constructor_View.Returned_Type.Type_Of.Pass_By_Address := True;
      end if;

      Ada_Constructor_Handler
        (Constructor_Ada_Call_Handler_Access.all).Type_Index :=
        Type_View.Index;

      Ada_Constructor_Handler
        (Constructor_Ada_Call_Handler_Access.all).Type_Name :=
        Binded_Type_Name;

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Code_Wrapper_Access'
           (new Java_Default_Super_Caller),
         Lang    => Java_Lang,
         Target  => High_Sb,
         Context => Start);

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Constructor_Ada_Call_Handler_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Parameter_Resolution);

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Constructor_Ada_Call_Handler_Access,
         Lang    => Ada_Lang,
         Target  => JNI_Sb,
         Context => Parameter_Resolution);

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Code_Wrapper_Access'
           (new Java_Constructor_Code_Node_Resolver),
         Lang    => Java_Lang,
         Target  => High_Sb,
         Context => Code_Node_Resolution);
   end Initialize_Constructor_View;

   ---------------------------------------
   -- Initialize_Constructor_Parameters --
   ---------------------------------------

   procedure Initialize_Constructor_Parameters
     (Constructor_View : Simple_Subprogram_View_Access) is
   begin
      if Constructor_View.Returned_Type.Type_Of.Ref.Target_Type.Ref.Indexes
        /= null
      then
         Constructor_View.Parameters := new Parameter_Array
           (1 .. Constructor_View.Returned_Type.Type_Of.Ref.Target_Type.Ref
            .Indexes'Length);

         for J in Constructor_View.Returned_Type.Type_Of.Ref.Target_Type
           .Ref.Indexes'Range
         loop
            Constructor_View.Parameters (J) := Create (Constructor_View);

            Constructor_View.Parameters (J).Name :=
              Constructor_View.Returned_Type.Type_Of.Ref.Target_Type.Ref
                .Indexes (J).Name;
            Constructor_View.Parameters (J).Mode := In_Mode;

            Constructor_View.Parameters (J).Type_Of :=
              Constructor_View.Returned_Type.Type_Of.Ref.Target_Type.Ref
                .Indexes (J).
                Type_Of;
            Constructor_View.Parameters (J).Enclosing_Sb := Constructor_View;
         end loop;
      end if;
   end Initialize_Constructor_Parameters;

   ---------------------
   -- Bind_Destructor --
   ---------------------

   procedure Bind_Destructor
     (Handle   : not null access Kernel.Kernel_Record;
      Unit     : Bound_Unit;
      The_Type : Simple_Type_View_Access)
   is
      Destructor_View        : constant Simple_Subprogram_View_Access :=
       Create (The_Type);
      Destructor_Wrappers    : Wrapping_Map;
      Destructor_Ada_Wrapper : Ada_Finalize_Wrapper;
      Primitive_Param : Parameter_Simple_View_Access;
   begin
      Primitive_Param := Create (Destructor_View);

      Primitive_Param.Index          := The_Type.Index;
      Primitive_Param.Location       := The_Type.Location;
      Primitive_Param.Is_Primitive   := True;
      Primitive_Param.Is_Controlling := True;
      Primitive_Param.Type_Of        :=
        Create_Access_To ((The_Type, others => <>), False);
      Primitive_Param.Name           := To_Dynamic_Expression ("Obj");
      Primitive_Param.Attached       := True;
      Primitive_Param.Assume_Stored  := False;

      Destructor_View.Returned_Type     := null;
      Destructor_View.Parameters        := new Parameter_Array'
        (1 => Primitive_Param);
      Destructor_View.Is_Final          := False;
      Destructor_View.Name              := To_Dynamic_Expression
        ("deallocateNativeObject");
      Destructor_View.Locking_State     :=
        Ada2Java.Default_Finalize_Locking_State;
      Destructor_View.Is_Real_Primitive := False;
      Destructor_View.Location          := The_Type.Location;
      Destructor_View.Index             := The_Type.Index;

      Add_Wrapper
        (Map     => Destructor_Wrappers,
         Wrapper => Destructor_Ada_Wrapper'Unrestricted_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Parameter_Resolution);

      Bind_Subprogram
        (Handle   => Handle,
         View     => Destructor_View,
         Unit     => Unit,
         Wrappers => Destructor_Wrappers);
   end Bind_Destructor;

   -----------------
   -- Bind_Fields --
   -----------------

   procedure Bind_Fields
     (Handle    : not null access Kernel.Kernel_Record;
      Type_View : Simple_Type_View_Access;
      Unit      : Bound_Unit)
   is
      procedure Bind_Field
        (Field : Simple_Object_View_Access);
      --  Generate a binding for a record field or discriminant.

      procedure Bind_Field
        (Field : Simple_Object_View_Access)
      is
         Getter_View : Simple_Subprogram_View_Access;
         Setter_View : Simple_Subprogram_View_Access;

         Getter_Ada_Call_Handler : aliased Ada_Field_Modifiers_Handler;
         Setter_Ada_Call_Handler : aliased Ada_Field_Modifiers_Handler;

         Getter_Java_Call_Handler : aliased Java_Component_Wrapper;

         Field_Type      : Simple_Type_Reference;

         Used_Field_Type : Simple_Type_Reference;
         --  We generate a false access type if needed. This field type is
         --  supposed to be always an access type for objects passed by
         --  reference.

         Primitive_Type : Simple_Type_Reference;

      begin
         Field_Type := Field.Type_Of;
         Create_Unrestricted_Access
           (Handle       => Handle,
            Unit         => Unit,
            Initial_Type => Field_Type,
            Is_Constant  => False,
            New_Type     => Used_Field_Type,
            Created      =>
              Getter_Ada_Call_Handler.Is_Unrestricted_Access);

         Getter_Java_Call_Handler.Is_From_Static_Field :=
           Getter_Ada_Call_Handler.Is_Unrestricted_Access;

         Setter_Ada_Call_Handler.Is_Unrestricted_Access :=
           Getter_Ada_Call_Handler.Is_Unrestricted_Access;

         Primitive_Type := Create_Access_To ((Type_View, others => <>), False);

         Getter_Ada_Call_Handler.Object_View := Field;
         Setter_Ada_Call_Handler.Object_View := Field;

         declare
            Getter_Map : Wrapping_Map;
            Setter_Map : Wrapping_Map;
         begin
            Getter_View := Create (Type_View);

            Getter_View.Is_Dispatching := False;
            Getter_View.Name := Field.Name;
            Getter_View.Returned_Type := Create (Getter_View);
            Getter_View.Returned_Type.Index   := Type_View.Index;
            Getter_View.Returned_Type.Type_Of := Used_Field_Type;
            Getter_View.Location := Field.Location;

            Getter_View.Is_Final := True;
            Getter_View.Parameters := new Parameter_Array (1 .. 1);

            Getter_View.Parameters (1) := Create (Getter_View);
            Getter_View.Parameters (1).Name :=
              To_Dynamic_Expression ("This");
            Getter_View.Parameters (1).Mode := In_Mode;
            Getter_View.Parameters (1).Type_Of := Primitive_Type;
            Getter_View.Parameters (1).Attached := True;
            Getter_View.Parameters (1).Assume_Stored := False;

            Getter_Ada_Call_Handler.This_Exp :=
              Empty_Dynamic_Expression;
            Getter_Ada_Call_Handler.Mode := Getter;
            Getter_Ada_Call_Handler.Var_Name := Getter_View.Name;

            Add_Wrapper
              (Map     => Getter_Map,
               Wrapper => Getter_Ada_Call_Handler'Unrestricted_Access,
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Parameter_Resolution);

            Add_Wrapper
              (Map     => Getter_Map,
               Wrapper => Getter_Java_Call_Handler'Unrestricted_Access,
               Lang    => Java_Lang,
               Target  => High_Sb,
               Context => Return_Type_Converted);

            Bind_Subprogram
              (Handle   => Handle,
               View     => Getter_View,
               Unit     => Unit,
               Wrappers => Getter_Map);

            --  Do not create a setter if this field is either limited or
            --  constant. Note that although a field with a constant type is
            --  not legal Ada, we model a record discriminant as a constant
            --  field.

            if not (Field_Type.Ref.Is_Limited or Field.Is_Constant) then
               Setter_View := Create (Type_View);

               Setter_View.Is_Dispatching := False;
               Setter_View.Name := Field.Name;
               Setter_View.Is_Final := True;
               Setter_View.Parameters := new Parameter_Array (1 .. 2);
               Setter_View.Location := Field.Location;

               Setter_View.Parameters (1) := Create (Setter_View);
               Setter_View.Parameters (1).Name :=
                 To_Dynamic_Expression ("This");
               Setter_View.Parameters (1).Mode := In_Mode;
               Setter_View.Parameters (1).Type_Of := Primitive_Type;
               Setter_View.Parameters (1).Attached := True;
               Setter_View.Parameters (1).Assume_Stored := False;

               Setter_View.Parameters (2) := Create (Setter_View);
               Setter_View.Parameters (2).Name :=
                 To_Dynamic_Expression ("Value");
               Setter_View.Parameters (2).Mode := In_Mode;
               Setter_View.Parameters (2).Type_Of := Field_Type;

               Setter_Ada_Call_Handler.This_Exp :=
                 Empty_Dynamic_Expression;
               Setter_Ada_Call_Handler.Mode := Setter;
               Setter_Ada_Call_Handler.Var_Name := Setter_View.Name;

               Add_Wrapper
                 (Map     => Setter_Map,
                  Wrapper => Setter_Ada_Call_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Parameter_Resolution);

               Add_Wrapper
                 (Map     => Setter_Map,
                  Wrapper => Setter_Ada_Call_Handler'Unrestricted_Access,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Code_Node_Resolution);

               Bind_Subprogram
                 (Handle   => Handle,
                  View     => Setter_View,
                  Unit     => Unit,
                  Wrappers => Setter_Map);
            end if;
         end;
      exception
         when Silent_Not_Supported =>
            null;

         when E : others =>
            Trace
              (Conversions.To_Wide_String (Exception_Information (E)),
               Errors_Only);
      end Bind_Field;

   begin
      if Type_View.Components /= null then
         for I in Type_View.Components'Range loop
            Bind_Field (Type_View.Components (I));
         end loop;
      end if;

      if Type_View.Discriminants /= null then
         for I in Type_View.Discriminants'Range loop
            Bind_Field (Type_View.Discriminants (I));
         end loop;
      end if;
   end Bind_Fields;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Default_Super_Caller;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Prepend
        (Expression.Bloc.Declarations,
         New_Line
         & "super (new com.adacore.ajis.internal.ada.AdaAccess (null));");
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Constructor_Code_Node_Resolver;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Append
        (Expression.Bloc.Code_Node,
         New_Line & "this.adaAccess = "
         & Expression.Expression & ";");
      Append
        (Expression.Bloc.Statements_After,
         New_Line & "this.myOwner = com.adacore.ajis.IProxy.Owner.PROXY;");
      Append
        (Expression.Bloc.Statements_After,
         New_Line
         & "this.myAllocator = com.adacore.ajis.IProxy.Allocator.DYNAMIC;");
      Expression.Expression := Empty_Dynamic_Expression;
   end Wrap;

   ------------------------
   -- Get_Constructor_Id --
   ------------------------

   function Get_Constructor_Id
     (Element : Bound_Type) return Dynamic_Expression is
   begin
      return Element.Enclosing_Unit.Ada_Pckg_Name & "."
        & Element.Constructor_Id;
   end Get_Constructor_Id;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Equals_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Parameter_Handlings_Params
         renames Parameter_Handlings_Params (Parameters.all);
   begin
      if Params.Is_First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Get_Bound_Package_Name (Wrapper.Type_Of.Base_Package) & ".""="" ("
            & Expression.Expression & ".all, ");
      else
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression & ".all)");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Equals_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Parameter_Handlings_Params
         renames Parameter_Handlings_Params (Parameters.all);
   begin
      if Params.Is_First then
         Wrapper.Condition := New_Line & "if (this.getClass () != ";

         Append
           (Expression.Bloc.Statements_Before,
            Wrapper.Condition);
      else
         Append
           (Wrapper.Condition,
            Copy (Expression.Expression) & ".getClass ()) return false;");

         Expression.Expression := "((com.adacore.ajis.internal.ada.AdaProxy) "
           & Expression.Expression & ")";
      end if;
   end Wrap;

   -----------------
   -- Bind_Equals --
   -----------------

   procedure Bind_Equals
     (Handle    : not null access Kernel.Kernel_Record;
      Type_View : Simple_Type_View_Access;
      Unit      : Bound_Unit)
   is
      Java_Wrapper : aliased Java_Equals_Wrapper;
      Ada_Wrapper  : aliased Ada_Equals_Wrapper;

      Equals_Map : Wrapping_Map;

      Sb_View : constant Simple_Subprogram_View_Access := Create (Type_View);
      Access_Type : Simple_Type_Reference;
   begin
      if Type_View.Is_Limited then
         return;
      end if;

      Sb_View.Returned_Type := Create (Sb_View);
      Sb_View.Returned_Type.Type_Of.Ref := Create (Sb_View);
      Sb_View.Returned_Type.Type_Of.Ref.Kind := Standard_Boolean_Kind;
      Sb_View.Returned_Type.Type_Of.Ref.Full_Ada_Name :=
        To_Dynamic_Expression ("Standard.Boolean");
      Sb_View.Returned_Type.Index := Type_View.Index;
      Sb_View.Location := Type_View.Location;

      Access_Type := Create_Access_To ((Type_View, others => <>), True);
      Access_Type.Ref.Target_Type.Ref := new Simple_Type_View'Class'
        (Access_Type.Ref.Target_Type.Ref.all);
      Access_Type.Ref.Target_Type.Ref.Full_Java_Name := To_Dynamic_Expression
        ("java.lang.Object");

      Sb_View.Name := To_Dynamic_Expression ("equals");
      Sb_View.Parameters := new Parameter_Array (1 .. 2);
      Sb_View.Is_Final := Type_View.Kind /= Tagged_Record_Kind;

      Sb_View.Parameters (1) := Create (Sb_View);
      Sb_View.Parameters (1).Name := To_Dynamic_Expression ("Left");
      Sb_View.Parameters (1).Is_Primitive := False;
      Sb_View.Parameters (1).Is_Controlling := False;
      Sb_View.Parameters (1).Type_Of := Access_Type;
      Sb_View.Parameters (1).Assume_Stored := False;
      Sb_View.Parameters (1).Attached := True;

      Sb_View.Parameters (2) := Create (Sb_View);
      Sb_View.Parameters (2).Name := To_Dynamic_Expression ("Right");
      Sb_View.Parameters (2).Is_Primitive := False;
      Sb_View.Parameters (2).Is_Controlling := False;
      Sb_View.Parameters (2).Type_Of := Access_Type;
      Sb_View.Parameters (2).Assume_Stored := False;

      if Type_View.Kind = Tagged_Record_Kind then
         Sb_View.Parameters (1).Type_Of.Is_Class_Wide := True;
         Sb_View.Parameters (2).Type_Of.Is_Class_Wide := True;
      end if;

      Ada_Wrapper.Type_Of := Type_View;

      Add_Wrapper
        (Map     => Equals_Map,
         Wrapper => Ada_Wrapper'Unrestricted_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Parameter_Resolution);

      Add_Wrapper
        (Map     => Equals_Map,
         Wrapper => Java_Wrapper'Unrestricted_Access,
         Lang    => Java_Lang,
         Target  => High_Sb,
         Context => Add_Parameter_Original);

      Bind_Subprogram
        (Handle   => Handle,
         View     => Sb_View,
         Unit     => Unit,
         Wrappers => Equals_Map);
   end Bind_Equals;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Finalize_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : Parameter_Handlings_Params
      renames Parameter_Handlings_Params (Parameters.all);
      Conversion : constant Type_Record := Get_Or_Create_Conversion_Type
        (Params.Handle,
         Params.Subprogram_View.Parameters (1).Type_Of.Ref.Target_Type.Ref);
      Garbage_Id : constant Wide_String := Get_Unique_Id;

      Pointer_Name : Dynamic_Expression;
   begin
      if Is_Unconstrained_Array
        (Params.Parameter.Type_Of.Ref.Target_Type.Ref)
      then
         Pointer_Name := To_Dynamic_Expression ("Array_Pointer");
      else
         Pointer_Name := To_Dynamic_Expression ("Object_Pointer");
      end if;

      Append
        (Expression.Bloc.Declarations,
         New_Line & Garbage_Id & " : " & Conversion.Conversion_Package_Name.all
         & "." & Pointer_Name & " := "
         & Conversion.Conversion_Package_Name.all
         & "." & Pointer_Name & " (" & Expression.Expression & ");");

      Append
        (Expression.Enclosing_Expression.Expression,
         Conversion.Conversion_Package_Name.all
         & ".Free (" & Garbage_Id & ")");
   end Wrap;

   ----------------
   -- Bind_Clone --
   ----------------

   procedure Bind_Clone
     (Handle      : not null access Kernel.Kernel_Record;
      Type_View   : Simple_Type_View_Access;
      Unit        : Bound_Unit;
      Ada_Wrapper : Code_Wrapper_Access := null)
   is
      Clone_View : Simple_Subprogram_View_Access;
      Clone_Wrapper_Map : Wrapping_Map;
      This_Parameter    : Parameter_Simple_View_Access;
      Returned_Type     : constant Simple_Type_View_Access :=
        Simple_Type_View_Access (Type_View.Copy);
      Returned_Type_Ref : Simple_Type_Reference;
      Default_Clone_Wrapper : aliased Ada_Clone_Wrapper;
      Limited_Clone_Wrapper : aliased Java_Limited_Clone_Wrapper;
   begin
      Clone_View := Create (Type_View);
      This_Parameter := Create (Clone_View);
      This_Parameter.Type_Of :=
        Create_Access_To ((Ref => Type_View, others => <>), False);
      This_Parameter.Name           := To_Dynamic_Expression ("This");
      This_Parameter.Is_Primitive   := True;
      This_Parameter.Is_Controlling := True;
      This_Parameter.Attached       := True;
      This_Parameter.Assume_Stored  := False;

      if Type_View.Kind = Tagged_Record_Kind then
         Returned_Type_Ref := Create_Access_To
           ((Ref => Returned_Type, Is_Class_Wide => True, others => <>),
            False);
         Returned_Type_Ref.Is_Class_Wide := True;
         --  ??? Strange - the 'Class on the target type should be enough...
      elsif Is_Unconstrained_Array (Type_View) then
         --  ??? Factorize the following in some kind of 'generate named access
         --  type' - would be useful for the constructor as well.

         Returned_Type_Ref := Create_Access_To
           ((Ref => Type_View, others => <>), False);
      else
         Returned_Type_Ref := Create_Access_To
           ((Ref => Returned_Type, Is_Class_Wide => False, others => <>),
            False);
         Returned_Type_Ref.Is_Class_Wide := False;
      end if;

      Returned_Type_Ref.Pass_By_Address := True;

      Clone_View.Returned_Type     := Create (Clone_View);
      Clone_View.Returned_Type.Location := Type_View.Location;
      Clone_View.Returned_Type.Type_Of  := Returned_Type_Ref;
      Clone_View.Parameters        :=
        new Parameter_Array'(1 => This_Parameter);
      Clone_View.Bind_Java         := True;
      Clone_View.Name              := To_Dynamic_Expression ("internalClone");
      Clone_View.Is_Real_Primitive := False;
      Clone_View.Is_Final          := False;
      Clone_View.Location          := Type_View.Location;

      if Type_View.Is_Limited then
         if Type_View.Kind = Tagged_Record_Kind
           and then Type_View.Target_Type /= Null_Type_Reference
           and then Type_View.Target_Type.Ref.Is_Limited then
            --  In this case, the final clone has already been generated.
            --  Nothing has to be done.

            return;
         end if;

         Clone_View.Bind_Ada := False;
         Clone_View.Is_Final := True;

         Add_Wrapper
           (Map     => Clone_Wrapper_Map,
            Wrapper => Limited_Clone_Wrapper'Unrestricted_Access,
            Lang    => Java_Lang,
            Target  => High_Sb,
            Context => Code_Node_Resolution);
      elsif Ada_Wrapper /= null then
         Add_Wrapper
           (Map     => Clone_Wrapper_Map,
            Wrapper => Ada_Wrapper,
            Lang    => Ada_Lang,
            Target  => High_Sb,
            Context => Parameter_Resolution);
      else
         Add_Wrapper
           (Map     => Clone_Wrapper_Map,
            Wrapper => Default_Clone_Wrapper'Unrestricted_Access,
            Lang    => Ada_Lang,
            Target  => High_Sb,
            Context => Parameter_Resolution);
      end if;

      Bind_Subprogram
        (Handle   => Handle,
         View     => Clone_View,
         Unit     => Unit,
         Wrappers => Clone_Wrapper_Map);
   end Bind_Clone;

   ---------------------------------
   -- Get_Array_Manipulation_Kind --
   ---------------------------------

   function Get_Array_Manipulation_Kind
     (Ref : Simple_Type_Reference) return Array_Manipulation_Kind
   is
   begin
      if Ref.Ref.Kind /= Access_Kind
        or else Ref.Ref.Target_Type.Ref.Kind /= Array_Kind
        or else not Is_Unconstrained_Array (Ref.Ref.Target_Type.Ref)
      then
         return Constrained_Object;
      elsif Ref.Is_From_Unrestricted_Access then
         return Wrapped_Access;
      else
         return Direct_Array_Access;
      end if;
   end Get_Array_Manipulation_Kind;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Clone_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Result : constant Wide_String := Get_Unique_Id;

      Params : Parameter_Handlings_Params renames
        Parameter_Handlings_Params (Parameters.all);
      Conversion : constant Type_Record := Get_Or_Create_Conversion_Type
        (Params.Handle,
         Params.Parameter.Type_Of.Ref.Target_Type.Ref);
   begin
      Append
        (Expression.Bloc.Declarations,
         New_Line & Result & " : "
         & Conversion.Conversion_Package_Name.all & ".");

      if Get_Array_Manipulation_Kind
        (Params.Parameter.Type_Of) = Direct_Array_Access
      then
         Append
           (Expression.Bloc.Declarations,
            "Array_Pointer");
      else
         Append
           (Expression.Bloc.Declarations,
            "Object_Pointer");
      end if;

      Append
        (Expression.Bloc.Declarations,
         " := "
         & "new "
         & Params.Parameter.Type_Of.Ref.Target_Type.Ref.Full_Ada_Name
         & "'(" & Expression.Expression & ".all);");

      Append
        (Expression.Enclosing_Expression.Expression, Result);
   end Wrap;

   --------------------------------
   -- Java_Limited_Clone_Wrapper --
   --------------------------------

   procedure Wrap
     (Wrapper    : in out Java_Limited_Clone_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Append
        (Expression.Bloc.Code_Node,
         New_Line
         & "throw new com.adacore.ajis.NativeException "
         & "(""Cannot clone an Ada limited type."");");
   end Wrap;

   ----------------------------
   -- Java_Component_Wrapper --
   ----------------------------

   procedure Wrap
     (Wrapper    : in out Java_Component_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : Subprogram_Params renames
        Subprogram_Params (Parameters.all);
      Tmp : constant Wide_String := Get_Unique_Id;
   begin
      if Wrapper.Is_From_Static_Field then
         Append
           (Expression.Bloc.Statements_Before,
            New_Line
            & Params.Subprogram_View.Returned_Type.Type_Of.Ref.
              Target_Type.Ref.Full_Java_Name & " " & Tmp & " = "
            & Expression.Expression & ";");
         Expression.Expression := To_Dynamic_Expression (Tmp);
         Append
           (Expression.Bloc.Statements_After,
            New_Line & Expression.Expression & ".container = this;");
      end if;
   end Wrap;

   ----------------------------------
   -- Bind_Extra_String_Primitives --
   ----------------------------------

   procedure Bind_Extra_String_Primitives
     (Handle : not null access Kernel_Record;
      Unit   : Bound_Unit;
      View   : Simple_Type_View_Access)
   is
      To_String_View       : Simple_Subprogram_View_Access;
      To_String_Map        : Wrapping_Map;
      Ada_To_String_Hander : aliased Ada_To_String_Wrapper;

      Constructor_View : Simple_Subprogram_View_Access;
      Constructor_Map  : Wrapping_Map;
      Ada_Simple_Constructor_Handler : aliased Ada_Constructor_Handler;
      Ada_String_Constructor_Handler : aliased Ada_String_Constructor_Wrapper;
      Java_Constructor_Handler       : aliased
        Java_Constructor_Code_Node_Resolver;

      Conv : constant Type_Record :=
        Get_Or_Create_Conversion_Type (Handle, View);
   begin
      To_String_View               := Create (View);
      To_String_View.Returned_Type := Create (To_String_View);
      To_String_View.Returned_Type.Type_Of.Ref := Create (To_String_View);
      To_String_View.Returned_Type.Type_Of.Ref.Kind := JNI_Kind;
      To_String_View.High_Sb_With_Env := True;
      To_String_View.Location := View.Location;

      To_String_View.Returned_Type.Type_Of.Ref.Full_Java_Name :=
        To_Dynamic_Expression ("java.lang.String");
      To_String_View.Returned_Type.Type_Of.Ref.Full_Ada_Name :=
        JNI_Pckg & ".J_String";
      To_String_View.Returned_Type.Type_Of.Ref.Exported_Name :=
        To_Dynamic_Expression ("J_String");

      To_String_View.Parameters := new Parameter_Array (1 .. 1);

      To_String_View.Parameters (1) := Create (To_String_View);
      To_String_View.Parameters (1).Type_Of.Ref := View;
      To_String_View.Parameters (1).Name :=
        To_Dynamic_Expression ("This");
      To_String_View.Parameters (1).Mode := In_Mode;
      To_String_View.Parameters (1).Attached := True;
      To_String_View.Parameters (1).Assume_Stored := False;

      To_String_View.Is_Final := True;
      To_String_View.Name     := To_Dynamic_Expression ("toString");

      Add_Wrapper
        (Map     => To_String_Map,
         Wrapper => Ada_To_String_Hander'Unchecked_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Code_Node_Resolution);

      Bind_Subprogram
        (Handle   => Handle,
         View     => To_String_View,
         Unit     => Unit,
         Wrappers => To_String_Map);

      Constructor_View                  := Create (View);
      Constructor_View.High_Sb_With_Env := True;
      Constructor_View.Is_Constructor   := True;
      Constructor_View.Name             := View.Exported_Name;
      Constructor_View.Force_Attach     := View;
      Constructor_View.Is_Final         := False;
      Constructor_View.Location         := View.Location;

      Constructor_View.Returned_Type := Create (To_String_View);
      Constructor_View.Returned_Type.Type_Of :=
        Create_Access_To ((Ref => View, others => <>), False);

      if Is_Unconstrained_Array (View) then
         Constructor_View.Returned_Type.Type_Of.Ref.Full_Ada_Name :=
           To_Dynamic_Expression
             (Conv.Conversion_Package_Name.all & ".Array_Pointer");
      else
         Constructor_View.Returned_Type.Type_Of.Ref.Full_Ada_Name :=
           To_Dynamic_Expression
             (Conv.Conversion_Package_Name.all & ".Object_Pointer");
      end if;

      Constructor_View.Parameters := new Parameter_Array (1 .. 1);

      Constructor_View.Parameters (1) := Create (Constructor_View);
      Constructor_View.Parameters (1).Type_Of.Ref := Create (Constructor_View);
      Constructor_View.Parameters (1).Type_Of.Ref.Full_Java_Name :=
        To_Dynamic_Expression ("java.lang.String");
      Constructor_View.Parameters (1).Type_Of.Ref.Full_Ada_Name :=
        JNI_Pckg & ".J_String";
      Constructor_View.Parameters (1).Type_Of.Ref.Exported_Name :=
        To_Dynamic_Expression ("J_String");
      Constructor_View.Parameters (1).Type_Of.Ref.Kind := JNI_Kind;
      Constructor_View.Parameters (1).Name := To_Dynamic_Expression ("Str");

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Java_Constructor_Handler'Unchecked_Access,
         Lang    => Java_Lang,
         Target  => High_Sb,
         Context => Code_Node_Resolution);

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Ada_String_Constructor_Handler'Unchecked_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Code_Node_Resolution);

      Add_Wrapper
        (Map     => Constructor_Map,
         Wrapper => Ada_Simple_Constructor_Handler'Unchecked_Access,
         Lang    => Ada_Lang,
         Target  => JNI_Sb,
         Context => Parameter_Resolution);

      Bind_Subprogram
        (Handle   => Handle,
         View     => Constructor_View,
         Unit     => Unit,
         Wrappers => Constructor_Map);
   end Bind_Extra_String_Primitives;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_To_String_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      C_Tmp           : constant Wide_String := Get_Unique_Id;
      Returned_Object : constant Wide_String := Get_Unique_Id;
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);
   begin
      Append
        (Expression.Bloc.Declarations,
         New_Line
         & C_Tmp & " : Interfaces.C.Strings.chars_ptr;"
         &  New_Line & Returned_Object & " : " & JNI_Pckg & ".J_String;");

      Append
        (Expression.Bloc.Code_Node,
         New_Line & C_Tmp
         & " := Interfaces.C.Strings.New_String (String ("
         & Params.Subprogram_View.Parameters (1).Name & "));"
         & New_Line & Returned_Object & " := " & JNI_Pckg & ".New_String_UTF ("
         & Env_Parameter_Name & ", " & C_Tmp & ");"
         & New_Line & "Interfaces.C.Strings.Free (" & C_Tmp & ");"
         & New_Line & "return " & Returned_Object & ";");
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_String_Constructor_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);

      Target_Type_Name : constant Dynamic_Expression :=
        Params.Subprogram_View.Returned_Type.Type_Of
          .Ref.Target_Type.Ref.Full_Ada_Name;

      --  Should not use Conv - the general encapsulation is created in the
      --  JNI subprogram.
   begin
      Append
        (Expression.Bloc.Declarations,
         New_Line & "Is_Copy : aliased " & JNI_Pckg & ".J_Boolean;"
         & New_Line & "C_Str : constant Interfaces.C.Strings.chars_ptr := "
         & JNI_Pckg & ".Get_String_UTF_Chars (" & Env_Parameter_Name
         & ", Str, Is_Copy'Unchecked_Access);"
         & New_Line & "Ada_Str : constant " & AJIS_Pckg
         & ".Unchecked_String_Access := "
         & AJIS_Pckg & ".To_Unchecked_String ("
         & AJIS_Pckg & ".Convert (C_Str));"
         & New_Line & "pragma Suppress (Access_Check, Ada_Str);"
         & New_Line & "Result : "
         & Params.Subprogram_View.Returned_Type.Type_Of.Ref.Full_Ada_Name
         & " := new " & Target_Type_Name & "'("
         & Target_Type_Name
         & " (Ada_Str "
         & "(1 .. Integer (Interfaces.C.Strings.Strlen (C_Str)))));");

      Append
        (Expression.Bloc.Code_Node,
         New_Line & JNI_Pckg & ".Release_String_UTF_Chars ("
         & Env_Parameter_Name & ", Str, C_Str);"
         & New_Line & "return Result;");
   end Wrap;

end Ada2Java.Bound_Elements.Types;
