------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada.Containers.Doubly_Linked_Lists;

with Ada2Java.Code_Wrappers;        use Ada2Java.Code_Wrappers;

package Ada2Java.Bound_Elements.Subprograms is

   ---------------------
   -- Bind_Subprogram --
   ---------------------

   type Wrapping_Map is private;
   --  A wrapping map hold the wrappers that have to be applied to subprograms
   --  generated by the binder.

   Empty_Wrapping_Map : constant Wrapping_Map;

   type Wrapping_Language is (Ada_Lang, Java_Lang);

   type Wrapped_Target is
     (High_Sb,
      --  This subprogram has a High callable parameter profile.

      JNI_Sb
      --  The parameters of this subprogram are modified so that it can be
      --  called from JNI
     );

   type Wrapping_Context is
     (Start,
      --  These wrappers are called at the beginning of the binding process,
      --  after Set_Subprogram.

      Add_Parameter_Original,
      --  These wrappers are called each time a parameter is added, and before
      --  the parameter transformation from or to JNI.

      Add_Parameter_Converted,
      --  These wrappers are called each time a parameter is added, after
      --  the conversion to JNI.

      Return_Type_Original,
      --  Called before the conversion of the return type of a function.

      Return_Type_Converted,
      --  Called once the returned type of a function has been converted.

      Parameter_Resolution,
      --  These wrappers are responsible of taking care of the parameter
      --  value, and to add it in the enclosing expression or
      --  anywhere else relevant. There can be only one wrapper of this type
      --  for each Language / Target pair. If not set, then a default wrapper
      --  will be used.

      Finish,
      --  These wrappers are called at the end of the binding process

      Code_Node_Resolution
      --  This is a special kind of wrapper. There can be only one for each
      --  Language / Target pair. It purpose is to move the computed
      --  expression into the code node. If not set, then a default wrapper
      --  will be used. In the case of a function call, this wrapper is
      --  responsible of writing the proper return statement.
     );

   type Subprogram_Params is new Wrapper_Parameters with record
      Handle           : access Kernel.Kernel_Record;
      Subprogram_View  : Simple_Subprogram_View_Access;
      Unit             : Bound_Unit;
      Context          : Wrapping_Context;
      Target           : Wrapped_Target;
      Java_JNI_Name    : Dynamic_Expression;
      Ada_High_Sb_Name : Dynamic_Expression;
      Ada_JNI_Sb_Name : Dynamic_Expression;
   end record;

   type Subprogram_Params_Access is access all Subprogram_Params'Class;

   type Parameter_Handlings_Params is new Subprogram_Params with record
      Parameter         : Parameter_Simple_View_Access;
      Is_First, Is_Last : Boolean;
      Param_Number      : Integer;
   end record;

   type Parameter_Handlings_Params_Access is
     access all Parameter_Handlings_Params;

   procedure Add_Wrapper
     (Map     : in out Wrapping_Map;
      Wrapper : Code_Wrapper_Access;
      Lang    : Wrapping_Language;
      Target  : Wrapped_Target;
      Context : Wrapping_Context);
   --  Adds a wrapper to the wrapping map. The same wrapper can be added to
   --  several set of {Lang, Target, Context}.

   procedure Bind_Subprogram
     (Handle            : not null access Kernel.Kernel_Record;
      View              : Simple_Subprogram_View_Access;
      Unit              : Bound_Unit;
      Wrappers          : Wrapping_Map;
      Body_For_Abstract : Boolean := False;
      Add_Java_High_Sb  : Boolean := True);
   --  Create the actual binding to Ada in the relevant files stored in the
   --  Handle.

   function To_Original_Expanded_Profile
     (Handle : not null access Kernel.Kernel_Record;
      View   : Simple_Subprogram_View_Access) return Dynamic_Expression;
   --  Return the original profile, expanding all the parameter names.

   type Bound_Subprogram is new Bound_Element_Record with private;

   overriding
   procedure Bind
     (Element_Bound : access Bound_Subprogram;
      Handle        : not null access Kernel.Kernel_Record);
   --  See inherited documentation

   function Get_View
     (This : access Bound_Subprogram)
      return Simple_Subprogram_View_Access;

   type Ada_Function_Wrapper is new Code_Wrapper with record
      In_JNI : Boolean;
      Returned_Exp : Dynamic_Expression;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Function_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

private

   type Bound_Subprogram is new Bound_Element_Record with
      record
         Handle : access Kernel.Kernel_Record;
         View   : Simple_Subprogram_View_Access;
      end record;

   ------------------
   -- Wrapping_Map --
   ------------------

   package Wrappers_List is new Ada.Containers.Doubly_Linked_Lists
     (Code_Wrapper_Access);

   use Wrappers_List;

   type Wrappers_Array is array
     (Wrapping_Language, Wrapped_Target, Wrapping_Context)
   of Wrappers_List.List;

   type Wrapping_Map is record
      Wrappers : Wrappers_Array;
   end record;

   Empty_Wrapping_Map : constant Wrapping_Map := (others => <>);

   procedure Set_Default_Wrappers
     (Map  : in out Wrapping_Map;
      View : Simple_Subprogram_View_Access);

   procedure Call_Wrappers
     (Map                  : Wrapping_Map;
      Context              : Wrapping_Context;
      Params               : access Subprogram_Params'Class;
      Ada_High_Expression  : Wrapping_Expression;
      Ada_JNI_Expression   : Wrapping_Expression;
      Java_JNI_Expression  : Wrapping_Expression;
      Java_High_Expression : Wrapping_Expression);

end Ada2Java.Bound_Elements.Subprograms;
