------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada2Java.Kernel; use Ada2Java.Kernel;

package Ada2Java.Bound_Elements.Subprograms.Data is

   type Subprogram_Properties is record
      Parameters_Written           : Natural := 0;
      Java_Call_Parameters_Written : Natural := 0;
      Ada_Call_Parameters_Written  : Natural := 0;
   end record;
   --  This type is used to store the properties and the process of a binding
   --  progress.

   type Data_Information is (Yes, No, Unknown);

   type Data_Record is abstract tagged private;

   function To_Type_Name
     (D : Data_Record) return Dynamic_Expression is abstract;

   function To_JNI_Type_Name
     (D : Data_Record) return Dynamic_Expression is abstract;

   function Create_Parameters
     (D    : Data_Record;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access)
      return Dynamic_Expression is abstract;
   --  Return a string containing a part of the subprogram profile, adding the
   --  data given in parameter.

   function Create_JNI_Parameters
     (D    : Data_Record;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression
      is abstract;
   --  Return a string containing a part of the subprogram JNI profile, adding
   --  the data given in parameter.

   procedure Create_Conversion_From_JNI
     (D          : in out Data_Record;
      Expression : Wrapping_Expression) is null;
   --  If the data needs to have some kind of initialization before the call,
   --  it's appenned to Initialization. Initialized_Input gets the value of
   --  the result of the initialization.

   procedure Adjust_Unit_For_Use (D : in out Data_Record; Unit : Bound_Unit);
   --  There might be things that have to be done in order to make the data
   --  usable in the file, for example with / use / import statements. This
   --  subprogram, which doesn't do anything by default, is responsible of
   --  adding these information when required.

   function Get_Type_View (D : Data_Record) return Simple_Type_Reference;
   --  Return the simple type view associated to this data

   ---------------
   -- Java_Data --
   ---------------

   type Java_Data is abstract new Data_Record with private;

   function Create_Parameters
     (D    : Java_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression;
   --  See inherited documentation

   function Create_JNI_Parameters
     (D    : Java_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_Data;
      Expression : Wrapping_Expression) is abstract;
   --  Adds needing information in Call, Initialization and Finalization,
   --  assuming that Data_Expression is denoting an expression coming from JNI
   --  through the data type D.

   function Get_JNI_Type (D : Java_Data) return Dynamic_Expression is abstract;
   --  Return the mangled expression used in JNI profiles for the type of the
   --  data given in parameter.

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties);
   --  This subprogram return the expression that gets the value of the object
   --  referenced by this data.

   type Parameter_Mode_Type is
     (Value_Native,
      --  The object is a native type, passed by value

      Value_Reference,
      --  The object is a reference, passed by copy

      Escapable_Reference,
      --  The object is an escapable reference

      Non_Escapable_Reference,
      --  The object is a non escapable reference

      Static_Reference
      --  The object is a static reference
     );

   function Get_Mode (D : Java_Data) return Parameter_Mode_Type;
   --  Returns the mode in which the data is passed. This is only really
   --  relevant when dealing with Java references.

   --------------
   -- Ada_Data --
   --------------

   type Ada_Data is abstract new Data_Record with private;

   function To_Ada_Internal_Type_Name (D : Ada_Data) return Dynamic_Expression;
   --  Return the type name that should be used for temporary variables
   --  declared of this data type. Usually no different than To_Type_Name,
   --  exept on special cases (e.g. anonymous access types).

   function Create_Parameters
     (D    : Ada_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression;
   --  See inherited documentation

   function Create_JNI_Parameters
     (D    : Ada_Data;
      Prof : access Subprogram_Properties;
      View : Simple_Subprogram_View_Access) return Dynamic_Expression;
   --  See inherited documentation

   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Data;
      Expression : Wrapping_Expression) is abstract;
   --  Adds needing information in Call, Initialization and Finalization,
   --  assuming that Data_Expression is denoting an expression that we want to
   --  JNI pass to JNI through the data type D.

   function Get_JNI_Call (D : Ada_Data) return Dynamic_Expression is abstract;
   --  Return the JNI function to call if we have to call a Java function
   --  returning this type.

   function Get_Default_JNI_Value
     (D : Ada_Data) return Dynamic_Expression is abstract;
   --  Return a default value for this kind of data.

   function Should_Be_Class_Wide (D : Ada_Data'Class) return Boolean;
   --  Return true if D should be bound into a class wide data, based
   --  on the target type and the bound context.

   -----------------
   -- Bound_Data --
   -----------------

   type Bound_Data is record
      Ada_Part      : access Ada_Data'Class := null;
      Java_Part     : access Java_Data'Class := null;
   end record;

   Null_Bound_Data : Bound_Data := (others => <>);

   function Create_Bound_Data
     (View   : Simple_Object_View_Access;
      Unit   : Bound_Unit;
      Kernel : access Kernel_Record;
      Call_Convention : Call_Conventions)
      return Bound_Data;
   --  Process the element given in parameter, and return the corresponding
   --  binding data. Will raise a Not_Supported exception if the element cannot
   --  be bound. If the element given is a function, the returned type will
   --  be returned.

   type Check_Loc is (Before, After);

   procedure Generate_Access_Checks
     (D              : in out Java_Data'Class;
      Expression     : Wrapping_Expression;
      Loc            : Check_Loc);
   --  Generate test with regards to access management.
   --
   --  If the object pointed by this java data is an access parameter pointer
   --  that should be allowed to be stored, checks are generated to ensure
   --  that non-storable objects are not passed, e.g. owned objects.
   --
   --  If the object pointed by this java data is not an access parameter, then
   --  null values will be forbidden.

private

   type Data_Record is abstract tagged record
      Name             : Unbounded_Wide_String;
      Decl_View        : Simple_Object_View_Access;
      Kernel           : access Kernel_Record;
      Simple_Type_View : Simple_Type_Reference := Null_Type_Reference;
      Unit             : Bound_Unit;
      Call_Convention  : Call_Conventions;
   end record;

   type Java_Data is abstract new Data_Record with record
      null;
   end record;

   type Ada_Data is abstract new Data_Record with record
      null;
   end record;

   function Has_Out_Mode (D : Data_Record'Class) return Boolean;
   --  Return true if D is an "out" or "in out" parameter

   ---------------
   -- Java Data --
   ---------------

   type Java_Primitive_Data is new Java_Data with record
      null;
   end record;

   overriding
   function To_Type_Name (D : Java_Primitive_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Java_Primitive_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_Primitive_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Type (D : Java_Primitive_Data) return Dynamic_Expression;

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Java_Primitive_Data; Unit : Bound_Unit);

   type Java_Reference_Data is new Java_Data with record
      null;
   end record;

   function To_Type_Name (D : Java_Reference_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Java_Reference_Data; Unit : Bound_Unit);
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Java_Reference_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Java_Reference_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties);

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_Reference_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Type (D : Java_Reference_Data) return Dynamic_Expression;

   type Java_Callback_Data is new Java_Data with record
      null;
   end record;

   overriding
   function To_Type_Name (D : Java_Callback_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Java_Callback_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Java_Callback_Data; Unit : Bound_Unit);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Java_Callback_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_Callback_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   function Get_JNI_Type (D : Java_Callback_Data) return Dynamic_Expression;

   type Java_JNI_Data is new Java_Data with record
      null;
   end record;
   --  This type is used to map data that are already managed as Java objects
   --  in the code. No transformation is required.

   overriding
   function To_Type_Name (D : Java_JNI_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Java_JNI_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Java_JNI_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_JNI_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   function Get_JNI_Type (D : Java_JNI_Data) return Dynamic_Expression;

   type Java_Enum_Data is new Java_Data with record
      null;
   end record;

   overriding
   function To_Type_Name (D : Java_Enum_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Java_Enum_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Java_Enum_Data;
      Expression : Wrapping_Expression);

   procedure Create_Conversion_To_JNI
     (D          : in out Java_Enum_Data;
      Expression : Wrapping_Expression;
      Prop       : access Subprogram_Properties);

   overriding
   function Get_JNI_Type (D : Java_Enum_Data) return Dynamic_Expression;

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Java_Enum_Data; Unit : Bound_Unit);

   --------------
   -- Ada Data --
   --------------

   type Ada_Scalar_Data is new Ada_Data with record
      Initial_Input : Dynamic_Expression;
   end record;

   overriding
   function To_Type_Name (D : Ada_Scalar_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name (D : Ada_Scalar_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Scalar_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Scalar_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Call (D : Ada_Scalar_Data) return Dynamic_Expression;

   overriding
   function Get_Default_JNI_Value
     (D : Ada_Scalar_Data) return Dynamic_Expression;

   procedure Adjust_Unit_For_Use
     (D : in out Ada_Scalar_Data; Unit : Bound_Unit);

   type Ada_Object_Data is new Ada_Data with record
      null;
   end record;

   overriding
   function To_Type_Name (D : Ada_Object_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name (D : Ada_Object_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Object_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Object_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   function Get_JNI_Call (D : Ada_Object_Data) return Dynamic_Expression;

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Ada_Object_Data; Unit : Bound_Unit);

   overriding
   function Get_Default_JNI_Value
     (D : Ada_Object_Data) return Dynamic_Expression;

   type Ada_Pointer_To_Object_Data is new Ada_Data with record
      Conversion_Package_Name : Wide_String_Access;
      Initial_Input    : Dynamic_Expression;
   end record;

   overriding
   function To_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_Ada_Internal_Type_Name
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Pointer_To_Object_Data;
      Expression : Wrapping_Expression);
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Pointer_To_Object_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Call
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression;

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Ada_Pointer_To_Object_Data; Unit : Bound_Unit);

   overriding
   function Get_Default_JNI_Value
     (D : Ada_Pointer_To_Object_Data) return Dynamic_Expression;

   type Ada_Pointer_To_Scalar_Data is new Ada_Data with record
      null;
   end record;

   overriding
   function To_Type_Name
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Pointer_To_Scalar_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Call
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression;

   overriding
   function Get_Default_JNI_Value
     (D : Ada_Pointer_To_Scalar_Data) return Dynamic_Expression;

   type Ada_Callback_Data is new Ada_Data with record
      null;
   end record;

   overriding
   function To_Type_Name (D : Ada_Callback_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Ada_Callback_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Ada_Callback_Data;
      Expression : Wrapping_Expression);

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_Callback_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Call (D : Ada_Callback_Data) return Dynamic_Expression;

   overriding
   procedure Adjust_Unit_For_Use
     (D : in out Ada_Callback_Data; Unit : Bound_Unit);

   overriding
   function Get_Default_JNI_Value
     (D : Ada_Callback_Data) return Dynamic_Expression;

   type Ada_JNI_Data is new Ada_Data with record
      null;
   end record;
   --  This type is used to map data that are already managed as Java objects
   --  in the code. No transformation is required.

   overriding
   function To_Type_Name (D : Ada_JNI_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   function To_JNI_Type_Name
     (D : Ada_JNI_Data) return Dynamic_Expression;
   --  See inherited documentation

   overriding
   procedure Create_Conversion_From_JNI
     (D          : in out Ada_JNI_Data;
      Expression : Wrapping_Expression);

   overriding
   procedure Create_Conversion_To_JNI
     (D          : in out Ada_JNI_Data;
      Expression : Wrapping_Expression);

   overriding
   function Get_JNI_Call (D : Ada_JNI_Data) return Dynamic_Expression;

   overriding
   function Get_Default_JNI_Value
     (D : Ada_JNI_Data) return Dynamic_Expression;

end Ada2Java.Bound_Elements.Subprograms.Data;
