------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2008-2012, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

with Ada2Java.Utils; use Ada2Java.Utils;

with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Simplifications; use Ada2Java.Simplifications;
with Ada2Java.Bound_Units;     use Ada2Java.Bound_Units;
with Ada2Java.Code_Wrappers;   use Ada2Java.Code_Wrappers;

with Ada2Java.Bound_Elements.Subprograms;
use Ada2Java.Bound_Elements.Subprograms;

with Ada2Java.Kernel; use Ada2Java.Kernel;

with Ada2Java.Bound_Elements.Common; use Ada2Java.Bound_Elements.Common;

package body Ada2Java.Bound_Elements.Exceptions is

   type Ada_Exception_Handler is new Code_Wrapper with record
      Exception_Name : Dynamic_Expression;
   end record;

   overriding
   procedure Wrap
     (Handler    : in out Ada_Exception_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Element_Bound : access Bound_Exception;
      Handle        : not null access Kernel.Kernel_Record)
   is
      Simple_View : constant Simple_Exception_View_Access :=
        Simple_Exception_View_Access (Element_Bound.Simple_Element);

      Unit : constant Bound_Unit := Get_Or_Create_Bound_Unit
        (Handle, Element_Bound.Simple_Element);

      Exception_Creator_View : constant Simple_Subprogram_View_Access :=
        Create (Element_Bound.Simple_Element);
      Exception_Creator_Handler  : aliased Ada_Exception_Handler;
      Exception_Creator_Wrappers : Wrapping_Map;
      Type_Of : constant Simple_Type_View_Access :=
        Get_Exception_Occurence (Handle);
      Except_Field : constant Wide_String := Get_Unique_Id;
   begin
      Unit.Java_File.Java_Parent_Class := To_Dynamic_Expression
        ("com.adacore.ajis.NativeException");
      String_Lists.Insert
        (Unit.Java_File.Interfaces_List,
         "com.adacore.ajis.internal.ada.AdaException");

      Append
        (Unit.Ada_Body_File.Elab_Part,
         New_Line &
         AJIS_Pckg & ".Reference_Exception ("
         & Get_Bound_Package_Name (Simple_View.Base_Package) & "."
         & Simple_View.Name & "'Identity, """
         & Ada2Java.Utils.Replace_Dots_By_Slashes
           (To_Wide_String (Unit.Java_File.Full_Class_Name))
         & """);");

      Append
        (Unit.Java_File.Public_Body_Part,
         New_Line & New_Line
         & Type_Of.Full_Java_Name & " " & Except_Field & ";"
         & New_Line & New_Line
         & "private " & Simple_View.Name & " (String message, int [] addr) {"
         & New_Line (1)
         & "super (message);"
         & New_Line & Except_Field & " = new "
         & Type_Of.Full_Java_Name
         & " (new com.adacore.ajis.internal.ada.AdaAccess (addr));"
         & New_Line & Except_Field & ".myAllocator = "
         & "com.adacore.ajis.IProxy.Allocator.DYNAMIC;"
         & New_Line & Except_Field & ".myOwner = "
         & "com.adacore.ajis.IProxy.Owner.PROXY;"
         & New_Line (-1)
         & "}"
         & New_Line & New_Line
         & "public " & Simple_View.Name & " ("
         & Get_String (Handle).Ref.Full_Java_Name & " message) {"
         & New_Line (1)
         & "super (message.toString());"
         & New_Line & Except_Field
         & " = createOccurrence (message);"
         & New_Line (-1)
         & "}"
         & New_Line
         & New_Line & "public int [] getExceptionAddr () {"
         & New_Line (1) & "return " & Except_Field & ".getAccess ();"
         & New_Line (-1) & "}");

      --   ??? There should be a string associated to the creation !
      --   ??? How do we manage garbage collection & co here ? It's not derived
      --   from native object...

      Exception_Creator_View.Name :=
        To_Dynamic_Expression ("createOccurrence");
      Exception_Creator_View.Returned_Type :=
        Create (Exception_Creator_View);
      Exception_Creator_View.Returned_Type.Type_Of.Ref :=
        Type_Of;
      Exception_Creator_View.Returned_Type.Type_Of :=
        Create_Access_To (Exception_Creator_View.Returned_Type.Type_Of, False);
      Exception_Creator_View.Parameters := new Parameter_Array (1 .. 1);
      Exception_Creator_View.Parameters (1) := Create (Exception_Creator_View);
      Exception_Creator_View.Parameters (1).Type_Of := Get_String (Handle);
      Exception_Creator_View.Parameters (1).Name :=
        To_Dynamic_Expression ("Message");
      Exception_Creator_View.Location := Simple_View.Location;

      Exception_Creator_Handler.Exception_Name :=
        Get_Bound_Package_Name (Simple_View.Base_Package) & "."
        & Simple_View.Name;

      Add_Wrapper
        (Map     => Exception_Creator_Wrappers,
         Wrapper => Exception_Creator_Handler'Unchecked_Access,
         Lang    => Ada_Lang,
         Target  => High_Sb,
         Context => Code_Node_Resolution);

      Bind_Subprogram
        (Handle   => Handle,
         View     => Exception_Creator_View,
         Unit     => Unit,
         Wrappers => Exception_Creator_Wrappers);

      Append
        (Unit.Ada_Spec_File.Dependencies_Part,
         New_Line & "with Ada.Exceptions; use Ada.Exceptions;");
   end Bind;

   -----------
   -- Wrap --
   -----------

   procedure Wrap
     (Handler    : in out Ada_Exception_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Parameters);

      Except_Name    : constant Wide_String := Get_Unique_Id;
   begin
      Append
        (Expression.Bloc.Code_Node,
         New_Line & "begin"
         & New_Line (1)
         & "raise " & Handler.Exception_Name & " with Message;"
         & New_Line (-1) & "exception"
         & New_Line (1) & "when "
         & Except_Name & " : " & Handler.Exception_Name & " =>"
         & New_Line (1) & "return Ada.Exceptions.Save_Occurrence ("
         & Except_Name & ");"
         & New_Line (-2) & "end;");
   end Wrap;

   procedure Bind_Standard_Exceptions
     (Handle : not null access Kernel.Kernel_Record)
   is
      procedure Create_Standard_Exception (Name : Wide_String);

      procedure Create_Standard_Exception (Name : Wide_String) is
         Error_View : Simple_Exception_View_Access;
      begin
         Error_View := Create (null);
         Initialize_Configurable_Properties (Handle, Error_View);
         Error_View.Base_Package := Get_Standard_Package (Handle);
         Error_View.Index := Create_Dummy_Unique_Location;
         Error_View.Name :=
           To_Dynamic_Expression (Name);
         Add_Bound_Element
           (Handle, Simple_Element_View_Access (Error_View));
         Add_Element
           (Get_Or_Create_Bound_Unit
              (Handle, Simple_Element_View_Access (Error_View)),
            Simple_Element_View_Access (Error_View));
      end Create_Standard_Exception;

   begin
      Create_Standard_Exception ("Constraint_Error");
      Create_Standard_Exception ("Program_Error");
      Create_Standard_Exception ("Storage_Error");
   end Bind_Standard_Exceptions;

end Ada2Java.Bound_Elements.Exceptions;
