------------------------------------------------------------------------------
--                                                                          --
--                      GNAT METRICS TOOLS COMPONENTS                       --
--                                                                          --
--           A S I S _ U L . M E T R I C S . F L O W _ G R A P H            --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- GNAT Metrics Toolset  is free software;  you can  redistribute it and/or --
-- modify it under terms of the  GNU General Public License as published by --
-- the Free Software Foundation;  either version 2, or (at your option) any --
-- later version.  GNAT Metrics Toolset is  distributed in the hope that it --
-- will be useful, but  WITHOUT ANY WARRANTY; without even the implied war- --
-- ranty of  MERCHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the --
-- GNU General Public License for more details.  You should have received a --
-- copy of the  GNU General Public License distributed with  GNAT; see file --
-- COPYING.  If not,  write to the  Free Software  Foundation,  51 Franklin --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- GNAT Metrics Toolset is maintained by AdaCore (http://www.adacore.com).  --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Containers.Ordered_Sets;
with Ada.Containers.Vectors;

with Asis.Extensions.Flat_Kinds; use Asis.Extensions.Flat_Kinds;
with Asis.Text;                  use Asis.Text;

with ASIS_UL.Debug;              use ASIS_UL.Debug;
with ASIS_UL.Output;             use ASIS_UL.Output;
with ASIS_UL.Utilities;          use ASIS_UL.Utilities;

package body ASIS_UL.Metrics.Flow_Graph is

   pragma Warnings (Off);

   -------------------------
   --  Control Flow Graph --
   -------------------------

   type CFG_Node_Id is new Natural;

   No_CFG_Node    : constant CFG_Node_Id := CFG_Node_Id'First;
   First_CFG_Node : constant CFG_Node_Id := No_CFG_Node + 1;
   Entry_Node_Id  : constant CFG_Node_Id := First_CFG_Node;
   Exit_Node_Id   : constant CFG_Node_Id := First_CFG_Node + 1;

   subtype Existing_CFG_Node_Id is CFG_Node_Id range
     First_CFG_Node .. CFG_Node_Id'Last;

   package Edges is new Ada.Containers.Ordered_Sets
     (Element_Type => Existing_CFG_Node_Id);

   type CFG_Node_Kinds is (
     Not_A_Node,
     Entry_Node,
     Exit_Node,
     Ordinary_Node);

   type CFG_Node_Record is record
      Line : Line_Number;
      Col  : Character_Position;
      --  Start of the corresponding statement

      Reduced : Boolean := False;

      In_Edges  : Edges.Set;
      Out_Edges : Edges.Set;

      Hash_Link : CFG_Node_Id := No_CFG_Node;
   end record;

   function "=" (L, R : CFG_Node_Record) return Boolean;
   --  Compares Line and Col positions only, it is enough

   package CFG_Nodes_Container is new Ada.Containers.Vectors
      (Index_Type   => Existing_CFG_Node_Id,
       Element_Type => CFG_Node_Record);

   CFG_Nodes_Table : CFG_Nodes_Container.Vector;

   ---------------------------------------------
   -- Hash table for control flow nodes table --
   ---------------------------------------------

   Hash_Num : constant Integer := 2**10;
   --  Number of headers in the hash table. The body longer then 2 ** 100 lines
   --  definitely is an exceptional case

   Hash_Max : constant Integer := Hash_Num - 1;
   --  Indexes in the hash header table run from 0 to Hash_Num - 1

   subtype Hash_Index_Type is Integer range 0 .. Hash_Max;
   --  Range of hash index values

   Hash_Table : array (Hash_Index_Type) of CFG_Node_Id;
   --  The hash table is used to locate existing entries in the nodes table.
   --  The entries point to the first nodes table entry whose hash value
   --  matches the hash code. Then subsequent nodes table entries with the
   --  same hash code value are linked through the Hash_Link fields.

   function Hash (El : Asis.Element) return Hash_Index_Type;
   --  Computes hash code for its argument as an Element line
   --  number mod Hash_Num

   ----------------------------------
   -- Control Flow Graph utilities --
   ----------------------------------

   procedure Add_Control_Flow
     (From_Statements : Asis.Element_List;
      Starting_From   : CFG_Node_Id);
   --  Adds a subgraph that corresponds to the statement sequence passed as the
   --  actual for From_Statements, Starting_From is the entry node for this
   --  subgraph.

   function Last_CFG_Node return CFG_Node_Id;
   --  Returtns the last node stored in the node table.

   procedure Print_Edges (S : Edges.Set);
   --  Prints out debug image of S, adds EOL to the end.

   function Register_Node
     (Statement     : Asis.Element;
      CFG_Node_Kind : CFG_Node_Kinds)
      return          CFG_Node_Id;
   --  Adds a new node to the node table and returns its Id as a result. Does
   --  not add any adges. A caller shoudl make sure that the node corresponding
   --  to this element has not been added to the graph yet.

   type CFG_Node_Record_Access is access CFG_Node_Record;

   function Table (N : CFG_Node_Id) return CFG_Node_Record_Access;
   --  Mimics the notation Instantce_Name.Table (N) in the instantiation of the
   --  GNAT Table package. Returns the (pointer to the) Node with the index N
   --  from CFG_Nodes_Table (see the body of the package). Raises
   --  Constraint_Error if a node with this index does not exsist.

   ----------------------------------------------------
   -- Building and processing the control flow graph --
   ----------------------------------------------------

   procedure Build_Control_Flow_Graph (Body_Element : Asis.Element);
   --  ???

   procedure Reduce_Control_Flow_Graph;
   --  ???

   function Compute_Complexity return Metric_Count;
   --  ???

   procedure Control_Flow_Graph_Debug_Image;
   --  Prints out the debug image of the control flow graph.

   ----------
   --  "=" --
   ----------

   function "=" (L, R : CFG_Node_Record) return Boolean is
   begin
      return L.Line = R.Line and then L.Col = R.Col;
   end "=";

   ----------------------
   -- Add_Control_Flow --
   ----------------------

   procedure Add_Control_Flow
     (From_Statements : Asis.Element_List;
      Starting_From   : CFG_Node_Id)
   is
      pragma Warnings (Off);
   begin
      null;
--      raise Program_Error;
      pragma Warnings (On);
   end Add_Control_Flow;

   ------------------------------
   -- Build_Control_Flow_Graph --
   ------------------------------

   procedure Build_Control_Flow_Graph (Body_Element : Asis.Element) is
      Next_Node : CFG_Node_Id;
      pragma Warnings (Off, Next_Node);
   begin
      --  Clear the CFG structures
      Hash_Table := (others => No_CFG_Node);

      for J in First_CFG_Node .. Last_CFG_Node loop
         Edges.Clear (Table (J).In_Edges);
         Edges.Clear (Table (J).Out_Edges);
      end loop;

      CFG_Nodes_Container.Clear (CFG_Nodes_Table);

      --  Add entry and exit nodes:
      Next_Node := Register_Node (Body_Element, Entry_Node);
      Next_Node := Register_Node (Body_Element, Exit_Node);

      Add_Control_Flow
        (Get_Statements (Body_Element),
         Starting_From => Entry_Node_Id);
   end Build_Control_Flow_Graph;

   ------------------------
   -- Compute_Complexity --
   ------------------------

   function Compute_Complexity return Metric_Count is
   begin
      return 0;
   end Compute_Complexity;

   ------------------------------------
   -- Control_Flow_Graph_Debug_Image --
   ------------------------------------

   procedure Control_Flow_Graph_Debug_Image is
   begin
      for N in First_CFG_Node .. Last_CFG_Node loop
         if not Table (N).Reduced then
            Info ("Node =" & N'Img &
                  "(" & Table (N).Line'Img & ":" &  Table (N).Col'Img & ")");
            Info_No_EOL ("Out edges:");
            Print_Edges (Table (N).Out_Edges);
            Info_No_EOL ("In edges :");
            Print_Edges (Table (N).Out_Edges);
            Info ("");
         end if;
      end loop;
   end Control_Flow_Graph_Debug_Image;

   ----------------------------------
   -- Compute_Essential_Complexity --
   ----------------------------------

   function Compute_Essential_Complexity
     (Body_Element : Asis.Element)
      return         Metric_Count
   is
      Result   : Metric_Count := 0;
      Arg_Span : Span;
   begin
      Build_Control_Flow_Graph (Body_Element);

      if Debug_Flag_1 then
         Arg_Span := Element_Span (Body_Element);
         Info ("  *** Original control flow graph ***");
         Info ("  built for " & Flat_Element_Kind (Body_Element)'Img &
               " (line" & Arg_Span.First_Line'Img & ")");
         Control_Flow_Graph_Debug_Image;
      end if;

      Reduce_Control_Flow_Graph;

      if Debug_Flag_1 then
         Info ("  *** Reduced control flow graph ***");
         Control_Flow_Graph_Debug_Image;
         Info ("");
      end if;

      Result := Compute_Complexity;
      pragma Assert (Result /= 2);

      return Result;

   end Compute_Essential_Complexity;

   ----------
   -- Hash --
   ----------

   function Hash (El : Asis.Element) return Hash_Index_Type is
   begin
      return Hash_Index_Type (Element_Span (El).First_Line) mod Hash_Num;
   end Hash;

   -------------------
   -- Last_CFG_Node --
   -------------------

   function Last_CFG_Node return CFG_Node_Id is
   begin
      return CFG_Nodes_Container.Last_Index (CFG_Nodes_Table);
   end Last_CFG_Node;

   -----------------
   -- Print_Edges --
   -----------------

   procedure Print_Edges (S : Edges.Set) is
      Next_Node : Edges.Cursor := Edges.First (S);
      use Edges;
   begin
      if Next_Node = Edges.No_Element then
         Info_No_EOL (" ...nothing...");
      else

         while Next_Node /= Edges.No_Element loop
            Info_No_EOL (Edges.Element (Next_Node)'Img);
            Next_Node := Edges.Next (Next_Node);
         end loop;

      end if;

      Info ("");

   end Print_Edges;

   -------------------------------
   -- Reduce_Control_Flow_Graph --
   -------------------------------

   procedure Reduce_Control_Flow_Graph is
   begin
      null;
   end Reduce_Control_Flow_Graph;

   -------------------
   -- Register_Node --
   -------------------

   function Register_Node
     (Statement     : Asis.Element;
      CFG_Node_Kind : CFG_Node_Kinds)
      return          CFG_Node_Id
   is
      New_Node :           CFG_Node_Record;
      Result   :           CFG_Node_Id;
      Res_Span : constant Span := Element_Span (Statement);
   begin
      New_Node.Reduced   := False;
      New_Node.In_Edges  := Edges.Empty_Set;
      New_Node.Out_Edges := Edges.Empty_Set;
      New_Node.Hash_Link := No_CFG_Node;

      if CFG_Node_Kind = Exit_Node then
         New_Node.Line := Res_Span.Last_Line;
         New_Node.Col  := Res_Span.Last_Column;
      else
         New_Node.Line := Res_Span.First_Line;
         New_Node.Col  := Res_Span.First_Column;
      end if;

      CFG_Nodes_Container.Append
        (Container => CFG_Nodes_Table,
         New_Item  => New_Node);

      Result := Last_CFG_Node;

      return Result;
   end Register_Node;

   -----------
   -- Table --
   -----------

   function Table (N : CFG_Node_Id) return CFG_Node_Record_Access is
      Result : CFG_Node_Record_Access;

      procedure Process (E : in out CFG_Node_Record);

      procedure Process (E : in out CFG_Node_Record) is
      begin
         Result := E'Unrestricted_Access;
      end Process;
   begin

      CFG_Nodes_Container.Update_Element
        (Container => CFG_Nodes_Table,
         Index     => N,
         Process   => Process'Access);

      return Result;
   end Table;

end ASIS_UL.Metrics.Flow_Graph;
