------------------------------------------------------------------------------
--                                   JNI                                    --
--                                                                          --
--                     Copyright (C) 2005-2007, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
------------------------------------------------------------------------------

with JNI;     use JNI;
with JNI_JVM; use JNI_JVM;

package body Java_Arrays is

   ---------------------
   -- New_Boolean_Arr --
   ---------------------

   function New_Boolean_Arr (Length : Int) return Boolean_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : J_Boolean_J_Array;
      Result : Boolean_Arr := new Boolean_Typ_Arr;
   begin
      --  Get the Environment

      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor

      Arr := New_Boolean_Array (Env, Length);
      if Arr = J_Null_Boolean_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Boolean_Arr;

   ---------------------
   -- New_Boolean_Arr --
   ---------------------

   function New_Boolean_Arr (Arr : J_Boolean_Array) return Boolean_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Boolean_Arr := new Boolean_Typ_Arr;
      A   : J_Boolean_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Boolean_Array (Env, Arr'Length);
      Set_Boolean_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Boolean_Arr;

   -----------------
   -- Get_Boolean --
   -----------------

   function Get_Boolean
     (Arr : access Boolean_Typ_Arr;
      Pos : Int) return Java_Primitives.Boolean
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Boolean_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Boolean_Array_Region
        (Env, J_Boolean_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Boolean;

   -----------------
   -- Set_Boolean --
   -----------------

   procedure Set_Boolean
     (Arr   : access Boolean_Typ_Arr;
      Pos   : Int;
      Value : Java_Primitives.Boolean)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Boolean_J_Array := Arr.Get_J_Object;
      V   : J_Boolean_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Boolean_Array_Region (Env, A, Pos, 1, V);
   end Set_Boolean;

   ------------------
   -- New_Char_Arr --
   ------------------

   function New_Char_Arr (Length : Int) return Char_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Char_J_Array;
      Result : Char_Arr := new Char_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Char_Array (Env, Length);
      if Arr = J_Null_Char_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Char_Arr;

   ------------------
   -- New_Char_Arr --
   ------------------

   function New_Char_Arr (Arr : J_Char_Array) return Char_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Char_Arr := new Char_Typ_Arr;
      A   : J_Char_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Char_Array (Env, Arr'Length);
      Set_Char_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Char_Arr;

   --------------
   -- Get_Char --
   --------------

   function Get_Char
     (Arr : access Char_Typ_Arr;
      Pos : Int) return Char
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Char_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Char_Array_Region
        (Env, J_Char_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Char;

   --------------
   -- Set_Char --
   --------------

   procedure Set_Char
     (Arr   : access Char_Typ_Arr;
      Pos   : Int;
      Value : Char)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Char_J_Array := Arr.Get_J_Object;
      V   : J_Char_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Char_Array_Region (Env, A, Pos, 1, V);
   end Set_Char;

   ------------------
   -- New_Byte_Arr --
   ------------------

   function New_Byte_Arr (Length : Int) return Byte_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Byte_J_Array;
      Result : Byte_Arr := new Byte_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Byte_Array (Env, Length);
      if Arr = J_Null_Byte_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Byte_Arr;

   ------------------
   -- New_Byte_Arr --
   ------------------

   function New_Byte_Arr (Arr : Byte_Array) return Byte_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Byte_Arr := new Byte_Typ_Arr;
      A   : J_Byte_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Byte_Array (Env, Arr'Length);
      Set_Byte_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Byte_Arr;

   --------------
   -- Get_Byte --
   --------------

   function Get_Byte
     (Arr : access Byte_Typ_Arr;
      Pos : Int) return Byte
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Byte_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Byte_Array_Region
        (Env, J_Byte_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Byte;

   --------------
   -- Set_Byte --
   --------------

   procedure Set_Byte
     (Arr   : access Byte_Typ_Arr;
      Pos   : Int;
      Value : Byte)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Byte_J_Array := Arr.Get_J_Object;
      V   : J_Byte_Array   := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Byte_Array_Region (Env, A, Pos, 1, V);
   end Set_Byte;

   -------------------
   -- New_Short_Arr --
   -------------------

   function New_Short_Arr (Length : Int) return Short_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Short_J_Array;
      Result : Short_Arr := new Short_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Short_Array (Env, Length);
      if Arr = J_Null_Short_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Short_Arr;

   -------------------
   -- New_Short_Arr --
   -------------------

   function New_Short_Arr (Arr : Short_Array) return Short_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Short_Arr := new Short_Typ_Arr;
      A   : J_Short_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Short_Array (Env, Arr'Length);
      Set_Short_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Short_Arr;

   ---------------
   -- Get_Short --
   ---------------

   function Get_Short
     (Arr : access Short_Typ_Arr;
      Pos : Int) return Short
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Short_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Short_Array_Region
        (Env, J_Short_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Short;

   ---------------
   -- Set_Short --
   ---------------

   procedure Set_Short
     (Arr   : access Short_Typ_Arr;
      Pos   : Int;
      Value : Short)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Short_J_Array := Arr.Get_J_Object;
      V   : J_Short_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Short_Array_Region (Env, A, Pos, 1, V);

   end Set_Short;

   -----------------
   -- New_Int_Arr --
   -----------------

   function New_Int_Arr (Length : Int) return Int_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Int_J_Array;
      Result : Int_Arr := new Int_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Int_Array (Env, Length);
      if Arr = J_Null_Int_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Int_Arr;

   -----------------
   -- New_Int_Arr --
   -----------------

   function New_Int_Arr (Arr : Int_Array) return Int_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Int_Arr := new Int_Typ_Arr;
      A   : J_Int_J_Array;
   begin
      --  Get the Environment
      I := Get_Env (VM   => Current_VM,
                    Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Int_Array (Env, Arr'Length);
      Set_Int_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Int_Arr;

   -------------
   -- Get_Int --
   -------------

   function Get_Int
     (Arr : access Int_Typ_Arr;
      Pos : Int) return Int
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A : J_Int_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Int_Array_Region
        (Env, J_Int_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Int;

   -------------
   -- Set_Int --
   -------------

   procedure Set_Int
     (Arr   : access Int_Typ_Arr;
      Pos   : Int;
      Value : Int)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Int_J_Array := Arr.Get_J_Object;
      V   : J_Int_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Int_Array_Region (Env, A, Pos, 1, V);
   end Set_Int;

   -------------------
   -- New_Float_Arr --
   -------------------

   function New_Float_Arr (Length : Int) return Float_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Float_J_Array;
      Result : Float_Arr := new Float_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Float_Array (Env, Length);
      if Arr = J_Null_Float_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Float_Arr;

   -------------------
   -- New_Float_Arr --
   -------------------

   function New_Float_Arr (Arr : Float_Array) return Float_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Float_Arr := new Float_Typ_Arr;
      A   : J_Float_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Float_Array (Env, Arr'Length);
      Set_Float_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Float_Arr;

   ---------------
   -- Get_Float --
   ---------------

   function Get_Float
     (Arr : access Float_Typ_Arr;
      Pos : Int) return Java_Primitives.Float
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A : J_Float_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Float_Array_Region
        (Env, J_Float_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Float;

   -------------
   -- Set_Float --
   -------------

   procedure Set_Float
     (Arr   : access Float_Typ_Arr;
      Pos   : Int;
      Value : J_Float)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Float_J_Array := Arr.Get_J_Object;
      V   : J_Float_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Float_Array_Region (Env, A, Pos, 1, V);
   end Set_Float;

   ------------------
   -- New_Long_Arr --
   ------------------

   function New_Long_Arr (Length : Int) return Long_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Long_J_Array;
      Result : Long_Arr := new Long_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Long_Array (Env, Length);
      if Arr = J_Null_Long_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Long_Arr;

   ------------------
   -- New_Long_Arr --
   ------------------

   function New_Long_Arr (Arr : Long_Array) return Long_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Long_Arr := new Long_Typ_Arr;
      A   : J_Long_J_Array;
   begin
      --  Get the Environment
      I := Get_Env (VM   => Current_VM,
                    Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Long_Array (Env, Arr'Length);
      Set_Long_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Long_Arr;

   --------------
   -- Get_Long --
   --------------

   function Get_Long
     (Arr : access Long_Typ_Arr;
      Pos : Int) return Long
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A : J_Long_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Long_Array_Region
        (Env, J_Long_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Long;

   --------------
   -- Set_Long --
   --------------

   procedure Set_Long
     (Arr   : access Long_Typ_Arr;
      Pos   : Int;
      Value : Long)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Long_J_Array := Arr.Get_J_Object;
      V   : J_Long_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Long_Array_Region (Env, A, Pos, 1, V);
   end Set_Long;

   --------------------
   -- New_Double_Arr --
   --------------------

   function New_Double_Arr (Length : Int) return Double_Arr is
      Env    : aliased JNI_Env_Access;
      I      : J_Int;
      Arr    : JNI.J_Double_J_Array;
      Result : Double_Arr := new Double_Typ_Arr;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      --  Call the constructor
      Arr := New_Double_Array (Env, Length);
      if Arr = J_Null_Double_J_Array then
         raise Program_Error;
      end if;

      Result.Set_J_Object (J_Object (Arr));

      return Result;
   end New_Double_Arr;

   --------------------
   -- New_Double_Arr --
   --------------------

   function New_Double_Arr (Arr : Double_Array) return Double_Arr is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      R   : Double_Arr := new Double_Typ_Arr;
      A   : J_Double_J_Array;
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      A := New_Double_Array (Env, Arr'Length);
      Set_Double_Array_Region (Env, A, 0, Arr'Length, Arr);
      R.Set_J_Object (J_Object (A));

      return R;
   end New_Double_Arr;

   ----------------
   -- Get_Double --
   ----------------

   function Get_Double
     (Arr : access Double_Typ_Arr;
      Pos : Int) return Double
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A : J_Double_Array (1 .. 1);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Get_Double_Array_Region
        (Env, J_Double_J_Array (Arr.Get_J_Object), Pos, 1, A);

      return A (Integer (Pos));
   end Get_Double;

   ----------------
   -- Set_Double --
   ----------------

   procedure Set_Double
     (Arr   : access Double_Typ_Arr;
      Pos   : Int;
      Value : Double)
   is
      Env : aliased JNI_Env_Access;
      I   : J_Int;
      A   : J_Double_J_Array := Arr.Get_J_Object;
      V   : J_Double_Array := (1 => Value);
   begin
      --  Get the Environment
      I := Get_Env
        (VM   => Current_VM,
         Penv => Env'Access);

      if I /= JNI_OK then
         raise Program_Error;
      end if;

      Set_Double_Array_Region (Env, A, Pos, 1, V);
   end Set_Double;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Boolean_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Char_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Byte_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Short_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Int_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Long_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Float_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Double_Typ_Arr is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Boolean_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Char_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Byte_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Short_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Int_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Long_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Float_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

   ---------------------
   -- JNI_Constructor --
   ---------------------

   function JNI_Constructor
     (Params : not null access Parameters) return Double_Typ_Arr_2 is
   begin
      return (Java.Lang.Object.Root_Array with null record);
   end JNI_Constructor;

end Java_Arrays;
