------------------------------------------------------------------------------
--                                                                          --
--                     ASIS UTILITY LIBRARY COMPONENTS                      --
--                                                                          --
--     A S I S _ U L . S O U R C E _ T A B L E . P R O C E S S I N G .      --
--                             F I N A L I Z E                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                   Copyright (C) 2007-2011, AdaCore                       --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not, write to the Free Software Foundation, 59 Temple Place  --
--  - Suite 330, Boston,                                                    --
--                                                                          --
-- ASIS UL is maintained by ACT Europe (http://www.act-europe.fr).          --
--                                                                          --
------------------------------------------------------------------------------

--  This is a placeholder for the routine that implements a tool-specific
--  finalization for a tool that has a list of sources as a parameter. This
--  initialization should be performed after finishing the individual
--  processing of all the sources and it is supposed to collect and to analyze
--  some global information. Because this is tool-specific, this placeholder
--  does nothing.

with Ada.Characters.Conversions; use Ada.Characters;
with Ada.Directories;            use Ada.Directories;
with Ada.IO_Exceptions;
with Ada.Strings.Wide_Unbounded; use Ada.Strings.Wide_Unbounded;
with Ada.Wide_Text_IO;           use Ada.Wide_Text_IO;
with Ada.Characters.Handling; use Ada.Characters.Handling;
with Ada.Text_IO;

with Asis.Implementation;
with Asis.Iterator;
with Asis.Ada_Environments;
with Asis.Compilation_Units;
with Asis.Elements;          use Asis.Elements;

with GNAT.Directory_Operations; use GNAT.Directory_Operations;
with GNAT.Strings;

with Ada2Java;                     use Ada2Java;
with Ada2Java.Bound_Elements;     use Ada2Java.Bound_Elements;
with Ada2Java.Bound_Units;        use Ada2Java.Bound_Units;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Config;              use Ada2Java.Config;
with Ada2Java.Kernel;              use Ada2Java.Kernel;
with Ada2Java.Packages;            use Ada2Java.Packages;
with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Code_Templates;      use Ada2Java.Code_Templates;
with Ada2Java.Bound_Elements.Exceptions;
use Ada2Java.Bound_Elements.Exceptions;

separate (ASIS_UL.Source_Table.Processing)
procedure Finalize is

   function To_Path (Package_Name : String) return String;
   --  Converts a package name into a java path (basically, changes every . by
   --  Dir_Separator.

   function To_Path (Package_Name : String) return String is
      Returned : String := Package_Name;
   begin
      for J in Returned'Range loop
         if Returned (J) = '.' then
            Returned (J) := Dir_Separator;
         end if;
      end loop;

      return Returned;
   end To_Path;

   use Bound_Unit_List_Pckg;

   It : Bound_Units_Cursor;

   Conf_Class_Path : Unbounded_Wide_String;
   Conf_Ada_Dir    : Unbounded_Wide_String;

   Full_File_Path  : Unbounded_Wide_String;

   Library_Binding_Unit : constant Bound_Unit :=
     Get_Library_Binding_Unit (The_Kernel);
begin
   Bind_Standard_Exceptions (The_Kernel);

   Append
     (Library_Binding_Unit.Java_File.Public_Body_Part,
      New_Line &
      "public static final com.adacore.ajis.ILock lock" &
      " = new com.adacore.ajis.DefaultLock ();" &
      New_Line & "private static boolean fLoaded = false;" &
      New_Line &
      New_Line &
      "private static native void startLibrary ();" &
      New_Line &
      New_Line &
      "public static void load () {" &
      New_Line (1) & "if (!fLoaded) {"
      & New_Line (1) & "fLoaded = true;");

   if GNAT.Strings."/=" (Ada2Java.Native_Library_Name, null) then
      Append
        (Library_Binding_Unit.Java_File.Public_Body_Part,
         New_Line & "com.adacore.ajis.AJISLibrary.load ();" &
         New_Line & "System.loadLibrary (""" &
         Ada.Characters.Conversions.To_Wide_String
           (Ada2Java.Native_Library_Name.all) & """);" &
         New_Line & "startLibrary ();");
   end if;

   Append
     (Library_Binding_Unit.Java_File.Public_Body_Part,
      New_Line (-1) & "}" &
      New_Line (-1) & "}");

   Trace ("Binding files...", Limited_Verbosity);

   Generate_Missing_Bound_Units (The_Kernel);

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);
   --  Workaround for F802-006

   --  Check for name clashes

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Unit : constant Bound_Unit := Bound_Unit_List_Pckg.Element (It);
      begin
         Check_Name_Clashes (Unit);
      end;

      It := Next (It);
   end loop;

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);
   --  Workaround for F802-006

   --  Adjust the consistency of all units elements

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Unit : constant Bound_Unit := Bound_Unit_List_Pckg.Element (It);
      begin
         Adjust_Consistency (Unit);
      end;

      It := Next (It);
   end loop;

   --  Generate all units bindings

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Unit : constant Bound_Unit := Bound_Unit_List_Pckg.Element (It);
      begin
         Generate_Bindings (Unit);
      end;

      It := Next (It);
   end loop;

   --  Create references to all the bound code so that it gets compiled by
   --  Ada and Java.

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);

   Append
     (Library_Binding_Unit.Java_File.Private_Body_Part,
      New_Line & New_Line & "static {" & Indent (1));

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Package_Name : constant Wide_String := Key (It);
         Unit         : constant Bound_Unit :=
           Bound_Unit_List_Pckg.Element (It);
      begin
         if Unit.Ada_Spec_File /= null
           and then Unit.Create_Ada_Spec_File
         then
            Append
              (Library_Binding_Unit.Ada_Body_File.Dependencies_Part,
               New_Line & "with " & Package_Name & ";");
         end if;

         if Unit.Java_File /= null and then Unit.Create_Java_File then
            Append
              (Library_Binding_Unit.Java_File.Private_Body_Part,
               New_Line
               & Unit.Java_File.Full_Class_Name & " " & Get_Unique_Id & ";");
         end if;

         It := Next (It);
      end;
   end loop;

   Append
     (Library_Binding_Unit.Java_File.Private_Body_Part,
      New_Line (-1) & "}");

   --  Generate the dynamic links using register natives if needed

   if Ada2Java.Link_Method = Register_Natives then
      Append
        (Library_Binding_Unit.Ada_Spec_File.Public_Body_Part,
         New_Line & New_Line
         & "procedure Register_All_Natives (" & Env_Parameter_Name & " : "
         & JNI_Pckg & ".JNI_Env_Access);");

      Append
        (Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
         New_Line & New_Line
         & "procedure Register_All_Natives (" & Env_Parameter_Name & " : "
         & JNI_Pckg & ".JNI_Env_Access) is"
         & New_Line & "begin"
         & Indent (1)
         & Generate_Register_Native_Calls (The_Kernel)
         & New_Line (-1) & "end Register_All_Natives;");
   end if;

   --  Generates the global initialization method

   Append
     (Library_Binding_Unit.Ada_Spec_File.Public_Body_Part,
      New_Line & New_Line
      & "procedure Initialize_Library (" & Env_Parameter_Name & " : "
      & JNI_Pckg & ".JNI_Env_Access);");

   Append
     (Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
      New_Line & New_Line
      & "procedure Initialize_Library (" & Env_Parameter_Name & " : "
      & JNI_Pckg & ".JNI_Env_Access) is"
      & New_Line & "begin"
      & New_Line (1) & AJIS_Pckg & ".Initialize_JNI ("
      & Env_Parameter_Name & ");");

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Unit : constant Bound_Unit := Bound_Unit_List_Pckg.Element (It);
      begin
         if Unit.Ada_Spec_File /= null
           and then Unit.Create_Ada_Spec_File
         then
            Append
              (Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
               New_Line
               & Unit.Ada_Spec_File.Package_Name
               & "." & Initialize_Unit_Name
               & " (" & Env_Parameter_Name & ");");
         end if;

         It := Next (It);
      end;
   end loop;

   Append
     (Library_Binding_Unit.Ada_Body_File.Private_Body_Part,
      New_Line (-1) & "end Initialize_Library;");

   --  Create Ada & Java glue & specs.

   It := Bound_Unit_List_Pckg.First
     (Get_Binding_Units_DB (The_Kernel.all'Access).all);
   --  Workaround for F802-006

   while It /= Bound_Unit_List_Pckg.No_Element loop
      declare
         Unit : constant Bound_Unit := Bound_Unit_List_Pckg.Element (It);
         Package_Name : constant String := Conversions.To_String (Key (It));
         Ada_Base_File_Name : String := Package_Name;
         File_Contents : Dynamic_Expression;

         procedure Write_File (Path : String; Source : Binding_File);

         procedure Write_File (Path : String; Source : Binding_File) is
            File : Ada.Wide_Text_IO.File_Type;
         begin
            Trace
              ("Create file "
               & Conversions.To_Wide_String (Path), Limited_Verbosity);

            Create (File, Out_File, Path);

            File_Contents := Assemble (Source.all);

            Write (File_Contents, File);

            Close (File);
         end Write_File;
      begin
         if Get_Configuration (Unit.Base_Pckg) /= Null_Configuration then
            Conf_Class_Path := To_Unbounded_Wide_String
              (Get_Java_Class_Path (Get_Configuration (Unit.Base_Pckg)));
            Conf_Ada_Dir := To_Unbounded_Wide_String
              (Get_Ada_Output_Directory (Get_Configuration (Unit.Base_Pckg)));
         else
            Conf_Class_Path := Default_Class_Path;
            Conf_Ada_Dir := Default_Ada_Base_Folder;
         end if;

         for J in Ada_Base_File_Name'Range loop
            if Ada_Base_File_Name (J) = '.' then
               Ada_Base_File_Name (J) := '-';
            else
               Ada_Base_File_Name (J) := To_Lower (Ada_Base_File_Name (J));
            end if;
         end loop;

         Create_Path
           (Conversions.To_String (To_Wide_String (Conf_Ada_Dir)));
         Create_Path
           (Conversions.To_String (To_Wide_String (Conf_Class_Path)));

         if Unit.Ada_Spec_File /= null and then Unit.Create_Ada_Spec_File then
            Write_File
              (Conversions.To_String (To_Wide_String (Conf_Ada_Dir))
               & Dir_Separator & Ada_Base_File_Name & ".ads",
               Unit.Ada_Spec_File);
         end if;

         if Unit.Ada_Body_File /= null and then Unit.Create_Ada_Body_File then
            Write_File
              (Conversions.To_String (To_Wide_String (Conf_Ada_Dir))
               & Dir_Separator & Ada_Base_File_Name & ".adb",
               Unit.Ada_Body_File);
         end if;

         if Unit.Java_File /= null and then Unit.Create_Java_File then
            declare
               Java_Path : constant String :=
                 Conversions.To_String (To_Wide_String (Conf_Class_Path))
                 & Dir_Separator
                 & To_Path
                 (Conversions.To_String
                    (To_Wide_String (Unit.Java_File.Package_Name)))
                 & Dir_Separator;
            begin
               Create_Path (Java_Path);

               Write_File
                 (Java_Path
                  & Conversions.To_String
                    (To_Wide_String (Unit.Java_File.Class_Name)) & ".java",
                  Unit.Java_File);
            end;
         end if;

         Next (It);

      exception
         when Ada.IO_Exceptions.Name_Error =>
            Trace ("Cannot create file: "
                   & To_Wide_String (Full_File_Path), Errors_Only);
            raise;
      end;
   end loop;

   --  Generate support for a main subprogram or a native library

   if GNAT.Strings."/=" (Ada2Java.Native_Library_Name, null) then
      declare
         File : Ada.Text_IO.File_Type;
         Project_Path : constant String := Conversions.To_String
           (To_Wide_String (Conf_Ada_Dir))
           & Dir_Separator & Ada2Java.Native_Library_Name.all & ".gpr";

         Found : Boolean;
         Do_Comma : Boolean := False;

         procedure Generate_Source_List
           (All_Sources_Dirs : GNAT.Strings.String_List);
         --  Generate the "for Source_Dirs" statement of the project, according
         --  to the source dirs given in parameter. Ada base folder will be
         --  added if needed, and standard source directories removed.

         procedure Generate_Source_List
           (All_Sources_Dirs : GNAT.Strings.String_List)
         is
            Default_Sources_Dirs : constant GNAT.Strings.String_List :=
              Get_Source_Dirs ("ajis.gpr");
         begin
            Ada.Text_IO.Put (File, "   for Source_Dirs use (");

            for J in All_Sources_Dirs'Range loop
               Found := False;

               for K in Default_Sources_Dirs'Range loop
                  if All_Sources_Dirs (J).all
                    = Default_Sources_Dirs (K).all
                  then
                     Found := True;

                     exit;
                  end if;
               end loop;

               if not Found then
                  if Do_Comma then
                     Ada.Text_IO.Put (File, ", ");
                  end if;

                  Ada.Text_IO.Put
                    (File, """" & All_Sources_Dirs (J).all & """");

                  Do_Comma := True;
               end if;
            end loop;

            if To_Wide_String (Ada2Java.Default_Ada_Base_Folder) /= "" then
               if Do_Comma then
                  Ada.Text_IO.Put (File, ", ");

                  Ada.Text_IO.Put
                    (File,
                     """"
                     & Ada.Characters.Conversions.To_String
                       (To_Wide_String (Ada2Java.Default_Ada_Base_Folder))
                     & """");
               end if;
            end if;

            Ada.Text_IO.Put (File, ");" & ASCII.LF);
         end Generate_Source_List;

      begin
         Trace
           ("Create library project "
            & Conversions.To_Wide_String (Project_Path), Limited_Verbosity);

         Ada.Text_IO.Create (File, Ada.Text_IO.Out_File, Project_Path);

         Ada.Text_IO.Put
           (File,
            "with ""jni"";" & ASCII.LF &
            "with ""ajis"";" & ASCII.LF &
            "" & ASCII.LF &
            "project " & Ada2Java.Native_Library_Name.all &
            " is" & ASCII.LF & "" & ASCII.LF &
            "   for Object_Dir use ""obj"";" & ASCII.LF &
            "" & ASCII.LF);

         if ASIS_UL.Common.Project_File /= null then
            Generate_Source_List
              (Get_Source_Dirs (ASIS_UL.Common.Project_File.all));
         else
            Generate_Source_List ((1 => Ada2Java.Initial_Dir));
         end if;

         Ada.Text_IO.Put
           (File,
            "" & ASCII.LF &
            "   for Library_Name use """ &
            Ada2Java.Native_Library_Name.all & """;" & ASCII.LF &
            "   for Library_Kind use ""dynamic"";" & ASCII.LF &
            "   for Library_Dir use ""lib"";" & ASCII.LF &
            "   for Library_Auto_Init use ""false"";" & ASCII.LF &
            "   for Library_Interface use (""" &
            Ada.Characters.Conversions.
              To_String (Ada2Java.Bound_Package_Root.all) & """);" & ASCII.LF &
            "" & ASCII.LF &
            "   package Compiler is" & ASCII.LF &
            "      for Default_Switches use AJIS.Compiler'Default_Switches;" &
            ASCII.LF &
            "   end Compiler;" & ASCII.LF &
            "" & ASCII.LF &
            "   case JNI.OS is" & ASCII.LF &
            "      when ""Windows_NT"" =>" & ASCII.LF &
            "         for Shared_Library_Prefix use """";" & ASCII.LF &
            "      when others =>" & ASCII.LF &
            "         null;" & ASCII.LF &
            "   end case;" & ASCII.LF &
            "end "& Ada2Java.Native_Library_Name.all & ";");

         Ada.Text_IO.Close (File);
      end;
   elsif GNAT.Strings."/=" (Ada2Java.Main_Unit_Name, null) then
      declare
         Main_Template : constant Code_Template := Load_Template
           ("main_body");
         Main_Params : Template_Parameter_Association.Map;

         use Template_Parameter_Association;

         Lower_Case_Main : constant Wide_String := Conversions.To_Wide_String
           (To_Lower (Ada2Java.Main_Unit_Name.all));
         Project_File, Main_File : Ada.Wide_Text_IO.File_Type;
         Project_Path : constant Wide_String := To_Wide_String (Conf_Ada_Dir)
           & Conversions.To_Wide_String (String'(1 => Dir_Separator))
           & Lower_Case_Main & ".gpr";
         Main_File_Path : constant Wide_String := To_Wide_String (Conf_Ada_Dir)
           & Conversions.To_Wide_String (String'(1 => Dir_Separator))
           & Lower_Case_Main & ".adb";

         Project_Text : Dynamic_Expression :=
           New_Dynamic_Expression;
      begin
         Trace
           ("Create main project " & Project_Path, Limited_Verbosity);

         Append
           (Project_Text,
            "with ""jni"";"
            & New_Line & "with ""ajis"";");

         if ASIS_UL.Common.Project_File /= null then
            Append
              (Project_Text,
               New_Line & "with """
               & Conversions.To_Wide_String (ASIS_UL.Common.Project_File.all)
               & """;");
         end if;

         Append
           (Project_Text,
            New_Line & New_Line
            & "project "
            & Conversions.To_Wide_String (Ada2Java.Main_Unit_Name.all)
            &  " is"
            & New_Line (1) & New_Line
            & "for Object_Dir use ""obj"";");

         Append
           (Project_Text,
            New_Line & "for Main use (""" & Lower_Case_Main & """);"
            & New_Line & New_Line
            & "package Compiler is"
            & New_Line (1)
            & "for Default_Switches use AJIS.Compiler'Default_Switches;"
            & New_Line (-1) & "end Compiler;"
            & New_Line & New_Line
            & "package Linker is"
            & New_Line (1) & "case JNI.OS is"
            & New_Line (1) & "when ""Darwin"" =>"
            & New_Line (1)
            & "for Default_Switches (""Ada"") use (""-Wl,-framework,JavaVM"");"
            & New_Line (-1) & "when others =>"
            & New_Line (1) & "for Default_Switches (""Ada"") use (""-ljvm"");"
            & New_Line (-1) & "end case;"
            & New_Line (-1) & "end Linker;"
            & New_Line
            & New_Line (-1) & "end "
            & Conversions.To_Wide_String (Ada2Java.Main_Unit_Name.all) & ";");

         Ada.Wide_Text_IO.Create
           (Project_File,
            Ada.Wide_Text_IO.Out_File,
            Conversions.To_String (Project_Path));

         Write (Project_Text, Project_File);

         Ada.Wide_Text_IO.Close (Project_File);

         Insert
           (Main_Params,
            "library_package",
            Library_Binding_Unit.Ada_Pckg_Name);
         Insert
           (Main_Params,
            "main_name",
            To_Dynamic_Expression
              (Conversions.To_Wide_String (Ada2Java.Main_Unit_Name.all)));

         if Ada2Java.Java_Main_Class /= null then
            Insert
              (Main_Params,
               "main_java_class",
                To_Dynamic_Expression
                 (Replace_Dots_By_Slashes
                    (Conversions.To_Wide_String
                    (Ada2Java.Java_Main_Class.all))));
         else
            Insert
              (Main_Params,
               "main_java_class",
               To_Dynamic_Expression
                 (Replace_Dots_By_Slashes (Get_Java_Base_Package
                  (Get_Default_Configuration (The_Kernel))) & "/"
                  & Conversions.To_Wide_String (Ada2Java.Main_Unit_Name.all)));
         end if;

         Ada.Wide_Text_IO.Create
           (Main_File,
            Ada.Wide_Text_IO.Out_File,
            Conversions.To_String (Main_File_Path));

         Write (Instantiate (Main_Template, Main_Params), Main_File);

         Ada.Wide_Text_IO.Close (Main_File);
      end;
   end if;

end Finalize;
