-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2010, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Wide_Characters.Unicode; use Ada.Wide_Characters.Unicode;
with Ada.Characters.Conversions;  use Ada.Characters.Conversions;
with Ada.Strings.Fixed;           use Ada.Strings.Fixed; use Ada.Strings;

with Asis.Declarations;      use Asis.Declarations;
with Asis.Definitions;       use Asis.Definitions;
with Asis.Elements;          use Asis.Elements;
with Asis.Text;              use Asis.Text;

with ASIS_UL.Strings; use ASIS_UL.Strings;

with Ada2Java.Simplifications; use Ada2Java.Simplifications;
with Ada2Java.Packages;        use Ada2Java.Packages;

with GNAT.Regpat; use GNAT.Regpat;
with GNAT.Expect; use GNAT.Expect;

with Ada.Text_IO;

package body Ada2Java.Utils is

   ----------------------------------
   -- Create_Dummy_Unique_Location --
   ----------------------------------

   Current_Dummy_Loc : Integer := 0;

   function Create_Dummy_Unique_Location return Element_Index is
      Result : Element_Index := Anonymous_Element_Index;
   begin
      Result.Location := new String'("dummy:0:" & Current_Dummy_Loc'Img);
      Current_Dummy_Loc := Current_Dummy_Loc + 1;

      return Result;
   end Create_Dummy_Unique_Location;

   ----------------------
   -- To_Element_Index --
   ----------------------

   function To_Element_Index (Element : Asis.Element) return Element_Index is
   begin
      if Is_Standard_Package
        (Unit_Declaration (Enclosing_Compilation_Unit (Element)))
      then
         return
           (Asis.Extensions.Flat_Kinds.Not_An_Element,
            new String'
              ("Standard." & To_String (Get_First_Name (Element))));
      else
         return
           (Flat_Element_Kind (Element),
            new String'(Build_GNAT_Location (Element)));
      end if;
   end To_Element_Index;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Element_Index) return Boolean is
   begin
      return Left.Kind < Right.Kind
        or else
          (Left.Kind = Right.Kind
           and then Left.Location.all < Right.Location.all);
   end "<";

   ---------
   -- "=" --
   ---------

   function "=" (Left, Right : Element_Index) return Boolean is
   begin
      return Left.Kind = Right.Kind
        and then Left.Location.all = Right.Location.all;
   end "=";

   ----------
   -- Free --
   ----------

   procedure Free (Index : in out Element_Index) is
   begin
      Free (Index.Location);
   end Free;

   -------------------
   -- Remove_Blanks --
   -------------------

   function Remove_Blanks (Str : Wide_String) return Wide_String is
   begin
      for J in Str'Range loop
         if not Is_Space (Str (J)) then
            return Str (J .. Str'Last);
         end if;
      end loop;

      return "";
   end Remove_Blanks;

   ---------------
   -- Mangle_ID --
   ---------------

   function Mangle_ID (Str : Wide_String) return Wide_String is
      Size : Integer := Str'Length;
   begin
      for J in Str'Range loop
         case Str (J) is
            when '_' | ';' | '[' =>
               Size := Size + 1;

            when '$' =>
               Size := Size + 5;

            when others =>
               null;

         end case;
      end loop;

      declare
         Mangled       : Wide_String (1 .. Size);
         Mangled_Index : Natural := 1;
      begin
         for J in Str'Range loop
            if Str (J) = '.' or else Str (J) = '/' then
               Mangled (Mangled_Index) := '_';
               Mangled_Index := Mangled_Index + 1;
            elsif Str (J) = '_' then
               Mangled (Mangled_Index) := '_';
               Mangled (Mangled_Index + 1) := '1';
               Mangled_Index := Mangled_Index + 2;
            elsif Str (J) = ';' then
               Mangled (Mangled_Index) := '_';
               Mangled (Mangled_Index + 1) := '2';
               Mangled_Index := Mangled_Index + 2;
            elsif Str (J) = '[' then
               Mangled (Mangled_Index) := '_';
               Mangled (Mangled_Index + 1) := '3';
               Mangled_Index := Mangled_Index + 2;
            elsif Str (J) = '$' then
               Mangled (Mangled_Index .. Mangled_Index + 5) := "_00024";
               Mangled_Index := Mangled_Index + 6;
            else
               Mangled (Mangled_Index) := Str (J);
               Mangled_Index := Mangled_Index + 1;
            end if;
         end loop;

         return Mangled;
      end;
   end Mangle_ID;

   -----------------------------
   -- Replace_Dots_By_Slashes --
   -----------------------------

   function Replace_Dots_By_Slashes (Str : Wide_String) return Wide_String is
      Result : Wide_String := Str;
   begin
      for J in Result'Range loop
         if Result (J) = '.' then
            Result (J) := '/';
         end if;
      end loop;

      return Result;
   end Replace_Dots_By_Slashes;

   ---------------------
   -- Get_Root_Parent --
   ---------------------

   function Get_Root_Parent
     (Decl : Asis.Declaration) return Asis.Declaration
   is
      Parent : Asis.Element := Decl;
      Element_Type_Kind : Type_Kinds := Type_Kind
        (Type_Declaration_View (Parent));
   begin
      loop
         while Definition_Kind (Parent) = Not_A_Definition
           and then Declaration_Kind (Parent) = A_Subtype_Declaration
         loop
            Parent := Corresponding_First_Subtype (Parent);
            Element_Type_Kind := Type_Kind
              (Type_Declaration_View (Parent));
         end loop;

         exit when Element_Type_Kind /= A_Derived_Type_Definition
           and then Element_Type_Kind /= A_Derived_Record_Extension_Definition;

         Parent := Corresponding_Parent_Subtype
           (Type_Declaration_View (Parent));
         Element_Type_Kind := Type_Kind
           (Type_Declaration_View (Parent));
      end loop;

      return Parent;
   end Get_Root_Parent;

   --------------------
   -- Get_First_Name --
   --------------------

   function Get_First_Name (Obj : Asis.Element) return Wide_String is
   begin
      if Element_Kind (Obj) = A_Defining_Name then
         return Defining_Name_Image (Obj);
      elsif Element_Kind (Obj) /= A_Declaration then
         return Remove_Blanks (Element_Image (Obj));
      else
         return Remove_Blanks (Defining_Name_Image (Names (Obj)(1)));
      end if;
   end Get_First_Name;

   ------------------
   -- Get_Location --
   ------------------

   function Get_Location (Obj : Asis.Element) return Wide_String is
   begin
      return To_Wide_String (Build_GNAT_Location (Obj));
   end Get_Location;

   ------------------------
   -- Get_Full_Ada_Name --
   ------------------------

   function Get_Full_Ada_Name
     (Handle : not null access Ada2Java.Kernel.Kernel_Record;
      Obj : Asis.Element)
      return Wide_String
   is
      Pckg : Package_Handle;
   begin
      if Is_Standard_Package
        (Unit_Declaration (Enclosing_Compilation_Unit (Obj)))
      then
         return "Standard." & Get_First_Name (Obj);
      else
         Pckg := Get_Or_Create_Package (Handle, Obj);

         return Get_Bound_Package_Name (Pckg) & "." & Get_First_Name (Obj);
      end if;
   end Get_Full_Ada_Name;

   -------------------
   -- Get_Unique_Id --
   -------------------

   function Get_Unique_Id return Wide_String is
   begin
      Id_Counter := Id_Counter + 1;

      return Unique_Name_Prefix & To_Wide_String (Trim (Id_Counter'Img, Left));
   end Get_Unique_Id;

   ------------------
   -- Dump_Element --
   ------------------

   function Dump_Element (E : Asis.Element) return String is
      Buffer : String (1 .. 5000);
      Buffer_Pos : Integer := 0;

      procedure Add_Str (Str : String);

      procedure Add_Str (Str : String) is
      begin
         Buffer (Buffer_Pos + 1 .. Buffer_Pos + Str'Length) := Str;
         Buffer_Pos := Buffer_Pos + Str'Length;
      end Add_Str;

   begin
      Add_Str
        ("[" & Build_GNAT_Location (E)
         & " " & To_String (Get_First_Name (E)) & "]: ");
      Add_Str
        ("Element_Kind  (" & Element_Kinds'Image (Element_Kind (E)) & ")");

      if Pragma_Kind (E) /= Not_A_Pragma then
         Add_Str (", Pragma_Kind ("
              & Pragma_Kinds'Image (Pragma_Kind (E))
              & ")");
      end if;

      if Defining_Name_Kind (E) /= Not_A_Defining_Name then
         Add_Str (", Defining_Name_Kind ("
              & Defining_Name_Kinds'Image (Defining_Name_Kind (E))
              & ")");
      end if;

      if Declaration_Kind (E) /= Not_A_Declaration then
         Add_Str (", Declaration_Kind ("
              & Declaration_Kinds'Image (Declaration_Kind (E))
              & ")");
      end if;

      if Trait_Kind (E) /= Not_A_Trait then
         Add_Str (", Trait_Kind ("
              & Trait_Kinds'Image (Trait_Kind (E))
              & ")");
      end if;

      if Declaration_Origin (E) /= Not_A_Declaration_Origin then
         Add_Str (", Declaration_Origin ("
              & Declaration_Origins'Image (Declaration_Origin (E))
              & ")");
      end if;

      if Mode_Kind (E) /= Not_A_Mode then
         Add_Str (", Mode_Kind ("
              & Mode_Kinds'Image (Mode_Kind (E))
              & ")");
      end if;

      if Definition_Kind (E) /= Not_A_Definition then
         Add_Str (", Definition_Kind ("
              & Definition_Kinds'Image (Definition_Kind (E))
              & ")");
      end if;

      if Type_Kind (E) /= Not_A_Type_Definition then
         Add_Str (", Type_Kind ("
              & Type_Kinds'Image (Type_Kind (E))
              & ")");
      end if;

      if Formal_Type_Kind (E) /= Not_A_Formal_Type_Definition then
         Add_Str (", Formal_Type_Kind ("
              & Formal_Type_Kinds'Image (Formal_Type_Kind (E))
              & ")");
      end if;

      if Access_Type_Kind (E) /= Not_An_Access_Type_Definition then
         Add_Str (", Access_Type_Kind ("
              & Access_Type_Kinds'Image (Access_Type_Kind (E))
              & ")");
      end if;

      if Access_Definition_Kind (E) /= Not_An_Access_Definition then
         Add_Str (", Access_Definition_Kind ("
                  & Access_Definition_Kinds'Image (Access_Definition_Kind (E))
                  & ")");
      end if;

      if Root_Type_Kind (E) /= Not_A_Root_Type_Definition then
         Add_Str (", Root_Type_Kind ("
              & Root_Type_Kinds'Image (Root_Type_Kind (E))
              & ")");
      end if;

      if Constraint_Kind (E) /= Not_A_Constraint then
         Add_Str (", Constraint_Kind ("
              & Constraint_Kinds'Image (Constraint_Kind (E))
              & ")");
      end if;

      if Discrete_Range_Kind (E) /= Not_A_Discrete_Range then
         Add_Str (", Discrete_Range_Kind ("
              & Discrete_Range_Kinds'Image (Discrete_Range_Kind (E))
              & ")");
      end if;

      if Association_Kind (E) /= Not_An_Association then
         Add_Str (", Association_Kind ("
              & Association_Kinds'Image (Association_Kind (E))
              & ")");
      end if;

      if Expression_Kind (E) /= Not_An_Expression then
         Add_Str (", Expression_Kind ("
              & Expression_Kinds'Image (Expression_Kind (E))
              & ")");
      end if;

      if Operator_Kind (E) /= Not_An_Operator then
         Add_Str (", Operator_Kind ("
              & Operator_Kinds'Image (Operator_Kind (E))
              & ")");
      end if;

      if Attribute_Kind (E) /= Not_An_Attribute then
         Add_Str (", Attribute_Kind ("
              & Attribute_Kinds'Image (Attribute_Kind (E))
              & ")");
      end if;

      if Statement_Kind (E) /= Not_A_Statement then
         Add_Str (", Statement_Kind ("
              & Statement_Kinds'Image (Statement_Kind (E))
              & ")");
      end if;

      if Path_Kind (E) /= Not_A_Path then
         Add_Str (", Path_Kind ("
              & Path_Kinds'Image (Path_Kind (E))
              & ")");
      end if;

      if Clause_Kind (E) /= Not_A_Clause then
         Add_Str (", Clause_Kind ("
              & Asis.Clause_Kinds'Image (Clause_Kind (E))
              & ")");
      end if;

      if Representation_Clause_Kind (E) /= Not_A_Representation_Clause then
         Add_Str (", Representation_Clause_Kind ("
              & Representation_Clause_Kinds'Image
                (Representation_Clause_Kind (E))
              & ")");
      end if;

      return Buffer (1 .. Buffer_Pos);
   end Dump_Element;

   -------------------
   -- Print_Element --
   -------------------

   procedure Print_Element (E : Asis.Element) is
   begin
      Ada.Text_IO.Put_Line (Dump_Element (E));
   end Print_Element;

   ---------------------
   -- Get_Source_Dirs --
   ---------------------

   function Get_Source_Dirs (Project : String) return String_List is
      Result         : String_List (1 .. 1024);
      Ind            : Integer := 1;
      GNATLS_Process : Process_Descriptor;
      Expect_Result  : Expect_Match;
      Matched        : GNAT.Regpat.Match_Array (0 .. 2);
   begin
      declare
      begin
         if Project /= "" then
            Non_Blocking_Spawn
              (Descriptor  => GNATLS_Process,
               Command     => "gnat",
               Args        =>
                 (1 => new String'("ls"),
                  2 => new String'("-P" & Project),
                  3 => new String'("-v")));
         else
            Non_Blocking_Spawn
              (Descriptor  => GNATLS_Process,
               Command     => "gnat",
               Args        =>
                 (1 => new String'("ls"),
                  2 => new String'("-v")));
         end if;

         Expect
           (Descriptor => GNATLS_Process,
            Result     => Expect_Result,
            Regexp     => "Source Search Path:");

         if Expect_Result = 1 then
            loop
               Expect
                 (Descriptor => GNATLS_Process,
                  Result     => Expect_Result,
                  Regexp     => "   ([^\r\n]*)|(Object Search Path:)",
                  Matched    => Matched);

               exit when Expect_Result /= 1;

               if Matched (1).Last - Matched (1).First > 0 then
                  Result (Ind) := new String'
                    (Expect_Out (GNATLS_Process)
                     (Matched (1).First .. Matched (1).Last));
               else
                  exit;
               end if;

               Ind := Ind + 1;

            end loop;
         end if;

         Close (GNATLS_Process);
      exception
         when Process_Died =>
            null;
      end;

      return Result (1 .. Ind - 1);
   end Get_Source_Dirs;

   --------------
   -- To_Upper --
   --------------

   procedure To_Upper (Str : in out Wide_String) is
   begin
      for J in Str'Range loop
         Str (J) := To_Upper_Case (Str (J));
      end loop;
   end To_Upper;

   ------------
   -- Equals --
   ------------

   function Equals
     (Left, Right : Wide_String; Case_Sensitive : Boolean) return Boolean
   is
   begin
      if Case_Sensitive then
         return Left = Right;
      else
         declare
            Left_U : Wide_String := Left;
            Right_U : Wide_String := Right;
         begin
            To_Upper (Left_U);
            To_Upper (Right_U);

            return Left_U = Right_U;
         end;
      end if;
   end Equals;

   -----------------
   -- Gen_Ada_Loc --
   -----------------

   function Gen_Ada_Loc (Tag : Wide_String) return Dynamic_Expression is
   begin
      if Ada2Java.Debug_Glue then
         return New_Line & To_Dynamic_Expression ("--  DEBUG [" & Tag & "]");
      else
         return Exp_Empty;
      end if;
   end Gen_Ada_Loc;

end Ada2Java.Utils;
