------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                 L I N K                                  --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2010, AdaCore                          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- The GNAT Ada tool chain for the JVM and .NET platforms is  maintained by --
-- AdaCore - http://www.adacore.com                                         --
--                                                                          --
-- This work is partially  based on A#, an Ada  compiler for .NET by  Prof. --
-- Martin C. Carlisle of the United States Air Force Academy.               --
--                                                                          --
------------------------------------------------------------------------------
with Ada.Characters.Handling;            use Ada.Characters.Handling;
with Ada.Command_Line;                   use Ada.Command_Line;
with Ada.Containers.Indefinite_Vectors;  use Ada.Containers;
with Ada.Text_IO;                        use Ada.Text_IO;

with Version;
with GNAT.OS_Lib;                        use GNAT.OS_Lib;

procedure Link is

   Version_Switch : constant String := "--version";
   Help_Switch    : constant String := "--help";

   Verbose_Mode       : Boolean := False;
   Very_Verbose_Mode  : Boolean := False;

   Target_Object_Suffix : constant String := ".il";
   Output_File_Name     : String_Access;
   Linker_Path          : String_Access;

   Link_Max : Integer;
   pragma Import (C, Link_Max, "__gnat_link_max");
   --  Maximum number of bytes on the command line supported by the OS
   --  linker. Passed this limit the response file mechanism must be used
   --  if supported.

   Use_Response_File : Boolean := False;

   package String_List is new Ada.Containers.Indefinite_Vectors
     (Positive, String);

   Linker_Options : String_List.Vector := String_List.Empty_Vector;
   Linker_Objects : String_List.Vector := String_List.Empty_Vector;

   procedure Display_Version
     (Tool_Name      : String;
      Initial_Year   : String;
      Version_String : String := Version.Gnat_Version_String);

   function Base_Name (File_Name : String) return String;
   --  Return just the file name part without the extension (if present)

   procedure Process_Args;
   --  Go through all the arguments and build option tables

   procedure Usage;
   --  Display usage

   procedure Display_Version
     (Tool_Name      : String;
      Initial_Year   : String;
      Version_String : String := Version.Gnat_Version_String)
   is
   begin
      Put_Line (Tool_Name & " " & Version_String);

      Put ("Copyright (C) ");
      Put (Initial_Year);
      Put ('-');
      Put (Version.Current_Year);
      Put_Line (", AdaCore.");
   end Display_Version;

   ---------------
   -- Base_Name --
   ---------------

   function Base_Name (File_Name : String) return String is
      Findex1 : Natural;
      Findex2 : Natural;

   begin
      Findex1 := File_Name'First;

      --  The file might be specified by a full path name. However,
      --  we want the path to be stripped away.

      for J in reverse File_Name'Range loop
         if File_Name (J) = '/' or else File_Name (J) = '\' then
            Findex1 := J + 1;
            exit;
         end if;
      end loop;

      Findex2 := File_Name'Last;
      while Findex2 > Findex1
        and then File_Name (Findex2) /=  '.'
      loop
         Findex2 := Findex2 - 1;
      end loop;

      if Findex2 = Findex1 then
         Findex2 := File_Name'Last + 1;
      end if;

      return File_Name (Findex1 .. Findex2 - 1);
   end Base_Name;

   ------------------
   -- Process_Args --
   ------------------

   procedure Process_Args is
      Skip_Next : Boolean := False;
      --  Set to true if the next argument is to be added into the list of

      --  Start of processing for Process_Args

   begin
      --  First, check for --version and --help
      for J in 1 .. Argument_Count loop
         if Argument (J) = Version_Switch then
            Display_Version
              (To_Upper (Base_Name (Command_Name)),
               "2010");
            Put_Line (Version.Gnat_Free_Software);
            New_Line;
            OS_Exit (0);

         elsif Argument (J) = Help_Switch then
            Usage;
            New_Line;
            Put_Line ("Report bugs to report@adacore.com");
            OS_Exit (0);
         end if;
      end loop;

      --  Loop through arguments of gnatlink command

      for J in 1 .. Argument_Count loop
         declare
            Arg : constant String := Argument (J);

         begin
            --  Case of argument which is a switch

            --  We definitely need section by section comments here ???

            if Skip_Next then
               --  This argument must not be parsed, just add it to the
               --  list of linker's options.

               Skip_Next := False;

            elsif Arg'Length /= 0 and then Arg (1) = '-' then
               if Arg (2) = 'g'
                 and then (Arg'Length < 5 or else Arg (2 .. 5) /= "gnat")
               then
                  Linker_Options.Append ("/DEBUG");

               elsif Arg'Length >= 2
                 and then
                   (Arg (1 .. 2) = "-L"
                    or else Arg (1 .. 2) = "-l"
                    or else Arg (1 .. 2) = "-O")
               then
                  null;

               elsif Arg'Length >= 3
                 and then
                   (Arg (1 .. 3) = "-Wl"
                    or else Arg (1 .. 3) = "-sh")
               then
                  --  ignore
                  null;

               elsif Arg'Length = 2 then
                  case Arg (2) is
                     when 'v' =>
                        --  Support "double" verbose mode.  Second -v
                        --  gets sent to the linker and binder phases.

                        if Verbose_Mode then
                           Very_Verbose_Mode := True;

                        else
                           Verbose_Mode := True;

                        end if;

                     when 'o' =>
                        if J = Argument_Count then
                           Put_Line ("Missing argument for -o");
                           OS_Exit (1);
                        end if;

                        Skip_Next := True;
                        Output_File_Name := new String'(Argument (J + 1));

                     when others =>
                        null; --  ??? Display a warning ?

                  end case;

               --  Send all multi-character switches not recognized as
               --  a special case by gnatlink to the linker/loader stage.

               else
                  Linker_Options.Append (Arg);
               end if;

            elsif Arg (Arg'First) = '@' then
               declare
                  F : File_Type;
               begin
                  Open (F, In_File, Arg (Arg'First + 1 .. Arg'Last));
                  while not End_Of_File (F) loop
                     Linker_Objects.Append (Get_Line (F));
                  end loop;
                  Close (F);
               end;

            --  Here if argument is a file name rather than a switch

            else
               --  If target object file, record object file

               if Arg'Length > Target_Object_Suffix'Length
                 and then Arg
                   (Arg'Last - Target_Object_Suffix'Length + 1 .. Arg'Last)
                   = Target_Object_Suffix
               then
                  Linker_Objects.Append (Arg);

               else
                  Linker_Options.Append (Arg);

               end if;
            end if;
         end;
      end loop;

      if Output_File_Name = null then
         Put_Line ("Error: missing output file name");
         OS_Exit (1);
      end if;
   end Process_Args;

   -----------
   -- Usage --
   -----------

   procedure Usage is
   begin
      Put ("Usage: ");
      Put (Base_Name (Command_Name));
      Put (" switches mainprog.il [objects.il]");
      New_Line;
      New_Line;
      Put_Line ("  mainprog.il   the object file of the main program");
      New_Line;
      Put_Line ("  -g    Link with debug information");
      Put_Line ("  -v    verbose mode");
      Put_Line ("  -v -v very verbose mode");
      New_Line;
      Put_Line ("  -o nam     Use 'nam' as the name of the executable");
      New_Line;
      Put_Line ("  [non-Ada-objects]  list of non Ada object files");
      Put_Line ("  [linker-options]   other options for the linker");
   end Usage;

   Link_Bytes : Natural := 0;
   Status     : Boolean;

begin
   Process_Args;

   Linker_Options.Append ("/QUIET");
   Linker_Options.Append ("/OUTPUT=" & Output_File_Name.all);

   if Very_Verbose_Mode then
      Put_Line ("   Parsed command line:");
      for J in Linker_Options.First_Index .. Linker_Options.Last_Index loop
         Put_Line ("   - " & Linker_Options.Element (J));
      end loop;

      Put_Line ("   Objects to link:");
      for J in Linker_Objects.First_Index .. Linker_Objects.Last_Index loop
         Put_Line ("   - " & Linker_Objects.Element (J));
      end loop;
   end if;

   for J in Linker_Options.First_Index .. Linker_Options.Last_Index loop
      Link_Bytes := Link_Bytes + Linker_Options.Element (J)'Length + 1;
   end loop;
   for J in Linker_Objects.First_Index .. Linker_Objects.Last_Index loop
      Link_Bytes := Link_Bytes + Linker_Objects.Element (J)'Length + 1;
   end loop;

   if Link_Bytes > Link_Max then
      declare
         --  First file given in Linker_Objects is the binder-generated file.
         --  Let's create a temp file name from this file name, as we are sure
         --  no such other object file exists, and we want a file with .il
         --  extension that this file already has.
         Lname : String := Linker_Objects.First_Element;
      begin
         Lname (Lname'First) := 'l';

         for J in Linker_Objects.First_Index .. Linker_Objects.Last_Index loop
            Copy_File (Linker_Objects.Element (J), Lname,
                       Success => Status,
                       Mode    => Append);
            if not Status then
               Put_Line ("Issues while creating response file " & Lname);
               OS_Exit (1);
            end if;
         end loop;

         Linker_Objects.Clear;
         Linker_Objects.Append (Lname);

         Use_Response_File := True;
      end;
   end if;

   --  Prepare arguments for call to linker

   Linker_Path := Locate_Exec_On_Path ("ilasm");

   Call_Linker : declare
      Args        : Argument_List
                (1 .. Integer (Linker_Objects.Length + Linker_Options.Length));
      Index       : Integer := Args'First;
      Success     : Integer;
      Output_Fd   : File_Descriptor;
      Output_Name : GNAT.OS_Lib.Temp_File_Name;

   begin
      for J in Linker_Options.First_Index .. Linker_Options.Last_Index loop
         Args (Index) := new String'(Linker_Options.Element (J));
         Index := Index + 1;
      end loop;

      for J in Linker_Objects.First_Index .. Linker_Objects.Last_Index loop
         Args (Index) := new String'(Linker_Objects.Element (J));
         Index := Index + 1;
      end loop;

      if Verbose_Mode then
         Put (Linker_Path.all);

         for J in Args'Range loop
            Put (" ");
            Put (Args (J).all);
         end loop;

         New_Line;
      end if;

      GNAT.OS_Lib.Create_Temp_File (Output_Fd, Output_Name);
      GNAT.OS_Lib.Spawn (Linker_Path.all, Args, Output_Fd, Success);
      GNAT.OS_Lib.Close (Output_Fd);

      if Use_Response_File then
         GNAT.OS_Lib.Delete_File (Linker_Objects.Element (1), Status);
      end if;

      if Success /= 0 then
         Put_Line ("error when calling " & Base_Name (Linker_Path.all));
         declare
            S : String (1 .. 1024);
            N : Natural;
         begin
            Output_Fd := GNAT.OS_Lib.Open_Read (Output_Name, GNAT.OS_Lib.Text);
            loop
               N := GNAT.OS_Lib.Read (Output_Fd, S'Address, 1024);
               Put (S (1 .. N));
               exit when N < 1024;
            end loop;
            GNAT.OS_Lib.Close (Output_Fd);
         end;

         GNAT.OS_Lib.Delete_File (Output_Name, Status);
         OS_Exit (1);

      else
         GNAT.OS_Lib.Delete_File (Output_Name, Status);
      end if;
   end Call_Linker;
end Link;
