------------------------------------------------------------------------------
--                                Ada2Java                                  --
--                                                                          --
--                     Copyright (C) 2007-2010, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Deallocation;
with Ada.Unchecked_Conversion;
with Ada.Tags;
with Ada.Containers.Doubly_Linked_Lists;
with Ada.Finalization;
with Ada.Exceptions;

with Interfaces.C;
with Interfaces.C.Strings;

with System;

with Interfaces.Java.JNI; use Interfaces.Java.JNI;
with Ada.Containers.Indefinite_Ordered_Maps;

package AJIS.Internal.Java is

   type Cross_Language_Class is interface;

   procedure Set_Java_Ref
     (Object : in out Cross_Language_Class; Ref : J_Object) is abstract;

   procedure Set_Java_VM
     (Object : in out Cross_Language_Class; VM : Java_VM_Access) is abstract;

   function Get_Java_Ref
     (Object : access constant Cross_Language_Class) return J_Object is abstract;

   function Get_Java_VM
     (Object : Cross_Language_Class) return Java_VM_Access is abstract;

   function Copy_No_Clone
     (Object : Cross_Language_Class)
      return access Cross_Language_Class
      is abstract;
   --  Perform a copy of the cross language object, without cloning the
   --  backlinked reference. The returned object must be top level
   --  accessibility.

   function Get_Address_For_Proxy
     (Env    : JNI_Env_Access;
      Object : access Cross_Language_Class) return J_Int_J_Array is abstract;
   --  Return the java access of the current object.

   procedure Set_Global_Ref
     (This      : in out Cross_Language_Class;
      Is_Global : Boolean) is abstract;
   --  Set the status of the inner reference to the java object. Creates a weak
   --  or a regular reference depending on the state of Is_Global.

   type Cross_Lg_Container is record
      Ptr : access constant Cross_Language_Class'Class := null;
   end record;

   type Cross_Lg_Container_Acc is access all Cross_Lg_Container;

   type Backlinked_Java_Reference is new Ada.Finalization.Controlled
   with record
      Enclosing     : Cross_Lg_Container_Acc;
      Ref           : J_Object := J_Null_Object;
      VM            : Java_VM_Access := null;
      Is_Global_Ref : Boolean := False;
   end record;

   overriding
   procedure Initialize (This : in out Backlinked_Java_Reference);

   overriding
   procedure Adjust (This : in out Backlinked_Java_Reference);

   overriding
   procedure Finalize (This : in out Backlinked_Java_Reference);

   procedure Set_Global_Ref
     (This      : in out Backlinked_Java_Reference;
      Is_Global : Boolean);
   --  Set the status of the inner reference to the java object. Creates a weak
   --  or a regular reference depending on the state of Is_Global.

   procedure Set_Enclosing_Address
     (This      : Backlinked_Java_Reference;
      Enclosing : access constant Cross_Language_Class'Class);

   function Convert is new Ada.Unchecked_Conversion
     (Interfaces.C.Strings.chars_ptr, System.Address);

   subtype Unchecked_String is String (Positive);
   pragma Suppress (All_Checks, Unchecked_String);

   type Unchecked_String_Access is access all Unchecked_String;

   function To_Unchecked_String is new Ada.Unchecked_Conversion
     (System.Address, Unchecked_String_Access);

   function To_Unchecked_String is new Ada.Unchecked_Conversion
     (Interfaces.C.Strings.Chars_Ptr, Unchecked_String_Access);

   type Java_Method is private;
   --  This type hold the necessary data used to inialize a pointer to a Java
   --  method, but does not make the actual link. This way, it can be created
   --  without the pointer to the environment (which will be needed when
   --  extracting the actual J_Method_Id).

   type Java_Method_Access is access all Java_Method;

   function Get_Java_Method
     (Class_Name     : String;
      Method_Name    : String;
      Method_Profile : String;
      Is_Static      : Boolean := False) return Java_Method_Access;

   function Get_Id
     (This   : Java_Method_Access;
      Env    : JNI_Env_Access) return J_Method_ID;
   --  Return the id of this method access - retreives it from the environment
   --  if not yet computed

   function Get_Id (This : Java_Method_Access) return J_Method_Id;
   --  Same as above, but assumes that the method has already been computed.
   --  Will raise an exception if not.

   function Get_Class
     (This   : Java_Method_Access;
      Env    : JNI_Env_Access) return J_Class;
   --  Return the class of this method access - retreives it from the
   --  environment if not yet computed.

   function Get_Class (This : Java_Method_Access) return J_Class;
   --  Same as above, but assumes that the class has already been computed.

   procedure Initialize (This : Java_Method_Access; Env : JNI_Env_Access);
   --  Initializes the handle to the class and the method according to the
   --  environment given in parameter.

   type Java_Class is private;
   --  Stores a hande on a java class

   type Java_Class_Access is access all Java_Class;

   function Get_Java_Class
     (Class_Name : String) return Java_Class_Access;

   function Get_Class
     (This   : Java_Class_Access;
      Env    : JNI_Env_Access) return J_Class;
   --  Return the class of this class access - retreives it from the
   --  environment if not yet computed.

   function Get_Class (This   : Java_Class_Access) return J_Class;
   --  Same as above, but assumes that the class has already been computed.

   procedure Initialize (This : Java_Class_Access; Env : JNI_Env_Access);
   --  Initializes the handle to the class according to the
   --  environment given in parameter.

   function New_Java_Integer_Wrapper
     (Env : JNI_Env_Access; Val : J_Int) return J_Object;
   function New_Java_Long_Wrapper
     (Env : JNI_Env_Access; Val : J_Long) return J_Object;
   function New_Java_Boolean_Wrapper
     (Env : JNI_Env_Access; Val : J_Boolean) return J_Object;
   function New_Java_Double_Wrapper
     (Env : JNI_Env_Access; Val : J_Double) return J_Object;
   function New_Java_Character_Wrapper
     (Env : JNI_Env_Access; Val : J_Char) return J_Object;
   function New_Java_Object_Wrapper
     (Env : JNI_Env_Access; Val : J_Object) return J_Object;
   function New_Java_Access_Constructor
     (Env : JNI_Env_Access; Val : J_Object) return J_Object;

   function New_Java_Integer_Wrapper
     (Env : JNI_Env_Access) return J_Object;
   function New_Java_Long_Wrapper
     (Env : JNI_Env_Access) return J_Object;
   function New_Java_Boolean_Wrapper
     (Env : JNI_Env_Access) return J_Object;
   function New_Java_Double_Wrapper
     (Env : JNI_Env_Access) return J_Object;
   function New_Java_Character_Wrapper
     (Env : JNI_Env_Access) return J_Object;

   procedure Java_Set_Integer_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Int);
   procedure Java_Set_Long_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Long);
   procedure Java_Set_Boolean_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Boolean);
   procedure Java_Set_Double_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Double);
   procedure Java_Set_Character_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Char);
   procedure Java_Set_Object_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Object);
   procedure Java_Set_Enum_Value
     (Env : JNI_Env_Access; Obj : J_Object; Val : J_Int);

   function Java_Get_Integer_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Int;
   function Java_Get_Long_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Long;
   function Java_Get_Boolean_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Boolean;
   function Java_Get_Double_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Double;
   function Java_Get_Character_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Char;
   function Java_Get_Object_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Object;
   function Java_Get_Enum_Value
     (Env : JNI_Env_Access; Obj : J_Object) return J_Int;

   procedure Dump (Val : J_Int_Array);

   procedure Set_Values
     (Env : JNI_Env_Access; Dest : J_Int_J_Array; Src : J_Int_Array);

   function Get_Values
     (Env : JNI_Env_Access; Src : J_Int_J_Array) return J_Int_Array;

   function To_J_Int_J_Array
     (Env : JNI_Env_Access; V : System.Address) return J_Int_J_Array;

   function To_Address
     (Env : JNI_Env_Access; V : J_Int_J_Array) return System.Address;

   function Get_Serialized_Address
     (Env : JNI_Env_Access; V : J_Object) return J_Int_J_Array;
   --  Return the address of the native object stored in the Java object.

   -------------------
   -- Serialization --
   -------------------

   generic
      type Object_Type is private;
   package Serialization is

      function To_J_Int_J_Array
        (Env : JNI_Env_Access; V : Object_Type) return J_Int_J_Array;

      function To_Object_Type
        (Env : JNI_Env_Access; V : J_Int_J_Array) return Object_Type;

   end Serialization;

   --------------------------------
   -- Access_To_Jint_Conversions --
   --------------------------------

   type Array_Pointer_Kind is (Static, General_Access, Constant_Access);

   generic
      type The_Type (<>) is limited private;
   package Access_To_Jint_Conversions is

      type Object_Pointer is access all The_Type;

      type Object_Pointer_Cst is access constant The_Type;

      function To_Object_Pointer
        (V : access The_Type) return Object_Pointer;

      function To_Object_Pointer_Cst
        (V : access constant The_Type) return Object_Pointer_Cst;

      function To_JintArray
        (Env : JNI_Env_Access; V : Object_Pointer) return J_Int_J_Array;

      function To_JintArray_Cst
        (Env : JNI_Env_Access; V : Object_Pointer_Cst) return J_Int_J_Array;

      function To_Pointer
        (Env : JNI_Env_Access; V : J_Int_J_Array) return Object_Pointer;

      function To_Pointer_From_Object
        (Env : JNI_Env_Access; V : J_Object) return Object_Pointer;

      procedure Free is new
        Ada.Unchecked_Deallocation (The_Type, Object_Pointer);

   end Access_To_Jint_Conversions;

   function To_J_Value (Val : J_Boolean) return J_Value;
   function To_J_Value (Val : J_Byte) return J_Value;
   function To_J_Value (Val : J_Char) return J_Value;
   function To_J_Value (Val : J_Short) return J_Value;
   function To_J_Value (Val : J_Int) return J_Value;
   function To_J_Value (Val : J_Long) return J_Value;
   function To_J_Value (Val : J_Float) return J_Value;
   function To_J_Value (Val : J_Double) return J_Value;
   function To_J_Value (Val : J_Object) return J_Value;

   function Get_Env_For_Current_Thread
     (VM : Java_VM_Access) return JNI_Env_Access;

   function Create_Java_Object
     (Env          : JNI_Env_Access;
      The_Tag      : Ada.Tags.Tag;
      Access_Array : J_Int_J_Array) return J_Object;

   procedure Reference_Java_Object_Constructor
     (The_Tag : Ada.Tags.Tag; Constructor : Java_Method_Access);

   procedure Reference_Exception
     (Except : Ada.Exceptions.Exception_Id; Java_Class : String);
   --  Adds an exception in the known Ada exception map.

   function Create_Java_Exception
     (Env : JNI_Env_Access;
      Except : Ada.Exceptions.Exception_Occurrence)
      return J_Object;
   --  Return an instance of the java exception corresponding to the Ada
   --  exception occurence given in parameter.

   procedure Handle_Java_Exception (Env : JNI_Env_Access);
   --  If a Java exception has been raised, this procedure will raise the
   --  corresponding Ada exception;

   function Throwable_Image
     (Env : JNI_Env_Access; Exc : J_Throwable) return String;
   --  Return an string image of the exception given in parameter

   function Throwable_Value (Exc : String) return J_Throwable;
   --  Reconstruct the exception reference based on a string that should have
   --  been previously generated with Throwable_Image.

   type Owner is (Native, Proxy);

   procedure Set_Owner
     (Env       : JNI_Env_Access;
      Obj       : J_Object;
      The_Owner : Owner);
   --  Calls the setOwner method of the java object

   type Allocator is (Static, Dynamic, Unknown);

   procedure Set_Allocator
     (Env           : JNI_Env_Access;
      Obj           : J_Object;
      The_Allocator : Allocator);
   --  Set the allocator of the java object

   Access_Meth_Constructor : Java_Method_Access;

   Native_Exception_Class : Java_Class_Access;
   IProxy_Owner_Class     : Java_Class_Access;
   IProxy_Allocator_Class : Java_Class_Access;
   Ada_Proxy_Class        : Java_Class_Access;

   procedure Initialize_JVM_Parameters (Params : in out Java_VM_Init_Args);
   --  Initializes JVM parameters, taking into account environment variables
   --  such as CLASSPATH and command-line options such as -classpath.

   procedure Initialize_JNI (Env : JNI_Env_Access);
   --  To be call to Initialize the various subprogram handles.

private

   type String_Access is access all String;

   type Java_Method is record
      Class_Name     : String_Access;
      Method_Name    : String_Access;
      Method_Profile : String_Access;

      Class          : J_Class := J_Null_Class;
      Method_Id      : J_Method_ID := J_Null_Method_ID;

      Is_Static      : Boolean := False;
   end record;

   type Java_Class is record
      Class_Name : String_Access;
      Class      : J_Class := J_Null_Class;
   end record;

   type Type_Tree_Record;

   type Type_Tree_Record_Access is access all Type_Tree_Record;

   package Types_List is new
     Ada.Containers.Doubly_Linked_Lists (Type_Tree_Record_Access);

   use Types_List;

   type Type_Tree_Record is record
      The_Tag     : Ada.Tags.Tag;
      Constructor : Java_Method_Access;
      Children    : Types_List.List;
   end record;

   Root_Type : aliased Type_Tree_Record;

   function Get_Closest_Type_Tree_Record
     (The_Tag : Ada.Tags.Tag) return Type_Tree_Record_Access;

   function "<" (Left, Right : Ada.Exceptions.Exception_Id) return Boolean;

   package Ada_To_Java_Exception_Pckg is new
     Ada.Containers.Indefinite_Ordered_Maps
       (Ada.Exceptions.Exception_Id, Java_Method_Access);
   --  Package for the Ada -> Java exception correspondence

   function "=" (Left, Right : Ada.Exceptions.Exception_Id) return Boolean;

   function "<" (Left, Right : Java_Class_Access) return Boolean;

   package Java_To_Ada_Exception_Pckg is new
     Ada.Containers.Indefinite_Ordered_Maps
     (Java_Class_Access, Ada.Exceptions.Exception_Id);
   --  Package for the Java -> Ada exception correspondence

   Ada_To_Java_Exceptions : Ada_To_Java_Exception_Pckg.Map;
   Java_To_Ada_Exceptions : Java_To_Ada_Exception_Pckg.Map;

end AJIS.Internal.Java;
