-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                     Copyright (C) 2007, AdaCore                   --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with GNAT.Regpat;                use GNAT.Regpat;
with Ada.Characters.Handling;    use Ada.Characters.Handling;
with Ada.Characters.Conversions;

package body Ada2Java.Config is

   ---------------------------
   -- Set_Java_Base_Package --
   ---------------------------

   procedure Set_Java_Base_Package
     (Conf : in out Configuration; Base_Package : Wide_String) is
   begin
      Conf.Java_Base_Package := To_Unbounded_Wide_String (Base_Package);
   end Set_Java_Base_Package;

   ------------------------------
   -- Set_Ada_Output_Directory --
   ------------------------------

   procedure Set_Ada_Output_Directory
     (Conf : in out Configuration; Output_Directory : Wide_String) is
   begin
      Conf.Ada_Output_Directory := To_Unbounded_Wide_String (Output_Directory);
   end Set_Ada_Output_Directory;

   -------------------------
   -- Set_Java_Class_Path --
   -------------------------

   procedure Set_Java_Class_Path
     (Conf : in out Configuration; Class_Path : Wide_String) is
   begin
      Conf.Java_Class_Path := To_Unbounded_Wide_String (Class_Path);
   end Set_Java_Class_Path;

   ---------------------------
   -- Get_Java_Base_Package --
   ---------------------------

   function Get_Java_Base_Package
     (Conf : Configuration)
      return Wide_String
   is
   begin
      return To_Wide_String (Conf.Java_Base_Package);
   end Get_Java_Base_Package;

   ------------------------------
   -- Get_Ada_Output_Directory --
   ------------------------------

   function Get_Ada_Output_Directory
     (Conf : Configuration)
      return Wide_String
   is
   begin
      return To_Wide_String (Conf.Ada_Output_Directory);
   end Get_Ada_Output_Directory;

   -------------------------
   -- Get_Java_Class_Path --
   -------------------------

   function Get_Java_Class_Path
     (Conf : Configuration)
      return Wide_String
   is
   begin
      return To_Wide_String (Conf.Java_Class_Path);
   end Get_Java_Class_Path;

   ----------------
   -- Parse_File --
   ----------------

   procedure Parse_File (File : File_Type; Conf : in out Configuration) is
      Matcher : constant Pattern_Matcher := Compile ("^(\w+)\s*=\s*(.*)$");
   begin
      while not End_Of_File (File) loop
         declare
            Line    : constant String := Get_Line (File);
            Matches : Match_Array (0 .. 2);
         begin
            Match (Matcher, Line, Matches);

            if Matches (0) /= No_Match then
               declare
                  Var_Name : constant String :=
                    To_Lower (Line (Matches (1).First .. Matches (1).Last));
                  Var_Value : constant Wide_String :=
                    Ada.Characters.Conversions.To_Wide_String
                      (Line (Matches (2).First .. Matches (2).Last));
               begin
                  if Var_Name = "output_dir" then
                     Set_Ada_Output_Directory (Conf, Var_Value);
                  elsif Var_Name = "classpath" then
                     Set_Java_Class_Path (Conf, Var_Value);
                  elsif Var_Name = "base_java_package" then
                     Set_Java_Base_Package (Conf, Var_Value);
                  else
                     raise Program_Error with
                     Var_Name & " is not a known configuration variable";
                  end if;
               end;
            else
               raise Program_Error with "parse error in """ & Line & """";
            end if;
         end;
      end loop;
   end Parse_File;

end Ada2Java.Config;
