-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers.Indefinite_Ordered_Sets;
with Ada.Containers.Indefinite_Ordered_Maps;
with Ada.Containers.Doubly_Linked_Lists;
with Ada.Containers.Ordered_Maps;

with Ada2Java.Config;              use Ada2Java.Config;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
with Ada2Java.Packages;            use Ada2Java.Packages;
with Ada2Java.Simplifications;     use Ada2Java.Simplifications;
with Ada2Java.Utils;               use Ada2Java.Utils;

limited with Ada2Java.Kernel;

package Ada2Java.Bound_Units is

   ------------------
   -- Binding_File --
   ------------------

   package String_Lists is new
     Ada.Containers.Indefinite_Ordered_Sets (Wide_String);

   use String_Lists;

   type File_Language is (Ada_Lang, Java_Lang);

   type Type_Record is record
      Conversion_Package_Name : Wide_String_Access;
      Type_Of                 : Simple_Type_View_Access;
   end record;

   package Types_Map is new Ada.Containers.Ordered_Maps
     (Element_Index, Type_Record);

   use Types_Map;

   type Bound_Unit_Record;

   type Bound_Unit is access all Bound_Unit_Record;

   package Bound_Unit_List_Pckg is new
     Ada.Containers.Indefinite_Ordered_Maps (Wide_String, Bound_Unit);

   package Dependecy_List is new
     Ada.Containers.Indefinite_Ordered_Sets (Wide_String);

   use Bound_Unit_List_Pckg;

   type File_Kind is (Java_Class_Kind, Ada_Spec_Kind, Ada_Body_Kind);

   type Binding_File_Record (Kind : File_Kind) is limited record
      Unit                    : access Bound_Unit_Record;

      Package_Name            : Dynamic_Expression := New_Dynamic_Expression;
      Dependencies_Part       : Dynamic_Expression := New_Dynamic_Expression;
      Public_Body_Part        : Dynamic_Expression := New_Dynamic_Expression;
      Private_Body_Part       : Dynamic_Expression := New_Dynamic_Expression;
      Open_Part               : Dynamic_Expression := New_Dynamic_Expression;
      Close_Part              : Dynamic_Expression := New_Dynamic_Expression;
      Binding_Config          : Configuration;
      Base_Pckg               : Package_Handle;
      Extra_Unit_Dependencies : Dependecy_List.Set;

      case Kind is
         when Java_Class_Kind =>
            Class_Name        : Dynamic_Expression := New_Dynamic_Expression;
            Full_Class_Name   : Dynamic_Expression := New_Dynamic_Expression;
            Java_Parent_Class : Dynamic_Expression := New_Dynamic_Expression;
            Import_List       : String_Lists.Set;
            Interfaces_List   : String_Lists.Set;
            Is_Abstract       : Boolean := False;
            Is_Final          : Boolean := True;
            Is_Enum           : Boolean := False;
         when Ada_Spec_Kind =>
            Access_To_Address_Conversions : Types_Map.Map;
         when Ada_Body_Kind =>
            Elab_Part : Dynamic_Expression := New_Dynamic_Expression;

            Native_Number     : Integer := 0;
            Reg_Natives       : Dynamic_Expression := New_Dynamic_Expression;
            --  This expression is used to store the array initialization used
            --  for the Register_Native call if that's the way of linking to
            --  Java that has been chosen.

            Initialize_Body : Dynamic_Expression := New_Line & "null;";
            --  This contains a list of instruction to be called right after
            --  library load, in the context of the library loading thred.
            --  It can be used for early initialization potentially based on
            --  the java environment pointer. It can be used to e.g. initialize
            --  subprogram handlers.
      end case;
   end record;

   type Binding_File is access all Binding_File_Record;

   function Assemble
     (Parts : Binding_File_Record) return Dynamic_Expression;

   procedure Add_Import_Element (File : Binding_File; Element : Wide_String);
   --  Adds the given element in the import list of the binding file if it's
   --  not already there.

   procedure Add_Extra_Unit_Dependency
     (File : Binding_File; Unit : Bound_Unit);

   procedure Add_Extra_Unit_Dependency
     (File : Binding_File; Element : Wide_String);

   procedure Add_Dependency_On_Type
     (Handle   : not null access Kernel.Kernel_Record;
      File     : Binding_File;
      The_Type : Simple_Type_View_Access);

   -----------------
   -- Bound_Unit --
   -----------------

   package Simple_Elements_List is new Ada.Containers.Doubly_Linked_Lists
     (Simple_Element_View_Access);

   type Bound_Unit_Record is limited record
      Ada_Pckg_Name   : Dynamic_Expression := Empty_Dynamic_Expression;
      Java_Class_Name : Dynamic_Expression := Empty_Dynamic_Expression;
      Base_Pckg       : Package_Handle;

      Java_File     : Binding_File;
      Ada_Body_File : Binding_File;
      Ada_Spec_File : Binding_File;

      Create_Java_File     : Boolean := True;
      Create_Ada_Body_File : Boolean := True;
      Create_Ada_Spec_File : Boolean := True;

      Root_Unit : Bound_Unit;
      --  In the process of the binding of an Ada package, there may be several
      --  sub units generated. This field hold the value of the unit
      --  corresponding directly to that package.

      Elements : Simple_Elements_List.List;
   end record;

   subtype Bound_Units_DB is Bound_Unit_List_Pckg.Map;

   subtype Bound_Units_Cursor is Bound_Unit_List_Pckg.Cursor;

   function Get_Or_Create_Bound_Unit
     (Handle : not null access Kernel.Kernel_Record;
      View   : Simple_Element_View_Access) return Bound_Unit;

   function Get_Or_Create_Bound_Unit
     (Handle       : not null access Kernel.Kernel_Record;
      Base_Package : Package_Handle;
      Unit_Suffix  : Wide_String) return Bound_Unit;
   --  Return the bound unit which name is the Full_Name given in parameter.
   --  No Java nor Ada part would be created.

   procedure Generate_Missing_Bound_Units
     (Handle : not null access Kernel.Kernel_Record);
   --  Some units might have been generated without their child units. This
   --  function ensure that the whole architecture has been generated, and
   --  create empty packages when needed.

   function Get_Or_Create_Conversion_Type
     (Handle  : not null access Kernel.Kernel_Record;
      Element : Simple_Type_View_Access) return Type_Record;
   --  Return the conversion type corresponding to the element given in
   --  parameter, create one if no one exists.

   function Create_Java_Class
     (Unit         : access Bound_Unit_Record;
      Handle       : access Kernel.Kernel_Record;
      Pckg         : access Packages.Package_Record;
      Name         : Wide_String) return Binding_File;
   --  Creates a new Java class if it doesn't exit, and returns the
   --  corresponding binding file. If Extendint is True, then the created class
   --  expects to extend an other class.

   function New_Ada_Spec_File
     (Unit           : access Bound_Unit_Record;
      Name           : Wide_String;
      Binding_Config : Configuration) return Binding_File;

   function New_Ada_Body_File
     (Unit            : access Bound_Unit_Record;
      Name           : Wide_String;
      Binding_Config : Configuration) return Binding_File;

   procedure Insert_Type_In_Type_Map
     (Unit    : Bound_Unit;
      Index   : Element_Index;
      Element : Type_Record);

   procedure Add_Element
     (Unit : Bound_Unit;
      E    : Simple_Element_View_Access);
   --  Adds an element to the list of elements that's going to be checked
   --  and bound.

   procedure Check_Name_Clashes (Unit : Bound_Unit);
   --  Check that there is no clashes between types & subprograms names and
   --  deactivate binding for entities that would generate such a
   --  non-compilable code

   procedure Adjust_Consistency (Unit : Bound_Unit);
   --  Elements contained in this unit  that should not be bound, because
   --  e.g. they have a name clash or reference an other not bindable unit,
   --  will get marked as can't be bound.

   procedure Generate_Bindings (Unit : Bound_Unit);
   --  Generates the bindings for all elements added to that unit.

end Ada2Java.Bound_Units;
