------------------------------------------------------------------------------
--                                   JNI                                    --
--                                                                          --
--                     Copyright (C) 2005-2007, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
------------------------------------------------------------------------------

--  This package defines the root type of all the reference types generated
--  by the binding tool and the root type for all interface types.

--  The only type that directly inherits from the root reference type is
--  Java.Lang.Object.Typ . All other reference types inherit from
--  Java.Lang.Object.Typ or one of its descendant.

with JNI;      use JNI;
with Ada.Tags; use Ada.Tags;

package JNI_Object is

   ----------------------------------------------------
   -- Parameters for Generic_Dispatching_Constructor --
   ----------------------------------------------------

   type Parameters is null record;
   No_Param : aliased Parameters := (null record);
   --  Required parameter to instantiate Generic_Dispatching_Constructor

   -------------------------
   -- Root Interface Type --
   -------------------------

   --  TODO
   --  Root_Interface should be limited but this is not supported yet.
   --  type Root_Interface is limited Interface;

   type Root_Interface is interface;
   --  The root type of all types
   type Root_Interface_Access is access all Root_Interface'Class;

   function Get_J_Object (I : Root_Interface) return J_Object
      is abstract;
   procedure Set_J_Object (I : in out Root_Interface; O : J_Object)
      is abstract;
   --  Get and Set the JNI J_Object value, ie the reference to the
   --  concrete Java object defined in the JVM

   function Secure_Get_J_Object (O : access Root_Interface'Class)
                                 return J_Object;
   --  Secure version of Get_J_Object which support a null argument. Passing a
   --  null value too Get_J_Object is forbidden in Ada for dispatching
   --  parameters. Using a classwide parameter removes such a limitation.
   --  This allows to use the Ada null value the same way it is used in Java.

   function JNI_Constructor (Params : not null access Parameters)
      return Root_Interface is abstract;
   --  This is a constructor function that allows to create an object
   --  for each concrete type in the Root_Interface'Class hierarchy.

   function Create_Ada_Object (J_Obj : J_Object)
                               return access Root_Interface'Class;
   --  If J_Obj is J_Null_Object, null is returned.
   --
   --  Otherwise an Ada_Object corresponding to the Java object represented by
   --  J_Obj is created and it is ensured that the Java instance will not be
   --  garbage-collected by the JVM until a call to Free (see below) is done.
   --  The local reference J_Obj is deleted so J_Obj must not be used after a
   --  call to Create_Ada_Object !!

   procedure Free (I : in out Root_Interface_Access);
   --  If I is not null, Free ensures that the Java instance represented can
   --  be garbage-collected by the JVM and then deallocates I.

   ----------------------
   -- Root Object Type --
   ----------------------

   --  TODO
   --  JNI_Data should be limited but this is not supported yet when a type
   --  implements an interface

   type JNI_Data is abstract new Root_Interface with private;
   --  The root type of all the reference types

   overriding
   function  Get_J_Object (D : JNI_Data) return J_Object;
   overriding
   procedure Set_J_Object (D : in out JNI_Data; O : J_Object);
   --  Get and Set the JNI J_Object value, ie the reference to the
   --  concrete Java object defined in the JVM.

   -------------------------------------------------
   -- Java class name <-> Ada tagged type mapping --
   -------------------------------------------------

   procedure Register (Class_Name : String; T : Tag);
   --  Register the association between a Java Class_Name and the Tag T of
   --  the corresponding Ada tagged type.
   --  Class_Name has the following format "java.lang.String" and the Tag
   --  to register is Java.Lang.String.Typ'Tag.
   --  All packages defining an Ada tagged type corresponding to a concrete
   --  Java type must call Register during the package elaboration.

private
   type JNI_Data is abstract new Root_Interface with record
      --  Useful fields to use JNI
      J_Obj : J_Object;
   end record;
end JNI_Object;
