-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2010, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Asis;                        use Asis;
with Asis.Extensions.Flat_Kinds;  use Asis.Extensions.Flat_Kinds;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;
limited with Ada2Java.Kernel;

package Ada2Java.Utils is

   type Element_Index is record
      Kind     : Flat_Element_Kinds;
      Location : String_Access;
   end record;
   --  This type describes the location of an element in the Ada tree.

   function Create_Dummy_Unique_Location return Element_Index;
   --  This function return a unique location that doesn't correspond to
   --  anything in the code but can still be used where an index is needed.

   function To_Element_Index (Element : Asis.Element) return Element_Index;
   --  Return the element index corresponding to this element.

   function "<" (Left, Right : Element_Index) return Boolean;
   --  Return true if Left is lower than right. This function ensures that
   --  two elements will always have the same relationship.

   function "=" (Left, Right : Element_Index) return Boolean;

   procedure Free (Index : in out Element_Index);
   --  Free the data associated to an index.

   Anonymous_Element_Index : constant Element_Index := (Not_An_Element, null);

   function Remove_Blanks (Str : Wide_String) return Wide_String;
   --  Removes the blanks characters in front of the string given in parameter.

   function Mangle_ID (Str : Wide_String) return Wide_String;
   --  Mangle the identifier given in parameter, in the corresponding JNI
   --  symbol.

   function Replace_Dots_By_Slashes (Str : Wide_String) return Wide_String;
   --  Replace dots by slashes in the given list. This function is often used
   --  when translating a class name, like a.b.c, into a JNI class path,
   --  a/b/c

   function Get_Root_Parent (Decl : Asis.Declaration) return Asis.Declaration;
   --  Return the root parent, according by visiblity rules.

   function Get_First_Name (Obj : Asis.Element) return Wide_String;
   --  Return the first name of the given object

   function Get_Location (Obj : Asis.Element) return Wide_String;
   --  Return a string representation of the element location

   function Get_Full_Ada_Name
     (Handle : not null access Ada2Java.Kernel.Kernel_Record;
      Obj : Asis.Element)
      return Wide_String;
   --  Return the full name for the type designated by Obj, prefixed by the
   --  package name.

   function Get_Unique_Id return Wide_String;
   --  Return a unique id. The id returned by this function is different after
   --  each call.

   function Dump_Element (E : Asis.Element) return String;
   --  Return a a comprehensive string representation of the element given in
   --  parameter.

   procedure Print_Element (E : Asis.Element);
   pragma Export (C, Print_Element, "g_print_element");

   function Get_Source_Dirs (Project : String) return String_List;
   --  Return the source dirs where the sources of the project given in
   --  parameter are located.

   procedure To_Upper (Str : in out Wide_String);
   --  Upper-case the string given in parameter

   Env_Parameter_Name : constant Wide_String;
   --  This id should be used for the JNI_ENV ** parameter of JNI Profiles

   Object_Parameter_Name : constant Wide_String;
   --  This id should be used for the J_Object JNI parameter of JNI Profiles.

   Class_Parameter_Name : constant Wide_String;
   --  This id should be used for the J_Class JNI parameter of JNI Profiles.

   function Equals
     (Left, Right : Wide_String; Case_Sensitive : Boolean) return Boolean;

   JNI_Pckg : constant Dynamic_Expression :=
     To_Dynamic_Expression ("Interfaces.Java.JNI");

   AJIS_Pckg : constant Dynamic_Expression :=
     To_Dynamic_Expression ("AJIS.Internal.Java");

private

   Unique_Name_Prefix : constant Wide_String := "Id" & Ada_JNI_Suffix & "_";

   Id_Counter : Integer := 3;

   Env_Parameter_Name : constant Wide_String := Unique_Name_Prefix & "1";
   --  This id should be used for the JNI_ENV ** parameter of JNI Profiles

   Object_Parameter_Name : constant Wide_String := Unique_Name_Prefix & "2";
   --  This id should be used for the J_Object JNI parameter of JNI Profiles.

   Class_Parameter_Name : constant Wide_String := Unique_Name_Prefix & "3";

end Ada2Java.Utils;
