-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers.Indefinite_Ordered_Maps;

with Asis;             use Asis;
with Ada2Java.Config;  use Ada2Java.Config;

limited with Ada2Java.Kernel;

package Ada2Java.Packages is

   type Package_Record is limited private;
   --  This type is used to represent a package, from both the Ada and the Java
   --  side

   type Package_Handle is access all Package_Record;

   package Packages_Container is new Ada.Containers.Indefinite_Ordered_Maps
     (Wide_String, Package_Handle);

   use Packages_Container;

   subtype Bound_Packages_DB is Packages_Container.Map;

   procedure Free (This : in out Package_Handle);
   --  Free the memory associated to a package handle.

   function Get_Or_Create_Package
     (Handle : not null access Kernel.Kernel_Record;
      E      : Asis.Element)
      return Package_Handle;
   --  Get or create a package corresponding to the one enclosing the element
   --  given in parameter.

   function Get_Or_Create_Package
     (Handle             : not null access Kernel.Kernel_Record;
      Name               : Wide_String;
      Binding_Config     : Configuration)
      return Package_Handle;
   --  Get or Create a package of the name given in parameter. The name of the
   --  package is suposed to be a non-binding ada package name.

   function Get_Bound_Package_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the name of the bound package.

   function Get_Bound_Library_Unit_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the name of the bound library unit.

   function Get_Java_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the name of the package in the Java architecture, prefixed by
   --  the java base package if any.

   function Get_Binding_Package_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the name of the package where the binding is done, on the
   --  Ada side.

   function Get_JNI_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the JNI name of the package

   function Get_Configuration
     (Pckg : not null access Package_Record) return Configuration;
   --  Return the configuration data for the given package.

   function Get_Last_Child_Name
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return the name of the last element in the package name. E.g., in A.B.C,
   --  it will return C.

   type Clause_Kinds is (With_Clause, Use_Clause);

   procedure Add_Clause
     (Pckg : not null access Package_Record;
      Name : Wide_String;
      Kind : Clause_Kinds);
   --  Add a use or a with clause on with the resulting package is supposed to
   --  depend. The addition of a use clause will automatically add a with
   --  clause as well.

   function Get_Clauses
     (Pckg : not null access Package_Record) return Wide_String;
   --  Return a string containing the program text of all the clauses
   --  concatened for this package.

   function Get_Parent_Package
     (Pckg : not null access Package_Record)
      return access Package_Record;
   --  Return the parent package if any, null if none.

   function Is_From_Ada
     (Pckg : not null access Package_Record) return Boolean;
   --  Return true if this package has been created after an Ada package, false
   --  otherwise.

   procedure Set_Is_From_Ada
     (Pckg : not null access Package_Record; Value : Boolean);

   function Is_Library_Unit
     (Pckg : not null access Package_Record) return Boolean;
   --  Return true if this package is a library unit, false if it's a nested
   --  package

   function Get_Library_Unit
     (Pckg : not null access Package_Record) return Package_Handle;
   --  Return the library unit where this package is declared

private

   --  type Clause_Set is array (Clause_Kinds) of Boolean;
   --  This has been disabled, due to FC12-002

   type Clause_Set is record
      Use_Clause : Boolean := False;
      With_Clause : Boolean := False;
   end record;

   package Clause_Container is new Ada.Containers.Indefinite_Ordered_Maps
     (Wide_String, Clause_Set);

   use Clause_Container;

   type Package_Record is limited record
      Ada_Binding_Name : Unbounded_Wide_String;
      Ada_JNI_Name     : Unbounded_Wide_String;
      Java_Name        : Unbounded_Wide_String;
      Bound_Pckg       : Unbounded_Wide_String;

      Clauses          : Clause_Container.Map;

      Binding_Config   : Configuration;

      Parent_Package   : Package_Handle;

      From_Ada_Package : Boolean := True;

      Is_Library_Unit  : Boolean := True;
   end record;

end Ada2Java.Packages;
