-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                     Copyright (C) 2007, AdaCore                   --
--                                                                   --
-- GPS is free  software;  you can redistribute it and/or modify  it --
-- under the terms of the GNU General Public License as published by --
-- the Free Software Foundation; either version 2 of the License, or --
-- (at your option) any later version.                               --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters.Conversions;
with Ada.Containers.Ordered_Sets;
with Ada.Directories;            use Ada.Directories;
with Ada.Strings.Wide_Unbounded; use Ada.Strings.Wide_Unbounded;
with Ada.Wide_Text_IO;           use Ada.Wide_Text_IO;

with Asis;                   use Asis;
with Asis.Implementation;
with Asis.Iterator;
with Asis.Ada_Environments;
with Asis.Compilation_Units;
with Asis.Elements;          use Asis.Elements;
with Asis.Extensions;        use Asis.Extensions;

with GNAT.Command_Line;         use GNAT.Command_Line;
with GNAT.Directory_Operations; use GNAT.Directory_Operations;
with GNAT.OS_Lib;               use GNAT.OS_Lib;

with Ada2Java.Bound_Elements; use Ada2Java.Bound_Elements;
with Ada2Java.Config;          use Ada2Java.Config;
with Ada2Java.Kernel;          use Ada2Java.Kernel;
with Ada2Java.Packages;        use Ada2Java.Packages;

procedure Ada2Java.Main is

   My_Kernel : aliased Kernel_Record;

   function To_Path (Package_Name : String) return String;

   function Get_File_Name (Path : Wide_String) return Wide_String;

   function To_Path (Package_Name : String) return String is
      Returned : String := Package_Name;
   begin
      for J in Returned'Range loop
         if Returned (J) = '.' then
            Returned (J) := Dir_Separator;
         end if;
      end loop;

      return Returned;
   end To_Path;

   function Get_File_Name (Path : Wide_String) return Wide_String is
      Dir_Sep : constant Wide_Character := To_Wide_Character (Dir_Separator);
      Index   : Integer := Path'First;
   begin
      for J in Path'Range loop
         if Path (J) = Dir_Sep then
            Index := J + 1;
         end if;
      end loop;

      return Path (Index .. Path'Last);
   end Get_File_Name;

   procedure Pre_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer);

   procedure Post_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer);

   procedure Pre_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer) is
      pragma Unreferenced (Control, State);
   begin
      if Element_Kind (Element) = A_Declaration then
         Add_Bound_Element
           (Get_Bound_Elements_DB (My_Kernel'Access).all, Element);
      end if;
   end Pre_Procedure;

   procedure Post_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Integer) is
      pragma Unreferenced (Element, Control, State);
   begin
      null;
   end Post_Procedure;

   procedure Traverse_Node is new Asis.Iterator.Traverse_Element
     (Integer, Pre_Procedure, Post_Procedure);

   My_Context : Asis.Context;

   The_Declaration : Asis.Declaration;

   State   : Integer := 0;
   Control : Asis.Traverse_Control := Asis.Continue;

   Success : Boolean;

   Base_Package    : Unbounded_Wide_String;
   Class_Path      : Unbounded_Wide_String;
   Ada_Base_Folder : Unbounded_Wide_String;
   Project         : Unbounded_Wide_String;

   package File_List is new Ada.Containers.Ordered_Sets
     (Unbounded_Wide_String);
   use File_List;

   Files : File_List.Set;
begin
   Initialize_Option_Scan;

   loop
      case Getopt ("p: b: c: o:") is
         when ASCII.NUL => exit;

         when 'b' =>
            Base_Package := To_Unbounded_Wide_String
              (To_Wide_String (Parameter));

         when 'c' =>
            Class_Path := To_Unbounded_Wide_String
              (To_Wide_String (Parameter));

         when 'o' =>
            Ada_Base_Folder := To_Unbounded_Wide_String
              (To_Wide_String (Parameter));

         when 'p' =>
            Project := To_Unbounded_Wide_String
              (To_Wide_String (Project));

         when others =>
            raise Program_Error;         -- cannot occur!
      end case;
   end loop;

   Set_Java_Base_Package
     (My_Kernel'Access,
      Create_Package
        (My_Kernel'Access,
         To_Wide_String (Base_Package),
         Null_Configuration,
         False));

   Set_Java_Class_Path (My_Kernel'Access, To_Wide_String (Class_Path));

   Asis.Implementation.Initialize ("-asis05");

   Asis.Ada_Environments.Associate (My_Context, "CONTEXT");
   Asis.Ada_Environments.Open (My_Context);

   loop
      declare
         S : String_Access := new String'(Get_Argument (Do_Expansion => True));
      begin
         if S.all'Length = 0 then
            Free (S);
            exit;
         end if;

         if To_Wide_String (Project) /= "" then
            Asis.Extensions.Compile
              (Source_File  => S,
               Args         => (1 => new String'("-gnat05"),
                                2 => new String'("-P"),
                                3 => new String'
                                  (To_String (To_Wide_String (Project)))),
               Success      => Success,
               Use_GNATMAKE => True,
               GCC          => new String'("gnatmake"));
         else
            Asis.Extensions.Compile
              (Source_File  => S,
               Args         => (1 => new String'("-gnat05")),
               Success      => Success,
               Use_GNATMAKE => True);
         end if;

         Insert
           (Files,
            To_Unbounded_Wide_String (Get_File_Name (To_Wide_String (S.all))));

         Free (S);
      end;

   end loop;

   --  Generate the binding elements for all library units

   declare
      Units : constant Asis.Compilation_Unit_List :=
        Compilation_Units.Library_Unit_Declarations (My_Context);

      Unit_Spec : Asis.Compilation_Unit;
      Unit_Body : Asis.Compilation_Unit;

      use Compilation_Units;
   begin
      for J in Units'Range loop
         Unit_Spec := Units (J);
         Unit_Body := Corresponding_Body (Unit_Spec);

         if Unit_Kind (Unit_Spec) = A_Package then

            --  If the body or the spec of the unit is in one of the compiled
            --  units, then bind this unit

            if Contains
              (Files,
               To_Unbounded_Wide_String
                 (Get_File_Name (Text_Name (Unit_Spec))))
              or else
                (not Is_Nil (Unit_Body)
                 and then Contains
                   (Files,
                    To_Unbounded_Wide_String
                      (Get_File_Name (Text_Name (Unit_Body)))))
            then
               The_Declaration := Asis.Elements.Unit_Declaration (Units (J));
               Traverse_Node (The_Declaration, Control, State);
            end if;
         end if;
      end loop;
   end;

   Generate_Binding (My_Kernel'Access);

   declare
      use Binding_File_List_Pckg;
      It : Binding_Files_Cursor;

      Conf_Class_Path : Unbounded_Wide_String;
      Conf_Ada_Dir    : Unbounded_Wide_String;
   begin
      It := Binding_File_List_Pckg.First
        (Get_Binding_Files_DB (My_Kernel'Access).all);

      Create_Path (To_String (To_Wide_String (Class_Path)));
      Create_Path (To_String (To_Wide_String (Ada_Base_Folder)));

      while It /= Binding_File_List_Pckg.No_Element loop
         declare
            File   : Ada.Wide_Text_IO.File_Type;
            Source : constant Binding_File :=
              Binding_File_List_Pckg.Element (It);
            File_Name : constant String :=
              To_String (To_Wide_String (Key (It).File_Name));
            Package_Name : constant String :=
              To_String (To_Wide_String (Key (It).Package_Name));
         begin
            if Source.Binding_Config /= Null_Configuration then
               Conf_Class_Path := To_Unbounded_Wide_String
                 (Get_Java_Class_Path (Source.Binding_Config));
               Conf_Ada_Dir := To_Unbounded_Wide_String
                 (Get_Ada_Output_Directory (Source.Binding_Config));

               Create_Path (To_String (To_Wide_String (Conf_Class_Path)));
               Create_Path (To_String (To_Wide_String (Conf_Ada_Dir)));
            else
               Conf_Class_Path := Class_Path;
               Conf_Ada_Dir := Ada_Base_Folder;
            end if;

            if Source.Language = Ada_Lang then
               Create
                 (File,
                  Out_File,
                  To_String (To_Wide_String (Conf_Ada_Dir))
                  & Dir_Separator
                  & File_Name);
            elsif Source.Language = Java_Lang then
               declare
                  Java_Path : constant String :=
                    To_String (To_Wide_String (Conf_Class_Path))
                    & Dir_Separator
                    & To_Path (Package_Name);
               begin
                  Create_Path (Java_Path);

                  Create
                    (File, Out_File, Java_Path & Dir_Separator & File_Name);
               end;
            end if;

            Put_Line (File, To_Wide_String (Assemble (Source.all)));
            Close (File);
            Next (It);
         end;
      end loop;
   end;
end Ada2Java.Main;
