-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers.Doubly_Linked_Lists;

with Ada2Java.Bound_Elements;      use Ada2Java.Bound_Elements;
with Ada2Java.Config;              use Ada2Java.Config;
with Ada2Java.Packages;            use Ada2Java.Packages;
with Ada2Java.Bound_Units;         use Ada2Java.Bound_Units;
with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Simplifications;     use Ada2Java.Simplifications;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;

package Ada2Java.Kernel is

   type Kernel_Record is private;
   --  A kernel holds data that needs to be accessible widely throughout
   --  the binding process.

   function Get_Bound_Elements_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Elements_DB;
   --  Return the bound elements database.

   function Get_Binding_Units_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Units_DB;
   --  Return the bound files database.

   function Get_Bound_Packages_DB
     (Handle : not null access Kernel_Record)
      return not null access Bound_Packages_DB;

   procedure Set_Default_Configuration
     (Handle : not null access Kernel_Record;
      Conf   : Configuration);
   --  Define the default values of the configuration parameters.

   function Get_Default_Configuration
     (Handle : not null access Kernel_Record) return Configuration;
   --  Return the default configuration for the kernel

   ------------------------------
   -- Bound_Elements_Generator --
   ------------------------------

   type Create_Bound_Element is access
     function (Element : Simple_Element_View_Access)
               return Bound_Element;
   --  This function returns a bound element if it knows which type is able to
   --  bind the element given in parameter, null otherwise.

   procedure Add_Bound_Element_Creator
     (Handle : not null access Kernel_Record;
      Sb     : Create_Bound_Element);
   --  Add a bound element generator in the bound element list of the kernel.

   function Create_Corresponding_Bound_Element
     (Handle  : not null access Kernel_Record;
      Element : Simple_Element_View_Access)
      return Bound_Element;
   --  Call the various bound elements generators and return the first bound
   --  element found.

   function Get_Simple_Elements_DB
     (Handle : not null access Kernel_Record)
      return not null access Simple_Elements_DB_Pckg.Map;

   function Get_Library_Binding_Unit
     (Handle : not null access Kernel_Record)
      return Bound_Unit;
   --  Return the unit holding global properties & subprograms for the bound
   --  code.

   function Generate_Register_Native_Calls
     (Handle : not null access Kernel_Record) return Dynamic_Expression;
   --  Return a list of instructions calling all Reference_Native subprograms
   --  of the application. Return the null instruction if none.

private

   package Create_Bound_Elements_Lists is new
     Ada.Containers.Doubly_Linked_Lists (Create_Bound_Element);

   use Create_Bound_Elements_Lists;

   type Kernel_Record is record
      Elements             : aliased Bound_Elements_DB;
      Units                : aliased Bound_Units_DB;
      Packages             : aliased Bound_Packages_DB;
      Simplified_Elements  : aliased Simple_Elements_DB_Pckg.Map;
      Default_Conf         : Configuration;
      Bound_Elements_Fcts  : Create_Bound_Elements_Lists.List;
      Library_Binding_Unit : Bound_Unit;
   end record;

end Ada2Java.Kernel;
