-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Characters.Conversions; use Ada.Characters.Conversions;

package body Ada2Java.Dynamic_Expressions is

   New_Line_Constant : constant Wide_String :=
     (1 => To_Wide_Character (ASCII.LF));

   ----------------------------
   -- New_Dynamic_Expression --
   ----------------------------

   function New_Dynamic_Expression return Dynamic_Expression is
   begin
      return new Dynamic_Expression_Record;
   end New_Dynamic_Expression;

   ---------
   -- "&" --
   ---------

   function "&" (Right, Left : Dynamic_Expression) return Dynamic_Expression is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      if Right /= null then
         Append
           (Node.List,
            (The_Type => Expression, Size => 0, Contents_Exp => Right));
         Right.Ref_Counter := Right.Ref_Counter + 1;
      end if;

      if Left /= null then
         Append
           (Node.List,
            (The_Type => Expression, Size => 0, Contents_Exp => Left));

         Left.Ref_Counter := Left.Ref_Counter + 1;
      end if;

      return Node;
   end "&";

   ---------
   -- "&" --
   ---------

   function "&"
     (Right : Wide_String; Left : Dynamic_Expression)
      return Dynamic_Expression
   is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      Append
        (Node.List,
         (The_Type => Litteral, Size => Right'Length, Contents_Str => Right));

      if Left /= Empty_Dynamic_Expression then
         Append
           (Node.List,
            (The_Type => Expression, Size => 0, Contents_Exp => Left));

         Left.Ref_Counter := Left.Ref_Counter + 1;
      end if;

      return Node;
   end "&";

   ---------
   -- "&" --
   ---------

   function "&"
     (Right : Dynamic_Expression; Left : Wide_String)
      return Dynamic_Expression
   is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      if Right /= null then
         Append
           (Node.List,
            (The_Type => Expression, Size => 0, Contents_Exp => Right));

         Right.Ref_Counter := Right.Ref_Counter + 1;
      end if;

      Append
        (Node.List,
         (The_Type => Litteral, Size => Left'Length, Contents_Str => Left));

      return Node;
   end "&";

   ------------
   -- Append --
   ------------

   procedure Append
     (Exp : in out Dynamic_Expression; Content : Wide_String) is
   begin
      Append
        (Exp.List,
         (The_Type     => Litteral,
          Size         => Content'Length,
          Contents_Str => Content));
   end Append;

   ------------
   -- Append --
   ------------

   procedure Append
     (Exp : in out Dynamic_Expression; Content : Dynamic_Expression) is
   begin
      Append
        (Exp.List,
         (The_Type     => Expression,
          Size         => 0,
          Contents_Exp => Content));
   end Append;

   -------------
   -- Prepend --
   -------------

   procedure Prepend
     (Exp : in out Dynamic_Expression; Content : Wide_String) is
   begin
      Prepend
        (Exp.List,
         (The_Type     => Litteral,
          Size         => Content'Length,
          Contents_Str => Content));
   end Prepend;

   -------------
   -- Prepend --
   -------------

   procedure Prepend
     (Exp : in out Dynamic_Expression; Content : Dynamic_Expression) is
   begin
      Prepend
        (Exp.List,
         (The_Type     => Expression,
          Size         => 0,
          Contents_Exp => Content));
   end Prepend;

   ----------------------------
   -- Set_Indentation_Offset --
   ----------------------------

   procedure Set_Indentation_Offset
     (Exp : in out Dynamic_Expression; Offset : Integer) is
   begin
      Exp.Indentation_Offset := Offset;
   end Set_Indentation_Offset;

   --------------
   -- New_Line --
   --------------

   function New_Line (Offset : Integer := 0) return Dynamic_Expression is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      Node.Indentation_Offset := Offset;
      Append
        (Node.List,
         (The_Type     => Litteral,
          Size         => 1,
          Contents_Str => New_Line_Constant));

      return Node;
   end New_Line;

   ------------
   -- Indent --
   ------------

   function Indent (Offset : Integer) return Dynamic_Expression is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      Node.Indentation_Offset := Offset;
      Append
        (Node.List,
         (The_Type => Litteral, Size  => 0, Contents_Str => ""));

      return Node;
   end Indent;

   --------------------
   -- Set_Expression --
   --------------------

   function Set_Expression
     (Src : Dynamic_Expression; Value : Dynamic_Expression)
      return Dynamic_Expression
   is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      Append
        (Node.List,
         (The_Type => Set_Exp,
          Size     => 0,
          Src      => Src,
          Value    => Value));

      return Node;
   end Set_Expression;

   --------------------
   -- To_Wide_String --
   --------------------

   function To_Wide_String
     (Exp : Dynamic_Expression; Base_Indentation : access Integer := null)
      return Wide_String
   is
      Buffer  : Unbounded_Wide_String;
      Current : Cursor;

      Local_Base_Indentation : aliased Integer := 0;
      Local_Base_Indentation_Pointer : access Integer := Base_Indentation;

      function Make_Indent return Wide_String;
      --  Creates a string corresponding to the current level of indentation.

      -----------------
      -- Make_Indent --
      -----------------

      function Make_Indent return Wide_String is
         Ind : constant Wide_String (1 .. Base_Indentation.all * 3) :=
           (others => ' ');
      begin
         return Ind;
      end Make_Indent;

   begin
      if Exp = null then
         return "";
      end if;

      Current := First (Exp.List);

      if Local_Base_Indentation_Pointer = null then
         Local_Base_Indentation_Pointer := Local_Base_Indentation'Access;
      end if;

      Local_Base_Indentation_Pointer.all :=
        Local_Base_Indentation_Pointer.all + Exp.Indentation_Offset;

      while Current /= No_Element loop
         case Element (Current).The_Type is
            when Expression =>
               Append
                 (Buffer,
                  To_Wide_String
                    (Element (Current).Contents_Exp,
                     Local_Base_Indentation_Pointer));

            when Litteral =>
               --  ??? Should handle the case where a new line is not
               --  necessary a LF !

               if Element (Current).Contents_Str = New_Line_Constant then
                  Append (Buffer, New_Line_Constant & Make_Indent);
               else
                  Append (Buffer, Element (Current).Contents_Str);
               end if;

            when Set_Exp =>
               declare
                  Src, Value : Dynamic_Expression;
               begin
                  Src := Element (Current).Src;
                  Value := Element (Current).Value;
                  Empty (Src);
                  Append (Src, Value);
               end;

         end case;

         Current := Next (Current);
      end loop;

      return To_Wide_String (Buffer);
   end To_Wide_String;

   ---------------------------
   -- To_Dynamic_Expression --
   ---------------------------

   function To_Dynamic_Expression
     (Str : Wide_String) return Dynamic_Expression
   is
      Node : constant Dynamic_Expression := new Dynamic_Expression_Record;
   begin
      Append
        (Node.List,
         (The_Type => Litteral, Size => Str'Length, Contents_Str => Str));

      return Node;
   end To_Dynamic_Expression;

   ----------
   -- Free --
   ----------

   procedure Free (Exp : in out Dynamic_Expression) is
      pragma Unreferenced (Exp);
   begin
      null;
   end Free;

   -----------
   -- Write --
   -----------

   procedure Write
     (Exp : Dynamic_Expression; File : Ada.Wide_Text_IO.File_Type)
   is

      Last_Is_LF : Boolean := False;

      procedure Make_Indent (Indent : Integer);

      procedure Internal_Write
        (Exp : Dynamic_Expression; Indent : in out Integer);

      -----------------
      -- Make_Indent --
      -----------------

      procedure Make_Indent (Indent : Integer) is
         Ind_Str : constant Wide_String (1 .. Indent * 3) := (others => ' ');
      begin
         Put (File, Ind_Str);
      end Make_Indent;

      --------------------
      -- Internal_Write --
      --------------------

      procedure Internal_Write
        (Exp : Dynamic_Expression; Indent : in out Integer)
      is
         Current : Cursor;
      begin
         if Exp = null then
            return;
         end if;

         Current := First (Exp.List);

         Indent := Indent + Exp.Indentation_Offset;

         while Current /= No_Element loop
            case Element (Current).The_Type is
               when Expression =>
                  Internal_Write (Element (Current).Contents_Exp, Indent);
               when Litteral =>
                  if Element (Current).Contents_Str = New_Line_Constant then

                     --  ??? Should handle the case where a new line is not
                     --  necessary a LF !

                     Last_Is_LF := True;
                  elsif Last_Is_LF then
                     Make_Indent (Indent);
                     Last_Is_LF := False;
                  end if;

                  Put (File, Element (Current).Contents_Str);

               when Set_Exp =>
                  declare
                     Src, Value : Dynamic_Expression;
                  begin
                     Src := Element (Current).Src;
                     Value := Element (Current).Value;
                     Empty (Src);
                     Append (Src, Value);
                  end;

            end case;

            Current := Next (Current);
         end loop;
      end Internal_Write;

      Base_Indent : Integer := 0;

   begin
      Internal_Write (Exp, Base_Indent);
   end Write;

   ----------
   -- Copy --
   ----------

   function Copy (Exp : Dynamic_Expression) return Dynamic_Expression is
      New_Exp : constant Dynamic_Expression := New_Dynamic_Expression;
      Current : Cursor;
   begin
      Current := First (Exp.List);

      while Current /= No_Element loop
         Append (New_Exp.List, Element (Current));

         Current := Next (Current);
      end loop;

      return New_Exp;
   end Copy;

   ---------------
   -- Deep_Copy --
   ---------------

   function Deep_Copy (Exp : Dynamic_Expression) return Dynamic_Expression is
      New_Exp : constant Dynamic_Expression := New_Dynamic_Expression;
      Current : Cursor;
   begin
      if Exp = Empty_Dynamic_Expression then
         return New_Exp;
      end if;

      Current := First (Exp.List);

      while Current /= No_Element loop
         declare
            Item : constant Expression_Item := Element (Current);
         begin
            case Item.The_Type is
               when Expression =>
                  Append
                    (New_Exp.List, (Expression, 0, Copy (Item.Contents_Exp)));

               when Litteral =>
                  Append
                    (New_Exp.List,
                     (Litteral, Item.Contents_Str'Length, Item.Contents_Str));

               when Set_Exp =>
                  Append
                    (New_Exp.List,
                     (Set_Exp, 0, Copy (Item.Src), Copy (Item.Value)));

            end case;
         end;

         Current := Next (Current);
      end loop;

      return New_Exp;
   end Deep_Copy;

   procedure Empty (Exp : in out Dynamic_Expression) is
   begin
      Clear (Exp.List);
   end Empty;

   Last_Check_Val : Integer := 0;

   procedure Check_Cycles (Exp : Dynamic_Expression) is
      Current_Check_Val : constant Integer := Last_Check_Val + 1;

      procedure Internal_Check_Cycles (Exp : Dynamic_Expression);

      procedure Internal_Check_Cycles (Exp : Dynamic_Expression) is
         Current : Cursor;
      begin
         if Exp.Cycle_Tag = Current_Check_Val then
            raise Dynamic_Expression_Error with Exp.Cycle_Tag'Img;
         end if;

         Exp.Cycle_Tag := Current_Check_Val;
         Current := First (Exp.List);

         while Current /= No_Element loop
            case Element (Current).The_Type is
               when Expression =>
                  Internal_Check_Cycles (Element (Current).Contents_Exp);

               when Litteral =>
                  null;

               when Set_Exp =>
                  declare
                     Src, Value : Dynamic_Expression;
                  begin
                     Src := Element (Current).Src;
                     Value := Element (Current).Value;
                     Empty (Src);
                     Append (Src, Value);
                  end;
            end case;

            Current := Next (Current);
         end loop;

         Exp.Cycle_Tag := 0;
      end Internal_Check_Cycles;
   begin
      Last_Check_Val := Last_Check_Val + 1;

      Internal_Check_Cycles (Exp);
   end Check_Cycles;

   -----------------
   -- Debug_Print --
   -----------------

   procedure Debug_Print (Exp : access Dynamic_Expression) is
   begin
      Put_Line (Exp.all);
   end Debug_Print;

   --------------
   -- Put_Line --
   --------------

   procedure Put_Line (Exp : Dynamic_Expression) is
   begin
      Put_Line (To_Wide_String (Exp));
   end Put_Line;

end Ada2Java.Dynamic_Expressions;
