-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers.Ordered_Maps;

with Asis;                        use Asis;

with Ada2Java.Utils;           use Ada2Java.Utils;
with Ada2Java.Bound_Units;     use Ada2Java.Bound_Units;
with Ada2Java.Packages;        use Ada2Java.Packages;
with Ada2Java.Simplifications; use Ada2Java.Simplifications;
with Ada2Java.Dynamic_Expressions; use Ada2Java.Dynamic_Expressions;

limited with Ada2Java.Kernel;

package Ada2Java.Bound_Elements is

   ----------------------
   --  Binding_Element --
   ----------------------

   type Bound_Element_Record is abstract tagged private;

   type Bound_Element is access all Bound_Element_Record'Class;

   procedure Initialize
     (Handle        : not null access Kernel.Kernel_Record;
      Element       : Simple_Element_View_Access;
      Element_Bound : access Bound_Element_Record);
   --  Initializes default properties of the element

   procedure Bind
     (Element_Bound : access Bound_Element_Record;
      Handle        : not null access Kernel.Kernel_Record) is abstract;
   --  Implementors of this method should complete the various binding file to
   --  bind the given element.

   function "<" (Left, Right : Asis.Element) return Boolean;
   --  Needed to instantiate Bound_Element_DB

   package Bound_Elements_DB_Pckg is new Ada.Containers.Ordered_Maps
     (Element_Index, Bound_Element, "<");

   use Bound_Elements_DB_Pckg;

   subtype Bound_Elements_DB is Bound_Elements_DB_Pckg.Map;

   subtype Bound_Elements_DB_Cursor is Bound_Elements_DB_Pckg.Cursor;

   procedure Add_Bound_Element
     (Kernel  : not null access Ada2Java.Kernel.Kernel_Record;
      Element : Simple_Element_View_Access);
   --  Add an element to bind in the database, if the given element is not
   --  supported, then a limited verbosity message will be send.

   function Get_Bound_Element
     (Kernel  : access Ada2Java.Kernel.Kernel_Record;
      Element : Simple_Element_View_Access) return Bound_Element;
   --  Return the bound element corresponding to the given Asis element.
   --  Create one if needed.

   procedure  Add_Bound_Element
     (Kernel  : access Ada2Java.Kernel.Kernel_Record;
      Element : Bound_Element;
      Index   : Element_Index);
   --  Add the element given in parameter at the given address

   function Get_Bound_Element
     (Kernel : access Ada2Java.Kernel.Kernel_Record;
      Index  : Element_Index) return Bound_Element;
   --  Return the element stored at the given index.

   function Get_Package
     (Element : access Bound_Element_Record'Class) return Package_Handle;
   --  Return the package where this bound element is stored

   function Get_Array_Bounds
     (Element : Simple_Type_View_Access;
      Variable : Dynamic_Expression) return Dynamic_Expression;
   --  Return the list of bounds of a given type
   --  (e.g. A'First (1), A'Last (1), A'First (2), A'Last (2), etc...)

private

   type Bound_Element_Record is abstract tagged record
      Kernel         : access Kernel.Kernel_Record;
      Enclosing_Unit : Bound_Unit;
      Simple_Element : Simple_Element_View_Access;
   end record;

   Ind : constant Wide_String := "   ";

   procedure Create_Unrestricted_Access
     (Handle       : not null access Kernel.Kernel_Record;
      Unit         : Bound_Unit;
      Initial_Type : Simple_Type_Reference;
      New_Type     : out Simple_Type_Reference;
      Created      : out Boolean);
   --  Create an unrestricted access if the type given in parameter has to
   --  be acceded trough a reference and not a value. Created is False
   --  otherwise, and New_Type set to Initial_Type.

end Ada2Java.Bound_Elements;
