-----------------------------------------------------------------------
--                             Ada2Java                              --
--                                                                   --
--                  Copyright (C) 2007-2009, AdaCore                 --
--                                                                   --
-- Ada2Java is free software;  you can redistribute it and/or modify --
-- it under the terms of the GNU General Public License as published --
-- by the Free Software Foundation; either version 2 of the License, --
-- or (at your option) any later version.                            --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Containers;              use Ada.Containers;
with Ada.Characters.Conversions;  use Ada.Characters;
with Ada.Exceptions;              use Ada.Exceptions;
with Ada.Strings;                 use Ada.Strings;
with Ada.Strings.Fixed;           use Ada.Strings.Fixed;

with Ada2Java.Kernel;              use Ada2Java.Kernel;
with Ada2Java.Bound_Elements.Subprograms.Data;
use Ada2Java.Bound_Elements.Subprograms.Data;
with Ada2Java.Utils;               use Ada2Java.Utils;
with Ada2Java.Bound_Elements.Tagged_Types;
use Ada2Java.Bound_Elements.Tagged_Types;

package body Ada2Java.Bound_Elements.Subprograms is

   type Ada_Procedure_Wrapper is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Procedure_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Function_Wrapper is new Code_Wrapper with record
      In_JNI : Boolean;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Function_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Procedure_Wrapper is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Procedure_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Default_Ada_Parameter_Handler is new Code_Wrapper with record
      In_JNI : Boolean;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Default_Ada_Parameter_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Call_A_To_J_JNI_Param is new Code_Wrapper with record
      Params_Array_Id        : Dynamic_Expression;
      Params_Array_Aggregate : Dynamic_Expression;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Call_A_To_J_JNI_Param;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Call_A_To_J_High_Param is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Call_A_To_J_High_Param;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Default_Java_Parameter_Handler is new Code_Wrapper with record
      null;
   end record;

   overriding
   procedure Wrap
     (Wrapper    : in out Default_Java_Parameter_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Ada_Exception_Wrapper is new Code_Wrapper with null record;

   overriding
   procedure Wrap
     (Wrapper    : in out Ada_Exception_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   type Java_Locking_Wrapper is new Code_Wrapper with null record;
   --  This wrapper adds lock checks depending on the configuration of the
   --  entity.

   overriding
   procedure Wrap
     (Wrapper    : in out Java_Locking_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class);

   ---------------------
   -- Bind_Subprogram --
   ---------------------

   procedure Bind_Subprogram
     (Handle            : not null access Kernel.Kernel_Record;
      View              : Simple_Subprogram_View_Access;
      Unit              : Bound_Unit;
      Wrappers          : Wrapping_Map;
      Body_For_Abstract : Boolean := False;
      Add_Java_High_Sb  : Boolean := True)
   is
      Local_Wrappers : Wrapping_Map := Wrappers;

      Ada_JNI_Declaration   : Dynamic_Expression := New_Dynamic_Expression;
      Ada_High_Declaration  : Dynamic_Expression := New_Dynamic_Expression;
      Ada_High_Generic_Decl : Dynamic_Expression := New_Dynamic_Expression;
      Java_JNI_Function     : Dynamic_Expression := New_Dynamic_Expression;

      --  The various pieces of the subprograms that have to be created:

      Ada_Bound_Name : Dynamic_Expression := New_Dynamic_Expression;

      Ada_High_Profile : Dynamic_Expression := New_Dynamic_Expression;
      Ada_High_Name    : Dynamic_Expression := New_Dynamic_Expression;

      Ada_High_Subprogram : Dynamic_Expression := New_Dynamic_Expression;
      Ada_High_Bloc       : constant Wrapping_Bloc :=
        new Wrapping_Bloc_Record'
          (Code_Node         => New_Dynamic_Expression,
           Declarations      => New_Dynamic_Expression,
           Statements_Before => New_Dynamic_Expression,
           Statements_After  => New_Dynamic_Expression);
      Ada_High_Expression : constant Wrapping_Expression :=
        new Wrapping_Expression_Record'
          (Expression           => New_Dynamic_Expression,
           Bloc                 => Ada_High_Bloc,
           Enclosing_Expression => null,
           Kind                 => Unknown_Kind);

      Ada_Subprogram_Type   : Dynamic_Expression := New_Dynamic_Expression;
      Ada_Method_ID_Profile : Dynamic_Expression := New_Dynamic_Expression;
      Ada_Method_ID_Init    : Dynamic_Expression := New_Dynamic_Expression;

      Ada_JNI_Profile      : Dynamic_Expression := New_Dynamic_Expression;
      Ada_JNI_Name         : Dynamic_Expression := New_Dynamic_Expression;
      Ada_JNI_Reg_Natives  : Dynamic_Expression := New_Dynamic_Expression;

      Ada_JNI_Subprogram : Dynamic_Expression := New_Dynamic_Expression;
      Ada_JNI_Bloc       : constant Wrapping_Bloc :=
        new Wrapping_Bloc_Record'
          (Code_Node         => New_Dynamic_Expression,
           Declarations      => New_Dynamic_Expression,
           Statements_Before => New_Dynamic_Expression,
           Statements_After  => New_Dynamic_Expression);
      Ada_JNI_Expression : constant Wrapping_Expression :=
        new Wrapping_Expression_Record'
          (Expression           => New_Dynamic_Expression,
           Bloc                 => Ada_JNI_Bloc,
           Enclosing_Expression => null,
           Kind                 => Unknown_Kind);

      Mangled_Name            : Dynamic_Expression := New_Dynamic_Expression;
      Package_Mangled_Name    : Dynamic_Expression := New_Dynamic_Expression;
      Subprogram_Mangled_Name : Dynamic_Expression := New_Dynamic_Expression;
      Mangled_Parameters      : Dynamic_Expression := New_Dynamic_Expression;
      Mangled_Return_Type     : Dynamic_Expression := New_Dynamic_Expression;

      Java_JNI_Profile : Dynamic_Expression := New_Dynamic_Expression;
      Java_JNI_Name    : Dynamic_Expression := New_Dynamic_Expression;
      Java_JNI_Call    : Dynamic_Expression := New_Dynamic_Expression;

      Java_High_Profile  : Dynamic_Expression := New_Dynamic_Expression;
      Java_High_Name     : Dynamic_Expression := New_Dynamic_Expression;

      Java_High_Subprogram : Dynamic_Expression :=
        New_Dynamic_Expression;
      Java_High_Bloc       : constant Wrapping_Bloc :=
        new Wrapping_Bloc_Record'
          (Code_Node         => New_Dynamic_Expression,
           Declarations      => New_Dynamic_Expression,
           Statements_Before => New_Dynamic_Expression,
           Statements_After  => New_Dynamic_Expression);
      Java_High_Expression : constant Wrapping_Expression :=
        new Wrapping_Expression_Record'
          (Expression           => New_Dynamic_Expression,
           Bloc                 => Java_High_Bloc,
           Enclosing_Expression => null,
           Kind                 => Unknown_Kind);

      Java_JNI_Subprogram : Dynamic_Expression :=
        New_Dynamic_Expression;
      Java_JNI_Bloc       : constant Wrapping_Bloc :=
        new Wrapping_Bloc_Record'
          (Code_Node         => New_Dynamic_Expression,
           Declarations      => New_Dynamic_Expression,
           Statements_Before => New_Dynamic_Expression,
           Statements_After  => New_Dynamic_Expression);
      Java_JNI_Expression : constant Wrapping_Expression :=
        new Wrapping_Expression_Record'
          (Expression           => New_Dynamic_Expression,
           Bloc                 => Java_JNI_Bloc,
           Enclosing_Expression => null,
           Kind                 => Unknown_Kind);

      Returned_Type : Bound_Data;

      Properties : aliased Subprogram_Properties;

      Number_Of_Params_To_Java : Integer := 0;
      --  Number of parameters passed to java, in the case of an Ada_To_Java
      --  call. May be dependent on e.g. controlling parameters.

      Subprogram_Wrapper_Params : aliased Subprogram_Params :=
        (Handle           => Handle,
         Subprogram_View  => View,
         Unit             => Unit,
         Java_JNI_Name    => Java_JNI_Name,
         Ada_High_Sb_Name => Ada_High_Name,
         Ada_JNI_Sb_Name  => Ada_JNI_Name,
         others           => <>);

      ----------------------
      -- Handle_Parameter --
      ----------------------

      procedure Handle_Parameter
        (Element  : Parameter_Simple_View_Access;
         Number   : Integer;
         Is_Last  : Boolean);

      procedure Handle_Parameter
        (Element : Parameter_Simple_View_Access;
         Number  : Integer;
         Is_Last : Boolean)
      is
         Parameter_Wrapper_Params : aliased Parameter_Handlings_Params :=
           (Subprogram_Wrapper_Params with
            Parameter    => Element,
            Is_First     => Number = 1,
            Is_Last      => Is_Last,
            Param_Number => Number);

         Bound_Param : Bound_Data;

         Ada_JNI_Param_Expression : constant Wrapping_Expression :=
           new Wrapping_Expression_Record'
             (Expression           => Element.Get_Glue_Name,
              Bloc                 => Ada_JNI_Bloc,
              Enclosing_Expression => Ada_JNI_Expression,
              Kind                 => Unknown_Kind);

         Ada_High_Param_Expression : constant Wrapping_Expression :=
           new Wrapping_Expression_Record'
             (Expression           => Element.Get_Glue_Name,
              Bloc                 => Ada_High_Bloc,
              Enclosing_Expression => Ada_High_Expression,
              Kind                 => Unknown_Kind);

         Java_High_Param_Expression : constant Wrapping_Expression :=
           new Wrapping_Expression_Record'
             (Expression           => Element.Name,
              Bloc                 => Java_High_Bloc,
              Enclosing_Expression => Java_High_Expression,
              Kind                 => Unknown_Kind);

         Java_JNI_Param_Expression : constant Wrapping_Expression :=
           new Wrapping_Expression_Record'
             (Expression           => Element.Name,
              Bloc                 => Java_JNI_Bloc,
              Enclosing_Expression => Java_JNI_Expression,
              Kind                 => Unknown_Kind);
      begin
         Bound_Param := Create_Bound_Data
           (Simple_Object_View_Access (Element), Unit,
            Handle, View.Call_Convention);

         Call_Wrappers
           (Map                  => Local_Wrappers,
            Context              => Add_Parameter_Original,
            Params               => Parameter_Wrapper_Params'Access,
            Ada_High_Expression  => Ada_High_Param_Expression,
            Ada_JNI_Expression   => Ada_JNI_Param_Expression,
            Java_JNI_Expression  => Java_JNI_Param_Expression,
            Java_High_Expression => Java_High_Param_Expression);

         Adjust_Unit_For_Use (Bound_Param.Java_Part.all, Unit);
         Adjust_Unit_For_Use (Bound_Param.Ada_Part.all, Unit);

         Append
           (Java_High_Profile,
            Create_Parameters
              (Bound_Param.Java_Part.all, Properties'Access, View));

         Append
           (Java_JNI_Profile,
            Create_JNI_Parameters
              (Bound_Param.Java_Part.all, Properties'Access, View));
         Append
           (Ada_High_Profile,
            Create_Parameters
              (Bound_Param.Ada_Part.all, Properties'Access, View));
         Append
           (Ada_JNI_Profile,
            Create_JNI_Parameters
              (Bound_Param.Ada_Part.all, Properties'Access, View));

         Properties.Parameters_Written := Properties.Parameters_Written + 1;

         Append
           (Ada_Method_ID_Profile,
            Get_JNI_Type (Bound_Param.Java_Part.all));

         if View.Call_Convention = Java_To_Ada then
            Create_Conversion_From_JNI
              (D          => Bound_Param.Ada_Part.all,
               Expression => Ada_JNI_Param_Expression);

            Create_Conversion_To_JNI
              (D          => Bound_Param.Java_Part.all,
               Expression => Java_High_Param_Expression,
               Prop       => Properties'Access);
         else
            Create_Conversion_To_JNI
              (D          => Bound_Param.Ada_Part.all,
               Expression => Ada_High_Param_Expression);

            if Number > 1 or else not Element.Is_Primitive then
               Number_Of_Params_To_Java := Number_Of_Params_To_Java + 1;

               Create_Conversion_From_JNI
                 (D          => Bound_Param.Java_Part.all,
                  Expression => Java_JNI_Param_Expression);
            end if;
         end if;

         Call_Wrappers
           (Map                  => Local_Wrappers,
            Context              => Add_Parameter_Converted,
            Params               => Parameter_Wrapper_Params'Access,
            Ada_High_Expression  => Ada_High_Param_Expression,
            Ada_JNI_Expression   => Ada_JNI_Param_Expression,
            Java_JNI_Expression  => Java_JNI_Param_Expression,
            Java_High_Expression => Java_High_Param_Expression);

         Call_Wrappers
           (Map                  => Local_Wrappers,
            Context              => Parameter_Resolution,
            Params               => Parameter_Wrapper_Params'Access,
            Ada_High_Expression  => Ada_High_Param_Expression,
            Ada_JNI_Expression   => Ada_JNI_Param_Expression,
            Java_JNI_Expression  => Java_JNI_Param_Expression,
            Java_High_Expression => Java_High_Param_Expression);

         Append
           (Mangled_Parameters,
            Mangle_ID
              (To_Wide_String (Get_JNI_Type (Bound_Param.Java_Part.all))));

         Properties.Java_Call_Parameters_Written :=
           Properties.Java_Call_Parameters_Written + 1;
         Properties.Ada_Call_Parameters_Written :=
           Properties.Ada_Call_Parameters_Written + 1;
      exception
         when Silent_Not_Supported =>
            raise;

         when E : others =>
            Trace_With_Location
              ("unexpected exception - "
               & Conversions.To_Wide_String (Exception_Information (E)),
               Errors_Only);

            raise Silent_Not_Supported;
      end Handle_Parameter;

      Exception_Wrapper : aliased Ada_Exception_Wrapper;
      Loc : Location_Handle;

   begin --  Bind
      Loc := Push_Location (View.Location);

      --  Set code wrappers --

      if View.Is_Dispatching and then View.Call_Convention = Java_To_Ada
        and then View.Parameters /= null
        and then View.Parameters'Length > 0
        and then View.Parameters (1).Is_Controlling
        and then
          ((View.Parameters (1).Type_Of.Ref.Kind /= Access_Kind
            and then View.Parameters (1).Type_Of.Ref.Allow_Java_Child_Types)
           or else
             (View.Parameters (1).Type_Of.Ref.Kind = Access_Kind
              and then View.Parameters (1)
              .Type_Of.Ref.Target_Type.Ref.Allow_Java_Child_Types))
      then
         --  ??? This subprogram shouldn't be responsible of adding the
         --  dispatching wrapper. We should find an other mechanism for that.
         declare
            Dispatching_Wrapper : constant Code_Wrapper_Access :=
              new Tagged_Type_Ada_Dispatching_Wrapper;
         begin
            Add_Wrapper
              (Map     => Local_Wrappers,
               Wrapper => Dispatching_Wrapper,
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Start);

            Add_Wrapper
              (Map     => Local_Wrappers,
               Wrapper => Dispatching_Wrapper,
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Add_Parameter_Original);
         end;
      end if;

      Set_Default_Wrappers (Local_Wrappers, View);

      Add_Wrapper
        (Map     => Local_Wrappers,
         Wrapper => Exception_Wrapper'Unrestricted_Access,
         Lang    => Ada_Lang,
         Target  => JNI_Sb,
         Context => Start);

      Add_Wrapper
        (Map     => Local_Wrappers,
         Wrapper => Exception_Wrapper'Unrestricted_Access,
         Lang    => Ada_Lang,
         Target  => JNI_Sb,
         Context => Finish);

      --  Subprograms patterns  --

      Append
        (Ada_High_Subprogram,
         New_Line & New_Line
         & Ada_Subprogram_Type
         & " " & Ada_High_Name & Ada_High_Profile & " is"
         & Indent (1)
         & Ada_High_Bloc.Declarations
         & New_Line (-1)
         & "begin"
         & Indent (1)
         & Ada_High_Bloc.Statements_Before
         & Ada_High_Bloc.Code_Node
         & Ada_High_Bloc.Statements_After
         & New_Line (-1) & "end " & Ada_High_Name & ";");

      Append
        (Ada_JNI_Subprogram,
         New_Line & New_Line
         & Ada_Subprogram_Type
         & " " & Ada_JNI_Name & Ada_JNI_Profile & " is"
         & Indent (1)
         & Ada_JNI_Bloc.Declarations
         & New_Line (-1)
         & "begin"
         & Indent (1)
         & Ada_JNI_Bloc.Statements_Before
         & Ada_JNI_Bloc.Code_Node
         & Ada_JNI_Bloc.Statements_After
         & New_Line (-1) & "end " & Ada_JNI_Name & ";");

      if not View.Is_Abstract or else Body_For_Abstract then
         Append
           (Java_High_Subprogram,
            New_Line & New_Line
            & Java_High_Profile & "{"
            & Indent (1)
            & Java_High_Bloc.Declarations
            & Java_High_Bloc.Statements_Before
            & Java_High_Bloc.Code_Node
            & Java_High_Bloc.Statements_After
            & New_Line (-1) & "} // " & Java_High_Name);

      else
         Append
           (Java_High_Subprogram,
            New_Line & New_Line
            & Java_High_Profile & ";");
      end if;

      Append
        (Java_JNI_Subprogram,
         New_Line & New_Line
         & Java_JNI_Profile & "{"
         & Indent (1)
         & Java_JNI_Bloc.Declarations
         & Java_JNI_Bloc.Statements_Before
         & Java_JNI_Bloc.Code_Node
         & Java_JNI_Bloc.Statements_After
         & New_Line (-1) & "} // " & Java_JNI_Name);

      --  Start wrappers  --

      Call_Wrappers
        (Map                  => Local_Wrappers,
         Context              => Start,
         Params               => Subprogram_Wrapper_Params'Access,
         Ada_High_Expression  => Ada_High_Expression,
         Ada_JNI_Expression   => Ada_JNI_Expression,
         Java_JNI_Expression  => Java_JNI_Expression,
         Java_High_Expression => Java_High_Expression);

      --  Rest of the code --

      declare
         Name : constant Wide_String := To_Wide_String (View.Name);
      begin
         if Name'Length /= 0 and then Name (Name'First) = '"' then
            Trace_With_Location
              ("operator " & Name & " binding not supported",
               Errors_And_Warnings);
            raise Silent_Not_Supported;
         end if;
      end;

      --  The general scheme of the expressions:

      if View.Returned_Type /= null then
         Append (Ada_Subprogram_Type, "function");
      else
         Append (Ada_Subprogram_Type, "procedure");
      end if;

      Append
        (Ada_JNI_Declaration,
         New_Line
         & New_Line & Ada_Subprogram_Type
         & " " & Ada_JNI_Name & Ada_JNI_Profile & ";");

      Append
        (Ada_High_Declaration,
         New_Line
         & Ada_High_Generic_Decl
         & New_Line & Ada_Subprogram_Type
         & " " & Ada_High_Name & Ada_High_Profile & ";");

      Append (Ada_Method_ID_Profile, "(");

      if View.Call_Convention = Java_To_Ada then
         if Ada2Java.Link_Method = Export then
            Append
              (Ada_JNI_Declaration,
               New_Line
               & "pragma Export (C, "
               & Ada_JNI_Name & ", """ & Mangled_Name & """);");
         else
            Append
              (Ada_JNI_Declaration,
               New_Line & "pragma Convention (C, " & Ada_JNI_Name & ");");

            if Unit.Ada_Body_File.Native_Number /= 0 then
               Append
                 (Ada_JNI_Reg_Natives, ", ");
            end if;

            Append
              (Ada_JNI_Reg_Natives,
               New_Line & Conversions.To_Wide_String
                 (Integer'Image (Unit.Ada_Body_File.Native_Number + 1))
               & " => (Interfaces.C.Strings.New_String ("""
               & Java_JNI_Name & """), "
               & "Interfaces.C.Strings.New_String ("""
               & Ada_Method_ID_Profile & """), "
               & Ada_JNI_Name  & "'Address)");
         end if;
      else
         if View.JNI_Env_Origin = Generic_Parameter then
            Append
              (Ada_High_Generic_Decl,
               New_Line & "generic"
               & New_Line (1) & Env_Parameter_Name & " : "
               & JNI_Pckg & ".JNI_Env_Access;"
               & New_Line & Object_Parameter_Name & " : "
               & JNI_Pckg & ".J_Object;"
               & Indent (-1));
         end if;

         Append
           (Ada_Method_ID_Init,
            AJIS_Pckg & ".Get_Java_Method (""L"
            & Replace_Dots_By_Slashes
              (To_Wide_String (Get_Attached_Type (View.all).Full_Java_Name))
            & ";"", """
            & Java_JNI_Name & """, """ & Ada_Method_ID_Profile);
      end if;

      if View.Call_Convention = Java_To_Ada then
         Append
           (Java_JNI_Function,
            New_Line & New_Line
            & "native " & Java_JNI_Profile & ";");
      end if;

      if View.Get_Attached_Type = null then
         Prepend (Java_High_Profile, "static ");
         Prepend (Java_JNI_Profile, "static ");
      elsif View.Is_Final then
         Prepend (Java_High_Profile, "final ");
      elsif View.Is_Abstract and then not Body_For_Abstract then
         Prepend (Java_High_Profile, "abstract ");
      end if;

      Append
        (Mangled_Name,
         Package_Mangled_Name & "_"
         & Subprogram_Mangled_Name
         & "__" & Mangled_Parameters);

      --  Setting the various needed names

      Append (Ada_Bound_Name, View.Name);

      declare
         Suffix : constant Wide_String := "_" & Get_Unique_Id;
      begin
         if View.Call_Convention = Java_To_Ada then
            Append (Ada_JNI_Name, Ada_Bound_Name & Suffix);
            Append (Java_JNI_Name, Ada_Bound_Name & Suffix);
            Append (Java_High_Name, Ada_Bound_Name);
            Append (Ada_High_Name, Get_Unique_Id);
         else
            Append (Ada_JNI_Name, Ada_Bound_Name & Suffix & "_Callback");
            Append (Ada_High_Name, View.Original_Name);
            Append
              (Java_JNI_Name, Ada_Bound_Name & Suffix & "_Callback");
            Append (Java_High_Name, Ada_Bound_Name);
         end if;
      end;

      View.Ada_High_Sb_Name := Ada_High_Name;

      Append
        (Subprogram_Mangled_Name,
         Mangle_ID (To_Wide_String (Ada_JNI_Name)));

      Append (Java_High_Profile, "public ");
      Append (Java_JNI_Profile, "private ");

      --  Create the beginning of the profile, depending on the fact that the
      --  subprogram is a function or a procedure

      Append (Java_JNI_Call, Java_High_Name & " (");

      if View.Returned_Type = null then
         Append
           (Ada_JNI_Profile, " (" & Env_Parameter_Name & " : "
            & JNI_Pckg & ".JNI_Env_Access");
         Append (Java_High_Profile, "void " & Java_High_Name & " (");
         Append (Java_JNI_Profile, "void " & Java_JNI_Name & " (");
      else
         Append
           (Ada_JNI_Profile,
            " (" & Env_Parameter_Name & " : " & JNI_Pckg & ".JNI_Env_Access");

         Returned_Type := Create_Bound_Data
           (Simple_Object_View_Access (View.Returned_Type), Unit,
            Handle, View.Call_Convention);

         if not View.Is_Constructor then
            Append
              (Java_High_Profile,
               Returned_Type.Java_Part.To_Type_Name & " ");
         end if;

         Append
           (Java_JNI_Profile,
            Returned_Type.Java_Part.To_JNI_Type_Name & " ");

         Append (Java_High_Profile, Java_High_Name & " (");
         Append (Java_JNI_Profile, Java_JNI_Name & " (");
      end if;

      if View.Get_Attached_Type /= null then
         Append
           (Ada_JNI_Profile,
            "; " & Object_Parameter_Name & " : " & JNI_Pckg & ".J_Object");
      else
         Append
           (Ada_JNI_Profile,
            "; " & Class_Parameter_Name & " : " & JNI_Pckg & ".J_Class");
      end if;

      --  Handling parameters --

      if View.High_Sb_With_Env then
         Append
           (Ada_High_Profile,
            " (" & Env_Parameter_Name & " : " & JNI_Pckg & ".JNI_Env_Access");

         if View.Is_Constructor then
            Append
              (Ada_High_Profile,
               "; " & Object_Parameter_Name & " : " & JNI_Pckg & ".J_Object");
         end if;
      end if;

      if View.Parameters = null or else View.Parameters'Length = 0 then
         declare
            Parameter_Wrapper_Params : aliased Parameter_Handlings_Params :=
              (Subprogram_Wrapper_Params with
               Parameter => null,
               Is_First  => True,
               Is_Last   => True,
               Param_Number    => 0);

            Ada_High_Param_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record'
                (Expression           => Empty_Dynamic_Expression,
                 Bloc                 => Ada_High_Bloc,
                 Enclosing_Expression => Ada_High_Expression,
                 Kind                 => Unknown_Kind);

            Java_High_Param_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record'
                (Expression             => Empty_Dynamic_Expression,
                 Bloc                   => Java_High_Bloc,
                 Enclosing_Expression => Java_High_Expression,
                 Kind                 => Unknown_Kind);

            Ada_JNI_Param_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record'
                (Expression           => Empty_Dynamic_Expression,
                 Bloc                 => Ada_JNI_Bloc,
                 Enclosing_Expression => Ada_JNI_Expression,
                 Kind                 => Unknown_Kind);

            Java_JNI_Param_Expression : constant Wrapping_Expression :=
              new Wrapping_Expression_Record'
                (Expression           => Empty_Dynamic_Expression,
                 Bloc                 => Java_JNI_Bloc,
                 Enclosing_Expression => Java_JNI_Expression,
                 Kind                 => Unknown_Kind);
         begin
            Call_Wrappers
              (Map                  => Local_Wrappers,
               Context              => Parameter_Resolution,
               Params               => Parameter_Wrapper_Params'Access,
               Ada_High_Expression  => Ada_High_Param_Expression,
               Ada_JNI_Expression   => Ada_JNI_Param_Expression,
               Java_JNI_Expression  => Java_JNI_Param_Expression,
               Java_High_Expression => Java_High_Param_Expression);
         end;

         if View.High_Sb_With_Env then
            Append (Ada_High_Profile, ")");
         end if;
      else

         for J in View.Parameters'Range loop
            Handle_Parameter
              (View.Parameters (J), J, J = View.Parameters'Last);
         end loop;

         Append (Ada_High_Profile, ")");
      end if;

      Append (Ada_Method_ID_Profile, ")");

      if Returned_Type /= Null_Bound_Data then
         Adjust_Unit_For_Use (Returned_Type.Java_Part.all, Unit);
         Adjust_Unit_For_Use (Returned_Type.Ada_Part.all, Unit);

         Call_Wrappers
           (Map                  => Local_Wrappers,
            Context              => Return_Type_Original,
            Params               => Subprogram_Wrapper_Params'Access,
            Ada_High_Expression  => Ada_High_Expression,
            Ada_JNI_Expression   => Ada_JNI_Expression,
            Java_JNI_Expression  => Java_JNI_Expression,
            Java_High_Expression => Java_High_Expression);

         if View.Call_Convention = Java_To_Ada then
            Create_Conversion_To_JNI
              (D          => Returned_Type.Ada_Part.all,
               Expression => Ada_JNI_Expression);
         else
            Create_Conversion_From_JNI
              (D          => Returned_Type.Ada_Part.all,
               Expression => Ada_High_Expression);
         end if;

         if not View.Is_Constructor then
            if View.Call_Convention = Java_To_Ada then
               Create_Conversion_From_JNI
                 (D          => Returned_Type.Java_Part.all,
                  Expression => Java_High_Expression);
            else
               Create_Conversion_To_JNI
                 (D          => Returned_Type.Java_Part.all,
                  Expression => Java_JNI_Expression,
                  Prop       => Properties'Access);
            end if;
         end if;

         Append (Java_JNI_Call, ")");

         Call_Wrappers
           (Map                  => Local_Wrappers,
            Context              => Return_Type_Converted,
            Params               => Subprogram_Wrapper_Params'Access,
            Ada_High_Expression  => Ada_High_Expression,
            Ada_JNI_Expression   => Ada_JNI_Expression,
            Java_JNI_Expression  => Java_JNI_Expression,
            Java_High_Expression => Java_High_Expression);

         Append
           (Ada_Method_ID_Profile, Get_JNI_Type (Returned_Type.Java_Part.all));

         Append
           (Mangled_Return_Type,
            Mangle_ID
              (To_Wide_String (Get_JNI_Type (Returned_Type.Java_Part.all))));
      else
         Append (Ada_Method_ID_Profile, "V");
         Append (Mangled_Return_Type, "VOID");
      end if;

      Append (Java_High_Profile, ")");
      Append (Java_JNI_Profile, ")");

      Append (Ada_JNI_Profile, ")");

      if Returned_Type /= Null_Bound_Data then
         Append
           (Ada_JNI_Profile,
            " return " & Returned_Type.Ada_Part.To_JNI_Type_Name);
         Append
           (Ada_High_Profile,
            " return " & Returned_Type.Ada_Part.To_Type_Name);
      end if;

      Append (Ada_Method_ID_Init, """);");

      if View.Call_Convention = Ada_To_Java then
         Append
           (Ada_JNI_Declaration,
            New_Line & Ada_JNI_Name
            & "_Method_Access : " & AJIS_Pckg
            & ".Java_Method_Access := " & Ada_Method_ID_Init);

         Append
           (Unit.Ada_Body_File.Initialize_Body,
            New_Line & AJIS_Pckg & ".Initialize (" & Ada_JNI_Name
            & "_Method_Access, " & Env_Parameter_Name & ");");
      end if;

      Append
        (Package_Mangled_Name,
         "Java_"
         & Mangle_ID
           (To_Wide_String (Unit.Java_File.Full_Class_Name)));

      --  Code_Node_Resolution wrappers  --

      Call_Wrappers
        (Map                  => Local_Wrappers,
         Context              => Code_Node_Resolution,
         Params               => Subprogram_Wrapper_Params'Access,
         Ada_High_Expression  => Ada_High_Expression,
         Ada_JNI_Expression   => Ada_JNI_Expression,
         Java_JNI_Expression  => Java_JNI_Expression,
         Java_High_Expression => Java_High_Expression);

      --  Finish wrappers  --

      Call_Wrappers
        (Map                  => Local_Wrappers,
         Context              => Finish,
         Params               => Subprogram_Wrapper_Params'Access,
         Ada_High_Expression  => Ada_High_Expression,
         Ada_JNI_Expression   => Ada_JNI_Expression,
         Java_JNI_Expression  => Java_JNI_Expression,
         Java_High_Expression => Java_High_Expression);

      --  Everything is now computed. Add the dynamic expressions into the
      --  created units.

      if View.Bind_Ada then
         Append
           (Unit.Ada_Spec_File.Public_Body_Part,
            Ada_High_Declaration);

         Append
           (Unit.Ada_Spec_File.Public_Body_Part,
            Ada_JNI_Declaration);

         Append
           (Unit.Ada_Body_File.Public_Body_Part,
            Ada_High_Subprogram);

         Append
           (Unit.Ada_Body_File.Public_Body_Part,
            Ada_JNI_Subprogram);

         if View.Call_Convention = Ada_To_Java then
            Append
              (Unit.Java_File.Private_Body_Part,
               Java_JNI_Subprogram);
         else
            Append
              (Unit.Java_File.Private_Body_Part,
               Java_JNI_Function);

            if Ada2Java.Link_Method = Register_Natives then
               Unit.Ada_Body_File.Native_Number :=
                 Unit.Ada_Body_File.Native_Number + 1;

               Append (Unit.Ada_Body_File.Reg_Natives, Ada_JNI_Reg_Natives);
            end if;
         end if;
      end if;

      if Add_Java_High_Sb then
         Append
           (Unit.Java_File.Public_Body_Part,
            Java_High_Subprogram);
      end if;

      Pop_Location (Loc);
   exception
      when Silent_Not_Supported =>
         null;
      when E : others =>
         Trace_With_Location
           (Conversions.To_Wide_String (Exception_Message (E)),
            Errors_Only);
   end Bind_Subprogram;

   ----------------------------------
   -- To_Original_Expanded_Profile --
   ----------------------------------

   function To_Original_Expanded_Profile
     (Handle : not null access Kernel.Kernel_Record;
      View   : Simple_Subprogram_View_Access) return Dynamic_Expression
   is
      pragma Unreferenced (Handle);
      Result : Dynamic_Expression := New_Dynamic_Expression;
      --  ??? This function is currently simple minded, and does not take into
      --  account e.g. access modes. Perhaps should use the data package child
      --  instead.
   begin
      if View.Returned_Type = null then
         Append (Result, "procedure ");
      else
         Append (Result, "function ");
      end  if;

      if View.Parameters /= null and then View.Parameters'Length > 0 then
         Append (Result, "(");

         for J in View.Parameters'Range loop
            if J > 1 then
               Append (Result, "; ");
            end if;

            Append
              (Result,
               View.Parameters (J).Name & " : ");

            case View.Parameters (J).Mode is
               when In_Mode =>
                  Append (Result, "in ");
               when Out_Mode =>
                  Append (Result, "out ");
               when In_Out_Mode =>
                  Append (Result, "in out ");
            end case;

            Append
              (Result,
               View.Parameters (J).Type_Of.Initial_Subtype_Name);
         end loop;

         Append (Result, ")");
      end if;

      if View.Returned_Type /= null then
         Append
           (Result,
            " return " & View.Returned_Type.Type_Of.Initial_Subtype_Name);
      end if;

      return Result;
   end To_Original_Expanded_Profile;

   -----------------
   -- Add_Wrapper --
   -----------------

   procedure Add_Wrapper
     (Map     : in out Wrapping_Map;
      Wrapper : Code_Wrapper_Access;
      Lang    : Wrapping_Language;
      Target  : Wrapped_Target;
      Context : Wrapping_Context) is
   begin
      if Context = Code_Node_Resolution
        or else Context = Parameter_Resolution
      then
         if Length (Map.Wrappers (Lang, Target, Context)) /= 0 then
            raise Constraint_Error
              with "There can't be more than one wrapper for the key {"
                & Lang'Img & ", " & Target'Img & ", " & Context'Img & "}";
         end if;
      end if;

      Append (Map.Wrappers (Lang, Target, Context), Wrapper);
   end Add_Wrapper;

   --------------------------
   -- Set_Default_Wrappers --
   --------------------------

   procedure Set_Default_Wrappers
     (Map  : in out Wrapping_Map;
      View : Simple_Subprogram_View_Access) is
   begin
      if Length
        (Map.Wrappers (Ada_Lang, High_Sb, Parameter_Resolution)) = 0
      then
         if View.Call_Convention = Java_To_Ada then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Default_Ada_Parameter_Handler'(In_JNI => False)),
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Parameter_Resolution);
         else
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'(new Call_A_To_J_High_Param),
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Parameter_Resolution);
         end if;
      end if;

      if Length
        (Map.Wrappers (Ada_Lang, JNI_Sb, Parameter_Resolution)) = 0
      then
         if View.Call_Convention = Java_To_Ada then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Default_Ada_Parameter_Handler'(In_JNI => True)),
               Lang    => Ada_Lang,
               Target  => JNI_Sb,
               Context => Parameter_Resolution);
         else
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'(new Call_A_To_J_JNI_Param),
               Lang    => Ada_Lang,
               Target  => JNI_Sb,
               Context => Parameter_Resolution);
         end if;
      end if;

      if Length
        (Map.Wrappers (Java_Lang, JNI_Sb, Parameter_Resolution)) = 0
      then
         Add_Wrapper
           (Map     => Map,
            Wrapper => Code_Wrapper_Access'
              (new Default_Java_Parameter_Handler),
            Lang    => Java_Lang,
            Target  => JNI_Sb,
            Context => Parameter_Resolution);
      end if;

      if Length
        (Map.Wrappers (Java_Lang, High_Sb, Parameter_Resolution)) = 0
      then
         Add_Wrapper
           (Map     => Map,
            Wrapper => Code_Wrapper_Access'
              (new Default_Java_Parameter_Handler),
            Lang    => Java_Lang,
            Target  => High_Sb,
            Context => Parameter_Resolution);
      end if;

      if View.Returned_Type /= null then
         if Length
           (Map.Wrappers (Ada_Lang, High_Sb, Code_Node_Resolution)) = 0
         then
            declare
               Wrapper : constant Code_Wrapper_Access :=
                 new Ada_Function_Wrapper'(In_JNI => False, others => <>);
            begin
               Add_Wrapper
                 (Map     => Map,
                  Wrapper => Wrapper,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Code_Node_Resolution);

               Add_Wrapper
                 (Map     => Map,
                  Wrapper => Wrapper,
                  Lang    => Ada_Lang,
                  Target  => High_Sb,
                  Context => Finish);

            end;
         end if;

         if Length
           (Map.Wrappers (Ada_Lang, JNI_Sb, Code_Node_Resolution)) = 0
         then
            declare
               Wrapper : constant Code_Wrapper_Access :=
                 new Ada_Function_Wrapper'(In_JNI => True, others => <>);
            begin
               Add_Wrapper
                 (Map     => Map,
                  Wrapper => Wrapper,
                  Lang    => Ada_Lang,
                  Target  => JNI_Sb,
                  Context => Code_Node_Resolution);

               Add_Wrapper
                 (Map     => Map,
                  Wrapper => Wrapper,
                  Lang    => Ada_Lang,
                  Target  => JNI_Sb,
                  Context => Finish);
            end;
         end if;

         if Length
           (Map.Wrappers (Java_Lang, JNI_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Java_Function_Wrapper'(In_JNI => True)),
               Lang    => Java_Lang,
               Target  => JNI_Sb,
               Context => Code_Node_Resolution);
         end if;

         if Length
           (Map.Wrappers (Java_Lang, High_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Java_Function_Wrapper'(In_JNI => False)),
               Lang    => Java_Lang,
               Target  => High_Sb,
               Context => Code_Node_Resolution);
         end if;
      else
         if Length
           (Map.Wrappers (Ada_Lang, High_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Ada_Procedure_Wrapper),
               Lang    => Ada_Lang,
               Target  => High_Sb,
               Context => Code_Node_Resolution);
         end if;

         if Length
           (Map.Wrappers (Ada_Lang, JNI_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Ada_Procedure_Wrapper),
               Lang    => Ada_Lang,
               Target  => JNI_Sb,
               Context => Code_Node_Resolution);
         end if;

         if Length
           (Map.Wrappers (Java_Lang, JNI_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Java_Procedure_Wrapper),
               Lang    => Java_Lang,
               Target  => JNI_Sb,
               Context => Code_Node_Resolution);
         end if;

         if Length
           (Map.Wrappers (Java_Lang, High_Sb, Code_Node_Resolution)) = 0
         then
            Add_Wrapper
              (Map     => Map,
               Wrapper => Code_Wrapper_Access'
                 (new Java_Procedure_Wrapper),
               Lang    => Java_Lang,
               Target  => High_Sb,
               Context => Code_Node_Resolution);
         end if;
      end if;

      Add_Wrapper
        (Map     => Map,
         Wrapper => Code_Wrapper_Access'(new Java_Locking_Wrapper),
         Lang    => Java_Lang,
         Target  => High_Sb,
         Context => Start);
   end Set_Default_Wrappers;

   -------------------
   -- Call_Wrappers --
   -------------------

   procedure Call_Wrappers
     (Map                  : Wrapping_Map;
      Context              : Wrapping_Context;
      Params               : access Subprogram_Params'Class;
      Ada_High_Expression  : Wrapping_Expression;
      Ada_JNI_Expression   : Wrapping_Expression;
      Java_JNI_Expression  : Wrapping_Expression;
      Java_High_Expression : Wrapping_Expression)
   is
      procedure Do_Loop
        (Lang   : Wrapping_Language;
         Target : Wrapped_Target;
         Exp    : Wrapping_Expression);

      procedure Do_Loop
        (Lang   : Wrapping_Language;
         Target : Wrapped_Target;
         Exp    : Wrapping_Expression)
      is
         Cur : Wrappers_List.Cursor :=
           First (Map.Wrappers (Lang, Target, Context));
      begin
         Params.Target := Target;

         while Cur /= Wrappers_List.No_Element loop
            Wrap
              (Wrapper    => Wrappers_List.Element (Cur).all,
               Expression => Exp,
               Parameters => Params);

            Cur := Next (Cur);
         end loop;
      end Do_Loop;
   begin
      Params.Context := Context;

      Do_Loop (Ada_Lang, High_Sb, Ada_High_Expression);
      Do_Loop (Ada_Lang, JNI_Sb, Ada_JNI_Expression);
      Do_Loop (Java_Lang, High_Sb, Java_High_Expression);
      Do_Loop (Java_Lang, JNI_Sb, Java_JNI_Expression);
   end Call_Wrappers;

   ----------
   -- Bind --
   ----------

   procedure Bind
     (Element_Bound : access Bound_Subprogram;
      Handle        : not null access Kernel_Record)
   is
      Simple_View : constant Simple_Subprogram_View_Access :=
        Simple_Subprogram_View_Access (Element_Bound.Simple_Element);
   begin
      Element_Bound.Handle := Handle;
      Element_Bound.View := Simple_View;

      Bind_Subprogram
        (Handle,
         Simple_View,
         Get_Or_Create_Bound_Unit (Handle, Element_Bound.Simple_Element),
         Empty_Wrapping_Map);
   end Bind;

   --------------
   -- Get_View --
   --------------

   function Get_View
     (This : access Bound_Subprogram)
      return Simple_Subprogram_View_Access is
   begin
      return This.View;
   end Get_View;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Default_Ada_Parameter_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : constant Parameter_Handlings_Params_Access :=
        Parameter_Handlings_Params_Access (Parameters);
      Name : Dynamic_Expression;
   begin
      if Params.Is_First then
         if Wrapper.In_JNI then
            Name := Params.Ada_High_Sb_Name;
         else
            Name := Get_Bound_Package_Name (Params.Unit.Base_Pckg)
              & "." & Params.Subprogram_View.Original_Name;
         end if;

         Append (Expression.Enclosing_Expression.Expression, Name);

         if Params.Subprogram_View.High_Sb_With_Env then
            Append (Expression.Enclosing_Expression.Expression, " ("
                    & Env_Parameter_Name);

            if Expression.Expression /= Empty_Dynamic_Expression then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  ", ");
            elsif Params.Is_Last then
               Append
                 (Expression.Enclosing_Expression.Expression,
                  ")");
            end if;
         elsif Expression.Expression /= Empty_Dynamic_Expression then
            Append (Expression.Enclosing_Expression.Expression, " (");
         end if;

         Expression.Enclosing_Expression.Kind := Call_Kind;
      else
         Append
           (Expression.Enclosing_Expression.Expression,
            ", ");
      end if;

      if Expression.Expression /= Empty_Dynamic_Expression then
         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression);

         if Params.Is_Last then
            Append (Expression.Enclosing_Expression.Expression, ")");
         end if;
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Call_A_To_J_JNI_Param;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : constant Parameter_Handlings_Params_Access :=
        Parameter_Handlings_Params_Access (Parameters);
   begin
      if Params.Is_First then
         Wrapper.Params_Array_Id := To_Dynamic_Expression (Get_Unique_Id);
         Wrapper.Params_Array_Aggregate :=
           New_Line & Wrapper.Params_Array_Id
           & " : " & JNI_Pckg & ".J_Value_Array (1 .."
           & Conversions.To_Wide_String
           (Integer'Image (Params.Subprogram_View.Parameters'Length))
           & ")";

         Append (Expression.Bloc.Declarations, Wrapper.Params_Array_Aggregate);

         if Params.Subprogram_View.Returned_Type = null then
            Append
              (Expression.Enclosing_Expression.Expression,
               JNI_Pckg & ".Call_Void_Method_A");
         else
            declare
               Obj           : constant Simple_Object_View_Access :=
                 Create (Params.Subprogram_View);
               Returned_Data : Bound_Data;
            begin
               Obj.Type_Of := Params.Subprogram_View.Returned_Type.Type_Of;
               Returned_Data := Create_Bound_Data
                 (Obj,  Params.Unit, Params.Handle,
                  Params.Subprogram_View.Call_Convention);
               Append
                 (Expression.Enclosing_Expression.Expression,
                  Returned_Data.Ada_Part.Get_JNI_Call);
            end;
         end if;

         Append
           (Expression.Enclosing_Expression.Expression,
            " (" & Env_Parameter_Name & ", " & Object_Parameter_Name & ", "
            & AJIS_Pckg & ".Get_Id ("
            & Params.Ada_JNI_Sb_Name
            & "_Method_Access" & "), "
            & Wrapper.Params_Array_Id & ")");
      end if;

      if Expression.Expression /= Empty_Dynamic_Expression then
         if Params.Is_First then
            Append
              (Wrapper.Params_Array_Aggregate,
               " := " & New_Line (1) & "(");
         else
            Append
              (Wrapper.Params_Array_Aggregate,
               "," & New_Line & " ");
         end if;

         Append
           (Wrapper.Params_Array_Aggregate,
            Conversions.To_Wide_String (Trim (Params.Param_Number'Img, Left))
            & " => " & AJIS_Pckg & ".To_J_Value ("
            & Expression.Expression & ")");
      end if;

      if Params.Is_Last then
         if Expression.Expression /= Empty_Dynamic_Expression then
            Append (Wrapper.Params_Array_Aggregate, ");" & Indent (-1));
         else
            Append (Wrapper.Params_Array_Aggregate, ";");
         end if;
      end if;

   end Wrap;

   -----------
   -- Wrap --
   -----------

   procedure Wrap
     (Wrapper    : in out Call_A_To_J_High_Param;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : constant Parameter_Handlings_Params_Access :=
        Parameter_Handlings_Params_Access (Parameters);
   begin
      if Params.Is_First then
         Append
           (Expression.Enclosing_Expression.Expression,
            Params.Ada_JNI_Sb_Name
            & " (" & Env_Parameter_Name & ", " & Object_Parameter_Name);
      end if;

      if Expression.Expression /= Empty_Dynamic_Expression then
         Append
           (Expression.Enclosing_Expression.Expression,
            ", ");

         Append
           (Expression.Enclosing_Expression.Expression,
            Expression.Expression);
      end if;

      if Params.Is_Last then
         Append (Expression.Enclosing_Expression.Expression, ")");
      end if;
   end Wrap;

   -----------------------------
   -- Handle_Input_Expression --
   -----------------------------

   procedure Wrap
     (Wrapper    : in out Default_Java_Parameter_Handler;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : constant Parameter_Handlings_Params_Access :=
        Parameter_Handlings_Params_Access (Parameters);
   begin
      if Params.Is_First then
         if Params.Subprogram_View.Call_Convention = Java_To_Ada then
            Append
              (Expression.Enclosing_Expression.Expression,
               Params.Java_JNI_Name & " (" & Expression.Expression);
         else
            Append
              (Expression.Enclosing_Expression.Expression,
               Params.Subprogram_View.Name & " (" & Expression.Expression);
         end if;
      else
         Append
           (Expression.Enclosing_Expression.Expression,
            ", " & Expression.Expression);
      end if;

      if Params.Is_Last then
         Append
           (Expression.Enclosing_Expression.Expression, ")");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Function_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : constant Subprogram_Params_Access :=
        Subprogram_Params_Access (Parameters);

      Returned_Id : constant Wide_String := Get_Unique_Id;

      Returned_Data : constant Bound_Data := Create_Bound_Data
        (Simple_Object_View_Access (Params.Subprogram_View.Returned_Type),
         Params.Unit,
         Params.Handle,
         Params.Subprogram_View.Call_Convention);

      Pointer_Type : Dynamic_Expression;
   begin
      if Params.Context = Code_Node_Resolution then
         if Params.Subprogram_View.Returned_Type.Type_Of.Ref.Is_Limited then
            Wrapper.Returned_Exp := Expression.Expression;

            return;
         end if;

         Append
           (Expression.Bloc.Declarations,
            New_Line & Returned_Id
            & " : ");

         if Wrapper.In_JNI then
            Append
              (Expression.Bloc.Declarations,
               Returned_Data.Ada_Part.To_JNI_Type_Name);
         else
            if Returned_Data.Ada_Part.Get_Type_View.Ref.Kind = Access_Kind
              and then
                (Returned_Data.Ada_Part.Get_Type_View.Ref.Is_Anonymous
                 or else
                   Returned_Data.Ada_Part.Get_Type_View.Ref.Full_Ada_Name =
                   Empty_Dynamic_Expression)
            then
               Pointer_Type :=
                 Params.Subprogram_View.Returned_Type
                   .Type_Of.Initial_Subtype_Name;

               Append (Expression.Bloc.Declarations, Pointer_Type);

               Prepend (Expression.Expression, Pointer_Type & " (");

               Append (Expression.Expression, ")");
            else
               Append
                 (Expression.Bloc.Declarations,
                  Returned_Data.Ada_Part.To_Type_Name);
            end if;
         end if;

         if Expression.Kind = Call_Kind
           and then not Wrapper.In_JNI
           and then Params.Subprogram_View.Returned_Type.Type_Of.Ref.Kind
             /= Access_Kind
         then
            --  We only do renaming in the high subprogram - otherwise there
            --  may be conversions that can't be renamed. For the same reasons,
            --  we do not do renaming for pointers as they may get converted by
            --  the internal subprogram (which doesn't avoid big copies
            --  anyway).

            if Returned_Data.Ada_Part.Should_Be_Class_Wide then
               Append
                 (Expression.Bloc.Declarations,
                  " renames " & Returned_Data.Ada_Part.To_Type_Name
                  & " (" & Expression.Expression & ");");
            else
               Append
                 (Expression.Bloc.Declarations,
                  " renames " & Expression.Expression & ";");
            end if;
         else
            Append
              (Expression.Bloc.Declarations,
               " := " & Expression.Expression & ";");
         end if;

         Wrapper.Returned_Exp := To_Dynamic_Expression (Returned_Id);
         Expression.Expression := Wrapper.Returned_Exp;
      else
         Append
           (Expression.Bloc.Statements_After,
            New_Line & "return " & Wrapper.Returned_Exp & ";");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Procedure_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Append
        (Expression.Bloc.Code_Node, New_Line & Expression.Expression & ";");
      Expression.Expression := Empty_Dynamic_Expression;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Function_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      Params : constant Subprogram_Params_Access :=
        Subprogram_Params_Access (Parameters);

      Returned_Data : constant Bound_Data := Create_Bound_Data
        (Simple_Object_View_Access (Params.Subprogram_View.Returned_Type),
         Params.Unit,
         Params.Handle,
         Params.Subprogram_View.Call_Convention);

      Result_Id : constant Wide_String := Get_Unique_Id;
   begin
      if Wrapper.In_JNI then
         Append
           (Expression.Bloc.Code_Node,
            New_Line & Returned_Data.Java_Part.To_JNI_Type_Name & " "
            & Result_Id & " = " & Expression.Expression & ";");
      else
         Append
           (Expression.Bloc.Code_Node,
            New_Line & Returned_Data.Java_Part.To_Type_Name & " "
            & Result_Id & " = " & Expression.Expression & ";");
      end if;

      Append
        (Expression.Bloc.Statements_After,
         New_Line & "return " & Result_Id & ";");

      Expression.Expression := Empty_Dynamic_Expression;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Procedure_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper, Parameters);
   begin
      Append
        (Expression.Bloc.Code_Node, New_Line & Expression.Expression & ";");
      Expression.Expression := Empty_Dynamic_Expression;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Ada_Exception_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);

      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);

      Return_Statement : Dynamic_Expression := New_Dynamic_Expression;
   begin
      if Params.Subprogram_View.Call_Convention = Java_To_Ada
        and then Params.Context = Start
      then
         Append
           (Expression.Bloc.Statements_After,
            New_Line (-1) & "exception"
            & New_Line (1) & "when J : AJIS.Java.Java_Exception =>"
            & New_Line (1)
            & "declare"
            & New_Line (1) & "Result : " & JNI_Pckg & ".J_Int;"
            & New_Line (-1) & "begin"
            & New_Line (1)
            & "Result := " & JNI_Pckg & ".Throw (" & Env_Parameter_Name & ", "
            & AJIS_Pckg & ".Throwable_Value "
            & "(Ada.Exceptions.Exception_Message (J)));"
            & Return_Statement
            & New_Line (-1) & "end;"
            & New_Line (-1) & "when E : others =>"
            & New_Line (1)
            & "declare"
            & New_Line (1) & "Result : " & JNI_Pckg & ".J_Int;"
            & New_Line & "Except : " & JNI_Pckg & ".J_Object;"
            & New_Line (-1) & "begin"
            & New_Line (1)
            & "Except := " & AJIS_Pckg & ".Create_Java_Exception ("
            & Env_Parameter_Name & ", E);"
            & New_Line
            & "if " & JNI_Pckg & ".""="" (Except, "
            & JNI_Pckg & ".J_Null_Object) then"
            & New_Line (1)
            & "Result := " & JNI_Pckg & ".Throw_New ("
            & Env_Parameter_Name & ", "
            & AJIS_Pckg & ".Get_Class (" & AJIS_Pckg
            & ".Native_Exception_Class), "
            & "Interfaces.C.To_C "
            & "(Ada.Exceptions.Exception_Information (E)));"
            & New_Line (-1) & "else" & New_Line (1)
            & "Result := " & JNI_Pckg & ".Throw ("
            & Env_Parameter_Name & ", Except);"
            & New_Line (-1) & "end if;"
            & Return_Statement
            & New_Line (-1) & "end;" & Indent (-1));

         if Params.Subprogram_View.Returned_Type /= null then
            declare
               Returned_Type : constant Bound_Data := Create_Bound_Data
                 (View   => Simple_Object_View_Access
                    (Params.Subprogram_View.Returned_Type),
                  Unit   => Params.Unit,
                  Kernel => Params.Handle,
                  Call_Convention => Params.Subprogram_View.Call_Convention);
            begin
               Append
                 (Return_Statement,
                  New_Line
                  & "return "
                  & Returned_Type.Ada_Part.Get_Default_JNI_Value & ";");
            end;
         end if;

         Nest_Ada_Statements (Expression.Bloc, False);
      elsif Params.Subprogram_View.Call_Convention = Ada_To_Java
        and then Params.Context = Finish
      then
         Prepend
           (Expression.Bloc.Statements_After,
            New_Line & AJIS_Pckg & ".Handle_Java_Exception ("
            & Env_Parameter_Name & ");");
      end if;
   end Wrap;

   ----------
   -- Wrap --
   ----------

   procedure Wrap
     (Wrapper    : in out Java_Locking_Wrapper;
      Expression : Wrapping_Expression;
      Parameters : access Wrapper_Parameters'Class)
   is
      pragma Unreferenced (Wrapper);
      Params : Subprogram_Params renames Subprogram_Params (Parameters.all);
   begin
      case Params.Subprogram_View.Locking_State is
         when Disable =>
            null;

         when Check =>
            Append
              (Expression.Bloc.Statements_Before,
               New_Line & "if (!" & Params.Subprogram_View.Lock_Var
               & ".isHeldByCurrentThread()) {" &
               New_Line (1) & "throw new com.adacore.ajis.NativeException " &
               "(""native call is not protected"");" &
               New_Line (-1) & "}");

            Nest_Java_Statements (Expression.Bloc, False);

         when Protect =>
            Append
              (Expression.Bloc.Statements_Before,
               New_Line & Params.Subprogram_View.Lock_Var
               & ".lock ();"
               & New_Line & "try {" & Indent (1));

            Append
              (Expression.Bloc.Statements_After,
               New_Line (-1) & "} finally {"
               & New_Line (1) & Params.Subprogram_View.Lock_Var
               & ".unlock ();"
               & New_Line (-1) & "}");

            Nest_Java_Statements (Expression.Bloc, False);
      end case;
   end Wrap;

end Ada2Java.Bound_Elements.Subprograms;
