------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUN-TIME COMPONENTS                         --
--                                                                          --
--           S Y S T E M . T R A C E B A C K . S Y M B O L I C              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 1999-2015, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  Run-time symbolic traceback support for targets using DWARF debug data

pragma Polling (Off);
--  We must turn polling off for this unit, because otherwise we can get
--  elaboration circularities when polling is turned on.

with Ada.Exceptions.Traceback; use Ada.Exceptions.Traceback;

with System.Address_To_Access_Conversions;
with System.CRTL;
with System.Dwarf_Lines;
with System.Exception_Traces;
with System.Standard_Library;
with System.Storage_Elements;
with System.Traceback_Entries;

package body System.Traceback.Symbolic is

   subtype Big_String is String (Positive);

   function Value (Item : System.Address) return String;
   --  Return the String contained in Item, up until the first NUL character

   package Module_Name is

      function Get (Addr : access System.Address) return String;
      --  Returns the module name for the given address, Addr may be updated
      --  to be set relative to a shared library. This depends on the platform.
      --  Returns an empty string for the main executable.

      function Is_Supported return Boolean;
      pragma Inline (Is_Supported);
      --  Returns True if Module_Name is supported, so if the traceback is
      --  supported for shared libraries.

   end Module_Name;

   package body Module_Name is separate;

   function Executable_Name return String;
   --  Returns the executable name as reported by argv[0]. If gnat_argv not
   --  initialized or if argv[0] executable not found in path, function returns
   --  an empty string.

   function Get_Executable_Load_Address return System.Address;
   pragma Import (C, Get_Executable_Load_Address,
                    "__gnat_get_executable_load_address");
   --  Get the load address of the executable, or Null_Address if not known

   function Module_Symbolic_Traceback
     (Module_Name  : String;
      Traceback    : Tracebacks_Array;
      Load_Address : System.Address;
      Symbol_Found : in out Boolean) return String;
   --  Returns the Traceback for a given module or an empty string if not in
   --  a module. Parameter Load_Address is the load address of the module,
   --  or Null_Address is not rebased. Symbol_Found is as for
   --  Dwarf_Lines.Symbolic_Traceback.

   function Multi_Module_Symbolic_Traceback
     (Traceback    : Tracebacks_Array;
      Exec_Name    : String;
      Exec_Address : System.Address;
      Symbol_Found : in out Boolean) return String;
   --  Build string containing symbolic traceback for the given call chain,
   --  using Exec_Name for the path of the executable and Exec_Address for its
   --  load address. Symbol_Found is as for Dwarf_Lines.Symbolic_Traceback.

   -----------------
   -- Bounded_Str --
   -----------------

   --  Use our own verion of Bounded_Strings, to avoid depending on
   --  Ada.Strings.Bounded.

   type Bounded_Str (Max_Length : Natural) is limited record
      Length : Natural := 0;
      Chars  : String (1 .. Max_Length);
   end record;

   procedure Append (X : in out Bounded_Str; C : Character);
   procedure Append (X : in out Bounded_Str; S : String);
   function To_String (X : Bounded_Str) return String;
   function "+" (X : Bounded_Str) return String renames To_String;

   Max_String_Length : constant := 4096;
   --  Arbitrary limit on Bounded_Str length

   procedure Append_Address
     (Result : in out Bounded_Str;
      A      : Address);

   ------------
   -- Append --
   ------------

   procedure Append (X : in out Bounded_Str; C : Character) is
   begin
      --  If we have too many characters to fit, simply drop them

      if X.Length < X.Max_Length then
         X.Length           := X.Length + 1;
         X.Chars (X.Length) := C;
      end if;
   end Append;

   procedure Append (X : in out Bounded_Str; S : String) is
   begin
      for C of S loop
         Append (X, C);
      end loop;
   end Append;

   --------------------
   -- Append_Address --
   --------------------

   procedure Append_Address
     (Result : in out Bounded_Str;
      A      : Address)
   is
      S : String (1 .. 18);
      P : Natural;
      use System.Storage_Elements;
      N : Integer_Address;

      H : constant array (Integer range 0 .. 15) of Character :=
        "0123456789abcdef";
   begin
      P := S'Last;
      N := To_Integer (A);
      loop
         S (P) := H (Integer (N mod 16));
         P := P - 1;
         N := N / 16;
         exit when N = 0;
      end loop;

      S (P - 1) := '0';
      S (P) := 'x';

      Append (Result, S (P - 1 .. S'Last));
   end Append_Address;

   -----------
   -- Value --
   -----------

   function Value (Item : System.Address) return String is
      package Conv is new System.Address_To_Access_Conversions (Big_String);
   begin
      if Item /= Null_Address then
         for J in Big_String'Range loop
            if Conv.To_Pointer (Item) (J) = ASCII.NUL then
               return Conv.To_Pointer (Item) (1 .. J - 1);
            end if;
         end loop;
      end if;

      return "";
   end Value;

   ---------------------
   -- Executable_Name --
   ---------------------

   function Executable_Name return String is
      --  We have to import gnat_argv as an Address to match the type of
      --  gnat_argv in the binder generated file. Otherwise, we get spurious
      --  warnings about type mismatch when LTO is turned on.

      Gnat_Argv : System.Address;
      pragma Import (C, Gnat_Argv, "gnat_argv");

      type Argv_Array is array (0 .. 0) of System.Address;
      package Conv is new System.Address_To_Access_Conversions (Argv_Array);

      function locate_exec_on_path (A : System.Address) return System.Address;
      pragma Import (C, locate_exec_on_path, "__gnat_locate_exec_on_path");

   begin
      if Gnat_Argv = Null_Address then
         return "";
      end if;

      declare
         Addr   : constant System.Address :=
                    locate_exec_on_path (Conv.To_Pointer (Gnat_Argv) (0));
         Result : constant String := Value (Addr);

      begin
         --  The buffer returned by locate_exec_on_path was allocated using
         --  malloc, so we should use free to release the memory.

         if Addr /= Null_Address then
            System.CRTL.free (Addr);
         end if;

         return Result;
      end;
   end Executable_Name;

   -------------------------------
   -- Module_Symbolic_Traceback --
   -------------------------------

   function Module_Symbolic_Traceback
     (Module_Name  : String;
      Traceback    : Tracebacks_Array;
      Load_Address : System.Address;
      Symbol_Found : in out Boolean) return String
   is
      use System.Dwarf_Lines;
      C : Dwarf_Context (In_Exception => True);

   begin
      Open (Module_Name, C);

      --  If a module can't be opened just return an empty string, we
      --  just cannot give more information in this case.

      if not Is_Open (C) then
         return "";
      end if;

      Set_Load_Address (C, Load_Address);

      declare
         Result : constant String :=
                    Dwarf_Lines.Symbolic_Traceback
                      (C, Traceback, Symbol_Found);

      begin
         Close (C);

         if Symbolic.Module_Name.Is_Supported then
            return '[' & Module_Name & ']' & ASCII.LF & Result;
         else
            return Result;
         end if;
      end;

      --  We must not allow an unhandled exception here, since this function
      --  may be installed as a decorator for all automatic exceptions.

   exception
      when others =>
         return "";
   end Module_Symbolic_Traceback;

   -------------------------------------
   -- Multi_Module_Symbolic_Traceback --
   -------------------------------------

   function Multi_Module_Symbolic_Traceback
     (Traceback    : Tracebacks_Array;
      Exec_Name    : String;
      Exec_Address : System.Address;
      Symbol_Found : in out Boolean) return String
   is
      TB : Tracebacks_Array (Traceback'Range);
      --  A partial copy of the possibly relocated traceback addresses. These
      --  addresses gets relocated for GNU/Linux shared library for example.
      --  This gets done in the Get_Module_Name routine.

   begin
      if Traceback'Length = 0 then
         return "";
      end if;

      declare
         Addr   : aliased System.Address := Traceback (Traceback'First);
         M_Name : constant String := Module_Name.Get (Addr'Access);
         Pos    : Positive;

      begin
         --  Will symbolize the first address...

         TB (TB'First) := Addr;

         Pos := TB'First + 1;

         --  ... and all addresses in the same module

         Same_Module : loop
            exit Same_Module when Pos > Traceback'Last;

            --  Get address to check for corresponding module name

            Addr := Traceback (Pos);

            exit Same_Module when Module_Name.Get (Addr'Access) /= M_Name;

            --  Copy the possibly relocated address into TB

            TB (Pos) := Addr;

            Pos := Pos + 1;
         end loop Same_Module;

         --  Symbolize the addresses in the same module, and do a recursive
         --  call for the remaining addresses.

         declare
            Module_Name : constant String :=
              (if M_Name = "" then Exec_Name else M_Name);
            Load_Address : constant System.Address :=
              (if M_Name = "" then Exec_Address else System.Null_Address);

         begin
            return
              Module_Symbolic_Traceback
                (Module_Name, TB (TB'First .. Pos - 1), Load_Address,
                 Symbol_Found) &
              Multi_Module_Symbolic_Traceback
                (Traceback (Pos .. Traceback'Last), Exec_Name, Exec_Address,
                 Symbol_Found);
         end;
      end;
   end Multi_Module_Symbolic_Traceback;

   function Symbolic_Traceback (Traceback : Tracebacks_Array) return String is
      Exec_Path : constant String         := Executable_Name;
      Exec_Load : constant System.Address := Get_Executable_Load_Address;

      Symbol_Found : Boolean := False;
      --  This will be set to True if any call to
      --  Dwarf_Lines.Symbolic_Traceback finds any symbols.

      Result : constant String :=
        (if Symbolic.Module_Name.Is_Supported then
           Multi_Module_Symbolic_Traceback
             (Traceback, Exec_Path, Exec_Load, Symbol_Found)
        else
           Module_Symbolic_Traceback
             (Exec_Path, Traceback, Exec_Load, Symbol_Found));

   begin
      --  If symbols were found, use the symbolic traceback

      if Symbol_Found then
         return Result;

      --  Otherwise (no symbols found), fall back to hexadecimal addresses

      elsif Traceback'Length > 0 then
         declare
            Hex : Bounded_Str (Max_Length => Max_String_Length);
            use System.Traceback_Entries;
         begin
            Append (Hex, "Call stack traceback locations:" & ASCII.LF);

            for J in Traceback'Range loop
               Append_Address (Hex, PC_For (Traceback (J)));

               if J /= Traceback'Last then
                  Append (Hex, " ");
               end if;
            end loop;

            return +Hex;
         end;

      else
         return "";
      end if;
   end Symbolic_Traceback;

   function Symbolic_Traceback
     (E : Ada.Exceptions.Exception_Occurrence) return String
   is
   begin
      return Symbolic_Traceback (Ada.Exceptions.Traceback.Tracebacks (E));
   end Symbolic_Traceback;

   ---------------
   -- To_String --
   ---------------

   function To_String (X : Bounded_Str) return String is
   begin
      return X.Chars (1 .. X.Length);
   end To_String;

   use Standard_Library;

   Exception_Tracebacks_Symbolic : Integer;
   pragma Import (C, Exception_Tracebacks_Symbolic,
                  "__gl_exception_tracebacks_symbolic");
   --  Boolean indicating whether symbolic tracebacks should be generated.

begin
   --  If this version of this package is available, and the binder switch -Es
   --  was given, then we want to use this as the decorator by default, and we
   --  want to turn on tracing for Unhandled_Raise_In_Main. Note that the user
   --  cannot have already set Exception_Trace, because the runtime library is
   --  elaborated before user-defined code.

   if Exception_Tracebacks_Symbolic /= 0 then
      Exception_Traces.Set_Trace_Decorator (Symbolic_Traceback'Access);
      pragma Assert (Exception_Trace = RM_Convention);
      Exception_Trace := Unhandled_Raise_In_Main;
   end if;
end System.Traceback.Symbolic;
