------------------------------------------------------------------------------
--                                                                          --
--                 GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                 --
--                                                                          --
--                SYSTEM.TASK_PRIMITIVES.OPERATIONS.SPECIFIC                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                      Copyright (C) 1998-2015, AdaCore                    --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This is a VxWorks version of this package for ravenscar-cert (VxWorks 6
--  Cert DKM/SKM, VxWorks 653 and VxWorks MILS vThreads). The implementation is
--  based on VxWorks taskVarLib.

separate (System.Task_Primitives.Operations)
package body Specific is

   ATCB_Key : aliased System.Address := System.Null_Address;
   --  Key used to find the Ada Task_Id associated with a thread

   ATCB_Key_Addr : System.Address := ATCB_Key'Address;
   pragma Export (Ada, ATCB_Key_Addr, "__gnat_ATCB_key_addr");
   --  Exported to support the temporary VxWorks 653 task registration
   --  implementation. This mechanism is used to minimize impact on other
   --  targets.

   Stack_Limit : aliased System.Address;

   pragma Import (C, Stack_Limit, "__gnat_stack_limit");

   type Set_Stack_Limit_Proc_Acc is access procedure;
   pragma Convention (C, Set_Stack_Limit_Proc_Acc);

   Set_Stack_Limit_Hook : Set_Stack_Limit_Proc_Acc;
   pragma Import (C, Set_Stack_Limit_Hook, "__gnat_set_stack_limit_hook");
   --  Procedure to be called when a task is created to set stack limit if
   --  limit checking is used.

   function Is_Valid_Task (T : Task_Id) return Boolean;

   -------------------
   -- Is_Valid_Task --
   -------------------

   function Is_Valid_Task (T : Task_Id) return Boolean is
   begin
      return taskVarGet (T.Common.LL.Thread, ATCB_Key'Access) /= ERROR;
   end Is_Valid_Task;

   ---------
   -- Set --
   ---------

   procedure Set (New_Task_Id : Task_Id) is

      --  Called from Create_Task because under VxWorks 653 all dynamic
      --  allocation must take place before putting the partition into
      --  NORMAL mode. Previous implementation had the new task call Set
      --  on itself from Enter_Task, causing the allocation of the task
      --  variables to occur too late.

      Result : STATUS;

   begin
      if not Is_Valid_Task (New_Task_Id) then
         Result := taskVarAdd (New_Task_Id.Common.LL.Thread, ATCB_Key'Access);
         pragma Assert (Result = OK);

         --  The first condition is only for VxWorks 653 1.x and 2.x, not 3.x
         pragma Warnings (Off);
         --  OS is a constant
         if OS /= VxWorks_653
           and then Result /= ERROR
           and then Set_Stack_Limit_Hook /= null
         then

            --  This will be initialized from taskInfoGet() once the task is
            --  is running.

            Result :=
              taskVarAdd (New_Task_Id.Common.LL.Thread, Stack_Limit'Access);
            pragma Assert (Result /= ERROR);
         end if;
         pragma Warnings (On);
      end if;

      Result :=
        taskVarSet
          (New_Task_Id.Common.LL.Thread,
           ATCB_Key'Access,
           To_Address (New_Task_Id));
      pragma Assert (Result /= ERROR);
   end Set;

   ----------
   -- Self --
   ----------

   function Self return Task_Id is
   begin
      return To_Task_Id (ATCB_Key);
   end Self;

end Specific;
