# Porting the run-time to the STM32F7-DISCOVERY

Now on to building a run-time library for the STM32F7. This is going to be a
bit more difficult for one reason: the STM32F7, being based on the Cortex-M7,
can benefit from Data and Instruction caches. These caches require explicit
initialization.

A nice addition compared to the base run-time is also to initialize the
tightly-coupled memory (TCM) present on this MCU, that can provide very fast
access to critical data. That's the DTCM and ITCM memory, specifically.

## Prepare the run-time

This step follows the same schema as the previous one.

The SVD file to use with svd2ada is $(SVD2ADA_DIR)/CMSIS-SVD/ST/STM32F7x.svd

The values specific to the STM32F74-Disco are:

 * `memory-map.ld`:
```
MEMORY
{
  itcm (x) : ORIGIN = 0x00000000, LENGTH = 16K
  flash (rx) : ORIGIN = 0x08000000, LENGTH = 1024K
  dtcm (rx) : ORIGIN = 0x20000000, LENGTH = 64K
  sram (rwx) : ORIGIN = 0x20010000, LENGTH = 240K
}
```

 * `s-bbbopa.ads`:
```
   Main_Clock_Frequency : constant := 200_000_000;
   HSE_Clock_Frequency : constant := 25_000_000;
```

 * `s-bbmcpa.ads`:
```
   Number_Of_Interrupts : constant := 99;

   function Is_PWR_Stabilized return Boolean
     is (Interfaces.STM32.PWR.PWR_Periph.CSR1.VOSRDY = 1);
```

 * `s-bbmcpa.adb`: change PWR_Periph.CR by PWR_Periph.CR1, and CSR by CSR1

## Enable Data and Instruction caches

Initialization of the cache is described in detail by ARM in the [Cortex-M7
processor technical reference manual](http://infocenter.arm.com/help/index.jsp?topic=/com.arm.doc.ddi0489d/BABGJGCH.html).

The necessary support has already been implemented in the GNAT run-time, so
we'll just need to follow how it is done.

### Identify the Cortex-M7

First, we need to modify the common startup code located in `arm/stm32/start-common.S`.

In there is a check in the common startup code for the MCU ID:

```asm
        /*************************
         * Check MCU = Cortex-M7 *
        *************************/

        .set    MCU_ID_REG, 0xE0042000
        ldr     r0, =MCU_ID_REG
        ldr     r1, [r0]
        ldr     r2, =#0xFFF
        and     r1, r1, r2  /* Retrieve the DEV_ID field of the MCU_ID reg */
        ldr     r2, =#0x449
        cmp     r1, r2      /* Compare to STM32F74xxx/75xxx Dev ID */
        beq      init_cache
        ldr     r2, =#0x451
        cmp     r1, r2      /* Compare to STM32F76xxx/77xxx Dev ID */
        bne     end         /* Do not initialize cache on STM32F4 */
```

The code above is rather explicit:

1. we read the MCU_ID register, as described in the technical reference manual
2. we mask the value with 0xFFF to retrieve the DEV_ID field (bits 0-11)
3. if equal to 0x449 or 0x451, then we have a STM32F74/75/76/77xxx MCU
4. in this case we jump to init_cache
5. else we finish the setup

### Cache initialization overview

The init_cache part takes verbatim the indications from ARM:

1. first we need to invalidate the cache, whatever the previous state:
2. only then can we enable the I and D caches
3. as a final step, well init the ITCM and DTCM tightly coupled memories.

### Registers used

```asm
        /* Register definition for cache handling */
        .set    CCSIDR,  0xE000ED80
        .set    CSSELR,  0xE000ED84
        .set    DCISW,   0xE000EF60
        .set    ICIALLU, 0xE000EF50
        .set    CCR,     0xE000ED14
```

### Data cache invalidation

Data cache invalidation is a bit complex. Basically there are two ways to
invalidate the data cache: via address or via set/way.

The invalidation by address approach is appropriate at run-time to ensure that some
portion of memory is properly flushed to the RAM at critical moments, but is
not appropriate here: using this method would mean selecting the full range of
addressable memory when the cache in itself is very small (and so is able to contain
only a small portion of this memory). That would result in highly inefficient
startup code.

The set/way approach, on the other hand, only takes care of the cache itself. This means
we invalidate only what's in the cache and just that.

So let's go with that approach:

```asm
        mov     r0, #0x0
        ldr     r11, =CSSELR
        str     r0, [r11]       /* Select the data cache size */
        dsb

        ldr     r11, =CCSIDR
        ldr     r2, [r11]       /* Cache size identification */
        and     r1, r2, #0x7    /* Number of words in a cache line */
        add     r7, r1, #0x4

        ubfx    r4, r2, #3, #10  /* r4 = number of ways - 1 of data cache */
        ubfx    r2, r2, #13, #15 /* r2 = number of sets - 1 of data cache */
        clz     r6, r4           /* Calculate bit offset for "way" in DCISW */

        ldr     r11, =DCISW

inv_loop1:                       /* For each set */
        mov     r1, r4
        lsls    r8, r2, r7

inv_loop2:                       /* For each way */
        lsls    r3, r1, r6
        orrs    r3, r3, r8

        str     r3, [r11]        /* Invalidate the D-Cache line */
        subs    r1, r1, #1
        bge     inv_loop2
        subs    r2, r2, #1
        bge     inv_loop1

        dsb
```

This algorithm essentially says:

* for each way
  * for each set
    * invalidate a cache line

Note that a data barrier is mandatory after the operation.

### Instruction cache invalidation

The instruction cache invalidation is much easier to perform, requiring only a single
write to the ICIALLU register:

```asm
        mov     r0, #0x0
        ldr     r11, =ICIALLU
        str     r0, [r11]

        dsb
        isb
```

### Finally, enable the caches:

Enabling the caches is also a simple write to a register:

```asm
        ldr     r11, =CCR
        ldr     r0, [r11]
        orr     r0, r0, #(0x30000) /* Sets the I and D cache enabled fields */
        str     r0, [r11]

        dsb
        isb
```

## Enable the Tightly Coupled Memories

The TCM initialization is also described in the Cortex-M7 Technical reference
manual:

```asm
tcm_init:
        .set    CM7_ITCMCR, 0xE000EF90
        .set    CM7_DTCMCR, 0xE000EF94

        ldr     r0, =CM7_ITCMCR
        ldr     r1, [r0]
        orr     r1, r1, #0x1 /* set the EN field */
        str     r1, [r0]

        ldr     r0, =CM7_DTCMCR
        ldr     r1, [r0]
        orr     r1, r1, #0x1 /* set the EN field */
        str     r1, [r0]

        dsb
        isb
```

And that's it! You can now install your new run-time using the usual commands.

[Next](5_CONCLUSION.md) - [Home](README.md)
