------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--              A U N I T . R E P O R T E R . G N A T T E S T               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                                                                          --
--                         Copyright (C) 2012, AdaCore                      --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT is maintained by AdaCore (http://www.adacore.com)                   --
--                                                                          --
------------------------------------------------------------------------------

with GNAT.IO;            use GNAT.IO;

--  Reporter intended to be used by test drivers generated by gnattest.

package body  AUnit.Reporter.GNATtest is

   procedure Dump_Result_List (L : Result_Lists.List);
   --  List failed assertions

   procedure Report_Test (Test : Test_Result);
   --  Report a single assertion failure or unexpected exception

   ------------------------
   --  Dump_Result_List  --
   ------------------------

   procedure Dump_Result_List (L : Result_Lists.List) is

      use Result_Lists;

      C : Cursor := First (L);

   begin

      --  Note: can't use Iterate because it violates restriction
      --  No_Implicit_Dynamic_Code

      while Has_Element (C) loop
         Report_Test (Element (C));
         Next (C);
      end loop;
   end Dump_Result_List;

   ------------
   -- Report --
   ------------

   procedure Report (Engine : GNATtest_Reporter;
                     R      : in out Result'Class)
   is
      pragma Unreferenced (Engine);

      Failures_Count : Integer;
      Crashes_Count  : Integer;
      Passed_Count   : Integer;
      Tests_Count    : Integer;
   begin

      Tests_Count := Integer (Test_Count (R));
      Failures_Count := Integer (Failure_Count (R));
      Crashes_Count := Integer (Error_Count (R));
      Passed_Count := Tests_Count - (Failures_Count + Crashes_Count);

      declare
         S : Result_Lists.List;
      begin

         Successes (R, S);
         Dump_Result_List (S);

      end;

      declare
         F : Result_Lists.List;
      begin
         Failures (R, F);
         Dump_Result_List (F);
      end;

      declare
         E : Result_Lists.List;
      begin
         Errors (R, E);
         Dump_Result_List (E);
      end;

      Put      (Tests_Count);
      Put      (" tests run: ");
      Put      (Passed_Count);
      Put      (" passed; ");
      Put      (Failures_Count);
      Put      (" failed; ");
      Put      (Crashes_Count);
      Put_Line (" crashed.");

   end Report;

   ------------------
   -- Report_Test --
   ------------------

   procedure Report_Test (Test : Test_Result) is
      Is_Assert : Boolean;
      Is_Condition : Boolean := False;

      N : Integer;
   begin

      Put      (Test.Test_Name.all);

      if Test.Failure /= null or else Test.Error /= null then
         if Test.Failure /= null then
            Is_Assert := True;
         else
            Is_Assert := False;
         end if;

         if Is_Assert then
            Put   (" corresponding test FAILED: ");
         else
            Put   (" corresponding test CRASHED: ");
         end if;

         if Is_Assert then

            if Test.Failure.Message'Length > 9 then
               N := Test.Failure.Message'First;
               if
                 Test.Failure.Message.all (N .. N + 8) = "req_sloc("
                 or else Test.Failure.Message.all (N .. N + 8) = "ens_sloc("
               then
                  for I in N + 9 .. Test.Failure.Message'Last - 2 loop
                     if Test.Failure.Message.all (I + 1 .. I + 2) = "):" then
                        Put (Test.Failure.Message.all
                               (I + 3 .. Test.Failure.Message'Last));

                        Put      (" (");
                        Put      (Test.Failure.Message.all (N + 9 .. I));
                        Put_Line (")");
                        Is_Condition := True;
                        exit;
                     end if;
                  end loop;
               end if;
            end if;

            if not Is_Condition then
               Put      (Test.Failure.Message.all);
               Put      (" (");
               Put      (Test.Failure.Source_Name.all);
               Put      (":");
               Put      (Test.Failure.Line);
               Put_Line (")");
            end if;

         else
            Put      (Test.Error.Exception_Name.all);

            if Test.Error.Exception_Message /= null then
               Put      (" : ");
               Put_Line (Test.Error.Exception_Message.all);
            end if;

         end if;
      else
         Put_Line (" corresponding test PASSED");
      end if;

   end Report_Test;
end AUnit.Reporter.GNATtest;
